/*
 * To change this template, choose Tools | Templates
 * and open the template in the editor.
 */

package exenne.components.swingworkers.progressworker;

import exenne.components.dialogs.DialogFactory;
import exenne.components.modalpanel.progresspanel.JModalProgressPanel;
import exenne.components.modalpanel.progresspanel.JProgressPanel;
import exenne.components.swingworkers.IWorker;
import java.awt.Frame;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.util.List;
import javax.swing.SwingWorker;

/**
 * A worker that executes a callable class and displays a progress
 * @author iulian
 */
public class JProgressSwingWorker<T> extends SwingWorker<T, String>
        implements IWorker, ProgressInfoManager, PropertyChangeListener {
    private final Frame rootFrame;
    protected JModalProgressPanel progressPanel;
    private boolean hasFinished;
    private int delay;

    //Valoare default pentru delay
    private static int DEFAULT_DELAY = 500;

    //Outside method to call
    protected AbstractProgressCallable <T> callableMethod;
    protected T result;

    //Option to display a standard error no mather what
    private String standardError = null;

    /**~~~~~~~~~~~~~~~~~~~~~~~~~~~~
     *     Constructor area
     *~~~~~~~~~~~~~~~~~~~~~~~~~~~~~*/

    /**
     * DELEGATE CONTRUCTION
     * @param rootFrame
     * @param callableMethod
     */
    public JProgressSwingWorker(Frame rootFrame, AbstractProgressCallable <T> callableMethod) {
        this(rootFrame, "", callableMethod);
    }

    /**
     * DELEGATE CONTRUCTION
     * @param rootFrame
     * @param callableMethod
     * @param mode      JProgressPanel.INDETERMINATE or JProgressPanel.DETERMINATE
     */
    public JProgressSwingWorker(Frame rootFrame, AbstractProgressCallable <T> callableMethod,
            int mode) {
        this(rootFrame, "", callableMethod, mode);
    }


    /**
     * DELEGATE CONTRUCTION
     * @param rootFrame
     * @param description
     * @param callableMethod
     */
    public JProgressSwingWorker(Frame rootFrame, String description,
            AbstractProgressCallable <T> callableMethod) {
        this(rootFrame, description, callableMethod, DEFAULT_DELAY, JProgressPanel.INDETERMINATE);
    }


    /**
     *
     * @param rootFrame
     * @param description
     * @param callableMethod
     * @param mode      JProgressPanel.INDETERMINATE or JProgressPanel.DETERMINATE
     */
    public JProgressSwingWorker(Frame rootFrame, String description,
            AbstractProgressCallable <T> callableMethod, int mode) {
        this(rootFrame, description, callableMethod, DEFAULT_DELAY, mode);
    }

    /**
     * MAIN CONSTRUCTOR
     * BUild a progress displayable worker
     * @param rootFrame     Root frame of the application
     * @param description   Description shown
     * @param callableMethod The method which will be called
     * @param delay         The delay waited before deciding to show the progress
     * @param mode      JProgressPanel.INDETERMINATE or JProgressPanel.DETERMINATE
     */
    public JProgressSwingWorker(Frame rootFrame, String description,
            AbstractProgressCallable <T> callableMethod, int delay, int mode) {
        this.rootFrame = rootFrame;

        //Show a progress panel
        progressPanel = new JModalProgressPanel(rootFrame);

        //Store the callable method
        this.callableMethod = callableMethod;
        callableMethod.setProgressInfoManager(this);
        this.addPropertyChangeListener(this);
        
        //Set the panel delay
        this.delay = delay;

        //Use a description
        progressPanel.setDisplayMode(description, mode);
    }

    /**
     * Set the panel displayed text
     * @param description
     */
    public void setDescription(String description) {
        progressPanel.setDisplayMode(description, JProgressPanel.INDETERMINATE);
    }

    /**
     * Set a standard error message. This will surpress errors from the processing
     *  and will display the message no mather what
     */
    public void setStandardErrorMessage(String errorMessage) {
        standardError = errorMessage;
    }


    /**~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
     *     Processing begining area
     *~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~*/

    /**
     * Make this worker process
     */
    public T executeTask() {
        //Initialize the state flag
        hasFinished = false;

        //Start the excution of background task
        this.execute();

        //BLOCKING Show a progress panel
        openProgressPanel();

        //Return the result
        return result;
    }

    /**
     * Open a progress panel
     */
    private void openProgressPanel() {
        //This must be run in the main thread
        try { Thread.sleep(delay);} catch (InterruptedException ex) {}
        if (!hasFinished) progressPanel.setVisible(true);
        progressPanel.dispose();
    }

    /**
     * The background executing task
     * @return
     * @throws Exception
     */
    @Override
    protected T doInBackground() throws Exception {
        return callableMethod.call();
    }

    /**~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
     *    Process monitoring area
     *~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~*/
    
    /**
     * Seteaza valoarea panoului. Metoda este apelata din Callable
     * @param value
     */
    public void setProgressValue(int value) {
        this.setProgress(value);
    }

     /**
      * Ascult modificarea valorii pentru progress din taskul de background
      * @param evt
      */
    public void propertyChange(PropertyChangeEvent evt) {
        if ("progress".equals(evt.getPropertyName())) {
            int progress = (Integer) evt.getNewValue();
            progressPanel.setProgress(progress);
        }
    }

    /**
     * Seteaza textul descriptiv al panoului
     * @param info
     */
    public void setProgressInfo(String info) {
        this.publish(info);
    }

    @Override
    protected void process(List<String> chunks) {
        String info = chunks.get(chunks.size() - 1);
        progressPanel.setText(info);
    }


    /**~~~~~~~~~~~~~~~~~~~~~~~~~~~~
     *     Finalizing area
     *~~~~~~~~~~~~~~~~~~~~~~~~~~~~~*/

    /**
     * Method invoked when background task finishes
     */
    @Override
    protected void done() {
        //This must be run in the main thread
        try {
            hasFinished = true;

            //Get the result
            result = get();
            
            //Close the progress
            closeProgressPanel();
        } catch (Exception ex) {
            hasFinished = true;

            //Return a null result
            result = null;

            ex.printStackTrace();
            closeProgressPanel();

            //Print an error
            String error;
            if (standardError != null)
                error = standardError;
            else error = ex.getMessage();
            DialogFactory.showErrorDialog(rootFrame, "Eroare",
                    error);
        }
    }

    /**
     * Close the progress panel
     */
    private void closeProgressPanel() {
        //This action will be tun in the main thread
        //sequencialy with the openProgressPanel method
        progressPanel.setVisible(false);
    }
}
