
package exenne.components.treetable;

import exenne.components.treetable.cellrenderers.AbstractTreetableCellRenderer;
import exenne.components.treetable.cellrenderers.DefaultTreetableCellRenderer;
import java.applet.Applet;
import java.awt.Component;
import java.awt.Font;
import java.awt.FontMetrics;
import java.awt.Graphics;
import java.awt.Insets;
import java.awt.KeyboardFocusManager;
import java.awt.Rectangle;
import java.awt.Window;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.ComponentAdapter;
import java.awt.event.ComponentEvent;
import java.awt.event.ComponentListener;
import java.awt.event.MouseEvent;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.EventObject;
import javax.swing.JScrollBar;
import javax.swing.JScrollPane;
import javax.swing.JTable;
import javax.swing.JViewport;
import javax.swing.ListSelectionModel;
import javax.swing.SwingUtilities;
import javax.swing.Timer;
import javax.swing.UIManager;
import javax.swing.event.TableModelEvent;
import javax.swing.table.DefaultTableCellRenderer;
import javax.swing.table.TableCellEditor;
import javax.swing.table.TableCellRenderer;
import javax.swing.table.TableModel;
import javax.swing.tree.AbstractLayoutCache;
import javax.swing.tree.DefaultMutableTreeNode;
import javax.swing.tree.DefaultTreeModel;
import javax.swing.tree.TreeNode;
import javax.swing.tree.TreePath;

/** An Outline, or tree-table component.  Takes an instance of OutlineModel,
 * an interface which merges TreeModel and TableModel.
 * <p>
 * Simplest usage:
 * <ol>
 * <li>Create a standard tree model for the tree node portion of the outline.</li>
 * <li>Implement RowModel.  RowModel is a subset of TableModel - it is passed
 * the value in column 0 of the Outline and a column index, and returns the
 * value in the column in question.</li>
 * <li>Pass the TreeModel and the RowModel to <code>DefaultOutlineModel.createModel()</code>
 * </ol>
 * This will generate an instance of DefaultOutlineModel which will use the
 * TreeModel for the rows/tree column content, and use the RowModel to provide
 * the additional table columns.
 * <p>
 * It is also useful to provide an implementation of <code>RenderDataProvider</code>
 * to supply icons and affect text display of cells - this covers most of the
 * needs for which it is necessary to write a custom cell renderer in JTable/JTree.
 * <p>
 * <b>Example usage:</b><br>
 * Assume FileTreeModel is a model which, given a root directory, will
 * expose the files and foldrs underneath it.  We will implement a
 * RowModel to expose the file size and date, and a RenderDataProvider which
 * will use a gray color for uneditable files and expose the full file path as
 * a tooltip.  Assume the class this is implemented in is a
 * JPanel subclass or other Swing container.
 * <br>
 * clean up formatting & edit for style
 * <pre>
 * public void initComponents() {
 *   setLayout (new BorderLayout());
 *   TreeModel treeMdl = new FileTreeModel (someDirectory);
 *
 *   OutlineModel mdl = DefaultOutlineModel.createOutlineModel(treeMdl,
 *       new FileRowModel(), true);
 *   outline = new Outline();
 *   outline.setRenderDataProvider(new FileDataProvider());
 *   outline.setRootVisible (true);
 *   outline.setModel (mdl);
 *   add (outline, BorderLayout.CENTER);
 * }
 *  private class FileRowModel implements RowModel {
 *     public Class getColumnClass(int column) {
 *          switch (column) {
 *              case 0 : return Date.class;
 *              case 1 : return Long.class;
 *              default : assert false;
 *          }
 *          return null;
 *      }
 *
 *      public int getColumnCount() {
 *          return 2;
 *      }
 *
 *      public String getColumnName(int column) {
 *          return column == 0 ? "Date" : "Size";
 *      }
 *
 *      public Object getValueFor(Object node, int column) {
 *          File f = (File) node;
 *          switch (column) {
 *              case 0 : return new Date (f.lastModified());
 *              case 1 : return new Long (f.length());
 *              default : assert false;
 *          }
 *          return null;
 *      }
 *
 *      public boolean isCellEditable(Object node, int column) {
 *          return false;
 *      }
 *
 *      public void setValueFor(Object node, int column, Object value) {
 *          //do nothing, nothing is editable
 *      }
 *  }
 *
 *  private class FileDataProvider implements RenderDataProvider {
 *      public java.awt.Color getBackground(Object o) {
 *          return null;
 *      }
 *
 *      public String getDisplayName(Object o) {
 *          return ((File) o).getName();
 *      }
 *
 *      public java.awt.Color getForeground(Object o) {
 *          File f = (File) o;
 *          if (!f.isDirectory() && !f.canWrite()) {
 *              return UIManager.getColor ("controlShadow");
 *          }
 *          return null;
 *      }
 *
 *      public javax.swing.Icon getIcon(Object o) {
 *          return null;
 *      }
 *
 *      public String getTooltipText(Object o) {
 *          return ((File) o).getAbsolutePath();
 *      }
 *
 *      public boolean isHtmlDisplayName(Object o) {
 *          return false;
 *      }
 *   }
 * </pre>
 *
 * @author  Tim Boudreau
 */
public final class Treetable extends JTable {
    private boolean initialized = false;
    private Boolean cachedRootVisible = null;
    private boolean needCalcRowHeight = true;
    private RenderDataProvider renderDataProvider = null;
    private ComponentListener componentListener = null;
    private AbstractTreetableCellRenderer cellRenderer;
    private PropertyChangeListener editorRemover = null;

    /**
     * Creeaza o instanta de outline si seteaza si Renderul
     * @param renderer
     */
    public Treetable(AbstractTreetableCellRenderer renderer) {
        this.cellRenderer = renderer;
        init(renderer);
    }

    /**
     * Constructorul null
     */
    public Treetable() {
        this(new DefaultTreetableCellRenderer());
    }

    /**
     * Initializeaza Renderul
     * @param renderer Renderul ce se va initializa
     */
    private void init(DefaultTableCellRenderer renderer) {
        initialized = true;

        setDefaultRenderer(Object.class,
                renderer);
        setDefaultRenderer(BigDecimal.class,
                renderer);
        setDefaultRenderer(Integer.class,
                renderer);
    }

    //========================================
    //               Setters
    //========================================

    /** 
     * Transmite layout-ului aceiasi dinemsiune ca a randului
     */
    @Override
    public void setRowHeight(int val) {
        super .setRowHeight(val);
        if (getLayoutCache() != null) {
            getLayoutCache().setRowHeight(val);
        }
    }

    /** 
     * Metoda mostenita de la Tree. Seteaza daca root-ul sa fie vizibil
     */
    public void setRootVisible(boolean val) {
        if (getOutlineModel() == null) {
            cachedRootVisible = val ? Boolean.TRUE : Boolean.FALSE;
        }
        if (val != isRootVisible()) {
            //need to force a property change on the model,
            //the layout cache doesn't have direct listener support
            getLayoutCache().setRootVisible(val);
            firePropertyChange("rootVisible", !val, val); //NOI18N
        }
    }

    /**
     * Seteaza renderDataProvider-ul pentru TreeTable
     * RenderDataProvider va oferi datele pentru afisarea nodurilor
     * @param provider Providerul pentru valorile nodurilor
     */
    public void setRenderDataProvider(RenderDataProvider provider) {
        if (provider != renderDataProvider) {
            RenderDataProvider old = renderDataProvider;
            renderDataProvider = provider;
            firePropertyChange("renderDataProvider", old, provider); //NOI18N
        }
    }

    /**
     * Suprascris pentru a arunca o exceptie in cazul in care
     * modelul nu este o instanta de table model
     * @param mdl Modelul din spatele outline-ului
     */
    @Override
    public void setModel(TableModel mdl) {
        if (initialized && (!(mdl instanceof  TreetableModel))) {
            throw new IllegalArgumentException(
                    "Table model for an Outline must be an instance of "
                            + "OutlineModel"); //NOI18N
        }
        if (mdl instanceof  TreetableModel) {
            AbstractLayoutCache layout = ((TreetableModel) mdl)
                    .getLayout();
            if (cachedRootVisible != null) {

                layout.setRootVisible(cachedRootVisible.booleanValue());

            }

            layout.setRowHeight(getRowHeight());

            if (((TreetableModel) mdl).isLargeModel()) {
                addComponentListener(getComponentListener());
                layout.setNodeDimensions(new ND());
            } else {
                if (componentListener != null) {
                    removeComponentListener(componentListener);
                    componentListener = null;
                }
            }
        }

        super .setModel(mdl);
    }

    //========================================
    //          boolean Getters
    //========================================

    /**
     * Verific daca coloana este index. Coloanele nu se muta
     * Index este 0
     * @param column Coloana
     * @return true - este prima coloana
     */
    public boolean isTreeColumnIndex(int column) {
        return column == 0;
    }

    /**
     * Wrapper peste TreePathSupport
     *    Verific daca o cale este vizibila
     * @param path
     * @return true daca e vizibila
     */
    public boolean isVisible(TreePath path) {
        if (getTreePathSupport() != null) {
            return getTreePathSupport().isVisible(path);
        }
        return false;
    }

    /**
     * Verific daca radacina este vizibila
     * @return true radacina este vizibila
     */
    public boolean isRootVisible() {
        if (getLayoutCache() == null) {
            return cachedRootVisible != null ? cachedRootVisible
                    .booleanValue() : true;
        } else {
            return getLayoutCache().isRootVisible();
        }
    }

    //========================================
    //                Getters
    //========================================

    /**
     * Obtin cell renderul pentru o anumita celula.
     * Ar trebui sa fie doar un cell renderer
     * @param row    Randul
     * @param column Coloana
     * @return CellRenderul celulei respective
     */
    @Override
    public TableCellRenderer getCellRenderer(int row, int column) {
        TableCellRenderer result;
        if (column == 0) {
            result = getDefaultRenderer(Object.class);
        } else {
            result = super .getCellRenderer(row, column);
        }
        return result;
    }

    /**
     * Obtin RenderProvider-ul din spatele primei coloane
     * RenderProvider-ul ofera informatiile despre nodurile utilizator
     * @return RenderProvider
     */
    public RenderDataProvider getRenderDataProvider() {
        return renderDataProvider;
    }

    /**
     * Obtin TreePath Support ce face managementul extinderii, restrangerii
     * arborelui
     * @return TreePathSupport
     */
    public TreePathSupport getTreePathSupport() {
        TreetableModel mdl = getOutlineModel();
        if (mdl != null) {
            return mdl.getTreePathSupport();
        } else {
            return null;
        }
    }

    /** Get the layout cache which manages layout data for the Outline.
     * <strong>Under no circumstances directly call the methods on the
     * layout cache which change the expanded state - such changes will not
     * be propagated into the table model, and will leave the model and
     * its layout in inconsistent states.  Any calls that affect expanded
     * state must go through <code>getTreePathSupport()</code>.</strong> */
    public AbstractLayoutCache getLayoutCache() {
        TreetableModel mdl = getOutlineModel();
        if (mdl != null) {
            return mdl.getLayout();
        } else {
            return null;
        }
    }

    /** Convenience getter for the <code>TableModel</code> as an instance of
     * OutlineModel.  If no OutlineModel has been set, returns null. */
    public TreetableModel getOutlineModel() {
        TableModel mdl = getModel();
        if (mdl instanceof  TreetableModel) {
            return (TreetableModel) getModel();
        } else {
            return null;
        }
    }

    /** Create a component listener to handle size changes if the table model
     * is large-model */
    private ComponentListener getComponentListener() {
        if (componentListener == null) {
            componentListener = new SizeManager();
        }
        return componentListener;
    }

    /**
     * Obtine ScrollPane-ul (daca acesta este in vreunul)
     * @return ScrollPane-ul
     */
    private JScrollPane getScrollPane() {
        JScrollPane result = null;
        if (getParent() instanceof  JViewport) {
            if (((JViewport) getParent()).getParent() instanceof  JScrollPane) {
                result = (JScrollPane) ((JViewport) getParent())
                        .getParent();
            }
        }
        return result;
    }

    //========================================
    //    Actiuni directe pe outline
    //========================================

    //BEGIN Utils
    // Sunt apelate din afara clasei

    /** Calculate the height of rows based on the current font.  This is
     *  done when the first paint occurs, to ensure that a valid Graphics
     *  object is available.  */
    private void calcRowHeight(Graphics g) {
        //Users of themes can set an explicit row height, so check for it
        Integer i = (Integer) UIManager
                .get("netbeans.outline.rowHeight"); //NOI18N

        int localRowHeight;
        if (i != null) {
            localRowHeight = i.intValue();
        } else {
            //Derive a row height to accomodate the font and expando icon
            Font f = getFont();
            FontMetrics fm = g.getFontMetrics(f);
            localRowHeight = Math.max(fm.getHeight() + 3,
                    cellRenderer
                            .getExpansionHandleHeight());
        }
        //Clear the flag
        needCalcRowHeight = false;
        //Set row height.  If displayable, this will generate a new call
        //to paint()
        setRowHeight(localRowHeight);
    }

    /**
     * Forteaza reincarcarea componentelor
     */
    private void change() {
        revalidate();
        repaint();
    }

    /**
     * Transforma un vector de obiecte
     *   Integer intr-un tip primitiv
     * @param irri
     * @return vector primitiv
     */
    private int[] toPrimitive(Integer[] irri) {
        int []array = new int[irri.length];
        for (int i=0;i<array.length;i++) {
            array[i] = irri[i].intValue();
        }
        return array;
    }

    /**
     * Obtin nodul selectat curent
     * @return Primul nod selectat
     */
    public DefaultMutableTreeNode getSelectedNode() {
        return ((DefaultMutableTreeNode) this.getValueAt(this
                .getSelectedRow(), 0));
    }

    /**
     * Obtin nodul selectat curent
     * @return Primul nod selectat
     */
    public DefaultMutableTreeNode getNodeAt(int row) {
        return ((DefaultMutableTreeNode) this.getValueAt(row, 0));
    }

    /**
     * Obtin nodul selectat curent
     * @return Primul nod selectat
     */
    public int getRowForNode(DefaultMutableTreeNode node) {
        DefaultTreeModel localTreeModel = (DefaultTreeModel) getOutlineModel().getTreeModel();
        TreePath treePath = new TreePath(localTreeModel.getPathToRoot(node));
        return this.getLayoutCache().getRowForPath(treePath);
    }

    /**
     * Selecteaza nodul in treetable
     * @param node
     */
    public void selectNode(DefaultMutableTreeNode node) {
        int row = getRowForNode(node);
        if (row != -1)
            this.addRowSelectionInterval(row, row);
    }

    /**
     * Selecteaza nodul in treetable
     * @param node
     */
    public void selectAndScrollToNode(DefaultMutableTreeNode node) {
        int row = getRowForNode(node);
        if (row != -1) {
            this.addRowSelectionInterval(row, row);
            Rectangle rectangle = this.getCellRect(row, 0, true);
            this.scrollRectToVisible(rectangle);
        }
    }

    /**
     * Obtin un vector cu indicii randurilor selectate
     * @return vector cu randurile selectate
     */
    public int[] getSelectedIndices() {
        ListSelectionModel lsm = this.getSelectionModel();
        int min = lsm.getMinSelectionIndex();
        int max = lsm.getMaxSelectionIndex();

        ArrayList al = new ArrayList();
        for (int i = min; i <= max; i++) {
            if (lsm.isSelectedIndex(i)) {
                al.add(new Integer(i));
            }
        }
        Integer[] ints = (Integer[]) al.toArray(new Integer[0]);
        return (int[]) this.toPrimitive(ints);
    }

    //BEGIN PATH ACTIONS

    /** Expand a tree path */
    public void expandPath(TreePath path) {
       Object lastPathComponent = path.getLastPathComponent();

       //We should never expand a node with no children
       if (lastPathComponent instanceof TreeNode) {
           TreeNode treeNode = (TreeNode) lastPathComponent;
           if (treeNode.getChildCount() == 0)
               return;
       }

       //Expand it only if it is not expanded
       if (!getTreePathSupport().isExpanded(path))
            getTreePathSupport().expandPath(path);
    }

    /** Collapse a tree path */
    public void collapsePath(TreePath path) {
        if (getTreePathSupport().isExpanded(path))
            getTreePathSupport().collapsePath(path);
    }

    /**
     * Obtine granitele pentru calea respectiva
     * @param path TreePath
     * @return Dreptunghiul ce reprezinta granitele
     */
    public Rectangle getPathBounds(TreePath path) {
        Insets i = getInsets();
        Rectangle bounds = getLayoutCache().getBounds(path, null);

        if (bounds != null && i != null) {
            bounds.x += i.left;
            bounds.y += i.top;
        }
        return bounds;
    }

    /**
     * Obtine calea pentru o pozitie pe ecran
     */
    public TreePath getClosestPathForLocation(int x, int y) {
        Insets i = getInsets();
        if (i != null) {
            return getLayoutCache().getPathClosestTo(x - i.left,
                    y - i.top);
        } else {
            return getLayoutCache().getPathClosestTo(x, y);
        }
    }

    /**
     * Expand a certain node
     * @param node
     */
    public void expandNode(DefaultMutableTreeNode node) {
        DefaultTreeModel localTreeModel = (DefaultTreeModel) getOutlineModel().getTreeModel();
        TreePath treePath = new TreePath(localTreeModel.getPathToRoot(node));
        expandPath(treePath);
    }

    /**
     * Expand a certain node
     * @param node
     */
    public void collapseNode(DefaultMutableTreeNode node) {
        DefaultTreeModel localTreeModel = (DefaultTreeModel) getOutlineModel().getTreeModel();

        //The nodes needs to have childs
        if (node.getChildCount() > 0) {
            TreePath treePath = new TreePath(localTreeModel.getPathToRoot(node));
            collapsePath(treePath);
        }
    }

    /**
     * Check if a certain node is expanded
     * @param node
     * @return
     */
    public boolean isNodeExpanded(DefaultMutableTreeNode node) {
        DefaultTreeModel localTreeModel = (DefaultTreeModel) getOutlineModel().getTreeModel();
        TreePath treePath = new TreePath(localTreeModel.getPathToRoot(node));

        return getTreePathSupport().isExpanded(treePath);
    }

    //END PATH ACTIONS
    //BEGIN OVerrides

    /**
     * Aici se intercepteaza clickurile pe elementele tabelului
     */
    @Override
    public boolean editCellAt(int row, int column, EventObject e) {
        //If it was on column 0, it may be a request to expand a tree
        //node - check for that first.
        if (isTreeColumnIndex(column) && e instanceof  MouseEvent) {
            MouseEvent me = (MouseEvent) e;
            TreePath path = getLayoutCache().getPathClosestTo(
                    me.getX(), me.getY());
            
            //Fixes #43
            if (path == null ) return false;

            if (!getOutlineModel().isLeaf(path.getLastPathComponent())) {
                /**
                 * BEGIN Aici se trateaza clickul pe butonul de expand
                 * Modific astfel incat sa translatez evenimentele spre stanga
                 * cu un nivel pentru a putea ascunde root-ul
                 */
                int handleWidth = cellRenderer
                        .getExpansionHandleWidth();
                Insets ins = getInsets();
                int handleStart;
                // ~~~~~ Caz special pentru root ~~~~~ /
                if (path.getPathCount() == 1) handleStart = ins.left;
                else handleStart = ins.left
                        + ((path.getPathCount() - 2) * cellRenderer
                                .getNestingWidth());
                int handleEnd = ins.left + handleStart + handleWidth;

                //Translate x/y to position of column if non-0

                if ((me.getX() > ins.left && me.getX() >= handleStart && me
                        .getX() <= handleEnd)
                        || me.getClickCount() > 1) {

                    boolean expanded = getLayoutCache()
                            .isExpanded(path);
                    if (!expanded) {
                        getTreePathSupport().expandPath(path);
                    } else {
                        getTreePathSupport().collapsePath(path);
                    }
                    return false;
                }
            }
        }

        //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
        //Cod preluat din JTable pentru a permite CellEditorului
        //sa se extinda in voie (daca implementeaza SelfManagedComponent
        //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
        if(cellEditor!=null && !cellEditor.stopCellEditing()){
            return false;
        }

        if(row<0 || row>=getRowCount() ||
                column<0 || column>=getColumnCount()){
            return false;
        }

        if(!isCellEditable(row, column))
            return false;

        if(editorRemover==null){
            KeyboardFocusManager fm =
                    KeyboardFocusManager.getCurrentKeyboardFocusManager();
            editorRemover = new CellEditorRemover(fm);
            fm.addPropertyChangeListener("permanentFocusOwner", editorRemover);
        }

        TableCellEditor editor = getCellEditor(row, column);
        if(editor!=null && editor.isCellEditable(e)){
            editorComp = prepareEditor(editor, row, column);
            if(editorComp==null){
                removeEditor();
                return false;
            }

            Rectangle cellRect = getCellRect(row, column, false);
            if(editorComp instanceof SelfManagedComponent){
                ((SelfManagedComponent) editorComp).updateBounds(cellRect);
            }else
                editorComp.setBounds(cellRect);



            add(editorComp);
            editorComp.validate();
            editorComp.repaint(); //repaint the component

            setCellEditor(editor);
            setEditingRow(row);
            setEditingColumn(column);
            editor.addCellEditorListener(this);

            return true;
        }
        return false;
    }
    
    /**
     * Retrimite evenimentul catre parinte
     * @param e
     */
    @Override
    public void tableChanged(TableModelEvent e) {
        super.tableChanged(e);
    }

    /**
     * Fortez redesenarea unui anumit rand
     * @param tableRow
     */
    public void repaintRow(int tableRow) {
        TableModelEvent e = new TableModelEvent(this.getOutlineModel(), tableRow);
        super.tableChanged(e);
    }

    /**
     * Daca se modifica dimensiunea randului trebuie recalculat
     * @param g
     */
    @Override
    public void paint(Graphics g) {
        if (needCalcRowHeight) {
            calcRowHeight(g);
            //CalcRowHeight will trigger a repaint
            return;
        }
        super .paint(g);
    }

    /**
     * Face ca outline-ul sa umple tot timpul viewportul
     * @return
     */
    @Override
    public boolean getScrollableTracksViewportHeight(){
        if(getParent() instanceof JViewport)
            return getParent().getHeight()>getPreferredSize().height;
        else
            return false;
    }

    /**
     * Se forteaza redesenarea celulelor dupa ce s-a terminat editarea
     * @param g
     */
    @Override
    protected void paintComponent(Graphics g){
        super.paintComponent(g);
        if(isEditing()){
            Component component = getEditorComponent();
            component.repaint();
        }
    }

    //END OVerrides

    //========================================
    //               Editor remover
    //========================================

    @Override
    public void removeEditor(){
        KeyboardFocusManager.getCurrentKeyboardFocusManager().
                removePropertyChangeListener("permanentFocusOwner", editorRemover);

        editorRemover = null;
        TableCellEditor editor = getCellEditor();
        if(editor!=null){
            editor.removeCellEditorListener(this);
            Rectangle cellRect = getCellRect(editingRow, editingColumn, false);
            if(editorComp!=null){
                cellRect = cellRect.union(editorComp.getBounds());
                remove(editorComp);
            }

            setCellEditor(null);
            setEditingColumn(-1);
            setEditingRow(-1);
            editorComp = null;

            repaint(cellRect);
        }
    }

    @Override
    public void removeNotify(){
        KeyboardFocusManager.getCurrentKeyboardFocusManager().
                removePropertyChangeListener("permanentFocusOwner", editorRemover);
        editorRemover = null;
        super.removeNotify();
    }

    //========================================
    //               Clase interne
    //========================================

    /**
     * Provides node dimensions
     */
    private class ND extends AbstractLayoutCache.NodeDimensions {

        public Rectangle getNodeDimensions(Object value, int row,
                int depth, boolean expanded, Rectangle bounds) {

            int wid = Treetable.this .getColumnModel().getColumn(0)
                    .getPreferredWidth();
            bounds.setBounds(0, row * getRowHeight(), wid,
                    getRowHeight());
            return bounds;
        }

    }

    /** A component listener.  If we're a large model table, we need
     * to inform the FixedHeightLayoutCache when the size changes, so it
     * can update its mapping of visible nodes */
    private class SizeManager extends ComponentAdapter implements
            ActionListener {
        protected Timer timer = null;
        protected JScrollBar scrollBar = null;

        @Override
        public void componentMoved(ComponentEvent e) {
            if (timer == null) {
                JScrollPane scrollPane = getScrollPane();

                if (scrollPane == null) {
                    change();
                } else {
                    scrollBar = scrollPane.getVerticalScrollBar();
                    if (scrollBar == null
                            || !scrollBar.getValueIsAdjusting()) {
                        // Try the horizontal scrollbar.
                        if ((scrollBar = scrollPane
                                .getHorizontalScrollBar()) != null
                                && scrollBar.getValueIsAdjusting()) {

                            startTimer();
                        } else {
                            change();
                        }
                    } else {
                        startTimer();
                    }
                }
            }
        }

        protected void startTimer() {
            if (timer == null) {
                timer = new Timer(200, this );
                timer.setRepeats(true);
            }
            timer.start();
        }

        public void actionPerformed(ActionEvent ae) {
            if (scrollBar == null || !scrollBar.getValueIsAdjusting()) {
                if (timer != null)
                    timer.stop();
                change();
                timer = null;
                scrollBar = null;
            }
        }

        @Override
        public void componentHidden(ComponentEvent e) {
        }

        @Override
        public void componentResized(ComponentEvent e) {
        }

        @Override
        public void componentShown(ComponentEvent e) {
        }
    }

    /**
     * Clasa interna din JTable se ocupa cu afisarea si prezentarea editorului
     */
    private class CellEditorRemover implements PropertyChangeListener {
        KeyboardFocusManager focusManager;

        public CellEditorRemover(KeyboardFocusManager fm) {
            this.focusManager = fm;
        }

        public void propertyChange(PropertyChangeEvent ev) {
            if (!isEditing() || getClientProperty("terminateEditOnFocusLost") != Boolean.TRUE) {
                return;
            }

            Component c = focusManager.getPermanentFocusOwner();
            while (c != null) {
                if (c == Treetable.this) {
                    // focus remains inside the table
                    return;
                } else if ((c instanceof Window) ||
                           (c instanceof Applet && c.getParent() == null)) {
                    if (c == SwingUtilities.getRoot(Treetable.this)) {
                        if (!getCellEditor().stopCellEditing()) {
                            getCellEditor().cancelCellEditing();
                        }
                    }
                    break;
                }
                c = c.getParent();
            }
        }
    };

}
