/*
 * O versiune a renderului care obtine toate valorile din model
 * Se extinde acest renderer pentru a beneficia de acestea
 * Created on January 28, 2004, 7:49 PM
 */

package exenne.components.treetable.cellrenderers;

import exenne.components.treetable.JHandleIcons;
import exenne.components.treetable.RenderDataProvider;
import exenne.components.treetable.RowModel;
import exenne.components.treetable.Treetable;
import exenne.components.treetable.TreetableModel;
import exenne.components.utils.GraphicRenderingUtils;
import exenne.components.utils.NumberUtils;
import java.awt.Color;
import java.awt.Component;
import java.math.BigDecimal;
import java.util.Hashtable;
import javax.swing.BorderFactory;
import javax.swing.Icon;
import javax.swing.JTable;
import javax.swing.SwingConstants;
import javax.swing.UIManager;
import javax.swing.border.Border;
import javax.swing.table.DefaultTableCellRenderer;
import javax.swing.tree.AbstractLayoutCache;
import javax.swing.tree.DefaultMutableTreeNode;
import javax.swing.tree.TreePath;

/** An outline-aware TableCellRenderer which knows how to paint expansion
 * handles and indent child nodes an appropriate amount.
 *
 * @author  Tim Boudreau
 */
public abstract class AbstractTreetableCellRenderer extends
        DefaultTableCellRenderer {
    //Surse de date
    private Treetable treetable = null;
    private TreetableModel treetableModel = null;
    private AbstractLayoutCache layoutCache = null;
    private RenderDataProvider renderDataProvider = null;
    private RowModel rowModel = null;

    //Variabile booleane puse la dispozitie de catre clasa
    private boolean isSelected = false;
    private boolean hasSomeFocus = false;
    private boolean isCellEditable = false;
    private boolean isLeaf = true;
    private boolean isFirstColumn = false;
    private boolean isLastColumn = false;
    private boolean isExpanded = false;
    private boolean isCellImportant = false;
    private boolean isLeftCellImportant = false;
    private boolean isRightCellImportant = false;    
    private boolean isShowHandle = false;
    private boolean isGrouping = false;
    private boolean hasToolTipText = false;
    
    //Variabile de stare
    private int nestingDepth = 0; //Adancimea unui rand
    private Color prefferedBackground = null;
    private Color prefferedForeground = null;

    //Hashtable care specifica pentru ce nivele parinte nodul este utilmul
    private Hashtable<Integer,Boolean> hashtableIsLast;

    //Hashtable care specifica culoare de desenare pentru fiecare nivel parinte
    private Hashtable<Integer,Color> hashtableParentColor;

    //providere pentru desenare
    private Border expansionBorder;
    private JHandleIcons iconProvider;

    /**~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
     *      Construierea renderului
     *~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~*/

    /** Creates a new instance of DefaultOutlineTreeCellRenderer */
    public AbstractTreetableCellRenderer() {
        expansionBorder = new ExpansionHandleBorder();
        iconProvider = JHandleIcons.getInstance();

        //Initializez hastable-urile
        hashtableParentColor = new Hashtable<Integer, Color>();
        hashtableIsLast = new Hashtable<Integer, Boolean>();
    }

    /** Overridden to combine the expansion border (whose insets determine how
     * much a child tree node is shifted to the right relative to the ancestor
     * root node) with whatever border is set, as a CompoundBorder.  The expansion
     * border is also responsible for drawing the expansion icon.  */
    @Override
    public void setBorder(Border b) {
        if (b == expansionBorder) {
            super .setBorder(b);
        } else {
            super .setBorder(BorderFactory.createCompoundBorder(expansionBorder,
                    b));
        }
    }

    /**~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
     *      Acces la sursele de date
     *~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~*/

    /**
     * Obtin Layout Cache-ul
     * @return
     */
    public AbstractLayoutCache getLayoutCache() {
        return layoutCache;
    }

    /**
     * Obtin Data Providerul
     * @return
     */
    public RenderDataProvider getRenderDataProvider() {
        return renderDataProvider;
    }

    /**
     * Obtin Treetable-ul
     * @return
     */
    public Treetable getTreetable() {
        return treetable;
    }

    /**
     * Obtin modelul Treetable-ului
     * @return
     */
    public TreetableModel getTreetableModel() {
        return treetableModel;
    }

    /**
     * Obtin modelul de rand pentru acesta
     * @return
     */
    public RowModel getRowModel() {
        return rowModel;
    }

    /**~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
     *      Obtin icoanele pentru arbore
     *~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~*/

    /**
     * Iconul standard pentru deschis dat de Look and Feel
     * @return
     */
    public Icon getDefaultOpenIcon() {
        return UIManager.getIcon("Tree.openIcon"); //NOI18N
    }

    /**
     * Iconul standard pentru inchis dat de Look and Feel
     * @return
     */
    public Icon getDefaultClosedIcon() {
        return UIManager.getIcon("Tree.closedIcon"); //NOI18N
    }

    /**
     * Iconul standard pentru frunza dat de Look and Feel
     * @return
     */
    public Icon getDefaultLeafIcon() {
        return UIManager.getIcon("Tree.leafIcon"); //NOI18N
    }

    /**
     * Iconul propriu pentru inchis
     * @return
     */
    public Icon getExpandedIcon() {
        return iconProvider.getIcon(JHandleIcons.MINUS);
    }

    /**
     * Iconul propriu pentru deschis
     * @return
     */
    public Icon getCollapsedIcon() {
        return iconProvider.getIcon(JHandleIcons.PLUS);
    }

    /**~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
     *     Obtinerea variabilelor puse la
     *      dispozitie pentru desenare
     *~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~*/

    /**
     * Obtin daca randul este selectat
     * @return
     */
    public boolean isSelected() {
        return isSelected;
    }     

    /**
     * Obtin daca celula are focus in acest moment
     * @return
     */
    public boolean hasSomeFocus() {
        return hasSomeFocus;
    }

    /**
     * Obtin daca celula este editabila
     * @return
     */
    public boolean isCellEditable() {
        return isCellEditable;
    }

    /**
     * Specifica daca obiectul este frunza
     * @return
     */
    public boolean isLeaf() {
        return isLeaf;
    }

    /**
     * true daca aceasta celula reprezinta prima coloana
     * @return
     */
    public boolean isFirstColumn() {
        return isFirstColumn;
    }

    /**
     * true daca aceasta celula reprezinta ultima coloana
     * @return
     */
    public boolean isLastColumn() {
        return isLastColumn;
    }

    /**
     * Specifica daca nodul este restrans
     * @return
     */
    public boolean isExpanded() {
        return isExpanded;
    }    
    
    /**
     * Obtin daca celula respectiva trebuie desenata special
     * @return
     */
    public boolean isCellImportant() {
        return isCellImportant;
    }

    /**
     * Obtin daca celula vecina respectiva trebuie desenata special
     * @return
     */
    public boolean isLeftCellImportant() {
        return isLeftCellImportant;
    }

    /**
     * Obtin daca celula vecina respectiva trebuie desenata special
     * @return
     */
    public boolean isRightCellImportant() {
        return isRightCellImportant;
    }
    
    /**
     * Specifica daca handle-ul de expansiune este vizibil
     * @return
     */
    public boolean isShowHandle() {
        return isShowHandle;
    }

    /**
     * Obtin daca randul reprezinta o grupare
     * @return
     */
    public boolean isGrouping() {
        return isGrouping;
    }

    /**
     * Obtin daca cell rederul are tooltip text
     * @return
     */
    public boolean hasToolTipText() {
        return hasToolTipText;
    }

    /**
     * Seteaza identarea frunzelor
     * @return Identarea fata de urmatorul nivel
     */
    public int getNestingWidth() {
        return getExpansionHandleWidth();
    }

    /**
     * Dimensiune iconului de expansiune
     * @return
     */
    public int getExpansionHandleWidth() {
        return getExpandedIcon().getIconWidth();
    }

    /**
     * Inaltimea iconului de expansiune
     * @return
     */
    public int getExpansionHandleHeight() {
        return getExpandedIcon().getIconHeight();
    }

    /** Set the nesting depth - the number of path elements below the root.
     * This is set in getTableCellEditorComponent(), and retrieved by the
     * expansion border to determine how far to the right to indent the current
     * node. */
    public int getNestingDepth() {
        return nestingDepth;
    }

    /**
     * Obtin backgroundul preferat de model
     * @return
     */
    public Color getPrefferedBackground() {
        return prefferedBackground;
    }

    /**
     * Obtin foregroundul preferat de model
     * @return
     */
    public Color getPrefferedForeground() {
        return prefferedForeground;
    }

    /**
     * Obtin hashtable-ul cu culorile parintilor
     * @return
     */
    public Hashtable<Integer, Color> getHashtableParentColor() {
        return hashtableParentColor;
    }    

    /**
     * Obtin hashtable-ul informatii despre ultimul nod in parinte
     * @return
     */
    public Hashtable<Integer, Boolean> getHashtableIsLast() {
        return hashtableIsLast;
    }

    
    /**~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
     *     Setarea variabilelor puse la
     *      dispozitie pentru desenare
     *~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~*/

    /**
     * Setez faptul ca acest rand este selectat
     * @param isSelected
     */
    protected void setIsSelected(boolean isSelected) {
        this.isSelected = isSelected;
    }

    /**
     * Setez faptul ca aceasta componenta are focus
     * @param hasSomeFocus
     */
    protected void setHasSomeFocus(boolean hasSomeFocus) {
        this.hasSomeFocus = hasSomeFocus;
    }

    /**
     * Setez faptul ca celula este editabila
     * @param isCellEditable
     */
    protected void setIsCellEditable(boolean isCellEditable) {
        this.isCellEditable = isCellEditable;
    }

    /**
     * Specifica faptul ca nodul este frunza
     * @param val
     */
    protected void setIsLeaf(boolean val) {
        isLeaf = val;
    }

    /**
     * Setez faptul ca este prima coloana
     * @param isFirstColumn
     */
    protected void setIsFirstColumn(boolean isFirstColumn) {
        this.isFirstColumn = isFirstColumn;
    }

    /**
     * Setez faptul ca este ultima coloana
     * @param isLastColumn
     */
    protected void setIsLastColumn(boolean isLastColumn) {
        this.isLastColumn = isLastColumn;
    }
    
    /**
     * Seteaza daca nodul este restrans
     * @param val
     */
    protected void setIsExpanded(boolean val) {
        isExpanded = val;
    }    
    
    /**
     * Setez valoare variabilei ce specifia importanta celulei
     * @param isCellImportant
     */
    protected void setIsCellImportant(boolean isCellImportant) {
        this.isCellImportant = isCellImportant;
    }

    /**
     * Marchez faptul ca celula din stanga este importanta
     * @param isLeftCellImportant
     */
    protected void setIsLeftCellImportant(boolean isLeftCellImportant) {
        this.isLeftCellImportant = isLeftCellImportant;
    }

    /**
     * Marchez faptul ca celula din dreapta este importanta
     * @param isRightCellImportant
     */
    protected void setIsRightCellImportant(boolean isRightCellImportant) {
        this.isRightCellImportant = isRightCellImportant;
    }

    /**
     * Specifica faptul ca icoanele pentru handle-uri nu se vor afisa
     * @param val
     */
    protected void setIsShowHandle(boolean val) {
        isShowHandle = val;
    }

    /**
     * Specifica faptul ca aceast rand este o grupare
     * @param isGrouping
     */
    protected void setIsGrouping(boolean isGrouping) {
        this.isGrouping = isGrouping;
    }

    /**
     * Setez faptul ca are tooltiptext
     * @param hasToolTipText
     */
    protected void setHasToolTipText(boolean hasToolTipText) {
        this.hasToolTipText = hasToolTipText;
    }
    
    /**
     * Seteaza adancimea acestui nod
     * @param i
     */
    protected void setNestingDepth(int i) {
        nestingDepth = i;
    }

    /**
     * Setez culoare preferata de model
     * @param prefferedBackground
     */
    protected void setPrefferedBackground(Color prefferedBackground) {
        this.prefferedBackground = prefferedBackground;
    }

    /**
     * Setez culoare preferata de model
     * @param prefferedForeground
     */
    protected void setPrefferedForeground(Color prefferedForeground) {
        this.prefferedForeground = prefferedForeground;
    }

    /**
     * Setez hastable-ul cu culorile parintilor
     * @param hashtableParentColor
     */
    protected void setHashtableParentColor(Hashtable<Integer, Color> hashtableParentColor) {
        this.hashtableParentColor = hashtableParentColor;
    }

    /**
     * Setez hashtable-ul cu noduri finale
     * @param hashtableIsLast
     */
    protected void setHashtableIsLast(Hashtable<Integer, Boolean> hashtableIsLast) {
        this.hashtableIsLast = hashtableIsLast;
    }

    /**~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
     *      Construieste componenta ce
     *      va fi trimisa spre desenare
     *~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~*/

    /** Get a component that can render cells in an Outline.  If
     * <code>((Outline) table).isTreeColumnIndex(column)</code> is true,
     * it will paint as indented and with an expansion handle if the
     * Outline's model returns false from <code>isLeaf</code> for the
     * passed value.
     * <p>
     * If the column is not the tree column, its behavior is the same as
     * DefaultTableCellRenderer.
     */
    @Override
    public Component getTableCellRendererComponent(JTable table,
            Object value, boolean isSelected, boolean hasFocus,
            int row, int column) {

        //Apel in parinte pentru a seta el variabilele default
        super.getTableCellRendererComponent(table, value,
                        isSelected, hasFocus, row, column);

        //Surse de date
        treetable = (Treetable) table;
        treetableModel = treetable.getOutlineModel();
        layoutCache = treetable.getLayoutCache();
        renderDataProvider = treetable.getRenderDataProvider();
        if (renderDataProvider == null) {
            setText("no renderDataProvider");
            return this;
        }
        rowModel = treetableModel.getRowModel();

        //Valoarea principala din index
        DefaultMutableTreeNode indexColumnNode = (DefaultMutableTreeNode)
                treetableModel.getValueAt(row, 0);

        //Variabile boolene de stare
        this.isSelected = isSelected;
        this.hasSomeFocus = hasFocus;
        this.isCellEditable = treetableModel.isCellEditable(row, column);
        this.isLeaf = treetable.getOutlineModel().isLeaf(indexColumnNode);
        this.isFirstColumn = treetable.isTreeColumnIndex(column);
        this.isLastColumn = (column == (treetableModel.getColumnCount()-1));

        //Obtin nivelul nodului
        TreePath path = layoutCache.getPathForRow(row);
        this.isExpanded = layoutCache.isExpanded(path);
        this.nestingDepth = path.getPathCount() - 1;

        //Importanta celulei
        if (isFirstColumn) {
            this.isCellImportant = true;
            this.isLeftCellImportant = false;
            this.isRightCellImportant = rowModel
                    .isCellImportant(indexColumnNode, isExpanded, 0);
        } else {
            this.isCellImportant = rowModel
                    .isCellImportant(indexColumnNode, isExpanded, column-1);

            //Obtin importanta vecinului stanga
            if (column == 1) {
                this.isLeftCellImportant = true;
            } else {
                this.isLeftCellImportant = rowModel
                    .isCellImportant(indexColumnNode, isExpanded, column-2);
            }

            //Obtin importanta vecinului dreapta
            if (isLastColumn) {
                this.isRightCellImportant = false;
            } else {
                this.isRightCellImportant = rowModel
                    .isCellImportant(indexColumnNode, isExpanded, column);
            }
        }

        //Daca se afiseaza sau nu handle=ul
        if (isFirstColumn) {
            this.isShowHandle = true;
        } else {
            this.isShowHandle = false;
        }

        //Informatii despre tipul nodului
        this.isGrouping = renderDataProvider.isGrouping(indexColumnNode);

        //Textul ce se deseneaza
        String overwriteString = null;
        if (isFirstColumn) {
            String displayName = renderDataProvider.getDisplayName(indexColumnNode);
            if (displayName != null) {
                setText(displayName);
            }
        } else {
            //Daca este solicitat overwrite in folosesc
            overwriteString = rowModel
                    .getOverwriteString(indexColumnNode, isExpanded, column-1);
            if (overwriteString != null) {
                value = overwriteString;
            }

            //Pun valoare in Jlabel
            if (value instanceof String) {
                this.setText((String)value);
            } else
            if (value instanceof BigDecimal) {
                BigDecimal auxBigDecimal = (BigDecimal) value;
                this.setText(NumberUtils.formatBigDecimal(auxBigDecimal,
                        renderDataProvider.getDecimalScale()));
            } else
            if (value instanceof Double) {
                Double auxDouble = (Double) value;
                this.setText(NumberUtils.formatDouble(auxDouble,
                        renderDataProvider.getDecimalScale()));
            } else
            if (value instanceof Integer) {
                Integer auxInteger = (Integer) value;
                this.setText(NumberUtils.formatInteger(auxInteger));
            }
        }

        //Alinierea textului
        if (value instanceof Number)
            setHorizontalAlignment(SwingConstants.RIGHT);
        else if (overwriteString != null)
            setHorizontalAlignment(SwingConstants.CENTER);
        else
            setHorizontalAlignment(SwingConstants.LEFT);

        //obtin tooltip textul
        String tooltipText;
        if (isFirstColumn) {
            tooltipText = renderDataProvider.getTooltipText(indexColumnNode);
        } else {
            tooltipText = rowModel
                    .getTooltipTextFor(indexColumnNode, column - 1);
        }

        //Setez tooltip textul
        this.setToolTipText(tooltipText);
        if (tooltipText != null) {
            hasToolTipText = true;
        } else {
            hasToolTipText = false;
        }

        //Setez culorile de desenare
        Color overwriteBackground = null;
        Color overwriteForeground = null;
        if (!isFirstColumn) {
            //~~~ BG ~~~
            overwriteBackground = rowModel
                    .getOverwriteBackground(indexColumnNode, isExpanded, column - 1);
            //~~~ FG ~~~
            overwriteForeground = rowModel
                    .getOverwriteForeground(indexColumnNode, isExpanded, column - 1);
        }

        //Culorile pe intreg randul
        prefferedBackground = renderDataProvider.getBackground(indexColumnNode);
        prefferedForeground = renderDataProvider.getForeground(indexColumnNode);

        //~~~ Sel ~~~
        if (isSelected) {
            //Culoare speciala daca bg este suprascris
            if (overwriteBackground != null)
                setBackground(GraphicRenderingUtils.colorShift(
                        treetable.getSelectionBackground(), overwriteBackground));
            else setBackground(treetable.getSelectionBackground());

            //Culoare speciala daca fg e suprascris
            if (overwriteForeground != null)
                setForeground(GraphicRenderingUtils.colorShift(
                        treetable.getSelectionForeground(), overwriteForeground));
            else setForeground(treetable.getSelectionForeground());
        } else {
            if (isGrouping) {
                //Grouping Bg
                if (overwriteBackground != null) setBackground(overwriteBackground);
                else setBackground(Color.WHITE);
                //Grouping Fg
                if (overwriteForeground != null) setForeground(overwriteForeground);
                else setForeground(Color.BLACK);
            } else {
                //Normal Bg
                if (overwriteBackground != null) setBackground(overwriteBackground);
                else setBackground(prefferedBackground);
                //Normal Fg
                if (overwriteForeground != null) setForeground(overwriteForeground);
                else setForeground(prefferedForeground);
            }
        }

        //Setez iconul
        if (isFirstColumn) {
            Icon icon = renderDataProvider.getIcon(indexColumnNode,isExpanded);
            if (icon == null) {
                if (!isLeaf) {
                    if (isExpanded) {
                        setIcon(getDefaultOpenIcon());
                    } else {
                        setIcon(getDefaultClosedIcon());
                    }
                } else {
                    setIcon(getDefaultLeafIcon());
                }
            } else setIcon(icon);
        } else {
            setIcon(null);
        }

        //Compun hashtable-urile legat de culoare parintilor
        hashtableParentColor.clear();
        DefaultMutableTreeNode nod = indexColumnNode;
        for (int i = nestingDepth-1 ; i > 0; i--) {
            nod = (DefaultMutableTreeNode) nod.getParent();

            if (renderDataProvider.isGrouping(nod)) {
                hashtableParentColor.put(i, renderDataProvider.getBackground(nod));
            } else {
                //Verific daca nu cumva este selectat
                int[] sels = treetable.getSelectedIndices();

                //Fac lista de noduri bune
                DefaultMutableTreeNode auxNod;
                for (int j = 0; j < sels.length; j++) {
                    auxNod = (DefaultMutableTreeNode) treetable.getValueAt(sels[j], 0);
                    if (nod.equals(auxNod)) {
                        hashtableParentColor.put(i, treetable.getSelectionBackground());
                        break;
                    }
                }
            }
        }

        //Compun hashtable-ul legat de ultimul copil al parintilor
        hashtableIsLast.clear();
        nod = indexColumnNode;
        DefaultMutableTreeNode parent = (DefaultMutableTreeNode) nod.getParent();
        boolean state = true;
        for (int i = nestingDepth-1; i > 0; i--) {
            state = state && (parent.getLastChild() == nod);
            hashtableIsLast.put(i, state);

            //Urc in inaltime
            nod = parent;
            parent = (DefaultMutableTreeNode)nod.getParent();
        }

        //Intorc aceasta componenta pentru desenare
        return this ;
    }

    /**
     * Aici printez starea la aceasta celula
     * @return
     */
    @Override
    public String toString() {
        StringBuffer stringBuffer = new StringBuffer();

        //Variabile booleane puse la dispozitie de catre clasa
        stringBuffer.append("isSelected = "+isSelected); stringBuffer.append('\n');
        stringBuffer.append("hasSomeFocus = "+hasSomeFocus); stringBuffer.append('\n');
        stringBuffer.append("isCellEditable = "+isCellEditable); stringBuffer.append('\n');
        stringBuffer.append("isLeaf = "+isLeaf); stringBuffer.append('\n');
        stringBuffer.append("isFirstColumn = "+isFirstColumn); stringBuffer.append('\n');
        stringBuffer.append("isLastColumn = "+isLastColumn); stringBuffer.append('\n');
        stringBuffer.append("isCellImportant = "+isCellImportant); stringBuffer.append('\n');
        stringBuffer.append("isLeftCellImportant = "+isLeftCellImportant); stringBuffer.append('\n');
        stringBuffer.append("isRightCellImportant = "+isRightCellImportant); stringBuffer.append('\n');
        stringBuffer.append("isExpanded = "+isExpanded); stringBuffer.append('\n');
        stringBuffer.append("isShowHandle = "+isShowHandle); stringBuffer.append('\n');
        stringBuffer.append("isGrouping = "+isGrouping); stringBuffer.append('\n');
        stringBuffer.append("hasToolTipText = "+hasToolTipText); stringBuffer.append('\n');

        //Variabile de stare
        stringBuffer.append("text = "+getText()); stringBuffer.append('\n');
        stringBuffer.append("nestingDepth = "+nestingDepth); stringBuffer.append('\n');
        stringBuffer.append("background = "+getBackground()); stringBuffer.append('\n');
        stringBuffer.append("foreground = "+getForeground()); stringBuffer.append('\n');
        stringBuffer.append("prefferedBackground = "+prefferedBackground); stringBuffer.append('\n');
        stringBuffer.append("prefferedForeground = "+prefferedForeground); stringBuffer.append('\n');

        //Hashtable care specifica pentru ce nivele parinte nodul este utilmul
        stringBuffer.append("hashtableIsLast = "+hashtableIsLast); stringBuffer.append('\n');

        //Hashtable care specifica culoare de desenare pentru fiecare nivel parinte
        stringBuffer.append("hashtableParentColor = "+hashtableParentColor); stringBuffer.append('\n');

        //Intorc stringul compus
        stringBuffer.append("==========================");
        return stringBuffer.toString();
    }
}
