
package exenne.components.treetable;

import javax.swing.event.TableModelListener;
import javax.swing.event.TreeExpansionListener;
import javax.swing.event.TreeModelListener;
import javax.swing.table.TableModel;
import javax.swing.tree.AbstractLayoutCache;
import javax.swing.tree.DefaultMutableTreeNode;
import javax.swing.tree.DefaultTreeModel;
import javax.swing.tree.FixedHeightLayoutCache;
import javax.swing.tree.TreeModel;
import javax.swing.tree.TreePath;
import javax.swing.tree.VariableHeightLayoutCache;

/** Proxies a standard TreeModel and TableModel, translating events between
 * the two.  Note that the constructor is not public;  the TableModel that is
 * proxied is the OutlineModel's own.  To make use of this class, implement
 * RowModel - that is a mini-table model in which the TreeModel is responsible
 * for defining the set of rows; it is passed an object from the tree, which
 * it may use to generate values for the other columns.  Pass that and the
 * TreeModel you want to use to <code>createOutlineModel</code>.
 * <p>
 * A note on TableModelEvents produced by this model:  There is a slight
 * impedance mismatch between TableModelEvent and TreeModelEvent.  When the
 * tree changes, it is necessary to fire TableModelEvents to update the display.
 * However, TreeModelEvents support changes to discontiguous segments of the
 * model (i.e. &quot;child nodes 3, 4 and 9 were deleted&quot;).  TableModelEvents
 * have no such concept - they operate on contiguous ranges of rows.  Therefore,
 * one incoming TreeModelEvent may result in more than one TableModelEvent being
 * fired.  Discontiguous TreeModelEvents will be broken into their contiguous
 * segments, which will be fired sequentially (in the case of removals, in
 * reverse order).  So, the example above would generate two TableModelEvents,
 * the first indicating that row 9 was removed, and the second indicating that
 * rows 3 and 4 were removed.
 * <p>
 * Clients which need to know whether the TableModelEvent they have just
 * received is one of a group (perhaps they update some data structure, and
 * should not do so until the table's state is fully synchronized with that
 * of the tree model) may call <code>areMoreEventsPending()</code>.
 * <p>
 * In the case of TreeModelEvents which add items to an unexpanded tree node,
 * a simple value change TableModelEvent will be fired for the row in question
 * on the tree column index.
 * <p>
 * Note also that if the model is large-model, removal events may only indicate
 * those indices which were visible at the time of removal, because less data
 * is retained about the position of nodes which are not displayed.  In this
 * case, the only issue is the accuracy of the scrollbar in the model; in
 * practice this is a non-issue, since it is based on the Outline's row count,
 * which will be accurate.
 * <p>
 * A note to subclassers, if we even leave this class non-final:  If you do
 * not use ProxyTableModel and RowMapper (which probably means you are doing
 * something wrong), <strong>do not fire structural changes from the TableModel</strong>.
 * This class is designed such that the TreeModel is entirely in control of the
 * count and contents of the rows of the table.  It and only it may fire
 * structural changes.
 * <p>
 * Note that this class enforces access only on the event dispatch thread
 * with assertions.  All events fired by the underlying table and tree model
 * must be fired on the event dispatch thread.
 *
 * @author  Tim Boudreau
 */
public class DefaultTreetableModel implements  TreetableModel {
    private TreeModel treeModel;
    private ProxyTableModel tableModel;
    private RowModel rowModel;
    private AbstractLayoutCache layout;
    private TreePathSupport treePathSupport;
    private EventBroadcaster broadcaster;


    //Some constants we use to have a single method handle all translated
    //event firing
    private static final int NODES_CHANGED = 0;
    private static final int NODES_INSERTED = 1;
    private static final int NODES_REMOVED = 2;
    private static final int STRUCTURE_CHANGED = 3;

    /**
     * 4/19/2004 - Added ability to set the node column name.
     * David Botterill
     */

    private String nodeColumnName;

    //string version of the avoid constants debug output:
    private static final String[] types = new String[] {
            "nodesChanged", "nodesInserted", "nodesRemoved",
            "structureChanged" };

    /** Create an OutlineModel using the supplied tree model and row model,
     * specifying if it is a large-model tree */
    public static TreetableModel createOutlineModel(
            RowModel rowModel) {
        DefaultMutableTreeNode root = new DefaultMutableTreeNode();
        TreeModel treeMdl = new DefaultTreeModel(root, false);
        ProxyTableModel proxyTableModel = new ProxyTableModel(rowModel);
        return new DefaultTreetableModel(treeMdl, proxyTableModel,
                true);
    }

    /** Create an OutlineModel using the supplied tree model and row model,
     * specifying if it is a large-model tree */
    public static TreetableModel createOutlineModel(
            RowModel rowModel,TreeModel treeMdl) {
        ProxyTableModel proxyTableModel = new ProxyTableModel(rowModel);
        return new DefaultTreetableModel(treeMdl, proxyTableModel,
                true);
    }

    /** Creates a new instance of DefaultOutlineModel.  <strong><b>Note</b>
     * Do not fire table structure changes from the wrapped TableModel (value
     * changes are okay).  Changes that affect the number of rows must come
     * from the TreeModel.   */
    protected DefaultTreetableModel(TreeModel treeModel,
            ProxyTableModel tableModel, boolean largeModel) {
        this .treeModel = treeModel;
        this .tableModel = tableModel;
        this .rowModel = tableModel.getRowModel();

        layout = largeModel ? (AbstractLayoutCache) new FixedHeightLayoutCache()
                : (AbstractLayoutCache) new VariableHeightLayoutCache();

        broadcaster = new EventBroadcaster(this );

        layout.setRootVisible(true);
        layout.setModel(this );
       treePathSupport = new TreePathSupport(this , layout);
        treePathSupport.addTreeExpansionListener(broadcaster);
        treePathSupport.addTreeWillExpandListener(broadcaster);
        treeModel.addTreeModelListener(broadcaster);
        tableModel.addTableModelListener(broadcaster);
        if (tableModel instanceof  ProxyTableModel) {
            ((ProxyTableModel) tableModel).setOutlineModel(this );
        }
    }

    //========================================
    //      Implements the Outline Model
    //========================================

    public final TreePathSupport getTreePathSupport() {
        return treePathSupport;
    }

    public final AbstractLayoutCache getLayout() {
        return layout;
    }

    public boolean isLargeModel() {
        return layout instanceof  FixedHeightLayoutCache;
    }

    /**
     * Seteaza numele primei coloane
     */
    public void setNodeColumnName(String inName) {
        nodeColumnName = inName;
    }

    /**
     * Obtin NodeRowModel - Accesorul pentru nodurile din TreeTable
     * @return NodeRowModel
     */
    public NodeRowModel getRowNodeModel() {
        return (ProxyTableModel) tableModel;
    }

    /**
     * Obitin RowModelul - data provider pentru Celule
     * @return RowModel
     */
    public RowModel getRowModel() {
        return (RowModel) rowModel;
    }

    /** Accessor for EventBroadcaster */
    public TreeModel getTreeModel() {
        return treeModel;
    }

    /**
     * Setez modelul orbore din spate
     */
    public void setTreeModel(TreeModel treeModel) {
        this.treeModel = treeModel;
        treeModel.addTreeModelListener(broadcaster);
    }

    /**
     * Obtin brodcasterul acestui model.
     * Scopul sa il inregistrez la alt model
     * @return
     */
    public EventBroadcaster getBroadcaster() {
        return broadcaster;
    }

    /**
     * Inregistrez inca un ascultator pe evenimente
     * @param newBroadcaster
     */
    public void registerBroadcaster(EventBroadcaster newBroadcaster) {
        treePathSupport.addTreeExpansionListener(newBroadcaster);
        treePathSupport.addTreeWillExpandListener(newBroadcaster);
    }

    /**
     * Inregistrez un exapasion listenet
     */
    public void registerExpansionListener(TreeExpansionListener expansionListener) {
        treePathSupport.addTreeExpansionListener(expansionListener);
        treePathSupport.addTreeWillExpandListener(expansionListener);
    }

    /** Accessor for EventBroadcaster */
    public TableModel getTableModel() {
        return tableModel;
    }


    //========================================
    //      Implements the Tree Model
    //========================================

    public final Object getRoot() {
        return treeModel.getRoot();
    }    
    
    public final Object getChild(Object parent, int index) {
        return treeModel.getChild(parent, index);
    }

    public final int getChildCount(Object parent) {
        return treeModel.getChildCount(parent);
    }

     public final boolean isLeaf(Object node) {
        /**
         * Changed 1/13/2005 - David Botterill
         * We need to check for a null node here since the DefaultTreeModel does
         * not and will cause NPE if it's null.  This situation occurs when the
         * DefaultOutlineCellRenderer gets called from the accessibility context.
         */
        if (null == node)
            return true;
        return treeModel.isLeaf(node);
    }   

    public final void valueForPathChanged(
            javax.swing.tree.TreePath path, Object newValue) {
        //if the model is correctly implemented, this will trigger a change
        //event
        treeModel.valueForPathChanged(path, newValue);
    }

    public final int getIndexOfChild(Object parent, Object child) {
        return treeModel.getIndexOfChild(parent, child);
    }

    /** Delegates to the EventBroadcaster for this model */
    public final synchronized void addTreeModelListener(
            TreeModelListener l) {
        broadcaster.addTreeModelListener(l);
    }

    /** Delegates to the EventBroadcaster for this model */
    public final synchronized void removeTreeModelListener(
            TreeModelListener l) {
        broadcaster.removeTreeModelListener(l);
    }

    //========================================
    //      Implements the Table Model
    //========================================

    public final int getRowCount() {
        return layout.getRowCount();
    }    
    
    public final int getColumnCount() {
        return tableModel.getColumnCount() + 1;
    }
    
    public String getColumnName(int columnIndex) {
        /**
         * Changed 4/19/2004 to allow the node column to be named.
         * - David Botterill
         */
        if (columnIndex == 0) {
            return null == nodeColumnName ? "Nodes" : nodeColumnName;
        } else {
            return tableModel.getColumnName(columnIndex - 1);
        }
    }

    /** Delegates to the RowMapper for > 0 columns; column 0 always
     * returns Object.class */
    public final Class getColumnClass(int columnIndex) {
        if (columnIndex == 0) {
            return Object.class;
        } else {
            return tableModel.getColumnClass(columnIndex - 1);
        }
    }

    public boolean isCellEditable(int rowIndex, int columnIndex) {
        if (columnIndex == 0) {
            return false; //support editing of node names
        } else {
            return tableModel.isCellEditable(rowIndex, columnIndex - 1);
        }
    }

    public final Object getValueAt(int rowIndex, int columnIndex) {
        Object result;
        if (columnIndex == 0) { //need a column ID - columnIndex = 0 depends on the column model
            TreePath path = getLayout().getPathForRow(rowIndex);
            if (path != null) {
                result = path.getLastPathComponent();
            } else {
                result = null;
            }
        } else {
            result = (tableModel.getValueAt(rowIndex, columnIndex - 1));
        }
        return result;
    }

    /** Delegates to the RowModel (or TableModel) for non-0 columns */
    public final void setValueAt(Object aValue, int rowIndex,
            int columnIndex) {
        if (columnIndex != 0) {
            tableModel.setValueAt(aValue, rowIndex, columnIndex - 1);
        } else {
            //do something
        }
    }

    /** Delegates to the EventBroadcaster for this model */
    public final synchronized void addTableModelListener(
            TableModelListener l) {
        broadcaster.addTableModelListener(l);
    }

    /** Delegates to the EventBroadcaster for this model */
    public final synchronized void removeTableModelListener(
            TableModelListener l) {
        broadcaster.removeTableModelListener(l);
    }
   
}