package exenne.components.treetable.cellrenderers;

import exenne.components.utils.GraphicRenderingUtils;
import java.awt.BasicStroke;
import java.awt.Color;
import java.awt.Component;
import java.awt.Graphics;
import java.awt.Graphics2D;
import java.awt.geom.RoundRectangle2D;
import javax.swing.BorderFactory;
import javax.swing.JTable;
import javax.swing.SwingConstants;
import javax.swing.border.Border;

/** An outline-aware TableCellRenderer which knows how to paint expansion
 * handles and indent child nodes an appropriate amount.
 *
 * @author  Tim Boudreau
 */
public class DefaultTreetableCellRenderer extends
        AbstractTreetableCellRenderer {

    //Border pentru reprezentare tooltip
    private Border tooltipBorder;


    /** Creates a new instance of DefaultOutlineTreeCellRenderer */
    public DefaultTreetableCellRenderer() {
        super();

        tooltipBorder = new TreetableTooltipBorder();
    }

    /** Overridden to combine the expansion border (whose insets determine how
     * much a child tree node is shifted to the right relative to the ancestor
     * root node) with whatever border is set, as a CompoundBorder.  The expansion
     * border is also responsible for drawing the expansion icon.  */
    @Override
    public void setBorder(Border b) {
        if (b == tooltipBorder) {
            super .setBorder(b);
        } else {
            super .setBorder(BorderFactory.createCompoundBorder(tooltipBorder,
                    b));
        }
    }

    //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~/
    //      Clase de desenare
    //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~/

    /** Get a component that can render cells in an Outline.  If
     * <code>((Outline) table).isTreeColumnIndex(column)</code> is true,
     * it will paint as indented and with an expansion handle if the
     * Outline's model returns false from <code>isLeaf</code> for the
     * passed value.
     * <p>
     * If the column is not the tree column, its behavior is the same as
     * DefaultTableCellRenderer.
     */
    @Override
    public Component getTableCellRendererComponent(JTable table,
            Object value, boolean isSelected, boolean hasFocus,
            int row, int column) {

        //Setez parametrii in parinte
        super.getTableCellRendererComponent(table, value,
                isSelected, hasFocus, row, column);

        //DEBUG
        //if (hasFocus)
        //    System.out.println(this);
        

        //Intorc aceasta componenta pentru a-i suprascrie paintul
        return this ;
    }

    /**
     * Aici fac custom paintingul al celule
     * @param g Contextul grafic in care desenez
     */
    @Override
    public void paint(Graphics g) {
        Graphics2D g2d = (Graphics2D) g;
        int startX = 0;
        int startY = 0; //Pozitia de unde se incepe desenarea
        int localWidth = 0; //latimea de desenare
        int localHeight = 0; //inaltimea de desenare
        int width = getWidth();
        int height = getHeight();
        int arcW = 5, arcH = 5;
        int iconWidth = getNestingWidth();
        int lineWidth = 2;
        int focusMarkerSize = 4;
        boolean isLast;

        g2d.setStroke(new BasicStroke(lineWidth));

        // <editor-fold defaultstate="collapsed" desc="Fundal pt fiecare celula">
        //Actiunea inaintea desenarii
        boolean shouldPaint = true;
        Color actualBgColor, actualFgColor; //Culoarea setata in pregatiri
        actualBgColor = getBackground();
        actualFgColor = getForeground();
        Color prefferedBgColor, prefferedFgColor;
        prefferedBgColor = getPrefferedBackground();
        prefferedFgColor = getPrefferedForeground();

        //Desenez backgroundul
        g2d.setColor(actualBgColor);
        g2d.fillRect(0, 0, width, height);
        setBackground(null);
        
        //Caz special daca celula este editabila
        if (isCellEditable()) {
            boolean isLeftAligned = getHorizontalAlignment() == SwingConstants.LEFT ||
                    getHorizontalAlignment() == SwingConstants.LEADING;

            //Aleg culorile pentru desenare
            Color colorLeft;
            Color colorRight;
            if (isSelected()) {
                colorLeft = g2d.getColor();
                colorRight = GraphicRenderingUtils.lightenColor(colorLeft, 20);
            } else {
                colorLeft = g2d.getColor();
                colorRight = Color.WHITE;
            }

            //Fac desenarea propriu-zisa
            GraphicRenderingUtils.drawEditableBackground(g2d,colorLeft,
                    colorRight,width,height,iconWidth,isLeftAligned);

            //Invalidez fundalul
            setBackground(null);
        }
        // </editor-fold>

        /**
         * Verificare speciala pentru fiecare celula. Daca este copil al unei
         * Categorii, atunci tre sa desenez  liniutze inainte
         */
        if (isFirstColumn()) {
            // <editor-fold defaultstate="collapsed" desc="Desenez identarea primei coloane">
            isLast = false;
            //Umplu fundalul
            g2d.setColor(Color.WHITE);
            g2d.fillRect(0, 0, (getNestingDepth()-1)*iconWidth, height);
            //Desenez liniutele verticale
            for (int i = 1; i < getNestingDepth(); i++) {
                Color colorItem = getHashtableParentColor().get(i);
                if (!isExpanded()) isLast = getHashtableIsLast().get(i);
                if (colorItem != null) {
                    startX = i * iconWidth;
                    startY = 0;
                    g2d.setColor(colorItem);
                    g2d.drawLine(startX, startY, startX, startY + height);
                    if (isLast)
                       g2d.drawLine(startX, startY + height-lineWidth+1,
                               width, startY + height-lineWidth+1);
                }
            }
            // </editor-fold>
        } else
        if (isLastColumn()) {
            // <editor-fold defaultstate="collapsed" desc="Desenez identarea ultimei coloane">
            isLast = false;
            for (int i = getNestingDepth() - 1; i > 0; i--) {
                Color colorItem = getHashtableParentColor().get(i);
                if (!isExpanded()) {
                    isLast = getHashtableIsLast().get(i);

                }
                if (colorItem != null) {
                    startX = width - 3;
                    startY = 0;
                    g2d.setColor(Color.WHITE);
                    g2d.fillRect(startX, startY, width, height);
                    g2d.setColor(colorItem);
                    g2d.drawLine(startX, startY, startX, startY + height);
                    if (isLast) {
                        g2d.drawLine(0, startY + height - lineWidth + 1,
                                startX, startY + height - lineWidth + 1);

                    }
                    break;
                }
            }
            // </editor-fold>
        } else {
            // <editor-fold defaultstate="collapsed" desc="Desenez mijloc">
            isLast = false;
            for (int i = 1; i < getNestingDepth(); i++) {
                Color colorItem = getHashtableParentColor().get(i);
                if (!isExpanded()) {
                    isLast = getHashtableIsLast().get(i);
                }
                if (colorItem != null) {
                    startX = 0;
                    startY = 0;
                    g2d.setColor(colorItem);
                    if (isLast) {
                        g2d.drawLine(startX, startY + height - lineWidth + 1,
                                width, startY + height - lineWidth + 1);
                    }
                }
            }
            // </editor-fold>
        }

        // <editor-fold defaultstate="collapsed" desc="Fundalul special">
        if (isGrouping()) {
                //Desenez fundalul pentru categorie
                if (isFirstColumn()) {
                    g.setColor(prefferedBgColor);
                    startX = getNestingDepth() * iconWidth-lineWidth + 1;
                    startY = 0;
                    localWidth = width - startX - 3;
                    //Difera inaltimea
                    if (!isExpanded())
                        localHeight = height - startY - 1;
                    else
                        localHeight = height + 100;

                    //Desenez shieldul
                    g2d.fill(new RoundRectangle2D.Double(startX, startY,
                            localWidth,localHeight,
                            arcW, arcH));
                    if (isExpanded())
                        g2d.drawLine(startX+lineWidth, height - 1, width + startX, height - 1);

                    //Fac setarile de background pentru desenare in super
                    setBackground(null);
                    setForeground(prefferedFgColor);
                } else
                if (isCellImportant()) {
                    g.setColor(prefferedBgColor);
                    startY = 0;
                    
                    //In functie de vecini decid setarile
                    if (isLeftCellImportant() && isRightCellImportant()) {
                        startX = -arcW;
                        localWidth = width - 2*startX;
                    } else if (isLeftCellImportant()) {
                        startX = -arcW;
                        localWidth = width - startX - 2;
                    } else if (isRightCellImportant()) {
                        startX = 2;
                        localWidth = width;
                    } else {
                        startX = 2;
                        localWidth = width - startX - 2;
                    }

                    //Difera inaltimea
                    if (isExpanded())
                        localHeight = height + 100;
                    else
                        localHeight = height - startY - 1;

                    //Desenez shieldul
                    g2d.fill(new RoundRectangle2D.Double(startX, startY,
                            localWidth,localHeight,
                            arcW, arcH));
                    if (isExpanded()) {
                        if (isLastColumn())
                            g2d.drawLine(0, height - 1, width - startX - 3, height - 1);
                        else
                            g2d.drawLine(0, height - 1, width, height - 1);
                    }

                    //Fac setarile de background pentru desenare in super
                    setBackground(null);
                    setForeground(prefferedFgColor);
                } else {
                    g.setColor(prefferedBgColor);
                    startX = 0;
                    if (isExpanded()) {
                        if (isLastColumn())
                            g2d.drawLine(0, height - 1, width - startX - 3, height - 1);
                        else
                            g2d.drawLine(0, height - 1, width, height - 1);
                    }
                }
        }// </editor-fold>

        //Desenez focusul
        if (hasSomeFocus()) {
            g2d.setColor(actualFgColor);

            //Stanga - Sus
            startX = 0;
            startY = 0;
            g2d.drawLine(startX, startY,
                    startX, focusMarkerSize);
            g2d.drawLine(startX, startY,
                    focusMarkerSize, startY);

            //Stanga - Jos
            startX = 0;
            startY = height;
            g2d.drawLine(startX, startY,
                    startX, startY-focusMarkerSize);
            g2d.drawLine(startX, startY,
                    focusMarkerSize, startY);

            //Dreapta - Jos
            startX = width;
            startY = height;
            g2d.drawLine(startX, startY,
                    startX, startY-focusMarkerSize);
            g2d.drawLine(startX, startY,
                    startX-focusMarkerSize, startY);

            //Dreapta - Sus
            startX = width;
            startY = 0;
            g2d.drawLine(startX, startY,
                    startX, focusMarkerSize);
            g2d.drawLine(startX, startY,
                    startX-focusMarkerSize, startY);
        }

        //Apelez metoda de desenare din renderul default al tabelului
        if (shouldPaint) super.paint(g);
    }
}
