=head1 PlugIns in Genquire

=head1 DESCRIPTION

Genquire has a simple XML-based plugins API which is client driven.
This means that all events are initiated from the plugin-side, and
responses are sent back from Genquire.

Communication between the plugin and Genquire occurs via
STDIN/STDOUT, so for most plugins a simple "print" statement
may be sufficient to initiate contact with Genquire.
This also allows plugins to be written in any language.

Genquire buffers its output and creates non-blocking filehandles
at the Genquire end of the I/O pipes, however it is the responsibiliy
of the plugin to ensure that it does not generate blocking calls.

=cut


=head2 Plugins.conf

the file plugins.conf must be in the PLUGINS directory of Genquire.
the format of this file is as follows:

  Name,  script,  arg1,  arg2, ...

for example, the Blast plugin is called by the following line in plugins.conf: 

  Blast, gq_blast.pl, /tmp/, /home/markw/BLAST/

where Blast is the program name (inserted into the drop-down menu of Genquire),
gq_blast.pl is the name of the script that starts the plugin, and the two paths
/tmp/ and /home/markw/BLAST are required by this script to find the working
directory and the blast binaries.

=cut

=head2 Client death

If a plugin dies it should try to send a message to Genquire indicating that fact.
the correct format of this message is:
  <KILLED>

this can appear anywhere in a message string, even in the middle of other XML
tags.  This allows SIG INT to be caught and the plugin handler to be cleaned up
nicely at the Genquire end.

=cut

=head2 Server death

if Genquire dies, it will attempt to send a single message to the plugin:
  <KILL>

this may appear anywhere in the response string, even between or within XML tags.
This can be caught by the plugin to allow it to cleanup and exit gracefully.

=cut


=head2 Exiting the plugin

When a pligin exits, it should send a single XML tag to genquire:
  <EXIT>

Genquire will then clean up and destroy that plugin handler.


=cut

=head2 Errors & failures

for many calls, genquire will report a failure if it was unable to execute the
requested task, for reasons of invalid input or internal errors.  In such a case,
genquire responds as follows:
  <GQ_RESPONSE>
    <method_requested>
      <FAILED>message or reason here </FAILED>
    </method_requested>
  </GQ_RESPONSE>

	
=cut

=head2 Standard request format

This is quite straightforward.  There is a <REQUEST></REQUEST> pair of tags
and within these are the <method_requested></method_requested> pair, with
any relevant data between them.  See the API section below for valid methods
and the required data

A correctly formatted query looks, for example, like this:
  <REQUEST>
    <getFullContigSequence>At23TY65</getFullContigSequence>
  </REQUEST>


=cut

=head2 Requests in parallel

Genquire processes a call in its entirety before it sends information back or
begins processing another call.  Thus the integrety of the response sent to
STDIN should be guaranteed.  Multiple calls will be processed in the order
they are read from the plugin STDOUT.  Calls are expected to be complete
and correctly formatted; genquire will not correctly interpret overlaping calls.

Multiple calls may be stacked up in the response queue, and for this reason part
of the response string is the name of the method that was called:
  <GQ_RESPONSE>
    <method_requested>
      <response_tag>response value</response_tag>
    </method_requested>
  </GQ_RESPONSE>


=cut


package GQ::Client::PlugInHandler;

use subs qw(run new);
use vars qw($AUTOLOAD);


use Carp;
use strict;
use IPC::Open2;
use FileHandle;
use Fcntl;
use POSIX;

BEGIN {
    {

	#Encapsulated class data

	my %_attr_data =	#             DEFAULT          ACCESSIBILITY
	  (
	   program =>		[undef, 	'read/write'],
	   command =>		[undef, 	'read/write'],
	   parameters =>	[undef,		'read/write'],
	   PARENT		=>	[undef,		'read/write'],
	   running		=>	["yes",		'read/write'],
	   readfh		=>	[undef, 	'read/write'],
	   writefh		=>	[undef, 	'read/write'],

	  );

	#Class methods, to operate on encapsulated class data

	# Is a specified object attribute accessible in a given mode
	sub _accessible  {
	    my ($self, $attr, $mode) = @_;
	    return $_attr_data{$attr}[1] =~ /$mode/ if exists $_attr_data{$attr};
	}

	# Classwide default value for a specified object attribute
	sub _default_for {
	    my ($self, $attr) = @_;
	    return $_attr_data{$attr}[0] if exists $_attr_data{$attr};
	}

	# List of names of all specified object attributes
	# A hash so names are not repeated
	sub _standard_keys {
	    my ($self)=@_;
	    my %list;
	    foreach (keys %_attr_data) {
		$list{$_}++;
	    }
	    return keys %list;
	}
	} #end of BEGIN block
}

sub new {
    my ($caller, %args) = @_;
    my $caller_is_obj = ref($caller);
    my $class = $caller_is_obj || $caller;
    my $self = {};

    bless $self, $class;

    foreach my $key($self->_standard_keys) {
    	$self->{$key} = $self->_default_for($key);
    }

    foreach my $attrname ( $self->_standard_keys ) {
    	if (exists $args{$attrname}) {
    	    $self->{$attrname} = $args{$attrname} }
    	elsif ($caller_is_obj) {
    	    $self->{$attrname} = $caller->{$attrname} }
    	else {
    	    $self->{$attrname} = $self->_default_for($attrname) }
    }

	# do I somehow need to catch SIG INT here to kill the plugin?
    return $self;
}

sub run {
	my ($self) = @_;
	my $command = $self->command;
	my $args = join " ", @{$self->{parameters}};
	my $PARENT = $self->PARENT;
	my $PLUGINS_DIR = _addSlash($PARENT->PLUGINS_DIR);
	
	return unless($command);
	my $commandline = "$command $args";
	$commandline =~ /(\s*)(.*)/;   # remove any leading spaces
	$commandline = $2;
	return if (!$commandline);
	#print "\t*** executing $PLUGINS_DIR/$commandline\n";
	my ($rdr, $wtr) = (FileHandle->new, FileHandle->new);
		
	$self->readfh($rdr);
	$self->writefh($wtr);
	
	my $pid = open2($rdr, $wtr, "perl $PLUGINS_DIR$commandline");

	return 0  unless $pid;	
	# needto make the filehandles non-blocking	

	my $flags = '';
	
	unless ($^O =~ /Win32/){
		fcntl($rdr, F_GETFL, $flags) or die "can't get flags for $rdr:  $!\n";
		$flags |= O_NONBLOCK;
		fcntl($rdr, F_SETFL, $flags) or die "can't set the flags for $rdr: $!\n";
		# now do for the write filehandle
		fcntl($wtr, F_GETFL, $flags) or die "can't get flags for $wtr:  $!\n";
		$flags |= O_NONBLOCK;
		fcntl($wtr, F_SETFL, $flags) or die "can't set the flags for $wtr: $!\n";
	}
	
	my $message;
	$rdr->autoflush;
	$wtr->autoflush;

	while ($self->running eq "yes"){
		my $receive = <$rdr>;
		chomp $receive;
		unless (length($receive)){
			$PARENT->update;
			next;
		}
		if ($receive =~ /<KILLED>/){print "\n\nplugin KILLED\n\n"; undef $self; return}
		if ($receive eq "<EXIT>"){print "EXITING plugin\n"; undef $self; return}
		if ($receive =~ /<\/REQUEST>/){  # look for end-of-request element
			$message .= $receive;			# if found then the request is complete; 
			my $return = $self->parseMessage($message); # parse it, and send it to the approprioate genqurie subroutine
			#print "sending $wtr $return\n"; # print what was sent to the swcreen
			my $done = $self->respondToPlugin($wtr, $return);  # send the result in small packages to be read
			#print "returned $done\n"; # print what was sent to the swcreen
			$message = "";
			$PARENT->update;
			next;
		
		} else {
			#print "\n\n\nintermediate $receive\n\n\n";
			$message .= $receive;		# if message is still incomplete then append current string
			$PARENT->update;
			next;
		}
	}
}

sub Quit {
	my ($self) = @_;
	my $fh = $self->writefh;
	print $fh "<KILL>";
	exit 0;
}

sub respondToPlugin {
	my ($self, $fh, $return) = @_;

	my $len = length($return);
	my $offset = 0; my $tries = 0;
	while ($len){

		my $written = syswrite $fh, $return, $len, $offset; # may only write partial data...
		unless (defined $written){next}
		$offset += $written;
		$len -= $written;
	}
	return $return;
}	

sub parseMessage {
	my ($self, $message)=@_;
	my $return;
	my $req_code = (($message =~ /<REQUEST><(\w+)>/) && $1);

	if ($req_code eq "getContigIDs"){$return = $self->getContigIDs($message)}
	elsif ($req_code eq "getUniqueContigSequence"){$return = $self->getUniqueContigSequence($message)}
	elsif ($req_code eq "getOrganismName"){$return = $self->getOrganismName($message)}
	elsif ($req_code eq "getSequenceVersion"){$return = $self->getSequenceVersion($message)}
	elsif ($req_code eq "getFullContigSequence"){$return = $self->getFullContigSequence($message)}
	elsif ($req_code eq "getSelectedContigs"){$return = $self->getSelectedContigs($message)}
	elsif ($req_code eq "selectContigs"){$return = $self->selectContigs($message)}
	elsif ($req_code eq "openContig"){$return = $self->openContig($message)}
	elsif ($req_code eq "getFeatureIDs"){$return = $self->getFeatureIDs($message)}
	elsif ($req_code eq "getFeatureByID"){$return = $self->getFeatureByID($message)}
	elsif ($req_code eq "getSelectedFeatures"){$return = $self->getSelectedFeatures($message)}
	elsif ($req_code eq "getActiveSequences"){$return = $self->getActiveSequences($message)}
	elsif ($req_code eq "selectFeatures"){$return = $self->selectFeatures($message)}
	elsif ($req_code eq "mapFeature"){$return = $self->mapFeature($message)}
	else {print "\n\ncant understand code $req_code \n\tfrom message $message\n\n";$self->defaultDisplay($message)
		  }
	$return = "<GQ_RESPONSE>\n<$req_code>\n$return\n</$req_code>\n</GQ_RESPONSE>\n";
	return $return;

}

=head1 API

The calls below go between the <REQUEST>...</REQUEST> tags
the response, as indicated above, is contained between the
original request call tags (indicated here as <..> </..>)


=head2 getContigIDs

 request: <getContigIDs></getContigIDs>
 return: <..><contig>$contigID1</contig><contig>$contigID2</contig>....</..>
 function:  to return the names of all contigs in the genome

=cut


sub getContigIDs {
	# gets the list of contig names in the format <contig>NAME</contig>...
	my ($self, $message) = @_;  # $message is not used
	my @contigs = $self->PARENT->plugin_getContigIDs;
	my $return_string;
	#print "\nreturned @contigs\n";
	foreach my $contig(@contigs){
		$return_string .= "<contig>\n$contig\n</contig>\n";
	}
	return $return_string;
}

=head2 getUniqueContigSequence

 request: <getUniqueContigSequence>$contigID</getUniqueContigSequence>
 return: <..><contig>$contigID<seq>$seq</seq></contig><..>
 function:  to return the non-redundant sequence of a given contig
 non-redundant sequences are calculated such that a given portion of
 sequence will be returned on only one contig.  These can be used to
 build, for example, a Blast database.

=cut

sub getUniqueContigSequence{
	# gets the sequence of a requested contig
	# ignoring overlaps between contigs (i.e. non-redundant sequence, good for Blast databases)
	my ($self, $message) = @_;
	my $name = ($message =~ /<getUniqueContigSequence>(.*)<\/getUniqueContigSequence>/ && $1);
	my $seq = $self->PARENT->plugin_getNonRedundantContigSequence($name);
	#print "$seq $name\n\n";
	return "<contig>\n$name\n<seq>\n$seq\n</seq>\n</contig>\n";
}	

=head2 getFullContigSequence

 request: <getFullContigSequence>$contigID</getFullContigSequence>
 return: <..><contig>$contigID<seq>$seq</seq></contig><..>
 function:  to return the entire sequence of a given contig
 If this contig overlaps other contigs then the returned sequence
 will include this overlap region. Retrieving the full set of
 full contig sequences will result in a largely redundant sequence set.

=cut

sub getFullContigSequence{
	# gets the full sequence of a requested contig
	# this includes sequences that overlap other contigs.
	my ($self, $message) = @_;
	my $name = (($message =~ /<getFullContigSequence>(.*)<\/getFullContigSequence>/) && $1);
	#print "searching for $name\n";
	my $seq = $self->PARENT->getFullContigSequence($name);
	#print "got back $seq\n";
	$seq =~ s/\W//g;
	$seq =~ s/([^\n]{40})/$1\n/g;
	return "<contig>\n$name\n<seq>\n$seq</seq>\n</contig>\n";
}	

=head2 getOrganismName

 request: <getOrganismName></getOrganismName>
 return: <..>$name</..>
 function:  to return the latin name of the organism

=cut


sub getOrganismName {
	my ($self) = @_;
	return $self->PARENT->plugin_getOrganismName;
}

=head2 getSequenceVersion

 request: <getSequenceVersion></getSequenceVersion>
 return: <..>$version</..>
 function:  to return the version identifier

=cut


sub getSequenceVersion {
	my ($self) = @_;
	return $self->PARENT->plugin_getSequenceVersion;
}

=head2 getSelectedContigs

 request: <getSelectedContigs></getSelectedContigs>
 return: <..><contig>$contig_id</contig>....</..>
 function:  to return the names of the contigs currently selected
 in the genome map

=cut

sub getSelectedContigs {
	# returns names of currently highlighted contigs
	my ($self, $message) = @_;
	my $genome = $self->PARENT->GenomeMap;
	my @ids = @{$genome->getLitIDs};
	my $response;
	foreach my $id(@ids){
		$response .= "<contig>$id</contig>";
	}
	return $response;
}

=head2 selectContigs

 request: <selectContigs><contig>$contig1</contig>...</selectContigs>
 return: <..>1</..>
 function:  to set a contig as 'selected' in the genome map
 in the genome map

=cut

sub selectContigs {
	# highlights contigs by name
	my ($self, $message) = @_;
	my $list = ($message =~ /<selectContigs>(.*)<\/selectContigs>/ && $1);
	my @contigs = ($list =~ m/<contig>\n?\n?(.*?)\n?\n?<\/contig>/g);
	my $Genome = $self->PARENT->GenomeMap;
	unless ($Genome) {return &FAILED("genome Map is not open")};
	$Genome->dimAllContigs;
	foreach my $contigID(@contigs){
		$Genome->lightContig($contigID);
	}
	return 1;
}

=head2 openContig

 request:
 <openContig>
  <contig>$contig1</contig>
  <start>$start</start>
  <stop>$stop</stop>
 </openContig>
 return: <..>1|0</..>
 function:  to open the detailed display of a contig from position $start to $stop
 the values of $start and $stop may be left blank to open the entire contig, but
 the tags must be present.

=cut

sub openContig {
	# open contig by name with optional start/stop
	my ($self, $message) = @_;
	my $info = ($message =~ /<openContig>(.*)<\/openContig>/ && $1);
	my ($contig, $start, $stop) = (($info =~ /<contig>\n?\n?(.*?)\n?\n?<\/contig>\n?\n?<start>\n?\n?(.*?)\n?\n?<\/start>\n?\n?<stop>\n?\n?(.*?)\n?\n?<\/stop>/)  && ($1, $2, $3));
	$self->PARENT->{start} = $start;
	$self->PARENT->{stop} = $stop;
	$self->PARENT->{contig} = $contig;
	return 0 unless $contig;
	my $Genome = $self->PARENT->GenomeMap;
	$Genome->lightContig($contig);
	$self->PARENT->btnBegin->invoke;
	return 1;
}

=head2 getActiveSequences

 request: <getActiveSequences></getActiveSequences>
 return:
 <..>
 <contig id='$MAP_id1'>
  <start>$start</start>
  <length>$length</length>
  <sequence>$seq</sequence>
 </contig>
 </..>
 function:  to retrieve the sequence of all currently open map displays
 Only the sequence of the dispayed portion of the contig is retrieved,
 NOT the entire contig.  N.B. $MAP_id1 is what is required in the contig
 field of the GFF string when you call mapFeatures!!!!

=cut

sub getActiveSequences {
	my ($self, $message) = @_;
	my $result;
	my @features;
	foreach my $Map(@{$self->PARENT->MapWindows}){
		#print "found map $Map\n";
		my ($SeqCanvas, $start, $stop, $contig, $MapWindow) = @{$Map};  # get all info about the map
		my $SeqObj = $SeqCanvas->MapSeq;								# obtain the Seq object
		my $primary_id = $SeqObj->primary_id;							# the Seq object primary_id has format contig::start-length
		my ($ctg, $st, $length) = (($primary_id =~ /(.*?)\:\:(\d+)\-(\d+)/) && ($1, $2, $3));
		$result .= "<contig id='".($SeqObj->primary_id)."'>\n<start>$st</start>\n<length>$length</length>\n";
		my $seq = $SeqObj->seq;
		$seq =~ s/\W//g;
		$seq =~ s/([^\n]{40})/$1\n/g;
		$result .= "<sequence>$seq</sequence>\n</contig>";
	}
	return $result;

}

=head2 getFeatureIDs

 request: <getFeatureIDs></getFeatureIDs>
 return: <..><feature>$fid</feature>...</..>
 function:  retrieve the unique feature ids for all currently displayed
 features on all open maps (I don't think this is very useful...)

=cut

sub getFeatureIDs {
	# gets the ids of all features (?? is this useful?)
	my ($self, $message) = @_;
	my $result;
	my @features;
	foreach my $Map(@{$self->PARENT->MapWindows}){
		my ($SeqCanvas, $start, $stop, $contig, $MapWindow) = @{$Map};
		push @features, (keys %{$SeqCanvas->getFeaturesWithTag(["Canvas draft"])});
		push @features, (keys %{$SeqCanvas->getFeaturesWithTag(["Canvas finished"])});
	}
	unless ($features[0]){return &FAILED("No Mapped Features Open")}
	else {
		foreach my $fid(@features){
			$result .= "<feature>$fid</feature>/n";
		}
	}
	return $result;
}

=head2 getFeatureByID

 request: <getFeatureByID>$fid</getFeatureByID>
 return:
   <..>
     <contig>$contig_id</contig>
     <map>$map</map>
     <source>$source</source>
     <feature>$feature</feature>
     <contig_start>$start</contig_start>
     <map_start>$map_start</map_start>
     <contig_end>$stop</contig_end>
     <map_end>$map_stop</map_end>
     <score>$score</score>
     <strand>$strand</strand>
     <frame>$frame</frame>
	 <sequence>$seq</sequence>
     <notes>$msg</notes> 
   </..>
 function:  retrieve the extended GFF information for the given feature ID.
 Note the difference between contig coordinates, and map coordinates in the
 returned information.  This is because a map does not necessarily display
 an entire contig.  $msg takes the form of a GFF2 attributes string.
( tag val val...;tag val;tag val val;...... )

=cut

sub getFeatureByID {
	# gets the extended GFF string for a given feature FIDxxxx (including sequence)
	# since we can guarantee that there is at maximum *one*
	# instance of such a feature (they are uniquely nmumbered over all maps)
	# we can send back a single feature.
	my ($self, $message) = @_;
	my $id = ($message =~ /<getFeatureByID>(.*)<\/getFeatureByID>/ && $1);
	my $feature;
	my ($map, $mapseq, $map_start, $map_stop);
	foreach my $Map(@{$self->PARENT->MapWindows}){
		my ($SeqCanvas, $start, $stop, $contig, $MapWindow) = @{$Map};
		my %features = %{$SeqCanvas->getFeaturesWithTag([$id])}; # returns a hash ref of {id}{feature_object}
		if ($features{$id}){
			$map = $SeqCanvas->MapSeq->primary_id;
			$mapseq = $SeqCanvas->MapSeq;
			($map_start, $map_stop) = (($map =~ /.*?::(\d+)\-(\d+)/) && ($1, $2));  # match the digists from the map designation eg. T12DF4::14335-20987
			#print "\nMap Was $map   $map_start   $map_stop\n";
			$feature = $features{$id};
			last;
		}
	}
	my $gff_xml;
	unless ($feature){
		return &FAILED("Feature does not exist")
	} else {
		my $gff = $feature->gff_string;
		my $strt = $feature->start;         # get relevant feature information
        my $stp = $feature->end;
        my $stnd = $feature->strand;
        my $seq = $mapseq->subseq($strt, $stp);  # get the sequence for this feature
		if ($stnd eq "-1") {$seq =~ tr/ATCG/TAGC/; $seq = scalar reverse $seq;}    # reverse complement

		#print "\nGFF of $feature: $gff\nSequence: $seq";
		my ($contig, $source, $feature, $start, $stop, $score, $strand, $frame, $msg) = (($gff =~ /(.*?)\t(.*?)\t(.*?)\t(.*?)\t(.*?)\t(.*?)\t(.*?)\t(.*?)(\s+.*)?/) && ($1, $2, $3, $4, $5, $6, $7, $8, $9));
		$gff_xml = "<contig>$contig</contig>\n<map>$map</map>\n<source>$source</source>\n<feature>$feature</feature>\n<contig_start>$start</contig_start>\n<map_start>$map_start</map_start>\n<contig_end>$stop</contig_end>\n<map_end>$map_stop</map_end>\n<score>$score</score>\n<strand>$strand</strand>\n<frame>$frame</frame>\n<notes>$msg</notes>\n<sequence>$seq</sequence>\n";
	}
	return $gff_xml;
}


=head2 getSelectedFeatures

 request: <getSelectedFeatures></getSelectedFeatures>
 return: <..><feature>$fid</feature>...</..>
 function:  retrieve the unique feature IDs for all currently selected
 features on all open maps.

=cut

sub getSelectedFeatures {
	# gets the ids of selected features
	my ($self, $message) = @_;
	my $result;
	my @features;
	foreach my $Map(@{$self->PARENT->MapWindows}){
		my ($SeqCanvas, $start, $stop, $contig, $MapWindow) = @{$Map};
		@features = (keys %{$SeqCanvas->getSelectedFeatures()});
		unless ($features[0]){return &FAILED("No Features Selected")}
	}
	foreach my $feature(@features){
		$result .= "<feature>$feature</feature>\n";
	}
	return $result;
}


=head2 selectFeatures

 request: <selectFeatures><feature>$fid</feature>...</selectFeatures>
 return: <..>1</..>
 function:  "select" all of the features in the list on the feature map
 features that are not present are ignored.

=cut

sub selectFeatures {
	my ($self, $message) = @_;
	my $featurelist = ($message =~ /<selectFeatures>(.*)<\/selectFeatures>/ && $1);
	my @features = ($featurelist =~ m/<feature>(.*?)<\/feature>/g);
	foreach my $Map(@{$self->PARENT->MapWindows}){
		my ($SeqCanvas, $start, $stop, $contig, $MapWindow) = @{$Map};
		$SeqCanvas->clearSelections();
		$SeqCanvas->selectFeatures(\@features);
	}
	return 1;
}


=head2 mapFeature

 request:
	[<import>]
	<mapFeature>
	  <contig>$MapSequenceID</contig>
	  <start>$start</start>
	  <end>$end</end>
	  <feature>$feature_type</feature>
	  <source>$source</source>
	  <strand>$strand</strand>
	  <frame>$frame</frame>
	  <score>$score</score>
	  <attributes>$attrib</attributes>
	</mapFeature>
	[</import>]
 return: <..>1</..>
 function:  to map a feature onto a Sequence viewer and optionally import
 that feature into the database.  If the MapSequenceID is not recognized
 as a currently open map (i.e. if the sequence in-hand did not come from a
 call to getActiveSequences), but has the form of a map-name:
	contig_id::start-length
	eg:  T22H7::2500-5000
 then the call will be ignored to avoid writing spurious data into the database.
 This allows the mapFeatures call to double as a database import routine for contig
 analyses.
 Example: if you requested getFullContigSequence, instead of a getActiveSequences,
 you could do your sequence analysis and import the features into the database
 using mapFeature, regardless of being able to observe the mapped results.


=cut


sub mapFeature {
	my ($self, $message) = @_;
	my ($contig, $start, $stop, $feature, $source, $strand, $frame, $score, $msg, $import);
	if ($message =~ /<import>/){
		$import = 1;
		$message = (($message =~ /<import>(.*?)<\/import>/) && $1);
		unless ($message) {print "failed: improperly formatted string\n";return &FAILED("Improperly formatted import string")}
	} else {
		$import = 0
	}
	#print "\n\n$message\n\n";
	if ($message =~ /<contig>(.*?)<\/contig>/){$contig = $1};
	if ($message =~ /<start>(.*?)<\/start>/){$start = $1};
	if ($message =~ /<end>(.*?)<\/end>/){$stop = $1};
	if ($message =~ /<feature>(.*?)<\/feature>/){$feature = $1};
	if ($message =~ /<source>(.*?)<\/source>/){$source = $1};
	if ($message =~ /<strand>(.*?)<\/strand>/){$strand = $1};
	if ($message =~ /<frame>(.*?)<\/frame>/){$frame = $1};
	if ($message =~ /<score>(.*?)<\/score>/){$score = $1};
	if ($message =~ /<attributes>(.*?)<\/attributes>/){$msg = $1};

	# the case below is going to happen often enough that
	# we might as well fix it here rather than punish the user for not knowing GFF
	unless (($strand eq "+") || ($strand eq "-")){
		if ($strand =~ /-/){$strand = "-"}
		elsif ($strand =~ /\+/){$strand = "+"}
		elsif ($strand =~ /1/){$strand = "+"}
		else {$strand = "."}
	}
	
	unless (defined $score){$score = "."}
	unless (defined $frame){$frame = "."}
	
	
	# print "$contig && $start $stop && $feature && $source && $strand && $frame && $score\n";
	return &FAILED("Not properly formatted Genquire GFF") unless (defined($contig) && defined($start) && defined($stop) && defined($feature) && defined($source) && defined($strand) && defined($frame) && defined($score));
	
	my $MAP;	
	foreach my $Map(@{$self->PARENT->MapWindows}){
		my ($SeqCanvas, $stt, $stp, $ctg, $MapWindow) = @{$Map}; # only SeqCanvas is interesting here...
		my $ctgid = $SeqCanvas->MapSeq->primary_id; # based on MapSeq->primary_id (contig::start-length)
		if (($ctgid == $contig)){ # find the appropriate open map if there is one, 
			$MAP = $SeqCanvas;  # grab it
		}
	}
	my $feat; my $newcontig; my $is_map = 0; # is_contig is a boolean saying whether this is a full contig or an active map
	# they may a "pure" contig analysis or an open MapSeq contig analysis
	# These can be distingished by the name of the contig...
	# print "\n\ncontig *$contig* ";
	if ($contig =~ /(.*?)::(\d+)\-\d+/){  # this is a MapSeq, so the startand stop need to be translated into real contig coords.
		#print "matched!!\n";
		$newcontig = $1;
		my $cstart = $2;		# grab the c ontig start coordinate
		$start = $cstart + $start - 1;  # and add them -1 off-by-one be damned!
		$stop = $cstart + $stop - 1;  # same with stop
		$is_map = 1;  # flag it as a MapType
	}
	
	if ($import) {
		if ($is_map  && !$MAP){return}  # if this doesn't appear to be an open map, but the name has that form, then presume that the client has buggered up.
		# but, we *can* import features into the database that are not masquerading as MapSequences
		# since tese might represent legitimate contigs.
		my $gff_string = "$newcontig\t$source\t$feature\t$start\t$stop\t$score\t$strand\t$frame\t$msg";
		$feat = $self->PARENT->load_gff_string($gff_string);  # this creates a Genquire feature object, and loads it into the database
		$feat->par_pos($MAP->MapSeq->bc_start+$MAP->MapSeq->start-1);
		unless ($feat){return &FAILED("Failed to import feature into the database")}
		unless ($is_map && $MAP) {return 1}  # check again if it is a map open to draw on, and if so, then go ahead
		$MAP->mapFeatures("draft", [$feat]);  # and map the feature
		return;
	} else {
		# if they dont want to import, then we only care about it having map open
		unless ($MAP) {return}
		my $gff_string = "$contig\t$source\t$feature\t$start\t$stop\t$score\t$strand\t$frame\t$msg";
		#print "GFF was:  $gff_string\n";
		my $contig_id=$self->PARENT->context->get_contig_id($newcontig);
		my %taghash;
		($contig, $source, $feature, $start, $stop, $score, $strand, $frame, %taghash)
		  =$self->PARENT->parse_gff_string($gff_string);    #this is called just to parse $msg
		require GQ::Server::Feature::inMemory;
		$feat = GQ::Server::Feature::inMemory->new(
			context      => $self->PARENT->context,
			lockid       => $MAP->MapSeq->lockid,
			contig_id    => $contig_id,
			contig_start => $start,
			contig_stop  => $stop,
			source_tag   => $source,
			primary_tag  => $feature,
			score        => $score,
			strand       => $strand,
			frame        => $frame,
			length 		=> $stop-$start + 1,
			name		=> "Imported_From_Plugin_$source",
			tagvalues    => \%taghash,
												 );
		$feat->par_pos($MAP->MapSeq->bc_start+$MAP->MapSeq->start-1);
			
		unless ($feat){return &FAILED("Failed to create in-memory feature")};
		$MAP->mapFeatures("draft", [$feat]);	
	}
	return 1;
}

=head2 defaultDisplay

 request:
	any unrecognized request,
	may direct to a file by: <html_file>$full_path</html_file>
 return: <..>1</..>
 function: any request sent to Genquire without a reconized request tag
 is sent to the default web browser to be displayed

=cut

sub defaultDisplay {
	my ($self, $message) = @_;
	my $browser = $self->PARENT->BROWSER;
	my $file = (($message =~ /<html_file>(.*?)<\/html_file>/) && $1);
	if ($file){
		system "$browser $file";
		return 1;
	} else {
		my $id = $self->PARENT->_next_unique_index;
		my $tmp = $self->PARENT->TEMP_DIR;
		open OUT, ">$tmp/GQhtml$id.html";
		if ($message =~ /<REQUEST>/){$message = (($message =~ /<REQUEST>(.*?)<\/REQUEST>/s) && $1);}  # strip out request line to get to teh HTML if it is there
		print OUT $message;
		close OUT;
		system "$browser $tmp/GQhtml$id.html";
	}
}

sub FAILED {
	my ($reason) = @_;
	my $return = "<FAILED>$reason</FAILED>";
	return $return;
}

sub _addSlash {
	my ($path) = @_;
	if ($path =~ /\\/){
		if (!($path =~ /\\$/)){$path = $path . "\\"}
	} else {
		if (!($path =~ /\/$/)){$path = $path . "\/"}
	}
	#print "path is $path\n";
	return $path;
}
sub AUTOLOAD {
    no strict "refs";
    my ($self, $newval) = @_;

    $AUTOLOAD =~ /.*::(\w+)/;

    my $attr=$1;
    if ($self->_accessible($attr,'write')) {

	*{$AUTOLOAD} = sub {
	    if (defined $_[1]) { $_[0]->{$attr} = $_[1] }
	    return $_[0]->{$attr};
	};    ### end of created subroutine

###  this is called first time only
	if (defined $newval) {
	    $self->{$attr} = $newval
	}
	return $self->{$attr};

    } elsif ($self->_accessible($attr,'read')) {

	*{$AUTOLOAD} = sub {
	    return $_[0]->{$attr} }; ### end of created subroutine
	return $self->{$attr}  }


    # Must have been a mistake then...
    croak "No such method: $AUTOLOAD";
}

DESTROY {

	my ($self) = @_;
	undef $self;

}


1;
	
