# Transcript module for Workbench II

# by David Block <dblock@gene.pbir.nrc.ca>

# Copyright David Block and NRC-PBI

# POD Documentation - main docs before the code

=head1 NAME

Transcript - Transcript Object for Workbench II

=head1 SYNOPSIS

=head2 Loading Transcripts from the database

Loading Transcripts from the database from chromosome $chr_id
where the Transcript overlaps the region between $start and $stop:

use Transcript;

$hash_of_Transcripts is a reference to a hash of Transcripts.
$par_pos is the chromosomal position of the start of the sequence
region you are interested in (often the same as $start).  It is used to
define the relative location of the Transcript within that sequence.

The keys of $hash_of_Transcripts will be the ids of the Transcripts, and the
values will be the Transcript objects themselves.

Alternatively, finding all Transcripts from the database that have
length less than 1000:

use Transcript;


=head2 Creating a new Transcript:

use Transcript;

my $Transcript=Transcript->new(context=>$context,
		   features=>\@list_of_features,
                   comments=>$comments
                  );

=head2 Deleting a Transcript:

$Transcript->delete_from_db;

=head2 Finding out about a Transcript:

$start=$Transcript->start;

returns the start of the Transcript relative to its parent sequence ($par_pos).

$end=$Transcript->end;

returns the end of the Transcript relative to its parent sequence ($par_pos).

$absstart=$Transcript->absstart;

returns the Transcript's start position on the chromosome.

$absstop=$Transcript->absstop;

returns the Transcript's end position on the chromosome.

$strand=$Transcript->strand;

returns the Transcript's strand (+1,-1,0)

$gff_strand=$Transcript->gff_strand;

returns the Transcript's strand (+,-,.)

$length=$Transcript->length;

returns the Transcript's length.

=head2 Finding out about a Transcript's Features:

The List of features in a hash:

$hash_of_features=$Transcript->features;

An individual feature:

$feature=$Transcript->feature($feature_id);

Adding a feature:

$Transcript->features($feature);

=head1 DESCRIPTION

Transcript is a collection of features that has been 'anointed' as a real Transcript
by an annotator at some point.  It lives in the database in the Feature
table.  It is a GenericFeature, which means it implements Bio::SeqFeatureI
as well as all the methods common to Feature, Gene.

=head1 FEEDBACK

Like it or lump it, report to dblock@gene.pbi.nrc.ca.  Feel free to add
to the docs available at
http://bioinfo.pbi.nrc.ca/dblock/wiki

=head2 Reporting Bugs

Email the author with any bug reports.

=head1 AUTHOR - David Block

email dblock@gene.pbi.nrc.ca

=cut

package GQ::Server::Transcript;
$VERSION = 1.00;
use strict;

use vars qw( $AUTOLOAD @ISA);  #Keep 'use strict' happy

use Carp;
use DBI;

use GQ::Server::Container;
use GQ::Root;

@ISA = qw(GQ::Server::Container Bio::SeqFeature::Gene::Transcript);


{
    #Encapsulated class data
    #replace with your column names, and sensible defaults

                    #       DEFAULT      ACCESSIBILITY
    GQ::Root->create();
}

sub _db_keys {
    my ($self)=@_;
    my @list;
    foreach ($self->SUPER::_db_keys) {
	push @list,$_ unless /^id$/;
    }

    push @list,'id';
    return @list;
}


### write in any custom subroutines here
sub _initialize {
    my ($self) = @_;

    $self->adaptor($self->context->adaptor($self));

    ###  This section inserts the object's data into the database
    ###  only if the new method was called by the class, i.e.
    ###  not as a copy of a previous object,  OR
    ###  if the caller does not have an id at present
    ###  which it would if it were loaded from the database (see sub load)
    ######NEW ARGUMENT load true if you want to force loading into the db
    if (not $self->{id} ) {

	die "Must Acquire Lock On This Region To Proceed.\n"
	  if $self->context->checklock(lockid       => $self->lockid,
				       contig_id    => $self->contig_id,
				       contig_start => $self->contig_start,
				       contig_stop  => $self->contig_stop);
	
	$self->{id}=$self->adaptor->insert_new(keys  =>[$self->_db_keys],
					       values=>[$self->_db_values],
					      );
	$self->last_modified;   #put the db last_modified timestamp value into the object
    }

    $self->type($self->primary_tag) unless $self->type;
    $self->flesh_out_details if $self->features;
    $self->insert_tag_values if defined $self->tagvalues;
    return $self;
}

sub remove_feature {
    my ($self,$feature)=@_;
    return $self if $self->access eq 'ro';       #don't mess with read-only transcripts
    $self->adaptor->remove_feature($self,$feature);  #remove long-term connection
    delete $self->{features}{$feature->id};          #remove short-term connection
    if ($self->parent && $self->parent->is_feature_orphan($self,$feature)) {  #are there any other copies
	$self->entire_seq->add_SeqFeature($feature);  #tie $feature directly to the sequence object so it will be redrawn
    }                                                 #otherwise let the other transcript maintain the reference
    if (not $self->features) {                      #if there are no features
	if ($self->parent){$self->parent->remove_transcript($self)}    #I am no longer part of my parent gene
	$self->delete_from_db;                      #I have no reason for existence
	undef $self;                                #good-bye!
    } else {
	$self->flesh_out_details;        #I shall live on, albeit shrunken
    }
    return $self;
}

######TranscriptI compliance:

sub _add {
    my ($self, $fea, $type) = @_;
    if(! $fea->isa('Bio::SeqFeatureI') ) {
	$self->throw("$fea does not implement Bio::SeqFeatureI");
    }
    
    if (! $fea->isa('GQ::Server::GenericFeature')) {
	$self->throw("$fea is not a Genquire-type feature");
    }
    
    if(! $fea->isa($type) ) {
	$fea->add_to_isa($type);
    }
    
    $self->features($fea);
}

sub _flush {
    my ($self, $type, $pri) = @_;
    my @cut = $self->get_unordered_feature_type($type, $pri);
    foreach (@cut) {
	$self->{features}{$_->id}->delete_from_db;
	delete $self->{features}{$_->id};
    }
}

sub features_ordered {
    my ($self)=@_;
    return $self->_stranded_sort($self->features);
}


1;





