package edu.hust.go.model;

/**
 * <p>Title: GO4J</p>
 * <p>Description: </p>
 * <p>Copyright: Copyright (c) 2005</p>
 * <p>Company: </p>
 * @author GQ Zhang
 * @version 1.0
 */
import org._3pq.jgrapht.graph.SimpleDirectedGraph;
import org._3pq.jgrapht.edge.DirectedEdge;
import java.util.Vector;
import java.util.HashMap;
import java.util.HashSet;
import java.util.ArrayList;

import edu.hust.go.term.GO_term;

/**
 * A graph model is constructed by the connection in GO terms. It can be used to
 * construct GUI by edu.hust.go.gui.GraphCreator.
 */

public class GoGraphModel {
    private final String[] rootKeys = {
        "GO:0008150", "GO:0005575", "GO:0003674"};
    private HashMap rootVertexesMap;
    private HashSet obsoleteSet;
    private SimpleDirectedGraph graph;

    /**@param map the key is Go ids, the value is GO terms.
     * @throws InvalidGoDefinitionException there are problems in GO definition
     * file, such as one id is connected to another id, while the latter is not
     * exist in fact. Please make sure that the definition file comes from GO
     * official site: <a href="http://www.geneontology.org">www.geneontology.org</a>
     *  or <a href="ftp://ftp.geneontology.org">ftp.geneontology.org</a>.
     */
    public GoGraphModel(HashMap map) throws GoException {
        graph = init(map);
        rootVertexesMap = new HashMap();
        for (int i = 0; i < rootKeys.length; i++) {
            Object root = map.get(rootKeys[i]);
            if (root == null) {
                System.out.println(rootKeys[i] + "\t**\t" + map.size());
            }
            rootVertexesMap.put(root, new Integer(GraphPath.getNodeAllChildren(graph, root).size()));
        }
        obsoleteSet = removeObsolete();
    }

    /**organize all go ids in goMap as a graph
     * @param goMap
     * @return
     * @throws InvalidGoDefinitionException
     */
    private SimpleDirectedGraph init(HashMap goMap) throws GoException {
        SimpleDirectedGraph g = new SimpleDirectedGraph();
        GO_term targetTerm = null;
        GO_term sourceTerm = null;
        try {
            Object[] nodes = goMap.values().toArray();
            for (int i = 0; i < nodes.length; i++) {
                g.addVertex(nodes[i]);
            }
            for (int i = 0; i < nodes.length; i++) {
                targetTerm = (GO_term) nodes[i];
                Vector parents = targetTerm.getIs_a();
                if (parents != null) {
                    for (int j = 0; j < parents.size(); j++) {
                        sourceTerm = (GO_term) goMap.get(parents.get(j));
                        g.addEdge(new MyEdge(sourceTerm, targetTerm));
                    }
                }
                parents = targetTerm.getPart_of();
                if (parents != null) {
                    for (int j = 0; j < parents.size(); j++) {
                        sourceTerm = (GO_term) goMap.get(parents.get(j));
                        g.addEdge(new MyEdge(sourceTerm, targetTerm));
                    }
                }
            }
        }
        catch (Exception e) {
            System.out.println("The problems exists in!" + targetTerm + "**" + sourceTerm + "\n" + e);
            GoException exc = new GoException(sourceTerm, targetTerm);
            g = null;
            throw exc;
        }
        return g;
    }

    /**remove obsolete go ids in GoMap
     * @return all obsolte go ids
     */
    private HashSet removeObsolete() {
        HashSet _obsoleteSet = new HashSet();
        Object[] vertexs = graph.vertexSet().toArray();
        int counter = 0;
        for (int i = 0; i < vertexs.length; i++) {
            if (graph.inDegreeOf(vertexs[i]) == 0 && rootVertexesMap.get(vertexs[i]) == null) {
                _obsoleteSet.add(vertexs[i]);
                graph.removeVertex(vertexs[i]);
            }
        }
        return _obsoleteSet;
    }

    /**extract sub graph from GO graph, the sub graph contains a set of target
     * go terms , and related in the pathway from the root to its leaf nodes, for
     * example, one pathway is A->B->C->Target->D->E->F, while A is root and F is
     * leaf, and suppose parentLevel and childLevel are both 1, then the nodes C,
     * Taraget, and D are extracted intio the subGraph, the the related edges are
     * extracted into the  subGraph too.
     * @param goTerms query terms
     * @param parentLevel the depth from one target term to root
     * @param childLevel the depth from one target term to leaf
     * @return the extracted sub graph
     */
    public SimpleDirectedGraph buildSubGraph(ArrayList goTerms, int parentLevel, int childLevel) {
        SimpleDirectedGraph subGraph = new SimpleDirectedGraph();
        if (parentLevel == 0) {
            getAllSubGraph(goTerms, subGraph, true);
        }
        else if (parentLevel > 0) {
            getPartSubGraph(goTerms, subGraph, parentLevel, true);
        }
        if (childLevel == 0) {
            getAllSubGraph(goTerms, subGraph, true);
        }
        else if (childLevel > 0) {
            getPartSubGraph(goTerms, subGraph, childLevel, false);
        }

        return subGraph;
    }

    /**get the common parents in the pathway of target go terms.
     * Notice: the common parents in the pathways of one target node is not common
     * parents of all target nodes.
     * @param goTerms target go terms
     * @return the common parents
     */
    public HashSet getCommonParents(ArrayList goTerms) {
        HashMap map = new HashMap();
        HashSet commonSet = new HashSet();
        for (int i = 0; i < goTerms.size(); i++) {
            Object[][] nodePath = GraphPath.getPath2Root(graph, goTerms.get(i));
            GraphPath.getCommonElements(nodePath, map);
        }
        Object[] commonObjs = map.keySet().toArray();
        for (int i = 0; i < commonObjs.length; i++) {
            int value = ( (Integer) map.get(commonObjs[i])).intValue();
            if (value > 1) {
                commonSet.add(commonObjs[i]);
            }
        }
        return commonSet;
    }

    /** @see #evalSimilarity(GO_term node1,GO_term node2, byte type)
     */
    public float evalSimilarity(GO_term node1, GO_term node2) {
        return Similarity.evaluate(this, node1, node2, Similarity.CAO);
    }

    /** evaluate the similarity of two nodes in directed acylic graph
     * @param node1 one of the two nodes
     * @param node2 one of the two nodes
     * @param type one of the following five values.
     *  <br/>Similarity.CAO: basen on information content
     *  <br/>Similarity.LIN: basen on information content, see <a href="http://citeseer.ist.psu.edu/cache/papers/cs/1049/ftp:zSzzSzftp.cs.umanitoba.cazSzpubzSzlindekzSzpaperszSzsim.pdf/an-information-theoretic-definition.pdf">
     *  Lin D. An information-theoretic definition of similarity. In:  Proceedings
     *  of 15th International Conference on Machine Learning, San Francisco US, 1998, 296-304</a>
     *  <br/>Similarity.RESNIK: based on information content, see <a href="http://citeseer.ist.psu.edu/cache/papers/cs/2124/http:zSzzSzwww.cs.umbc.eduzSz891zSzresnik.pdf/resnik95using.pdf">
     *  Resnik P. Using information content to evaluate semantic similarity in a taxonomy.
     *  In: Proceedings of the 14th International Joint Conference on Artificial
     *  Intelligence, Montreal Canada, 1995, 448-453</a>
     *  <br/>Similarity.ZZL: refined ZZL based on distance, the LCA(latest common ancestor)
     *  is one with maximal information content in the of the same distance LCA
     *  <br/>Similarity.oldZZL: based on distance, see Zhong <a href="http://apex.sjtu.edu.cn/docs/iccs2002.pdf">JW, Zhu HP, Li JM, Yu Y.
     *  Conceptual Graph Matching for Semantic Search. In: Proceedings of the 10th
     *  International Conference on Conceptual Structures: Integration and Interfaces,
     *  2002, 92-196</a>
     * @return the semanic similarity value
     */
    public float evalSimilarity(GO_term node1, GO_term node2, byte type) {
        return Similarity.evaluate(this, node1, node2, type);
    }

    /**
     * @param node the target node
     * @return all pathways between the target node and its corresponing root.
     * Each element is a GO term.
     */
    public Object[][] getPathsOfNode2Root(GO_term node) {
        return GraphPath.getPath2Root(graph, node);
    }

    /**Tell whether two nodes are in one pathway
     * @param node1 one of the two nodes
     * @param node2 one of the two nodes
     * @return true means the two nodes are in one pathway, false means no.
     */
    public boolean inOnePathway(GO_term node1, GO_term node2) {
        return GraphPath.related(graph, node1, node2);
    }

    /** get pathway between two Go terms
     * @param node1
     * @param node2
     * @return there is only one key and one value.<bf/>
     *  The key is a HashSet of common ancestors, while the value is a Object[][] of
     *  common pathways.
     */
    public HashMap getPathsBetweenNodes(GO_term node1, GO_term node2) {
        return GraphPath.getPath2Node(graph, node1, node2);
    }

    private void getAllSubGraph(ArrayList goids, SimpleDirectedGraph subGraph, boolean toParent) {
        HashSet nodeSet = new HashSet(goids);
        if (toParent) {
            for (int i = 0; i < goids.size(); i++) {
                nodeSet.addAll(GraphPath.getNodeAllParents(graph, goids.get(i)));
            }
        }
        else {
            for (int i = 0; i < goids.size(); i++) {
                nodeSet.addAll(GraphPath.getNodeAllChildren(graph, goids.get(i)));
            }
        }
        subGraph.addAllVertices(nodeSet);
        Object[] nodes = nodeSet.toArray();
        if (toParent) {
            for (int i = 0; i < nodes.length; i++) {
                subGraph.addAllEdges(graph.incomingEdgesOf(nodes[i]));
            }
        }
        else {
            for (int i = 0; i < nodes.length; i++) {
                subGraph.addAllEdges(graph.outgoingEdgesOf(nodes[i]));
            }
        }
    }

    private void getPartSubGraph(ArrayList goids, SimpleDirectedGraph subGraph, int level, boolean toParent) {
        HashSet nodeSet = new HashSet(goids);
        subGraph.addAllVertices(goids);
        for (int i = 0; i < level; i++) {
            Object[] vectexes = nodeSet.toArray();
            nodeSet.clear();
            for (int m = 0; m < vectexes.length; m++) {
                Object[] edges;
                if (toParent) {
                    edges = graph.incomingEdgesOf(vectexes[m]).toArray();
                }
                else {
                    edges = graph.outgoingEdgesOf(vectexes[m]).toArray();
                }
                for (int n = 0; n < edges.length; n++) {
                    Object obj;
                    if (toParent) {
                        obj = ( (MyEdge) edges[n]).getSource();
                    }
                    else {
                        obj = ( (MyEdge) edges[n]).getTarget();
                    }
                    nodeSet.add(obj);
                    subGraph.addVertex(obj);
                    subGraph.addEdge( (MyEdge) edges[n]);
                }
            }
        }
    }

    /**the roots are defined as the nodes which has no incoming edges.
     * @param subGraph
     * @return
     */
    public static Object[] getRoots(SimpleDirectedGraph subGraph) {
        Object[] vertexs = subGraph.vertexSet().toArray();
        Vector subRoot = new Vector();
        for (int i = 0; i < vertexs.length; i++) {
            if (subGraph.incomingEdgesOf(vertexs[i]).size() == 0) {
                subRoot.add(vertexs[i]);
            }
        }
        return subRoot.toArray();
    }

    /**the leave are defindes as the nodes which has no outgoing edges.
     * @param subGraph
     * @return
     */
    public static Object[] getLeave(SimpleDirectedGraph subGraph) {
        Object[] vertexs = subGraph.vertexSet().toArray();
        Vector leave = new Vector();
        for (int i = 0; i < vertexs.length; i++) {
            if (subGraph.outgoingEdgesOf(vertexs[i]).size() == 0) {
                leave.add(vertexs[i]);
            }
        }
        return leave.toArray();
    }

    /**get the whole GO graph
     * @return
     */
    public SimpleDirectedGraph getGraph() {
        return graph;
    }

    /**get the roots of the whole GO graph
     * @return the key is go ids, the value is go terms
     */
    public HashMap getRootVertexesMap() {
        return rootVertexesMap;
    }

    /**get obsolete terms in the go definition
     * @return
     */
    public HashSet getObsoleteSet() {
        return obsoleteSet;
    }

    /**Construct a map between GO id and GO terms
     * @param definition GO defineion file
     * @param isXml the defintion file type, XML or OBO.
     * @return GO ids map, including obsolete GO ids
     */
    public static HashMap loadGoDefinition(String definition, byte defType)throws  GoException{
        GoParser parser = null;
        HashMap goMap = null;
        try {
            switch (defType) {
                case GoParser.OBO:
                    parser = new OboGoParser(definition);
                    break;
                case GoParser.OBOXML:
                    parser = new OboXmlGoParser(definition);
                    break;
                case GoParser.RDFXML:
                    parser = new RdfXmlGoParser(definition);
                    break;
                case GoParser.OWL:
                    parser = new OwlGoParser(definition);
                    break;
                default:
                    ;
            }
            goMap = parser.getTermMap();
        }
        catch (Exception e) {goMap = null;}
        if(parser==null || goMap==null || goMap.size()==0){
            GoException exe = new GoException(definition, defType);
            throw exe;
        }
        return goMap;
    }
}
