package edu.hust.go.model;

/**
 * <p>Title: GO4J</p>
 * <p>Description: Go api in Java</p>
 * <p>Copyright: Copyright (c) 2005</p>
 * <p>Company: Life Science and Technology, HUST, China</p>
 * @author GQ Zhang
 * @version 1.0
 */
import java.util.HashSet;
import java.util.HashMap;
import java.util.Vector;

import org._3pq.jgrapht.graph.SimpleDirectedGraph;

/**
 *
 */
public class Similarity {
    public static final byte CAO = 0;
    public static final byte LIN = 1;
    public static final byte RESNIK = 2;
    public static final byte ZZL = 3;
    public static final byte oldZZL = 4;

    /** evaluate the similarity of two nodes in directed acylic graph
     * @param graphModel the graph that contains the two nodes
     * @param obj1 one of the two nodes
     * @param obj2 one of the two nodes
     * @param type Similarity.CAO: basen on information content
     *  Similarity.LIN: basen on information content, see <a href="http://citeseer.ist.psu.edu/cache/papers/cs/1049/ftp:zSzzSzftp.cs.umanitoba.cazSzpubzSzlindekzSzpaperszSzsim.pdf/an-information-theoretic-definition.pdf">
     *  Lin D. An information-theoretic definition of similarity. In:  Proceedings
     *  of 15th International Conference on Machine Learning, San Francisco US, 1998, 296-304</a>
     *  Similarity.RESNIK: based on information content, see <a href="http://citeseer.ist.psu.edu/cache/papers/cs/2124/http:zSzzSzwww.cs.umbc.eduzSz891zSzresnik.pdf/resnik95using.pdf">
     *  Resnik P. Using information content to evaluate semantic similarity in a taxonomy.
     *  In: Proceedings of the 14th International Joint Conference on Artificial
     *  Intelligence, Montreal Canada, 1995, 448-453</a>
     *  Similarity.ZZL: refined ZZL based on distance, the LCA(latest common ancestor)
     *  is one with maximal information content in the of the same distance LCA
     *  Similarity.oldZZL: based on distance, see Zhong <a href="http://apex.sjtu.edu.cn/docs/iccs2002.pdf">JW, Zhu HP, Li JM, Yu Y.
     *  Conceptual Graph Matching for Semantic Search. In: Proceedings of the 10th
     *  International Conference on Conceptual Structures: Integration and Interfaces,
     *  2002, 92-196</a>
     * @return the semanic similarity value
     */
    public static float evaluate(GoGraphModel graphModel, Object obj1, Object obj2, byte type) {
        float rootChild = 0;
        SimpleDirectedGraph graph=graphModel.getGraph();
        Object[][] aPath = GraphPath.getPath2Root(graph,obj1);
        Object[][] bPath = GraphPath.getPath2Root(graph,obj2);
        if (sameRoot(aPath, bPath)) {
            rootChild = ( (Integer) graphModel.getRootVertexesMap().get(aPath[0][aPath[0].length - 1])).floatValue() +
                1;
        }
        else {
            return 0;
        }
        int[] lcaIndexes = findLatestCommonAncestor(aPath, bPath, graph);
        switch (type) {
            case CAO:
                return cao(aPath, bPath, graph, obj1, obj2, rootChild);
            case LIN:
                return lin(graph, bPath[lcaIndexes[0]][lcaIndexes[1]], obj1, obj2, rootChild);
            case RESNIK:
                return (float) -
                    Math.log( (GraphPath.getNodeAllChildren(graph,bPath[lcaIndexes[0]][lcaIndexes[1]]).size() + 1) / rootChild);
            case ZZL:
                return zzl(aPath, bPath, lcaIndexes[0], lcaIndexes[1]);
            case oldZZL:
                return oldZZL(aPath, bPath);
            default:
                return 0;
        }
    }

    private static float oldZZL(Object[][] aPath, Object[][] bPath) {
        Vector aVector = new Vector();
        int max = 0;
        for (int i = 0; i < aPath.length; i++) {
            if (max < aPath[i].length) {
                aVector.clear();
                aVector.add(new Integer(i));
                max = aPath[i].length;
            }
            else if (max == aPath[i].length) {
                aVector.add(new Integer(i));
            }
        }
        Vector bVector = new Vector();
        max = 0;
        for (int i = 0; i < bPath.length; i++) {
            if (max < bPath[i].length) {
                bVector.clear();
                bVector.add(new Integer(i));
                max = bPath[i].length;
            }
            else if (max == bPath[i].length) {
                bVector.add(new Integer(i));
            }
        }
        max = 0;
        for (int i = 0; i < aVector.size(); i++) {
            int M = ( (Integer) aVector.get(i)).intValue();
            for (int j = 0; j < bVector.size(); j++) {
                int N = ( (Integer) bVector.get(j)).intValue();
                int commonNum = 0;
                for (int m = aPath[M].length - 1, n = bPath[N].length - 1; m > -1 && n > -1; m--, n--) {
                    if (aPath[M][m].equals(bPath[N][n])) {
                        commonNum++;
                    }
                    else {
                        break;
                    }
                }
                max = (max < commonNum) ? commonNum : max;
            }
        }
        float d1 = (float) (0.5 / Math.pow(2, aPath[ ( (Integer) aVector.get(0)).intValue()].length - 1));
        float d2 = (float) (0.5 / Math.pow(2, bPath[ ( (Integer) bVector.get(0)).intValue()].length - 1));
        float d0 = (float) (0.5 / Math.pow(2, max - 1));
        return 2 * d0 - d1 - d2;
    }

    private static float zzl(Object[][] aPath, Object[][] bPath, int dim1, int dim2) {
        int size = bPath[dim1].length - dim2;
        HashMap map = new HashMap();
        for (int i = 0; i < aPath.length; i++) {
            int swap = aPath[i].length - size;
            if (swap > -1 && aPath[i][swap].equals(bPath[dim1][dim2])) {
                map.put(new Integer(i), new Integer(swap));
            }
        }
        Object[] keys = map.keySet().toArray();
        int[] indexes = new int[keys.length];
        for (int i = 0; i < keys.length; i++) {
            indexes[i] = ( (Integer) keys[i]).intValue();
            boolean hit = true;
            for (int m = ( (Integer) map.get(keys[i])).intValue(), n = dim2;
                 m < aPath[indexes[i]].length && n < bPath[dim1].length; m++, n++) {
                if (!aPath[indexes[i]][m].equals(bPath[dim1][n])) {
                    hit = false;
                    break;
                }
            }
            if (!hit) {
                indexes[i] = -1;
            }
        }
        int index = -1;
        int maxPathNumber = 0;
        for (int i = 0; i < indexes.length; i++) {
            if (indexes[i] > -1) {
                if (maxPathNumber < aPath[indexes[i]].length) {
                    maxPathNumber = aPath[indexes[i]].length;
                    index = indexes[i];
                }
            }
        }
        if (index < 0) {
            System.out.println("Error!");
            return 0;
        }
        else {
            float d1 = (float) (0.5 / Math.pow(2, aPath[index].length - 1));
            float d2 = (float) (0.5 / Math.pow(2, bPath[dim1].length - 1));
            float d0 = (float) (0.5 / Math.pow(2, size - 1));
            return 2 * d0 - d1 - d2;
        }
    }

    private static float lin(SimpleDirectedGraph graph, Object latestCommonAncestor, Object obj1, Object obj2,
                             float rootChildrenNum) {
        float d0 = (float) Math.log( (GraphPath.getNodeAllChildren(graph, latestCommonAncestor).size() + 1) / rootChildrenNum);
        float d1 = (float) Math.log( (GraphPath.getNodeAllChildren(graph,obj1).size() + 1) / rootChildrenNum);
        float d2 = (float) Math.log( (GraphPath.getNodeAllChildren(graph,obj2).size() + 1) / rootChildrenNum);
        return 2 * d0 / (d1 + d2);
    }

    private static float cao(Object[][] aPath, Object[][] bPath, SimpleDirectedGraph graph, Object obj1, Object obj2,
                                 float rootChildrenNum) {
        HashSet[] sets = {
            new HashSet(), new HashSet()};
        HashSet swapSet = new HashSet();
        for (int i = 0; i < aPath.length; i++) {
            for (int j = 0; j < aPath[i].length; j++) {
                sets[0].add(aPath[i][j]);
            }
        }
        for (int i = 0; i < bPath.length; i++) {
            for (int j = 0; j < bPath[i].length; j++) {
                sets[1].add(bPath[i][j]);
            }
        }

        Object[] values = sets[0].toArray();
        for (int i = 0; i < values.length; i++) {
            if (sets[1].contains(values[i])) {
                swapSet.add(values[i]);
            }
        }
        sets[0].addAll(sets[1]);
        sets[0].removeAll(swapSet);
        sets[1] = swapSet;
        float[] informs = {
            0, 0};
        for (int i = 0; i < sets.length; i++) {
            values = sets[i].toArray();
            for (int j = 0; j < values.length; j++) {
                float swap = (GraphPath.getNodeAllChildren(graph,values[j]).size() + 1) / rootChildrenNum;
                informs[i] += -Math.log(swap);
            }
        }
        return informs[1] / (informs[0] + informs[1]);
    }

    private static boolean sameRoot(Object[][] aPath, Object[][] bPath) {
        return aPath[0][aPath[0].length - 1].equals(bPath[0][bPath[0].length - 1]);
    }

    private static int[] findLatestCommonAncestor(Object[][] aPath, Object[][] bPath, SimpleDirectedGraph graph) {
        if (!sameRoot(aPath,bPath)) {
            return null;
        }
        HashMap map = new HashMap();
        HashSet set = new HashSet();
        for (int i = 0; i < aPath.length; i++) {
            for (int j = 0; j < aPath[i].length; j++) {
                set.add(aPath[i][j]);
            }
        }
        for (int i = 0; i < bPath.length; i++) {
            for (int j = bPath[i].length - 1; j > -1; j--) {
                if (set.contains(bPath[i][j])) {
                    map.put(new Integer(i), new Integer(j));
                }
            }
        }
        Object[] values = map.keySet().toArray();
        int max = 0;
        Vector vector = new Vector();
        for (int i = 0; i < values.length; i++) {
            int swap = bPath[ ( (Integer) values[i]).intValue()].length - ( (Integer) map.get(values[i])).intValue();
            if (max < swap) {
                vector.clear();
                vector.add(values[i]);
                max = swap;
            }
            else if (max == swap) {
                vector.add(values[i]);
            }
        }
        int[] array = new int[vector.size() * 2];
        values=vector.toArray();
        for (int i = 0; i < array.length; i += 2) {
            array[i] = ( (Integer) values[i / 2]).intValue();
            array[i + 1] = ( (Integer) map.get(values[i / 2])).intValue();
        }
        return getMaxInformationIndexes(bPath, array, graph);
    }

    private static int[] getMaxInformationIndexes(Object[][] path, int[] indexes, SimpleDirectedGraph graph) {
        float max = 0;
        int[] dim = {
            -1, -1};
        for (int i = 0; i < indexes.length; i += 2) {
            int swap = GraphPath.getNodeAllChildren(graph,path[indexes[i]][indexes[i + 1]]).size() + 1;
            if (max < swap) {
                dim[0] = indexes[i];
                dim[1] = indexes[i + 1];
                max = swap;
            }
            else if (max == swap) {
                if (path[dim[0]].length < path[indexes[i]].length) {
                    dim[0] = indexes[i];
                    dim[1] = indexes[i + 1];
                }
            }
        }
        return dim;
    }
}