package edu.hust.go.example;

import java.io.File;
import java.io.BufferedReader;
import java.io.InputStreamReader;
import java.io.IOException;
import java.text.DecimalFormat;
import java.util.*;
import java.util.regex.*;

import edu.hust.go.model.*;
import edu.hust.go.model.GoParser.OntologyDefineType;
import edu.hust.go.term.Constants;
import edu.hust.go.term.GO_term;

public class ConsoleExample {
	enum OpType {
		GETSIMILARITY, GETPATHWAY, EXIT
	};

	DecimalFormat df;

	String idPatternStr = null;

	String idPrefixStr = null;

	String connectStr = null;

	OntologyDefineType ontologyType;

	HashMap<String, GO_term> termMap;

	OntologyGraphModel graphModel;

	public ConsoleExample(File ontologyFile, OntologyDefineType _ontologyType, String _idPatternStr,
			String _idPrefixStr, String _connectStr, DecimalFormat _df) {
		ontologyType = _ontologyType;
		idPatternStr = _idPatternStr;
		idPrefixStr = _idPrefixStr;
		connectStr = _connectStr;
		df = _df;
		long begin = System.currentTimeMillis();
		try {
			termMap = OntologyGraphModel.loadGoDefinition(ontologyFile, idPatternStr, idPrefixStr, connectStr,
					ontologyType);
			graphModel = new OntologyGraphModel(termMap);
		} catch (GoException e) {
			e.printStackTrace();
			System.out.println("Fail to create graph model");
			System.exit(-1);
		}
		long end = System.currentTimeMillis();
		System.out.println("It took " + (end - begin) + " ms to load the ontology " + ontologyFile.getName());

	}

	public static void main(String[] args) throws IOException {
		DecimalFormat df = new DecimalFormat("0000000");
		String idPatternStr = null;

		String idPrefixStr = null;

		String connectStr = null;
		int argIndex = 0;
		File ontologyFile = null;
		OntologyDefineType ontologyType = OntologyDefineType.UNKNOWN;
		OpType actionType = null;
		ArrayList<String> idNameList = new ArrayList<String>();
		while (argIndex < args.length) {
			if (args[argIndex].equals("-ontology")) {
				ontologyFile = new File(args[argIndex + 1]);
				argIndex += 2;
			} else if (args[argIndex].equals("-ontologyType")) {
				try {
					ontologyType = OntologyDefineType.valueOf(args[argIndex + 1].toUpperCase());
				} catch (IllegalArgumentException e) {
				} finally {
					argIndex += 2;
				}
			} else if (args[argIndex].equals("-getSim")) {
				actionType = OpType.GETSIMILARITY;
				argIndex++;
			} else if (args[argIndex].equals("-getPathway")) {
				actionType = OpType.GETPATHWAY;
				argIndex++;
			} else if (args[argIndex].equals("-term")) {
				idNameList.add(args[argIndex + 1]);
				argIndex += 2;
			} else if (args[argIndex].equals("-idPatternStr")) {
				idPatternStr = args[argIndex + 1];
				argIndex += 2;
			} else if (args[argIndex].equals("-idPrefixStr")) {
				idPrefixStr = args[argIndex + 1];
				argIndex += 2;
			} else if (args[argIndex].equals("-connectStr")) {
				connectStr = args[argIndex + 1];
				argIndex += 2;
			}
		}
		BufferedReader in = new BufferedReader(new InputStreamReader(System.in));
		while (true) {
			if (ontologyFile == null) {
				System.out.println("Please input the ontology file:");
				ontologyFile = new File(in.readLine());
				if (!ontologyFile.exists() || !ontologyFile.isFile() || ontologyFile.length() == 0) {
					ontologyFile = null;
				}
			} else if (ontologyType.equals(OntologyDefineType.UNKNOWN)) {
				System.out.println("Please choose the ontology type: ");
				System.out.println("\t" + OntologyDefineType.OBO.name() + ", default;");
				System.out.println("\t" + OntologyDefineType.OBOXML.name() + ";");
				System.out.println("\t" + OntologyDefineType.RDFXML.name() + ";");
				System.out.println("\t" + OntologyDefineType.OWL.name() + ".");
				String str = in.readLine();
				if (str.trim().length() == 0)
					ontologyType = OntologyDefineType.OBO;
				else {
					try {
						ontologyType = OntologyDefineType.valueOf(str.toUpperCase());
					} catch (java.lang.IllegalArgumentException e) {
						ontologyType = OntologyDefineType.UNKNOWN;
					}
				}
			} else
				break;
		}
		if (idPatternStr == null) {
			idPatternStr = Constants.geneOntologyIdPatternStr;
		}
		if (idPrefixStr == null) {
			idPrefixStr = Constants.geneOntologyPrefix;
		}
		if (connectStr == null) {
			if (ontologyType.equals(OntologyDefineType.OBO))
				connectStr = Constants.geneOntologyOboConnector;
			else if (ontologyType.equals(OntologyDefineType.OWL))
				connectStr = Constants.geneOntologyOwlConnector;
			else if (ontologyType.equals(OntologyDefineType.OBOXML))
				connectStr = Constants.geneOntologyOboXmlConnector;
			else if (ontologyType.equals(OntologyDefineType.RDFXML))
				connectStr = Constants.geneOntologyRdfXmlConnector;
			else
				connectStr = Constants.geneOntologyOboConnector;
		}
		ConsoleExample app = new ConsoleExample(ontologyFile, ontologyType, idPatternStr, idPrefixStr, connectStr,
				df);

		while (true) {
			if (actionType == null) {
				System.out.println("Please choose the action type:");
				System.out.println("\t" + OpType.GETSIMILARITY.name()
						+ ", calculate the semantic similarity between to ontology nodes;");
				System.out.println("\t" + OpType.GETPATHWAY.name()
						+ ", calculate the semantic similarity between to ontology nodes;");
				System.out.println("\t" + OpType.EXIT.name() + ", exit.");
				try {
					actionType = OpType.valueOf(in.readLine().toUpperCase());
				} catch (java.lang.IllegalArgumentException e) {
					actionType = null;
				}
			} else if (actionType.equals(OpType.GETSIMILARITY)) {
				GO_term[] terms=app.getGetTerm(2, in, idNameList);
				app.getSimilarity(terms[0],terms[1], Similarity.CAO);
			} else if (actionType.equals(OpType.GETPATHWAY)) {
				GO_term[] terms=app.getGetTerm(1, in, idNameList);
				app.getPathway(terms[0]);
			} else if (actionType.equals(OpType.EXIT)) {
				System.out.println(app.termMap.size() + " Node in the ontology file");
				String[] ids = app.termMap.keySet().toArray(new String[0]);
				Arrays.sort(ids);
				System.out.println(ids.length + " * " + ids[0] + " * " + ids[ids.length - 1]);
				break;
			}
		}
		in.close();
	}

	GO_term[] getGetTerm(int termNum, BufferedReader in, ArrayList<String> idNameList) throws IOException {
		ArrayList<GO_term> termList = new ArrayList<GO_term>();
		while (termList.size() < termNum) {
			System.out.println(termNum + " node names are necessary to calculate semantic similarity");
			System.out.println("Please input " + (termNum - termList.size()) + " term name(s) again");
			String[] array = in.readLine().split("[, ]+");
			for (int i = 0; i < array.length; i++)
				idNameList.add(array[i]);
			for (int i = 0; i < idNameList.size(); i++) {
				GO_term term = transformString2Term(idNameList.get(i));
				if (term != null)
					termList.add(term);
			}
			idNameList.clear();
		}
		return termList.toArray(new GO_term[0]);
	}

	GO_term transformString2Term(String idName) {
		Pattern p = Pattern.compile(idPatternStr);
		Matcher m = p.matcher(idName);
		if (m.find()) {
			idName = df.format(m.group());
			return termMap.get(idName);
		} else {
			return null;
		}
	}

	void getSimilarity(GO_term term1, GO_term term2, byte methodType) {
		if (graphModel == null || term1 == null || term2 == null) {
			return;
		}
		double value = graphModel.evalSimilarity(term1, term2, methodType);
		System.out.println("The semantic similarity between ");
		System.out.print(term1.getId() + " " + term2.getId() + " is: " + value);
	}

	void getPathway(GO_term term) {
		if (graphModel == null || term == null) {
			return;
		}
		Object[][] objs = graphModel.getPathsOfNode2Root(term);
		System.out.print("The pathways between " + term.getId() + " and the root is:");
		for (int i = 0; i < objs.length; i++) {
			System.out.print("\nPath " + (i + 1) + ":\t");
			for (int j = 0; j < objs[i].length; j++) {
				System.out.print(((GO_term) objs[i][j]).getId() + " ");
			}
		}
	}

	void getPathway(GO_term term1, GO_term term2) {
		if( term1 == null || term2 == null) {
			return;
		}
		HashMap map = graphModel.getPathsBetweenNodes(term1, term2);
		HashSet commonSet = (HashSet) map.keySet().toArray()[0];
		Object[] commomTerms = commonSet.toArray();
		System.out.println("The common ancestors between " + term1.getId() + " and " + term2.getId() + " are:\n\t");
		for (int i = 0; i < commomTerms.length; i++) {
			System.out.print((((GO_term) commomTerms[i])).getId() + " ");
		}
		System.out.println("\nthe commonPathways are:");
		Object[][] values = (Object[][]) map.get(commonSet);
		for (int i = 0; i < values.length; i++) {
			System.out.print("\nPath " + (i + 1) + ":\t");
			for (int j = 0; j < values[i].length; j++) {
				System.out.print(((GO_term) values[i][j]).getId() + " ");
			}
		}
	}

	static void printUsage() {
		System.out.println("Usage: java edu.hust.go.example.ConsoleExample -option");
		System.out.println("-ontologyFile ./data/gene_ontology.obo.2005-06-01, necessary!");
		System.out.println("-ontologyType OBO, the value can be OBO, OBOXML, RDFXML or OWL, default value is OBO");
		System.out.println("-getSim, calcualte the similarity between two ontology nodes.");
		System.out.println("\tthus it should be followed by two term parameter");
		System.out.println("-getPathway, infer the pathway between the target node to root node.");
		System.out.println("\tthus it should be followed by one term parameter");
		System.out.println("-term 0001, the specified term ID, it can appear more than 1 times.");
	}
}
