##    Copyright (C) 2006 Djork-Arne Clevert (okko@clevert.de),
##       				 Sepp Hochreiter (hochreit@bioinf.jku.at),
##                       Klaus Obermayer (oby@cs.tu-berlin.de)
##    Berlin University of Technology,
##    Institute for Software Engineering and Theoretical Computer Science 
##    The software is maintained and developed by Djork-Arné Clevert. 
##    We offer a first implementation of the new 
##    ``Factor Analysis for Robust Microarray Summarization'' (FARMS) algorithm.
##    This program is free software; you can redistribute it and/or modify it under 
##    the terms of the GNU General Public License as published by the Free Software 
##    Foundation; either version 2 of the License, or (at your option) any later version. 
##    This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; 
##    without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. 
##    See the GNU General Public License for more details.
##    If you use this library, please cite:
##
##    @article{SeppHochreiter02102006,
##		author = {Hochreiter, Sepp and Clevert, Djork-Arne and Obermayer, Klaus},
##		title = {{A new summarization method for Affymetrix probe level data}},
##		journal = {Bioinformatics},
##		volume = {},
##		number = {},
##		pages = {btl033},
##		doi = {10.1093/bioinformatics/btl033},
##		year = {2006},
##		URL = {http://bioinformatics.oxfordjournals.org/cgi/content/abstract/btl033v1},
##		eprint = {http://bioinformatics.oxfordjournals.org/cgi/reprint/btl033v1.pdf}
##		}

##	@article{INI-Calls:07,
##		author = {Willem Talloen and Djork-Arne Clevert and Sepp Hochreiter and Dhammika Amaratunga and Luc Bijnens and Stefan Kass and Hinrich W.H. Ghlmann},
##		title = {/NI-calls for the exclusion of non-informative genes: a highly effective filtering tool for microarray data},
##		journal = {Bioinformatics},
##		volume = {},
##		number = {},
##		pages = {btm478},
##		doi = {doi:10.1093/bioinformatics/btm478},
##		year = {2007},
##		URL = {http://bioinformatics.oxfordjournals.org/cgi/content/short/btm478v1},
##		eprint = {http://bioinformatics.oxfordjournals.org/cgi/reprint/btm478v1}
##		}





upDate.generateExprSet.methods(c(generateExprSet.methods(), "farms"))
upDate.express.summary.stat.methods(c(express.summary.stat.methods(),"farms"))


exp.farms<-function(object, bgcorrect.method = "none", pmcorrect.method = "pmonly", 
        normalize.method = "quantiles", weight, mu,  weighted.mean, robust,...)
{
if (missing(weight)){weight <- 0.5}
if (missing(mu)){mu <- 0}
if (missing(robust)){robust <- TRUE}
if (missing(weighted.mean)){weighted.mean <- TRUE}

#if(!("farms"%in%express.summary.stat.methods())){
#	upDate.generateExprSet.methods(c(generateExprSet.methods(), "farms"))
#	upDate.express.summary.stat.methods(c(express.summary.stat.methods(),"farms"))
#}
    res <- expresso(object, bgcorrect.method=bgcorrect.method, pmcorrect.method=pmcorrect.method, 
        normalize.method=normalize.method, summary.method = "farms", 
        summary.param=list(weight=weight, mu=mu,  weighted.mean=weighted.mean, robust=robust))
    return(res)
} 


q.farms<-function (object, weight, mu, weighted.mean, robust,...) 
{
if (missing(weight)){weight <- 0.5}
if (missing(mu)){mu <- 0}
if (missing(robust)){robust <- TRUE}
if (missing(weighted.mean)){weighted.mean <- TRUE}
#if(!("farms"%in%express.summary.stat.methods())){
#	upDate.generateExprSet.methods(c(generateExprSet.methods(), "farms"))
#	upDate.express.summary.stat.methods(c(express.summary.stat.methods(),"farms"))
#}


    res <- expresso(object, bgcorrect.method = "none", pmcorrect.method = "pmonly", 
        normalize.method = "quantiles", summary.method = "farms", 
        summary.param=list(weight=weight, mu=mu, weighted.mean=weighted.mean,robust=robust))
    return(res)
}

l.farms<-function (object, weight, mu, weighted.mean, robust, ...) 
{
if (missing(weight)){weight <- 0.5}
if (missing(mu)){mu <- 0}
if (missing(robust)){robust <- TRUE}
if (missing(weighted.mean)){weighted.mean <- TRUE}
#if(!("farms"%in%express.summary.stat.methods())){
#	upDate.generateExprSet.methods(c(generateExprSet.methods(), "farms"))
#	upDate.express.summary.stat.methods(c(express.summary.stat.methods(),"farms"))
#}

    res <- expresso(object, bgcorrect.method = "none", pmcorrect.method = "pmonly", 
        normalize.method = "loess", summary.method = "farms", 
        summary.param=list(weight=weight, mu=mu,  weighted.mean=weighted.mean, robust=robust))
    return(res)
}



generateExprVal.method.farms <- function(probes, weight, mu,  cyc, tol, weighted.mean, robust,...){
	
	if (missing(weight)){weight <- 0.5}	
	
	if (missing(mu)){mu <- 0}
	
	if (missing(tol)){tol <- 0.00001}
	
	if (missing(robust)){robust <- TRUE}
	
	if (missing(cyc)){cyc <- 25}
	
	if (missing(weighted.mean)){weighted.mean <- TRUE}
	
	
	## probes - data matrix
	## weight - hyperparameter default (0.5)
	## mu - hyperparameter default (0)
	## scale - scaling parameter for quantiles- (1.5) and 
	## loess-normalization (2)
	## tol - termination tolerance (default = 0.00001)
	## cyc - maximum number of cycles of EM (default 100)
	## L - factor loadings
	## Ph - diagonal uniqueness matrix
	
	a_old <- 0.5
	n_array <-  ncol(probes)
	n_probes <- nrow(probes)
	
	## log2 probes
	probes <- log2(probes)
	mean.probes <- rowMeans(probes)  ## calculate mean of probes
	centered.probes <- probes - mean.probes
	sd.probes <- sqrt(diag(crossprod(t(centered.probes))) / n_array) ## calculate sd of probes
	if(0 %in% sd.probes){
		index <- which(sd.probes == 0)
		sd.probes[index] <- 1	## avoiding division by zero
		probes <- probes / sd.probes ## standardize probes to variance 1
		x <- t(probes)
		y_v <- colMeans(x)
		#xmean <- cbind(seq(1, 1, length=n_array)) %*% y_v
		xmean <- matrix(y_v, n_array, n_probes, byrow = TRUE)
		X <- x - xmean  ## center data (0 mean)
		XX <- crossprod(X,X) / n_array
		diag(XX)[index] <- 1 ## avoiding division by zero
		}
	else{
		probes <- probes / sd.probes ## standardize probes to variance 1
		x <- t(probes)
		y_v <- colMeans(x)
		#xmean <- cbind(seq(1, 1, length=n_array)) %*% y_v
		xmean <- matrix(y_v, n_array, n_probes, byrow = TRUE)
		X <- x - xmean  ## center data (0 mean)
		XX <- crossprod(X, X) / n_array
	}
	

	XX <- (XX + t(XX)) / 2 ## XX is now positive definit
	XX[which(XX < 0)] <- 0
	diagXX <- diag(XX)
	L <- sqrt(0.75 * diagXX)
	Ph <- diagXX - L^2
	alpha <- weight * n_probes
	bbeta <- mu * alpha
	for (i in 1:cyc){
		
		# E Step
		
		PsiL <- (1 / Ph) * L
		a <- as.vector(1 + crossprod(L, PsiL))
		if((1/a < 0.999) && robust){
		L_robust <- L
		Ph_robust <- Ph
		}
		bar <- PsiL / a
		beta <- t(bar)
		XXbeta <- XX %*% bar
		EZZ <- 1 - beta %*% L + beta %*% XXbeta
		t_XXbeta <- XXbeta + Ph * bbeta
		t_EZZ <- as.vector(EZZ) + Ph * alpha
		
		## M Step
		
		L <- t_XXbeta / t_EZZ
		Ph <- diagXX - XXbeta * L + Ph * alpha * L * (bbeta - L) 
	
		if (sqrt((1/a_old - 1/a)^2) < tol){
			break
		}
		a_old <- 1/a

	}
	
	

	c <- X %*% bar ## hidden variable c - factor
	if(EZZ == 0){
		var_z_scale <- 1 ## avoiding division by zero
	}
	else{
		var_z_scale <- sqrt(EZZ)
	}
	c <- c / as.vector(var_z_scale)
	L <- L * as.vector(var_z_scale)
	PsiL <- (1 / Ph) * L
	a <- as.vector(1 + crossprod(L,PsiL))
	SNR <- 1 / a ## INI-Call
	SIG <- as.vector(crossprod(L, diag(as.vector(1/Ph)))) %*% XX %*% diag(as.vector(1/Ph)) %*% L * a^-2 ## SIGNAL-Call
	
	
	signal_info <- numeric(length=n_array)
	if (n_array >= 4){
	signal_info[1] <- SNR
	signal_info[2] <- SIG
	signal_info[3] <- SIG * a^2
	signal_info[4] <- i
	}
	if (n_array == 3){
	signal_info[1] <- SNR
	signal_info[2] <- SIG
	signal_info[3] <- SIG * a^2
	}
	if (n_array == 2){
	signal_info[1] <- SNR
	signal_info[2] <- SIG
	}
	if (n_array == 1){
	signal_info[1] <- SNR
	}
	
	
	if(robust && (SNR >= 0.999)){
		L <- L_robust
		Ph <- Ph_robust
		PsiL <- (1 / Ph) * L
		a <- as.vector(1 + crossprod(L, PsiL))
		bar <- PsiL / a
		beta <- t(bar)
		XXbeta <- XX %*% bar
		EZZ <- 1 - beta %*% L + beta %*% XXbeta
		c <- X %*% bar ## hidden variable c - factor
		if(EZZ == 0){
			var_z_scale <- 1 ## avoiding division by zero
		}
		else{
			var_z_scale <- sqrt(EZZ)
		}
		c <- c / as.vector(var_z_scale)
		L <- L * as.vector(var_z_scale)
	}
	
	
	
	if (weighted.mean){
		PsiLL <- (1 / Ph) * L^2 
		sumPsiLL <- sum(PsiLL)
		propPsiLL <- PsiLL / sumPsiLL
		express <- as.vector(crossprod(L * sd.probes, propPsiLL)) * c + mean(y_v * sd.probes)
	} 
	else
	{
		express <- median(L * sd.probes) * c + mean(y_v * sd.probes)
	}
	

	
	return(list(exprs=as.numeric(express),se.exprs=signal_info))
}


setClass("INI_Calls", 
		representation(I_Calls="vector",
		NI_Calls="vector",
		I_Exprs="ExpressionSet",
		NI_Exprs="ExpressionSet",
		varZX="vector"))

# I/NI calls

setGeneric("INIcalls", function(object) standardGeneric("INIcalls"))
setGeneric("getI_Eset", function(object) standardGeneric("getI_Eset"))
setGeneric("getNI_Eset", function(object) standardGeneric("getNI_Eset"))
setGeneric("getI_ProbeSets", function(object) standardGeneric("getI_ProbeSets"))
setGeneric("getNI_ProbeSets", function(object) standardGeneric("getNI_ProbeSets"))
setGeneric("plotINIs", function(object) standardGeneric("plotINIs"))



setMethod("INIcalls","ExpressionSet", function(object){
   INIs <- new("INI_Calls")
   SNR <- object@assayData$se.exprs[,1]
   INIs@I_Calls <- names(SNR[SNR < .5])
   INIs@NI_Calls <- names(SNR[SNR >= .5])
   INIs@I_Exprs <- object[SNR < .5,]
   INIs@NI_Exprs <- object[SNR >= .5,]
   INIs@varZX <- SNR
   return(INIs)				# number of informative genes
})


setMethod("summary","INI_Calls",function(object){
   cat("Summary \n")
   cat("Informative probe sets      : ",round(100 * length(object@I_Calls)/(length(object@I_Calls)+length(object@NI_Calls)),digits=2),"% \n",sep="")
   cat("Non-Informative probe sets  : ",round(100 * length(object@NI_Calls)/(length(object@I_Calls)+length(object@NI_Calls)),digits=2),"% \n",sep="")	# number of informative genes
})


setMethod("plotINIs","INI_Calls",function(object){
   # plot var(z|x) of FARMS summarized probe sets
   truehist(as.numeric(object@varZX),col="lightgrey",border="darkgrey",xlab="Var(z|x)",ylab="density")
   abline(v = .5, lty = 1,col="black")				
})


setMethod("getI_Eset","INI_Calls",function(object){
   return(object@I_Exprs)
})

setMethod("getNI_Eset","INI_Calls",function(object){
   return(object@NI_Exprs)
})

setMethod("getI_ProbeSets","INI_Calls",function(object){
   return(object@I_Calls)
})

setMethod("getNI_ProbeSets","INI_Calls",function(object){
   return(object@NI_Calls)
})















