/**************************************************************************
 * This code is part of the supporting infrastructure for ATA Mapper. 
 * Copyright (C) 2002,2003,2004 Applera Corporation. All rights reserved.
 * Author: Nathan Edwards
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received (LICENSE.txt) a copy of the GNU General Public 
 * License along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *************************************************************************/

//================================================================
// Brian Walenz's code. Provides methods for buffered I/O.
//
//===============================================================

#include "bufferedFile.h"

#if !defined(NO_STD_NAMESPACE)
using namespace std;
#endif

BufferedFile::BufferedFile() {
  int  buffersize = 1024*1024;
  _buffer = new char[buffersize];
  if (_buffer == 0L) {
    fprintf(stderr, "Couldn't allocate the file buffer\n");
    exit(1);
  }

  _bufferMax = buffersize;
  _bufferLen = 0;
  _readPos   = 0;
  _size = _pos = 0;
}

BufferedFile::BufferedFile(std::string const &infile, int buffersize) {

  _size = _pos = 0;

  if (buffersize <= 0) {
    fprintf(stderr, "WARNING:  Illegal buffer size;resetting to 1024*1024.\n");
    buffersize = 1024*1024;
  }

  _buffer = new char[buffersize];
  if (_buffer == 0L) {
    fprintf(stderr, "Couldn't allocate the file buffer\n");
    exit(1);
  }
  _bufferMax = buffersize;
  openBuf(infile);
}

void BufferedFile::openBuf(std::string const &fname) {
  struct stat st;
  if (stat(fname.c_str(),&st)<0) {
    perror("BufferedFile::openBuf:stat");
    exit(1);
  }
  _size = st.st_size;

#if ! defined(__CYGWIN__) 
  bufF = fopen(fname.c_str(),"r");
#else
  bufF = fopen(fname.c_str(),"rb");
#endif
  if (!bufF) {
    cerr << "couldn't open input file: " << fname << endl;
    exit(1);
  }
  _bufferLen = 0;
  _readPos   = 0;
}


BufferedFile::~BufferedFile() {
  delete [] _buffer;
  fclose(bufF);
}

void BufferedFile::reset() {
  _bufferLen = 0;
  _readPos   = 0;
  rewind(bufF);
  _pos = 0;
}

FILE_POSITION_TYPE BufferedFile::getPos() const {
  return _pos;
}

void BufferedFile::setPos(FILE_POSITION_TYPE p) {
  _pos = p;
  FILE_POSITION_TYPE tg = FTELL(bufF);
  if (tg > p && tg <= p + _bufferLen) {
    _readPos = p - (tg - _bufferLen);
  } else {
    // bufF.clear();
    FSEEK(bufF,_bufferMax*(p/_bufferMax),SEEK_SET);
    _bufferLen = fread(_buffer,sizeof(unsigned char),_bufferMax,bufF);
    _readPos   = p%_bufferMax;
  }
}

unsigned char BufferedFile::getCharacter(void) {
  if (_readPos >= _bufferLen) {
    _bufferLen = fread(_buffer,sizeof(unsigned char),_bufferMax,bufF);
    _readPos   = 0;
  }
  _pos++;
  return(_buffer[_readPos++]);
}



