/**************************************************************************
 * This code is part of the supporting infrastructure for ATA Mapper. 
 * Copyright (C) 2002,2003,2004 Applera Corporation. All rights reserved.
 * Author: Nathan Edwards
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received (LICENSE.txt) a copy of the GNU General Public 
 * License along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *************************************************************************/


#ifndef _IBPEP_CHAR_IO_H_
#define _IBPEP_CHAR_IO_H_

#include "mapFile.h"
#include "bufferedFile.h"
#include "fileStar.h"
#include "util.h"

#if !defined(NO_STD_NAMESPACE)
using namespace std;
#endif

class CharacterProducer {
public:
  CharacterProducer() {};
  virtual ~CharacterProducer() {};
  virtual char getch()=0;
  virtual unsigned char getnch()=0;
  virtual char ch(unsigned char)=0;
  virtual int nch(char)=0;
  virtual unsigned int size() const =0;
  virtual bool eof() const =0;
  virtual FILE_POSITION_TYPE pos() const =0;
  virtual void pos(FILE_POSITION_TYPE p) =0;
  virtual void reset()=0;
  virtual float progress() const =0;
};

class IStreamChars : public CharacterProducer {
private:
  bool delstream;
  std::istream & is_;
  unsigned char cached_char;
public:
  IStreamChars(const std::string & filename) : is_(*(new std::ifstream(filename.c_str()))) {
    delstream = true;
    cached_char = is_.get();
  }
  IStreamChars(std::istream & is) : is_(is) {
    delstream = false;
    cached_char = is_.get();
  }
  ~IStreamChars() {
    if (delstream) {
      delete (&is_);
    }
  }
  char getch() { char tmp(cached_char); cached_char = is_.get(); return tmp; }
  unsigned char getnch() { return (unsigned char) getch(); }
  inline char ch(unsigned char nch) { return nch; }
  inline int nch(char ch) { return ch; }
  inline unsigned int size() const { return 256; }
  bool eof() const { return is_.eof(); }
  FILE_POSITION_TYPE pos() const { return ((FILE_POSITION_TYPE)(is_.tellg())-1); }
  void pos(FILE_POSITION_TYPE p) { is_.clear(); is_.seekg(p); cached_char = is_.get();}
  void reset() { is_.clear(); is_.seekg(0); }
  float progress() const { return 0.0; }
};

class CharStarChars : public CharacterProducer {
  const char *ptr_;
  const char *p_;
public:
  CharStarChars(const char *chars) : ptr_(chars), p_(chars) {}
  ~CharStarChars() {}
  inline char getch() { return *p_++; }
  inline unsigned char getnch() { return *p_++; }
  inline char ch(unsigned char nch) { return nch; }
  inline int nch(char ch) { return ch; }
  inline unsigned int size() const { return 256; }
  inline bool eof() const { return (*p_=='\0'); }
  FILE_POSITION_TYPE pos() const { return (p_-ptr_); }
  void pos(FILE_POSITION_TYPE p) { p_ = ptr_+p; }
  void reset() { p_ = ptr_; }
  float progress() const { 
    return 0; 
  }
};

class MapFileChars : public MapFile, public CharacterProducer {
public:
  MapFileChars(const char *filename) : MapFile(filename) {}
  ~MapFileChars() {}
  inline char getch() { return (char) MapFile::getCharacter(); }
  inline unsigned char getnch() { return MapFile::getCharacter(); }
  inline char ch(unsigned char nch) { return nch; }
  inline int nch(char ch) { return ch; }
  inline unsigned int size() const { return 256; }
  inline bool eof() const { return MapFile::endOfFile(); }
  FILE_POSITION_TYPE pos() const { return MapFile::getPos(); }
  void pos(FILE_POSITION_TYPE p) { MapFile::setPos(p); }
  void reset() { MapFile::reset(); }
  float progress() const { 
    return ((float)MapFile::getPos())/MapFile::map_size(); 
  }
};

class BufferedFileChars : public BufferedFile, public CharacterProducer {
public:
  BufferedFileChars(std::string const & filename) : BufferedFile(filename) {}
  ~BufferedFileChars() {}
  inline char getch() { return (char) BufferedFile::getCharacter(); }
  inline unsigned char getnch() { return BufferedFile::getCharacter(); }
  inline char ch(unsigned char nch) { return nch; }
  inline int nch(char ch) { return ch; }
  inline unsigned int size() const { return 256; }
  inline bool eof() const { return BufferedFile::isEndOfFile(); }
  FILE_POSITION_TYPE pos() const { return BufferedFile::getPos(); }
  void pos(FILE_POSITION_TYPE p) { BufferedFile::setPos(p); }
  void reset() { BufferedFile::reset(); }
  float progress() const { 
    return ((float)BufferedFile::getPos())/BufferedFile::getSize(); 
  }
};

class FileStarChars : public FileStarFile, public CharacterProducer {
public:
  FileStarChars(std::string const & filename) : FileStarFile(filename) {}
  ~FileStarChars() {}
  inline char getch() { return (char) FileStarFile::getCharacter(); }
  inline unsigned char getnch() { return FileStarFile::getCharacter(); }
  inline char ch(unsigned char nch) { return nch; }
  inline int nch(char ch) { return ch; }
  inline unsigned int size() const { return 256; }
  inline bool eof() const { return FileStarFile::isEndOfFile(); }
  FILE_POSITION_TYPE pos() const { return FileStarFile::getPos(); }
  void pos(FILE_POSITION_TYPE p) { FileStarFile::setPos(p); }
  void reset() { FileStarFile::reset(); }
  float progress() const { 
    return ((float)FileStarFile::getPos())/FileStarFile::getSize(); 
  }

};

class CompressedCharUtil {
public:
  static unsigned int lcm(unsigned int, unsigned int);
};

// #include "char_io.t"

#endif

