/**************************************************************************
 * This code is part of the supporting infrastructure for ATA Mapper. 
 * Copyright (C) 2002,2003,2004 Applera Corporation. All rights reserved.
 * Author: Nathan Edwards
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received (LICENSE.txt) a copy of the GNU General Public 
 * License along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *************************************************************************/


#include <stdio.h>

/* posix compliant mmap */
# include <fcntl.h>
# include <errno.h>
# include <sys/types.h>
# include <unistd.h>
# include <stdlib.h>
# include <sys/stat.h>
# ifndef __USE_MISC
#   define __USE_MISC
# endif
# ifndef __MINGW32__
#   include <sys/mman.h>
#   if !defined(MAP_FILE)
#     error MAP_FILE
#   endif
#endif

#ifdef __alpha
#define MMAPFLAGS    (MAP_FILE | MAP_VARIABLE | MAP_SHARED)
#endif

#ifdef __linux
#define MMAPFLAGS    (MAP_FILE | MAP_SHARED)
#endif

#ifdef __FreeBSD__
#define MMAPFLAGS    (MAP_FILE | MAP_SHARED)
#endif

#ifdef _AIX
#define MMAPFLAGS    (MAP_FILE | MAP_SHARED)
#endif

#ifdef __CYGWIN__
#define MMAPFLAGS    (MAP_FILE | MAP_SHARED)
#endif

#include "mapFile.h"

/* posix mmap */

MapFile::MapFile() {
  fileString=0;
  _pos=0;
  mapsize=0;
}

MapFile::MapFile(const char *filename) {
  errno = 0;

  int f = open(filename, O_RDONLY);
  if ((f < 0) || (errno)) {
    fprintf(stderr, "Couldn't open '%s'\n", filename);
    perror("open");
    exit(1);
  }

  struct stat  sb;
  fstat(f, &sb);
  if (errno) {
    fprintf(stderr, "Couldn't stat '%s'\n", filename);
    perror("fstat\n");
    exit(1);
  }

  if (sizeof(FILE_SIZE_TYPE) > sizeof(void*) &&
        ((((FILE_SIZE_TYPE)1)<<(8*sizeof(void*)))-1) < sb.st_size) {
    fprintf(stderr, "Couldn't map '%s'\n", filename);
    exit(1);
  }

  mapsize = sb.st_size;

  // fprintf(stderr, "%40.40s: Mapping %lu bytes.\n", filename, mapsize);
  // fflush(stderr);

# ifndef __MINGW32__
  fileString = (unsigned char *)mmap(0L, mapsize, PROT_READ, MMAPFLAGS, f, 0);
# else
  fileString = new unsigned char[mapsize];
  read(f,fileString,mapsize);
# endif

  if (errno) {
    fprintf(stderr, "Couldn't map '%s'\n", filename);
    perror("mmap");
    exit(1);
  }

  _pos=0;

  close(f);
}

void MapFile::createMap(const char *filename) {
  errno = 0;

  int f = open(filename, O_RDONLY);
  if ((f < 0) || (errno)) {
    fprintf(stderr, "Couldn't open '%s'\n", filename);
    perror("open");
    exit(1);
  }

  struct stat  sb;
  fstat(f, &sb);
  if (errno) {
    fprintf(stderr, "Couldn't stat '%s'\n", filename);
    perror("fstat\n");
    exit(1);
  }

  if (sizeof(FILE_SIZE_TYPE) > sizeof(void*) &&
        ((((FILE_SIZE_TYPE)1)<<(8*sizeof(void*)))-1) < sb.st_size) {
    fprintf(stderr, "Couldn't map '%s'\n", filename);
    exit(1);
  }

  mapsize = sb.st_size;

  // fprintf(stderr, "%40.40s: Mapping %lu bytes.\n", filename, mapsize);
  // fflush(stderr);

# ifndef __MINGW32__
  fileString = (unsigned char *)mmap(0L, mapsize, PROT_READ, MMAPFLAGS, f, 0);
# else
  fileString = new unsigned char[mapsize];
  read(f,fileString,mapsize);
# endif

  if (errno) {
    fprintf(stderr, "Couldn't map '%s'\n", filename);
    perror("mmap");
    exit(1);
  }

  _pos=0;
  
  close(f);
}

MapFile::~MapFile() {
  if (fileString) {
#ifdef __CYGWIN__
    (void)munmap((char*)fileString, mapsize);
#elif defined(__MINGW32__) 
    delete [] fileString;
#else
    (void)munmap(fileString, mapsize);
#endif
  }
}

int MapFile::setPos(const FILE_POSITION_TYPE pos) {
  _pos=pos;
  return (pos>=mapsize);  
}









