/**************************************************************************
 * This code is part of the supporting infrastructure for ATA Mapper. 
 * Copyright (C) 2002,2003,2004 Applera Corporation. All rights reserved.
 * Author: Nathan Edwards
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received (LICENSE.txt) a copy of the GNU General Public 
 * License along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *************************************************************************/


#ifndef _IBPEP_pattern_alignment_h
#define _IBPEP_pattern_alignment_h

#include <assert.h>
#include <iostream>
#include <vector>
#include "char_io.h"
#include "alignment_code.h"
#include "memory_debug.h"

#if !defined(NO_STD_NAMESPACE)
using namespace std;
#endif

class pattern_hit {
 private:
  unsigned long pattern_;
  FILE_POSITION_TYPE pos_;
 public:
  pattern_hit(unsigned long k=0, FILE_POSITION_TYPE p=0) 
    : pattern_(k), pos_(p) {}
  virtual ~pattern_hit() {};  
  unsigned long const & pattern_id() const {
    return pattern_;
  }
  void pattern_iter(unsigned long const & id) {
    pattern_ = id;
  }
  FILE_POSITION_TYPE const & pos() const {
    return pos_;
  }
  void pos(FILE_POSITION_TYPE const & e) {
    pos_ = e;
  }
  MEMORY_DEBUG(pattern_hit);
};

class pattern_hit_w_dist : public pattern_hit {
  int dist_;
 public:
  pattern_hit_w_dist(unsigned long k=0, FILE_POSITION_TYPE p=0, int d=-1) 
    : pattern_hit(k,p), dist_(d) {};
  pattern_hit_w_dist() {};
  void editdist(int const & d) {
    dist_ = d;
  }
  int const & editdist() const {
    return dist_;
  }
  MEMORY_DEBUG(pattern_hit_w_dist);
};

class pattern_alignment {  
public:
  pattern_alignment(FILE_POSITION_TYPE e=0, bool yn=false)
    : end_(e), alignment_done_(false), yesno_(yn) { 
    if (!yesno_) {
      stats_.resize(alignment_codes); 
      fill(stats_.begin(),stats_.end(),0); 
    }
  }
  pattern_alignment(pattern_hit const & ph, bool yn=false) 
    : end_(ph.pos()), alignment_done_(false) {
    if (!yesno_) {
      stats_.resize(alignment_codes); 
      fill(stats_.begin(),stats_.end(),0);
    }
  }
  virtual ~pattern_alignment();
  FILE_POSITION_TYPE const & end() const {
    return end_;
  }
  void end(FILE_POSITION_TYPE const & e) {
    end_ = e;
  }
  virtual bool align(CharacterProducer &, std::string const & pattern) =0;
  void reset() {
    alignment_done_ = false;
    if (!yesno_) {
      fill(stats_.begin(),stats_.end(),0);
    }
  }
  int size() const {
    assert(alignment_done_);
    return alignment_.size();
  };
  alignment_code operator[](int i) const {
    assert(alignment_done_);
    return alignment_[i];
  }
  FILE_POSITION_TYPE start() const {
    assert(alignment_done_);
    return start_;
  }
  FILE_POSITION_TYPE length() const {
    assert(alignment_done_);
    return end_-start_+1;
  }
  std::string const & matching_text() const {
    assert(alignment_done_);
    return matching_text_;
  }
  unsigned int stats(alignment_code ac) const {
    assert(alignment_done_);
    return stats_[ac];
  }
  unsigned int editdist() const {
    assert(alignment_done_);
    if (stats_[alignment_constraint_violation] > 0) {
      // checkpoint;
      return MAXINT;
    } else {
      return stats_[alignment_substitution] + 
	stats_[alignment_insertion] +
	stats_[alignment_deletion];
    }
  }
  std::string alignment_string() const {
    assert(alignment_done_);
    char *r = new char[size()+1];
    for (int i=0; i<size(); i++) {
      char ch;
      switch ((*this)[i]) {
      case alignment_equal :
	ch = '|';
	break;
      case alignment_wildcard_equal : 
	ch = '+';
	break;
      case alignment_substitution :
	ch = '*';
	break;
      case alignment_insertion :
	ch = '^';
	break;
      case alignment_deletion :
	ch = 'v';
	break;
      case alignment_constraint_violation :
	ch = '!';
	break;
      default:
	ch = ' ';
	break;
      }
      r[i] = ch;
    }
    r[size()] = '\0';
    std::string r1(r);
    delete r;
    return r1;
  }
  std::string alignment_text() const {
    assert(alignment_done_);
    std::string r("");
    std::string const & mt = matching_text();
    int p=0;
    for (int i=0; i<size(); i++) {
      if ((*this)[i] != alignment_deletion) {
	r += mt[p];
	p++;
      } else {
	r += "-";
      }
    }
    return r;
  }
  std::string alignment_pattern(std::string const & pat) const {
    assert(alignment_done_);
    std::string r("");
    int p=0;
    for (int i=0; i<size(); i++) {
      if ((*this)[i] != alignment_insertion) {
	r += pat[p];
	p++;
      } else {
	r += "-";
      }
    }
    return r;
  }
  void yesno(bool yn) {
    yesno_ = yn;
  }
  // void write(ostream & os, 
  // FILE_POSITION_TYPE const seq_pos,
  // std::string const & pattern, 
  // long unsigned int id, bool revcomp);
  virtual void write(ostream & os) const;
  virtual void read(istream & is) {
    is >> pattern_ >> end_;
  }
  MEMORY_DEBUG(pattern_alignment);
private:
  unsigned long pattern_;
  FILE_POSITION_TYPE end_;

protected:
  std::vector<alignment_code> alignment_;
  std::vector<int> stats_;
  std::string matching_text_;
  FILE_POSITION_TYPE start_;
  bool alignment_done_;
  bool yesno_;
};

istream & operator>>(istream & is, pattern_alignment & ka);
ostream & operator<<(ostream & os, pattern_alignment const & ka);

class exact_alignment : public pattern_alignment {
public:
  exact_alignment(FILE_POSITION_TYPE e=0)
    : pattern_alignment(e) {};
  exact_alignment(pattern_hit const & ph)
    : pattern_alignment(ph) {};
  ~exact_alignment() {};
  bool align(CharacterProducer &, std::string const & pattern);
  MEMORY_DEBUG(exact_alignment)
};

class exact_peptide_alignment : public pattern_alignment {
 private:
  char lcontext_;
  char rcontext_;
 public:
  exact_peptide_alignment(FILE_POSITION_TYPE e=0)
    : pattern_alignment(e) {};
  exact_peptide_alignment(pattern_hit const & ph)
    : pattern_alignment(ph) {};
  ~exact_peptide_alignment() {};
  bool align(CharacterProducer &, std::string const & pattern);
  char lcontext() const {
    assert(alignment_done_);
    return lcontext_;
  }
  char rcontext() const {
    assert(alignment_done_);
    return rcontext_;
  }
  MEMORY_DEBUG(exact_peptide_alignment)
};

class exact_wc_alignment : public pattern_alignment {
 private:
  bool textn_;
public:
  exact_wc_alignment(FILE_POSITION_TYPE e=0, bool tn=false)
    : pattern_alignment(e), textn_(tn) {};
  exact_wc_alignment(pattern_hit const & ph, bool tn=false)
    : pattern_alignment(ph), textn_(tn) {};
  ~exact_wc_alignment() {};
  bool align(CharacterProducer &, std::string const & pattern);
  MEMORY_DEBUG(exact_wc_alignment)
};

class mismatch_alignment : public pattern_alignment {
public:
  mismatch_alignment(FILE_POSITION_TYPE e=0)
    : pattern_alignment(e) {};
  mismatch_alignment(pattern_hit const & ph)
    : pattern_alignment(ph) {};
  ~mismatch_alignment() {};
  bool align(CharacterProducer &, std::string const & pattern);
  MEMORY_DEBUG(mismatch_alignment)
};

class editdist_alignment : public pattern_alignment {
  FILE_POSITION_TYPE end2_;
  unsigned int k_;
  char eos_;
  bool wc_;
  bool textn_;
  bool indels_;
  int lconst_;
  int rconst_;
  char *buffer_;
  FILE_POSITION_TYPE bufstart_;
  FILE_POSITION_TYPE bufend_;
  long unsigned int bufsize_;
  unsigned int maxpatlen_;
  unsigned int matsize_;
  unsigned int *dp_;
  int *best_;
public:
  editdist_alignment(FILE_POSITION_TYPE e=0, 
		     FILE_POSITION_TYPE e2=0,
		     unsigned int k=0, char eos='\n', 
		     bool wc=false, bool tn=false, bool id=true,
		     int lconst=0, int rconst=0, bool yn=false)
    : pattern_alignment(e,yn), end2_(e2), k_(k), eos_(eos), 
    wc_(wc), textn_(tn), indels_(id), lconst_(lconst), rconst_(rconst),
    buffer_(0), bufstart_(0), bufend_(0), maxpatlen_(0), matsize_(0), dp_(0), best_(0) {};
  editdist_alignment(FILE_POSITION_TYPE e=0, 
		     unsigned int k=0, char eos='\n', 
		     bool wc=false, bool tn=false, bool id=true,
		     int lconst=0, int rconst=0, bool yn=false)
    : pattern_alignment(e,yn), end2_(e), k_(k), eos_(eos), wc_(wc), textn_(tn),
      indels_(id), lconst_(lconst), rconst_(rconst),
      buffer_(0), bufstart_(0), bufend_(0), maxpatlen_(0), matsize_(0), dp_(0), best_(0) {};
  editdist_alignment(pattern_hit const & ph, unsigned int k=0, char eos='\n', 
		     bool wc=false, bool tn=false, bool id=true,
		     int lconst=0, int rconst=0, bool yn=false)
    : pattern_alignment(ph,yn), end2_(ph.pos()), k_(k), eos_(eos), 
    wc_(wc), textn_(tn), indels_(id), lconst_(lconst), rconst_(rconst),
    buffer_(0), bufstart_(0), bufend_(0), maxpatlen_(0), matsize_(0), dp_(0), best_(0) {};
  ~editdist_alignment() {
    delete [] buffer_;
    delete [] dp_;
    delete [] best_;
  };
  void poslb(FILE_POSITION_TYPE p) {
    pattern_alignment::end(p);
  }
  void posub(FILE_POSITION_TYPE p) {
    end2_ = p;
  }
  void pos(FILE_POSITION_TYPE p) {
    pattern_alignment::end(p);
    end2_ = p;
  }
  void exact_start_bases(unsigned int esb) {
    lconst_ = esb;
  }
  void exact_end_bases(unsigned int eeb) {
    rconst_ = eeb;
  }
  void eos(char ch) {
    eos_ = ch;
  }
  char eos() {
    return eos_;
  }
  void kmax(int k) {
    k_ = k;
  }
  void wc(bool wc) {
    wc_ = wc;
  }
  void tn(bool tn) {
    textn_ = tn;
  }
  void indels(bool id) {
    indels_ = id;
  }
  void maxpatlen(long unsigned int mpl) {
    maxpatlen_ = mpl;
  }
  bool align(CharacterProducer &, std::string const & pattern);
  MEMORY_DEBUG(editdist_alignment)
};

#endif
