%
% t-CWT Copyright (C) 2003, 2014-2015 Vladimir Bostanov. Type "tcwt_help".
%

% This file is part of t-CWT.
%
% t-CWT is free software: you can redistribute it and/or modify
% it under the terms of the GNU General Public License as published
% by the Free Software Foundation, either version 3 of the License,
% or (at your option) any later version.
%
% t-CWT is distributed in the hope that it will be useful,
% but WITHOUT ANY WARRANTY; without even the implied warranty
% of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
% See the GNU General Public License for more details.
%
% You should have received a copy of the GNU General Public License
% along with t-CWT; see the file COPYING or the file tcwt_gpl.m
% If not, see <http://www.gnu.org/licenses/>.

%# tcwt_f2holdout(DSNL,statsNAME,Cnd,Nstd,MinGood,Npc,sdAlpha) computes
%# the error rates of LDA classificaion of single trials using the
%# hold-out method <tcwt_help math>.
%#
%# The function iterates over DSNL reading and processing data from
%# the INPUT FILES and then saves the results into the OUTPUT FILES.
%#
%# For each DSN, the hold-out method is applied by iterating through all
%# trials of the dataset, excluding the current trial, doing t-CWT with
%# the other trials <tcwt_help pc2ri2cnd> <tcwt_help f2x> <tcwt_help x2ld>
%# and applying the obtained LDF to the current trial <tcwt_help math>.
%# Classification errors and binomial p-values are computed and saved
%# in the OUTPUT FILES <tcwt_help stats>.
%#
%# FUNCTION ARGUMENTS
%#
%#   DSNL       Test DataSet Name List (DSN List) <tcwt_help dir2dsnl>
%#
%#   statsNAME  Name of the OUTPUT FILES (without the extensions)
%#
%#   Cnd        List of exactly 2 experimental conditions <tcwt_help t2f>
%#
%#   Nstd       Criterion for single-trial outlier detection <tcwt_help f2pc>
%#
%#   MinGood    Criterion for dataset outlier detection <tcwt_help f2pc>
%#
%#   Npc        PCA criterion for the step-down test <tcwt_help f2pc>
%#
%#   sdAlpha    Overall alpha level for the step-down test <tcwt_help math>
%#
%# INPUT FILES               INPUT VARIABLES
%#
%#   ./DSN.f.mat              Verp,CIerp <tcwt_help t2f>
%#   ./DSN.pc.mat             pcTerp,pcEV <tcwt_help f2pc>
%#   ./DSN.cwss.mat           cwVerpSS <tcwt_help f2cwss>
%#   ./DSN.ri0.mat            RIerp, DSNLpool <tcwt_help f2pool>
%#   ./DSN.ri1.mat            RIerp, DSNLpool
%#
%# OUTPUT FILES                          OUTPUT VARIABLES <tcwt_help f2stats>
%#
%#   ./statsNAME.1trial.out.stats0.mat    dsnl = DSNL,
%#   ./statsNAME.1trial.out.statsA.mat    Ntrials,Nerrors,ErrP100,Pbinomi
%#                                        NtrialsSum,NerrorsSum,ErrP100Sum,
%#                                        PbinomiSum
%#
%#   ./statsNAME.1trial.out.stats0A.txt   Text output
%#
%#   Suffix '0' denotes that the priori probabilities are taken to be equal;
%#   'A' means they are computed; from the number of trials in each condition.

%       10        20        30        40        50        60        70        80

function tcwt_f2holdout(dsnl,statsName,Cnd,Nstd,MinGood,Npc,sdAlpha);

rin1='0';
rin2='1';
countN=0;
Alpha=0.05;
statsName  = [statsName '.1trial.out.stats'];

tI = clock();

Nds = size(dsnl,1);

Ntrials  = zeros(Nds,3);
Nerrors  = zeros(Nds,3);
NerrorsA = zeros(Nds,3);
ErrP100  = zeros(Nds,3);
ErrP100A = zeros(Nds,3);
Pbinomi  = zeros(Nds,1);
PbinomiA = zeros(Nds,1);

statSig = ' *';

spaceS = repmat(' ',1,size(dsnl,2));
tildeS = repmat('~',1,size(dsnl,2));
minusS = repmat('-',1,size(dsnl,2));
equalS = repmat('=',1,size(dsnl,2));
ulineS = repmat('_',1,size(dsnl,2));

if (Npc==round(Npc)) NpcStr = sprintf('%u',Npc); else NpcStr = sprintf('%f',Npc); end

O=tcwt_txt('\n');
O=[O,tcwt_txt('%s~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~\n',tildeS)];
O=[O,tcwt_txt('tcwt_f2holdout: Hold-out LDA error rates (1 single trial out)\n')];
O=[O,tcwt_txt('%s-------------------------------------------------------------------------\n',minusS)];
O=[O,tcwt_txt('PCA criterion = %s, Step-Down Alpha = %f\n', NpcStr,sdAlpha)];
O=[O,tcwt_txt('LDA error rates computed for both equal a priori probabilities (1st row) and\n')];
O=[O,tcwt_txt('for those computed from the number of trials in each condition (2nd row).\n')];
O=[O,tcwt_txt('%s-------------------------------------------------------------------------\n',minusS)];
O=[O,tcwt_txt('%s COND.1  COND.2  TOTAL = COND.1 + COND.2  T I M E\n',spaceS)];
O=[O,tcwt_txt('%s Error   Error   Error   A      binomial  REMAINING\n',spaceS)];
O=[O,tcwt_txt('%s Rate(%%) Rate(%%) Rate(%%) Priori  p-value    h:mm:ss\n',spaceS)];
%              ?? 123.1   123.1   123.1   12.1    1.1234*  hhh:mm:ss
formatS =     '%s %5.1f   %5.1f   %5.1f   %4.1f%%   %6.4f%1s  %s\n';
O=[O,tcwt_txt('%s-------------------------------------------------------------------------\n',minusS)];

load('_A_const.mat');
for n=1:size(dsnl,1)
 dsn=dsnl(n,:);
 load([dsn '.f.mat']);
 load([dsn '.pc.mat']);
 load([dsn '.cwss.mat']);
 load([dsn '.ri' rin1 '.mat']);
 RIerp1 = RIerp;
 load([dsn '.ri' rin2 '.mat']);

 Sout = logical(RIerp1(:,1));
 Scnd1 = RIerp1(:,2)==Cnd(1);
 Scnd2 = RIerp1(:,2)==Cnd(2);
 Scnd = Scnd1 | Scnd2;

 Nt1 = sum(Scnd1); Nt2 = sum(Scnd2);
 p1 = Nt1/(Nt1+Nt2);
 p2 = 1 - p1;
 logRatio = log(p2/p1);
 pApriori = min(p1,p2);

 RIerp = RIerp(~Sout & Scnd,:);
 Verp = Verp(~Sout & Scnd,:);

 pSout = logical(RIerp(:,1));

 Nt1 = 0; Nt2 = 0; Ne1 = 0; Ne2 = 0; Ne1A = 0; Ne2A = 0;

 NtrTot = size(RIerp,1);
 for nn=1:NtrTot
  Verp1out = Verp(nn,:);
  Cnd1out = RIerp(nn,2);
  RIerp1out = RIerp;
  RIerp1out(nn,1) = 1;

  RIerp1out = tcwt_pcoutlcnd(RIerp1out,Verp*pcTerp,Cnd,Nstd,MinGood);

  Sout = logical(RIerp1out(:,1));
  nSout = Sout & ~pSout;
  cwSS1out = cwVerpSS - tcwt_cwtss(Verp(nSout,:),CIerp,pcTerp,cwTerp,NcwMax,countN);

  Scnd1 = RIerp(~Sout,2)==Cnd(1);  Scnd2 = RIerp(~Sout,2)==Cnd(2);

  [tcwVerp,tcwCIerp] = ...
    tcwt_tcwterp(Verp(~Sout,:),RIerp(~Sout,:),CIerp,pcTerp,cwTerp,logGrid,cwSS1out,Cnd);
  [xTerp,xCIerp] = tcwt_xtcwt(tcwVerp,tcwCIerp,pcTerp,cwTerp);
  xVerp = Verp(~Sout,:)*xTerp;
  [VerpSD,pcTerpSD] = tcwt_stepdown(xVerp,Scnd1,Scnd2,Npc,sdAlpha);
  [ldTerp,ldMean] = tcwt_ldt(VerpSD,Scnd1,Scnd2);
  ldTerp = xTerp*pcTerpSD*ldTerp;

  if (Cnd1out==Cnd(1))
   Nt1  = Nt1  + 1;
   Ne1  = Ne1  + [Verp1out*ldTerp - ldMean < 0];
   Ne1A = Ne1A + [Verp1out*ldTerp - ldMean < logRatio];
  else
   Nt2  = Nt2  + 1;
   Ne2  = Ne2  + [Verp1out*ldTerp - ldMean >= 0];
   Ne2A = Ne2A + [Verp1out*ldTerp - ldMean >= logRatio];
  end
 end

 Nt  = [Nt1,  Nt2,  Nt1+Nt2];
 Ne  = [Ne1,  Ne2,  Ne1+Ne2];
 NeA = [Ne1A, Ne2A, Ne1A+Ne2A];
 Ep  = 100*Ne./Nt;
 EpA = 100*NeA./Nt;
 Pb  = binocdf(Ne(3),  Nt(3), 0.5);
 PbA = binocdf(NeA(3), Nt(3), pApriori);
 Sb  = statSig([Pb<Alpha]+1);
 SbA = statSig([PbA<Alpha]+1);

 Ntrials(n,:)  = Nt;
 Nerrors(n,:)  = Ne;
 NerrorsA(n,:) = NeA;
 ErrP100(n,:)  = Ep;
 ErrP100A(n,:) = EpA;
 Pbinomi(n)    = Pb;
 PbinomiA(n)   = PbA;

 tE = etime(clock(),tI);
 hms = tcwt_hms((Nds-n)*tE/n);

 O=[O,tcwt_txt(formatS, dsn, Ep(1),Ep(2),Ep(3),50,Pb,Sb,'')];
 O=[O,tcwt_txt(formatS, spaceS, EpA(1),EpA(2),EpA(3),100*pApriori,PbA,SbA,hms)];

end

Nt  = sum(Ntrials);
Ne  = sum(Nerrors);
NeA = sum(NerrorsA);
Ep  = 100*Ne./Nt;
EpA = 100*NeA./Nt;
Pb  = binocdf(Ne(3), Nt(3), 0.5);
PbA = binocdf(NeA(3), Nt(3), pApriori);
Sb  = statSig([Pb<Alpha]+1);
SbA = statSig([PbA<Alpha]+1);

Ns = Nt(3);
Nc = Nchannels;
Nf = size(fTerp,2);
Nv = size(logGrid,2);

tE = etime(clock(),tI);
tEs = tE/Ns;
tEsc = tEs/Nc;
tEscf = tEsc/Nf;
tEscfv = tEscf/Nv;

O=[O,tcwt_txt('%s-------------------------------------------------------------------------\n',minusS)];
O=[O,tcwt_txt(formatS, spaceS, Ep(1),Ep(2),Ep(3),50,Pb,Sb,'')];
O=[O,tcwt_txt(formatS, spaceS, EpA(1),EpA(2),EpA(3),100*pApriori,PbA,SbA,'')];
O=[O,tcwt_txt('%s=========================================================================\n',equalS)];
O=[O,tcwt_txt('Datasets processed: %u\n',Nds)];
O=[O,tcwt_txt('Scalograms (trials total): Ns = %u\n',Ns)];
O=[O,tcwt_txt('Channels per scalogram:    Nc = %u\n',Nc )];
O=[O,tcwt_txt('Frequency components per channel:  Nf = %u\n', Nf )];
O=[O,tcwt_txt('CWT log-grid vertexes per channel: Nv = %u\n', Nv )];
O=[O,tcwt_txt('CWTmatrixElementsTotal  =  Nc * Nf * Nv  =  %u\n', Nc*Nf*Nv )];
O=[O,tcwt_txt('%s-------------------------------------------------------------------------\n',minusS)];
O=[O,tcwt_txt('ELAPSED TIME: %s\n', tcwt_hms(tE))];
O=[O,tcwt_txt('  Total: %f seconds\n', tE )];
O=[O,tcwt_txt('    per dataset: %f seconds\n', tE/Nds)];
O=[O,tcwt_txt('    per scalogram: %f seconds\n', tEs )];
O=[O,tcwt_txt('      per channel: %f seconds\n', tEsc )];
O=[O,tcwt_txt('        per frequency component: %f milliseconds\n', 1e3*tEscf )];
O=[O,tcwt_txt('          per CWT log-grid vertex: %f microseconds\n', 1e6*tEscfv )];
O=[O,tcwt_txt('%s-------------------------------------------------------------------------\n',minusS)];
O=[O,tcwt_txt('tcwt_f2x: ElapsedTimeTotal = Ns * Nc * Nf * Nv * ElapsedTimePerVertex\n')];

NtrialsSum = Nt;
NerrorsSum = Ne;
ErrP100Sum = Ep;
PbinomiSum = Pb;

save([statsName '0.mat'], 'dsnl', ...
 'Ntrials','Nerrors','ErrP100','Pbinomi', ...
 'NtrialsSum','NerrorsSum','ErrP100Sum','PbinomiSum');

Nerrors = NerrorsA;
ErrP100 = ErrP100A;
Pbinomi = PbinomiA;

NerrorsSum = NeA;
ErrP100Sum = EpA;
PbinomiSum = PbA;

save([statsName 'A.mat'], 'dsnl', ...
 'Ntrials','Nerrors','ErrP100','Pbinomi', ...
 'NtrialsSum','NerrorsSum','ErrP100Sum','PbinomiSum');

O=[O,tcwt_txt('%s----------------------------------------------------------------------\n',minusS)];
O=[O,tcwt_txt('tcwt_f2holdout: Results saved to: %s.mat\n',statsName)];
O=[O,tcwt_txt('                          and to: %sA.mat\n',statsName)];
tcwt_txtsave([statsName '0A.txt'],O);
O=[O,tcwt_txt('                  Text output to: %s0A.txt\n',statsName)];
O=[O,tcwt_txt('%s______________________________________________________________________\n',ulineS)];
