%
% t-CWT Copyright (C) 2003, 2014-2015 Vladimir Bostanov. Type "tcwt_help".
%

% This file is part of t-CWT.
%
% t-CWT is free software: you can redistribute it and/or modify
% it under the terms of the GNU General Public License as published
% by the Free Software Foundation, either version 3 of the License,
% or (at your option) any later version.
%
% t-CWT is distributed in the hope that it will be useful,
% but WITHOUT ANY WARRANTY; without even the implied warranty
% of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
% See the GNU General Public License for more details.
%
% You should have received a copy of the GNU General Public License
% along with t-CWT; see the file COPYING or the file tcwt_gpl.m
% If not, see <http://www.gnu.org/licenses/>.

%# tcwt_f2x(is1out,DSNL,Cnd) t-CWT & Extremum Detection <tcwt_help math>.
%#
%# The job constants are loaded from './_A_const.mat' <tcwt_help prm2mat>.
%# Then, the function iterates over DSNL reading and processing data from
%# the INPUT FILES and saving results to the corresponding OUTPUT FILES.
%# If is1out = 1, a second DSNLpool of 1out index names <tcwt_help ri2ri1out>
%# is created at each step, and the function iterates throuh DSNLpool.
%#
%# FUNCTION ARGUMENTS
%#
%#   is1out = 0 or 1; defines whether 1out indexes are used (see below)
%#
%#   DSNL     DataSet Name List (DSN List) <tcwt_help dir2dsnl>
%#
%#   Cnd      List of exactly 2 experimental conditions <tcwt_help t2f>
%#
%# INPUT FILES               INPUT VARIABLES
%#
%#   ./DSN.f.mat              Verp,CIerp <tcwt_help t2f>
%#   ./DSN.pc.mat             pcTerp,pcEV <tcwt_help f2pc>
%#   ./DSN.cwss.mat           cwVerpSS <tcwt_help f2cwss>
%#   ./DSN.ri1.mat            RIerp, DSNLpool <tcwt_help f2pool>
%#   ./DSN.ri2.mat            RIerp, DSNLpool
%#
%#  If is1out == 1
%#
%#   ./DSNp.out.DSN.ri2.mat   RIerp
%#
%# DSNp is taken from DSNLpool.
%#
%# OUTPUT FILES              OUTPUT VARIABLES
%#
%#   ./DSN.tcw.mat            tcwVerp,tcwCIerp
%#   ./DSN.x.mat              xTerp,xCIerp
%#
%#   If is1out == 1
%#
%#   ./DSNp.out.DSN.tcw.mat   tcwVerp,tcwCIerp
%#   ./DSNp.out.DSN.x.mat     xTerp,xCIerp
%#
%# OUTPUT VARIABLES
%#
%#   tcwCIerp =
%#
%#   [ chnum(1) chnum(1)... chnum(1) chnum(1)... chnum(2) chnum(2)... ... ;
%#     scale(1) scale(1)... scale(2) scale(2)... scale(1) scale(1)... ... ;
%#     time(1,1)time(1,2)...time(2,1)time(2,2)...time(1,1)time(1,2)... ... ]
%#
%#   tcwVerp =
%#
%#   [ t(1,1,1) t(1,1,2)... t(1,2,1) t(1,2,2)... t(2,1,1) t(2,1,2)... ... ]
%#
%#   xCIerp =
%#
%#   [ xchnum(1)    ... xchnum(2)    ...  ... ;
%#     xscale(1)    ... xscale(2)    ...  ... ;
%#     xtime(1)     ... xtime(2)     ...  ... ;
%#     xpolarity(1) ... xpolarity(2) ...  ... ;
%#     xtcwindex(1) ... xtcwindex(2) ...  ... ;
%#     xtcwvalue(1) ... xtcwvalue(2) ...  ... ]
%#
%# Notations used above:
%#
%#   tcwVerp    t-CWT scalogram
%#
%#   tcwCIerp   t-CWT column index
%#
%#   xTerp      Transformation matrix from the frequency domain ERP Verp
%#              to the t-CWT features (extrema) xVerp = Verp * xTerp
%#
%#   xCIerp     t-CWT extrema (i.e. xVerp column index)
%#
%#   t(K,N,M)   t-CWT-value corresponding to the Kth channel, and the Mth
%#              time point of the Nth scale line of logGrid <tcwt_help prm2mat>
%#
%#   chnum(K)   Channel number of the Kth channel
%#
%#   scale(N)   Scale in seconds of the Nth scale line of logGrid
%#
%#   time(N,M)  Time in seconds of the Mth time point of the Nth scale line
%#
%#   xchnum(I)      Channel number of the Ith extremum
%#
%#   xscale(I)      Scale in seconds of the Ith extremum
%#
%#   xtime(I)       Time in seconds of the Ith extremum
%#
%#   xpolarity(I)   Polarity of the Ith extremum (-1 = min, 1 = max)
%#
%#   xtcwindex(I)   Absolute tcwVerp index of the Ith extremum
%#
%#   xtcwvalue(I) = tcwVerp( xtcwindex(I) )

%       10        20        30        40        50        60        70        80

function tcwt_f2x(is1out,dsnl,Cnd);

rin1='1';
rin2='2';

tI = clock();
Ns = 0;

O=tcwt_txt('\n');
O=[O,tcwt_txt('~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~\n')];
O=[O,tcwt_txt('tcwt_f2x: t-CWT & Detection of Scalogram Extrema (t-CWT features)\n')];
O=[O,tcwt_txt('-------------------------------------------------------------------------\n')];

load('_A_const.mat');
for n=1:size(dsnl,1)
 dsn=dsnl(n,:);
 load([dsn '.f.mat']);
 load([dsn '.pc.mat']);
 load([dsn '.cwss.mat']);
 load([dsn '.ri' rin1 '.mat']);

 pSout = logical(RIerp(:,1));

 if is1out
  dsnl_ri = tcwt_dsnl2xdsnl('',DSNLpool,['.out.' dsn]);
 else
  dsnl_ri = dsn;
 end

 for n_ri=1:size(dsnl_ri,1)
  dsn_ri=dsnl_ri(n_ri,:);
  load([dsn_ri '.ri' rin2 '.mat']);

  Ns = Ns + 1;

  Sout = logical(RIerp(:,1));
  nSout = Sout & ~pSout;

  if ( sum(nSout) > 0 )
   cwSS = cwVerpSS - tcwt_cwtss(Verp(nSout,:),CIerp,pcTerp,cwTerp,NcwMax,0);
  else
   cwSS = cwVerpSS;
  end

  [tcwVerp,tcwCIerp] = tcwt_tcwterp(Verp(~Sout,:),RIerp(~Sout,:),CIerp,pcTerp,cwTerp,logGrid,cwSS,Cnd);
  [xTerp,xCIerp] = tcwt_xtcwt(tcwVerp,tcwCIerp,pcTerp,cwTerp);

  save([dsn_ri '.tcw.mat'],'tcwVerp','tcwCIerp');
  save([dsn_ri '.x.mat'],'xCIerp','xTerp');

  O=[O,tcwt_txt('%s: detected extrema: %6u\n', dsn_ri, size(xCIerp,2))];
 end
end

Nc = Nchannels;
Nf = size(fTerp,2);
Nv = size(logGrid,2);

tE = etime(clock(),tI);
tEs = tE/Ns;
tEsc = tEs/Nc;
tEscf = tEsc/Nf;
tEscfv = tEscf/Nv;

O=[O,tcwt_txt('-------------------------------------------------------------------------\n')];
O=[O,tcwt_txt('Scalograms processed:   Ns = %u\n',Ns )];
O=[O,tcwt_txt('Channels per scalogram: Nc = %u\n',Nc )];
O=[O,tcwt_txt('Frequency components per channel:  Nf = %u\n', Nf )];
O=[O,tcwt_txt('CWT log-grid vertexes per channel: Nv = %u\n', Nv )];
O=[O,tcwt_txt('CWTmatrixElementsTotal  =  Nc * Nf * Nv  =  %u\n', Nc*Nf*Nv )];
O=[O,tcwt_txt('-------------------------------------------------------------------------\n')];
O=[O,tcwt_txt('ELAPSED TIME: %s\n', tcwt_hms(tE))];
O=[O,tcwt_txt('  Total: %f seconds\n', tE )];
O=[O,tcwt_txt('    per scalogram: %f seconds\n', tEs )];
O=[O,tcwt_txt('      per channel: %f seconds\n', tEsc )];
O=[O,tcwt_txt('        per frequency component: %f milliseconds\n', 1e3*tEscf )];
O=[O,tcwt_txt('          per CWT log-grid vertex: %f microseconds\n', 1e6*tEscfv )];
O=[O,tcwt_txt('-------------------------------------------------------------------------\n')];
O=[O,tcwt_txt('tcwt_f2x: ElapsedTimeTotal = Ns * Nc * Nf * Nv * ElapsedTimePerVertex\n')];
O=[O,tcwt_txt('_________________________________________________________________________\n')];





