%
% t-CWT Copyright (C) 2003, 2014-2015 Vladimir Bostanov. Type "tcwt_help".
%

% This file is part of t-CWT.
%
% t-CWT is free software: you can redistribute it and/or modify
% it under the terms of the GNU General Public License as published
% by the Free Software Foundation, either version 3 of the License,
% or (at your option) any later version.
%
% t-CWT is distributed in the hope that it will be useful,
% but WITHOUT ANY WARRANTY; without even the implied warranty
% of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
% See the GNU General Public License for more details.
%
% You should have received a copy of the GNU General Public License
% along with t-CWT; see the file COPYING or the file tcwt_gpl.m
% If not, see <http://www.gnu.org/licenses/>.

%# tcwt_prm2mat(prmFILE) reads job input parameters <tcwt_help prm> from
%# the file prmFILE.m, computes other job constants and saves all of them
%# into the file './_A_const.mat'. The term 'job constants' means that,
%# once defined, these variables do not change any more during job execution.
%# This definition includes all job input parameters and all variables
%# computed from them by tcwt_prm2mat which is called from the main script
%# <tcwt_help main> before calling the job script <tcwt_help scr>.
%# During job execution, t-CWT functions using one or more job
%# constants access them by loading the file './_A_const.mat'.
%#
%# JOB CONSTANTS
%#
%#   fullTime   A vector consisting of the time-sampling points defined
%#              by the job input parameters a2dR and Epoch <tcwt_help prm>.
%#
%#   NtimePt    The length of fullTime.
%#
%#   Time       The part of fullTime contained in the time interval
%#              defined by the job input parameter Window <tcwt_help prm>.
%#
%#   fTerp      The matrix of the transformation of the EEG from the time
%#              domain into the frequency domain; defined by Time and the job
%#              input parameters cutoffScale and fadeInOut <tcwt_help prm>.
%#
%#   tTerp      The matrix of the transformation of the EEG from
%#              the frequency domain back into the time domain
%#
%#   cwTerp     The matrix of the continuous wavelet transform (CWT) from
%#              the frequency domain onto a log-grid in the time-scale plane.
%#
%#   logGrid    A two-row matrix of coordinates in the time-scale plane
%#              of the vertices of the CWT log-grid; defined by Time and
%#              job input parameters cutoffScale and logGridR <tcwt_help prm>.
%#
%# ALL JOB INPUT PARAMETERS ARE SAVED AS JOB CONSTANTS AS WELL.

%       10        20        30        40        50        60        70        80

function tcwt_prm2mat(prmfile);

O=tcwt_txt('\n');
O=[O,tcwt_txt('~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~\n')];
O=[O,tcwt_txt('tcwt_prm2mat: Computing Transformation Matrices: DFT, CWT, etc.\n')];
O=[O,tcwt_txt('----------------------------------------------------------------\n')];

if exist('_A_const.mat','file')
 O=[O,tcwt_txt('_A_const.mat already exists. No new job constants computed.\n')];
 load('_A_const.mat');
else
 eval(prmfile);
 if (~isscalar(NcwMax) || ~isnumeric(NcwMax) || ~isreal(NcwMax) || NcwMax~=round(NcwMax) || NcwMax<1)
  error('Job input parameter NcwMax must be a single positive integer number.');
 end
 if ( ~ischar(ASCIIext) || ~isrow(ASCIIext) )
  error('Job input parameter ASCIIext must be a string.');
 end
 if ( ~isscalar(isMultiplex) || ~isnumeric(isMultiplex) || ( isMultiplex~=0 &&  isMultiplex~=1))
  error('The value of the job input parameter isMultiplex must be either 0 or 1.');
 end
 if ( ~ischar(ChannelList) )
  error('Job input parameter ChannelList must be a character array.');
 end

 Nchannels = sum(ChannelList(:,size(ChannelList,2))==' ');
 [Time,fullTime,NtimePt,a2dR] = tcwt_erptime(NtimePt,a2dR,Epoch,Window,BaseLine);
 [fTerp,tTerp,cwTerp,logGrid] = tcwt_dftcwterp(Time,cutoffScale,logGridR,fadeInOut);

 save('_A_const.mat');
end

O=[O,tcwt_txt('----------------------------------------------------------------\n')];
O=[O,tcwt_txt('Time sampling points in Epoch: %u\n', NtimePt)];
O=[O,tcwt_txt('Time sampling rate: %u Hz\n', a2dR)];
O=[O,tcwt_txt('Time sampling points in Window: %u\n', length(Time))];
O=[O,tcwt_txt('CWT log-grid vertexes: %u\n', size(logGrid,2) )];
O=[O,tcwt_txt('Frequency domain components: %u\n', size(fTerp,2) )];
O=[O,tcwt_txt('Total number of EEG channels: %u\n', size(ChannelList,1))];
O=[O,tcwt_txt('Channels to be processed by t-CWT: %u\n', Nchannels)];
O=[O,tcwt_txt('----------------------------------------------------------------\n')];
O=[O,tcwt_txt('tcwt_prm2mat: Job constants saved to ./_A_const.mat\n')];
O=[O,tcwt_txt('________________________________________________________________\n')];
