%
% t-CWT Copyright (C) 2003, 2014-2015 Vladimir Bostanov. Type "tcwt_help".
%

% This file is part of t-CWT.
%
% t-CWT is free software: you can redistribute it and/or modify
% it under the terms of the GNU General Public License as published
% by the Free Software Foundation, either version 3 of the License,
% or (at your option) any later version.
%
% t-CWT is distributed in the hope that it will be useful,
% but WITHOUT ANY WARRANTY; without even the implied warranty
% of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
% See the GNU General Public License for more details.
%
% You should have received a copy of the GNU General Public License
% along with t-CWT; see the file COPYING or the file tcwt_gpl.m
% If not, see <http://www.gnu.org/licenses/>.

%# tcwt_t2f(DSNL) converts time domain ERP to frequency domain ERP.
%#
%# The job constants are loaded from './_A_const.mat' <tcwt_help prm2mat>.
%# Then, the function iterates over DSNL reading and processing data from
%# the INPUT FILES and saving results to the corresponding OUTPUT FILES.
%#
%# FUNCTION ARGUMENTS
%#
%#   DSNL   DataSet Name List (DSN List) <tcwt_help dir2dsnl>
%#
%# INPUT FILES           INPUT VARIABLES
%#
%#   ../_tmat/DSN.t.mat   Verp,RIerp,CIerp <tcwt_help ascii2tmat>
%#
%# OUTPUT FILES          OUTPUT VARIABLES (for details, see below)
%#
%#   ./DSN.f.mat          Verp,CIerp
%#   ./DSN.ri0.mat        RIerp,DSNLpool (DSNLpool=DSN) <tcwt_help f2pool>
%#
%# The following steps precede the frequency domain transformation:
%#
%#   Channel selection:  Channels that are not used according to the entries
%#                       in ChannelList <tcwt_help prm> are deleted.
%#
%#   Baseline reference: BaseLine is a job input parameter <tcwt_help prm>.
%#
%#   Window selection:   Data points outside of the time interval defined
%#                       by the job input parameter Window are deleted.
%#
%# The transform from the time domain to the frequency domain is performed
%# by multiplication with the t-CWT CONSTANT fTerp <tcwt_help prm2mat>.
%# It comprises time domain multiplication with a Tukey window function
%# defined by the job input parameter fadeInOut, Discrete Fourier Transform
%# (DFT), and low-pass filtering with a cutoff frequency = 1/cutoffScale
%# <tcwt_help prm>. For math details: <tcwt_help math>.
%#
%# OUTPUT VARIABLES:     CIerp =
%#
%#                       [ chnum(1) chnum(1)...  chnum(2) chnum(2)...  ... ]
%#
%#   RIerp =             Verp =
%#
%#   [ out(1) cndN ;     [ v(1,1,1) v(1,1,2)...  v(1,2,1) v(1,2,2)...  ... ;
%#     out(2) cndN ;       v(2,1,1) v(2,1,2)...  v(2,2,1) v(2,2,2)...  ... ;
%#     out(3) cndN ;       v(3,1,1) v(3,1,2)...  v(3,2,1) v(3,2,2)...  ... ;
%#     ...         ;       ...                                             ;
%#     ...         ;       ...                                             ;
%#     ...         ]       ...                                             ]
%#
%# Notations used above:
%#
%#   Verp      ERP matrix in the frequency domain
%#
%#   CIerp     Verp column index
%#
%#   RIerp     Verp row index
%#
%#   v(N,K,M)  Value corresponding to the Nth trial,
%#             Kth channel, and Mth frequency.
%#
%#   chnum(K)  Channel number of the Kth channel.
%#
%#   out(N)    Outlier mark (0 or 1) of the Nth trial (1 = marked as outlier).
%#
%#   cndN      Condition (experimental condition denoted
%#             by a number), to which all trials of this dataset belong.

%       10        20        30        40        50        60        70        80

function tcwt_t2f(dsnl);


rin='0';

O=tcwt_txt('\n');
O=[O,tcwt_txt('~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~\n')];
O=[O,tcwt_txt('tcwt_t2f: Time Domain to Frequency Domain Transform\n')];
O=[O,tcwt_txt('---------------------------------------------------\n')];

if ispc() pslash = '\'; else pslash = '/'; end
tmatPath = ['..' pslash '_tmat' pslash];

load('_A_const.mat');
for n=1:size(dsnl,1)
 dsn = dsnl(n,:);
 load([tmatPath dsn '.t.mat']);

 [Verp,RIerp] = tcwt_chselect(Verp,RIerp,ChannelList); % Delete unused channels
 Verp = tcwt_baseref(Verp,fullTime,BaseLine); % Reference to baseline
 Verp = Verp(:, fullTime>=Window(1) & fullTime<Window(2) ); % Window selection

 Verp = Verp*fTerp; % Discrete Fourier Transform with Filters
 [Verp,RIerp,CIerp] = tcwt_vectorerp(Verp,RIerp);

 DSNLpool = dsn;

 save([dsn '.f.mat'],'Verp','CIerp');
 save([dsn '.ri' rin '.mat'],'RIerp','DSNLpool');
 O=[O,tcwt_txt('%s: *.t.mat -> *.f.mat\n',dsn)];
end
O=[O,tcwt_txt('_______________________________________________________\n')];


