/* here are all options for the gp programs, stored in a static variable called
 * programs, of type programopt_s. Functions in this source file deal with
 * preparing the menu entries and creating a suitable options window if one of
 * these entries is called. Then a commandline is passed to function "execute"
 * from main.c */


#include <stdio.h>
#include <ctype.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <gtk/gtk.h>
#include "main.h"

typedef enum { BOOLEAN, INT, DOUBLE, STRING, FILENAME, HIDDEN } opttypes_t ;

/* structure holding options for one of the program entries */
typedef struct {
	GtkWidget *widget ;

	char *designator ;
	char *description ;
	char *defvalue ;
	int set ;

	char *strvalue ;
	double doublevalue ;
	int intvalue ;

	int required ;
	opttypes_t type ; 

	opt_s *main_opts ;
} entry_option_s ;

/* structure holding entry */
typedef struct {
	int number ;
	char *name ; /* program name */
	char *entry ;
	char *branch ;
	char *short_description ; /* short program description */
	char *long_description ; /* long description of the program */
	int nopts ; /* how many options */
	entry_option_s *options ; } entry_s ;

/* two global variables. Sorry. */
entry_s *programs[ARKA_PROG_NUM_MAX] ;
int number_of_programs ;

/* function prototypes */

/* parses program file and returns an array with entries */
entry_s * parse_rc(FILE *in, int *i) ;

/* print an entry on stdout */
int entry_print(entry_s *e) ;
char *my_strdup(char *string) ;

/* -------------- some small, helping string funtions ------------- */

/* ANSI C doesn't have a strdup */
char *my_strdup(char *string) {
	int i ;
	char * res ;

	i = strlen(string) ;
	res = malloc(i + 1) ;
	strcpy(res, string) ;
	return res ;
}


/* exchange all occurencies of 'from' in 'string' by 'to' */
char *my_strsubst(char *string, char *from, char *to) {
	char *tmp, *t, *p, *q, *res ;

	res = NULL ;
	tmp = my_strdup(string) ;
	t = tmp ;

	if(!string || !from || !to) return NULL ;

	while( (p = strstr(t, from)) ) {
		*p = '\0' ;
		if(!res) {
			res = malloc(strlen(t) + strlen(to) + 1) ;
			*res = '\0' ;
		} else res = realloc(res, strlen(res) + strlen(t) + strlen(to) + 1) ;
		strcat(res, t) ;
		strcat(res, to) ;
		t = p + strlen(from) ;
	}

	if(!res) {
		res = malloc(strlen(t) + 1) ;
		strcpy(res, t) ;
	} else {
		res = realloc(res, strlen(res) + strlen(t) + 1) ;
		strcat(res, t) ;
	}

	free(tmp) ;
	return res ;
}


/* ----------------------- functions called from outside and related ----------- */

/* catches the signal from menu and executes the program */
void program_select(gpointer dane, guint signal, GtkWidget *kontrolka) {
	opt_s *o ;
	char *p, *cmdline ;

	o = (opt_s *) dane ;
	p = set_options(signal, o) ; /* do the dialog stuff */

	Warn("program_select: got: %s\n from set_options", p) ;

	if(!p) return ;

	cmdline = my_strsubst(p, "%input", o->input.tmpfil) ;
	cmdline = realloc(cmdline, 
		strlen(cmdline) + strlen(o->output.tmpfil) + strlen(o->error.tmpfil) + 10) ;

	strcat(cmdline, " > ") ;
	strcat(cmdline, o->output.tmpfil) ;
	strcat(cmdline, " 2> ") ;
	strcat(cmdline, o->error.tmpfil) ;

	Warn("program_select: executing\n%s", cmdline) ;

	free(p) ;
	execute(cmdline, o) ;
	free(cmdline) ;
}


/* this function sits in between append_programs() and functions actually
 * parsing the description file. It records the descriptions and makes an
 * apriopriate menu entry. */
int parse_and_append(opt_s *o, FILE* fp, int start) {
	GtkItemFactoryEntry element = {NULL, NULL,	program_select,	0} ;
	int i, line = 0 ;
	char *t, tmp1[100] ;
	entry_s *e ;

	i = start ; 

	while( (e = parse_rc(fp, &line)) != NULL ) {

		if( start >= ARKA_PROG_NUM_MAX) {
			Warn("Sorry. Arka can accept only %i programs right now.\n"
				"This should be easy enough to change, but right now I am too lazy to do it.\n",
			ARKA_PROG_NUM_MAX) ;
			return i ;
		}

		e->number = i ;

		/* preparing item_factory item */
		t = malloc(15 + strlen(e->branch) + strlen(e->entry)) ;
		if(e->branch && strlen(e->branch) > 0)
			sprintf(t, "/Programs/%s/_%c %s", e->branch, i + 97, e->entry) ;
		else
			sprintf(t, "/Programs/_%c %s", i + 97, e->entry) ;
		element.path = t ;
		element.callback_action = i ;

		/* creating menu entry */
		gtk_item_factory_create_items(o->fabryka, 1, &element, o) ;

		programs[i] = e ;
		free(t) ;
		i++ ;
	}
	fclose(fp) ;

	return i ;

}


/* appends programs to the main program menu bar */
void append_programs(opt_s *o) {
	int i = 0  ;
	FILE *fp ;
	char *rcfile ;

	/* reading program description from the global file */
	if( !(fp = fopen(ARKA_PROGRAMS, "r")) ) { 
		Warn("File %s could not be opened", ARKA_PROGRAMS) ;
	} else {
		Warn("reading %s", ARKA_PROGRAMS) ;
		i = parse_and_append(o, fp, i) ;
	}

	if( !(rcfile = my_strdup(getenv("HOME")) ) ) {
		Warn("$HOME undefined.") ;
		return ;
	}

	/* reading program description from the local file */
	rcfile = realloc(rcfile, strlen(rcfile) + strlen(ARKA_DIR) + 20) ;
	strcat(rcfile, ARKA_DIR) ;
	strcat(rcfile, "/programs") ;

	if( !(fp = fopen(rcfile, "r")) ) { 
		Warn("File %s could not be opened", rcfile) ;
	} else {
		Warn("reading %s", rcfile) ;
		i = parse_and_append(o, fp, i) ;
	}

	number_of_programs = i ;

	Warn("%i programs loaded", number_of_programs) ;
	return ;
}


/* ----------------------- program dialog - related functions ----------------- */

/* this is called by the "Browse..." button in the main options dialog */
void browse_files(GtkWidget *kontr, gpointer dane) {
	opt_s *main_o ;
	entry_option_s *o ;
	data_s *d ;
	char *tmp ;

	o = (entry_option_s *) dane ;
	main_o = o->main_opts ;

	if(o->strvalue) free(o->strvalue) ;
	o->strvalue = file_get_name(main_o, NULL) ;
	gtk_entry_set_text(GTK_ENTRY(o->widget), o->strvalue) ;

}


/* clicking on "check" button calls this function, which records the change */
void toggle_optstat(GtkWidget *kontr, gpointer dane) {
	int *ip ;

	ip = (int *) dane ;
	*ip = ! *ip ;
}


/* called if "OK" in the option dialog is pressed. Computates the option line */
void close_options(GtkWidget *kontr, gpointer dane) {
	data_s *d ;
	int i, j, *optstat ;
	char *c, *res, tmp[100] ;
	entry_s *p ;
	entry_option_s *o ;
	

	/* casting data-to-be-processed into apriopriate types */
	d = (data_s *) dane ;
	p = (entry_s *) d->data1 ;

	res = malloc(2000) ; /* yeah I know this is evil */
	Warn("close_options: chosen program is %s", p->name) ;
	sprintf(res, "%s ", p->name) ;

	/* for each option... */
	for(i = 0 ; res && i < p->nopts ; i++) {

		o = &p->options[i] ;

		switch(o->type){
			case BOOLEAN:
				if(o->set) {
					sprintf(tmp, "%s ", o->designator) ;
					strcat(res, tmp) ;
				}
				break ;

			case INT:
				if(o->set) {
					o->intvalue = gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(o->widget)) ;
					sprintf(tmp, "%s %i ", o->designator, o->intvalue) ;
					strcat(res, tmp) ;
				}
				break ;

			case DOUBLE:
				if(o->set) {
					o->doublevalue = 
						gtk_spin_button_get_value_as_float(GTK_SPIN_BUTTON(o->widget)) ;
					sprintf(tmp, "%s %f ", o->designator, o->doublevalue) ;
					strcat(res, tmp) ;
				}
				break ;

			case STRING:
			case FILENAME:
				/* getting the user data */
				if(o->strvalue) free(o->strvalue) ;
				o->strvalue = gtk_editable_get_chars(GTK_EDITABLE(o->widget), 0, -1) ;
				if(o->strvalue && strlen(o->strvalue) > 0) o->set = TRUE ;
				else o->set = FALSE ;

				/* if entry field empty, the option is not added */
				if(o->set){
					sprintf(tmp, "%s %s ", o->designator, o->strvalue) ;
					strcat(res, tmp) ;
				}

				break ;

			case HIDDEN:
				sprintf(tmp, "%s %s ", o->designator, o->defvalue) ;
				strcat(res, tmp) ;
			break ;

			default:
			break ;
		}

	}

	gtk_widget_destroy(d->window1) ;
	gtk_main_quit() ;
	d->data = res ;
}


/* produces a dialog for choosing option for a given program and returns the
 * programs option string */
char* set_options(int wh, opt_s *main_opts) {
	GtkWidget *tmp, *sep, *label, *okno, *ypole, *hpole, *optlab, *ok, *cancel ;
	GtkWidget *notebook ;
	GtkAdjustment * adj ;
	int i, j ;
	char *res = NULL ;
	data_s d ;
	entry_option_s *o ;

	/* this structure will hold the data passed to close_options() */
	d.data = NULL ;
	d.data1 = programs[wh] ;

	Warn("set_options: selection %i", wh) ;
	Warn("name: %s", programs[wh]->name) ;

	/* main option window */
	okno = gtk_window_new(GTK_WINDOW_DIALOG) ;
	gtk_window_set_title(GTK_WINDOW(okno), programs[wh]->name) ;
	gtk_widget_show(okno) ;
	gtk_signal_connect(GTK_OBJECT(okno), "destroy",
		GTK_SIGNAL_FUNC(gtk_main_quit), okno) ;

	d.window1 = okno ;

	label = gtk_label_new("Main") ;
	ypole = gtk_vbox_new(FALSE, 5) ;
	notebook = gtk_notebook_new() ;
	gtk_notebook_append_page(GTK_NOTEBOOK(notebook), ypole, label) ;
	gtk_container_add(GTK_CONTAINER(okno), notebook) ;

	gtk_widget_show(notebook) ;
	gtk_widget_show(label) ;
	gtk_widget_show(ypole) ;

	if(programs[wh]->long_description) {
		label = gtk_label_new("More..") ;
		tmp = gtk_label_new(programs[wh]->long_description) ;
		gtk_label_set_justify(GTK_LABEL(tmp), GTK_JUSTIFY_LEFT) ;
		gtk_notebook_append_page(GTK_NOTEBOOK(notebook), tmp, label) ;
		gtk_widget_show(label) ;
		gtk_widget_show(tmp) ;
	}

	/* brief program description */
	label = gtk_label_new(programs[wh]->short_description) ;
	gtk_box_pack_start(GTK_BOX(ypole), label, FALSE, FALSE, 2) ;
	gtk_label_set_line_wrap(GTK_LABEL(label), TRUE) ;
	gtk_widget_show(label) ;

	sep = gtk_hseparator_new() ;
	gtk_widget_show(sep) ;
	gtk_box_pack_start(GTK_BOX(ypole), sep, FALSE, FALSE, 2) ;

	/* for each option, we create a widget stored in opsent and record a state in
	 * optstat */
	for(i = 0, j = 0 ; i < programs[wh]->nopts ; i++, j++) {
		
		o = &programs[wh]->options[i] ;

		hpole = gtk_hbox_new(FALSE, 5) ;
		gtk_box_pack_start(GTK_BOX(ypole), hpole, FALSE, FALSE, 2) ;
		gtk_widget_show(hpole) ;

		switch(o->type){

			case BOOLEAN:

				/* just a simple check button */
				o->widget = gtk_check_button_new_with_label(o->description) ;
				gtk_box_pack_start(GTK_BOX(hpole), o->widget, FALSE, FALSE, 5) ;
				gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(o->widget), o->set) ;
				gtk_widget_show(o->widget) ;
				gtk_signal_connect(GTK_OBJECT(o->widget), "toggled",
					GTK_SIGNAL_FUNC(toggle_optstat), &o->set) ;
				break ;

			case INT:
			case DOUBLE:

				/* checkbox */
				if(!o->required) {
					tmp = gtk_check_button_new() ;
					gtk_box_pack_start(GTK_BOX(hpole), tmp, FALSE, FALSE, 5) ;
					gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(tmp), o->set) ;
					gtk_signal_connect(GTK_OBJECT(tmp), "toggled",
						GTK_SIGNAL_FUNC(toggle_optstat), &o->set) ;
					gtk_widget_show(tmp) ;
				}

				/* spin button */
				adj = (GtkAdjustment *) gtk_adjustment_new(0, INT_MIN, INT_MAX, 1, 10, 10) ;

				if(o->type == INT)
					o->widget = gtk_spin_button_new(adj, 1, 0) ;
				else
					o->widget = gtk_spin_button_new(adj, 1, 2) ;

				gtk_box_pack_start(GTK_BOX(hpole), o->widget, FALSE, FALSE, 5) ;
				if(o->set)
					gtk_spin_button_set_value(GTK_SPIN_BUTTON(o->widget), o->intvalue) ;
				gtk_widget_show(o->widget) ;
				gtk_spin_button_set_numeric(GTK_SPIN_BUTTON(o->widget), FALSE) ;

				/* label */
				optlab = gtk_label_new(o->description) ;
				gtk_box_pack_start(GTK_BOX(hpole), optlab, FALSE, FALSE, 5) ;
				gtk_label_set_justify(GTK_LABEL(optlab), GTK_JUSTIFY_LEFT) ;
				gtk_label_set_line_wrap(GTK_LABEL(optlab), TRUE) ;
				gtk_widget_show(optlab) ;

				break ;

			case STRING:
			case FILENAME:

				/* we need to pass the main_opts pointer to file_get_name */
				o->main_opts = main_opts ;

				/* option entry */
				o->widget = gtk_entry_new() ;
				gtk_box_pack_start(GTK_BOX(hpole), o->widget, FALSE, FALSE, 5) ;
				if(o->strvalue) gtk_entry_set_text(GTK_ENTRY(o->widget), o->strvalue) ;
				gtk_widget_show(o->widget) ;

				/* when clicked, we go through the directory */
				if(o->type == FILENAME) {
					tmp = gtk_button_new_with_label("Browse...") ;
					gtk_box_pack_start(GTK_BOX(hpole), tmp, FALSE, FALSE, 5) ;
					gtk_signal_connect(GTK_OBJECT(tmp), "clicked",
						GTK_SIGNAL_FUNC(browse_files), o) ;
					gtk_widget_show(tmp) ;
				}
	
				/* description of this option */
				optlab = gtk_label_new(o->description) ;
				gtk_box_pack_start(GTK_BOX(hpole), optlab, FALSE, FALSE, 5) ;
				gtk_label_set_justify(GTK_LABEL(optlab), GTK_JUSTIFY_LEFT) ;
				gtk_label_set_line_wrap(GTK_LABEL(optlab), TRUE) ;
				gtk_widget_show(optlab) ;

				break ;
			default:
				break ;
		}
			
	}

	/* separator + OK + cancel buttons */
	sep = gtk_hseparator_new() ;
	gtk_widget_show(sep) ;
	gtk_box_pack_start(GTK_BOX(ypole), sep, FALSE, FALSE, 5) ;

	hpole = gtk_hbox_new(FALSE, 5) ;
	gtk_widget_show(hpole) ;
	gtk_box_pack_start(GTK_BOX(ypole), hpole, FALSE, FALSE, 5) ;

	ok = gtk_button_new_with_label("OK") ;
	gtk_widget_show(ok) ;
	gtk_box_pack_start(GTK_BOX(hpole), ok, FALSE, FALSE, 10) ;
	gtk_signal_connect(GTK_OBJECT(ok), "clicked",
		GTK_SIGNAL_FUNC(close_options), &d) ;

	cancel = gtk_button_new_with_label("Cancel") ;
	gtk_widget_show(cancel) ;
	gtk_box_pack_start(GTK_BOX(hpole), cancel, FALSE, FALSE, 10) ;
	gtk_signal_connect(GTK_OBJECT(cancel), "clicked",
		GTK_SIGNAL_FUNC(zakoncz_dialog), okno) ;

	gtk_main() ;

	/* the resulting command line gets stored by close_options() in d.data */
	res = (char*) d.data ;
	if(res) printf("command line: %s \n", res) ;
	return res ;
}


/* ---------- f. related to parsing the config files ------------------ */

/* prints an entry */
int entry_print(entry_s *e) {
	int i ;
	entry_option_s *o ;

	printf("%s - %s\n\n", e->entry, e->short_description) ;
	printf("Usage:\n\t%s ", e->name) ;
	for(i = 0 ; i < e->nopts ; i++) {
		o = &e->options[i] ;
		if(o->type == 4) continue ;
		if(o->required) printf("<") ;
		else printf("[") ;
		if(o->designator) {
			printf("-%s", o->designator) ;
			if(o->type) printf(" ") ;
		}
		switch(o->type){
			case 0:
				break ;
			case 1:
				printf("int_value") ;
				break ;
			case 2:
				printf("float_value") ;
				break ;
			case 3:
				printf("file") ;
				break ;
			default:
				break ;
		}
		if(o->required) printf("> ") ;
		else printf("] ") ;
	}
	printf("\n") ;


	for(i = 0 ; i < e->nopts ; i++) {
		if(!i) printf("Options:\n") ;
		o = &e->options[i] ;
		if(o->type == 4) continue ;
		printf("\t") ;
		if(o->designator) printf("-%s ", o->designator) ;
		switch(o->type){
			case 0:
				break ;
			case 1:
				printf("int_value ") ;
				break ;
			case 2:
				printf("float_value ") ;
				break ;
			case 3:
				printf("file ") ;
				break ;
			default:
				break ;
		}

		if(o->description) printf(": %s ", o->description) ;
		if(o->defvalue) printf(" (default: %s) ", o->defvalue) ;
		printf("\n") ;
	}
	printf("\n") ;

	if(e->long_description) printf("Description:\n\n%s\n", e->long_description) ;

	return EXIT_SUCCESS ;
}


/* parses the line starting the program section, which contains program
 * name and the short description */
int parse_program(char *line, entry_s *entries) {
	int current ;
	char *name, *p, *q, *r ;

	p = strchr(line, '(') ;

	/* reading program name; if no name is given, do not do anything */
	if(!p) return EXIT_FAILURE ;

	/* initializing the new entry_s structure */
	entries->short_description = NULL ;
	entries->branch = NULL ;
	entries->long_description = NULL ;
	entries->options = NULL ;
	entries->nopts = 0 ;

	/* start the actual parsing */
	p++ ;
	q = strchr(line, ')') ;
	if(q) { *q = '\0' ; q++ ; }
	entries->name = my_strdup(p) ;

	Warn("parse: %s", p) ;

	p = strchr(q, '(') ;
	if(!p) {
		entries->entry = my_strdup(entries->name) ;
		return EXIT_SUCCESS ;
	}

	/* reading menu entry */
	p++ ;
	q = strchr(p, ')') ;
	if(q) { *q = '\0' ; q++ ; }
	entries->entry = my_strdup(p) ;

	p = strchr(q, '(') ;
	if(!p) return EXIT_SUCCESS ;

	/* reading branch information */
	p++ ;
	q = strchr(p, ')') ;
	if(q) { *q = '\0' ; q++ ; }  
	entries->branch = my_strdup(p) ;

	/* reading short description */
	p = strchr(q, '(') ;
	if(!p) {
		entries->short_description = my_strdup("no description") ;
		return EXIT_SUCCESS ;
	}

	p++ ;
	q = strchr(p, ')') ;
	if(q) *q = '\0' ; 
	entries->short_description = my_strdup(p) ;

	return EXIT_SUCCESS ;
}


/* adds a line to the long description of the program, allocating memory if
 * needed */
int parse_description(char *line, entry_s *e) {
	int i ;

	if(!e->long_description) {
		e->long_description = malloc(strlen(line) + 1) ;
		*e->long_description = '\0' ;
	} else {
		i = strlen(line) + strlen(e->long_description) + 1 ;
		e->long_description = realloc(e->long_description, i) ;
	}

	if(!e->long_description) return EXIT_FAILURE ;
	strcat(e->long_description, line) ;

	return EXIT_SUCCESS ;
}


/* parses an option line and stores it in the apropriate structure */
/* see the sample rcfile for more information */
int parse_option(char *line, entry_s *e, int i) {
	entry_option_s * o ;
	char *p, *q ;

	p = strchr(line, '(') ;
	if(!p) return EXIT_FAILURE ;

	e->nopts++ ;

	/* reserving necessary memory; options are stored in an array */
	if(!e->options) e->options = malloc(sizeof(*(e->options))) ;
	else e->options = realloc(e->options, 
		e->nopts * sizeof(*(e->options))) ;

	/* current option */
	o = &e->options[e->nopts - 1] ;

	/* default settings. From now on, we can exit anytime */
	o->designator = NULL ;
	o->description = NULL ;
	o->defvalue = NULL ;
	o->required = FALSE ;
	o->type = BOOLEAN ;
	o->set = FALSE ;

	o->strvalue = NULL ;
	o->intvalue = 0 ;
	o->doublevalue = 0.0 ;

	/* reading designator, that is e.g. 'h' for option '-h' */
	q = strchr(p, ')') ;
	p++ ;

	if(p != q) {
		if(q) { *q = '\0' ; q++ ; } 
		o->designator = my_strdup(p) ;
	} else {
		o->designator = my_strdup("") ;
	}

	/* reading next token: option type, e.g. "float" for a floating number */
	p = strchr(q, '(') ;
	if(!p) return EXIT_SUCCESS ;
	p++ ;
	q = strchr(p, ')') ;
	if(p == q) o->type = BOOLEAN ;
	else {
		if(q) { *q = '\0' ; q++ ; }
		if(strcmp(p, "int") == 0) o->type = INT ;
		else if(strcmp(p, "float") == 0) o->type = DOUBLE ;
		else if(strcmp(p, "file") == 0) o->type = FILENAME ;
		else if(strcmp(p, "string") == 0) o->type = STRING ;
		else if(strcmp(p, "invisible") == 0) o->type = HIDDEN ;
		else Warn("bad option in line %i: %s", i, p) ;
	}

	/* reading default value */
	p = strchr(q, '(') ;
	if(!p) return EXIT_SUCCESS ;
	q = strchr(p, ')') ;
	p++ ;

	if(p != q) {
		if(q) { *q = '\0' ; q++ ; } 
		o->defvalue = my_strdup(p) ;
		o->set = TRUE ;
		switch(o->type){
			case BOOLEAN:
			case INT:
				o->intvalue = atoi(o->defvalue) ;
				break ;
			case DOUBLE:
				o->doublevalue = atof(o->defvalue) ;
				break ;
			case FILENAME:
			default:
				o->strvalue = my_strdup(o->defvalue) ;
				break ;
		}
	}

	/* reading description of the option */
	p = strchr(q, '(') ;
	if(!p) return EXIT_SUCCESS ;
	q = strchr(p, ')') ;
	p++ ;

	if(p != q) {
		if(q) { *q = '\0' ; q++ ; } 
		o->description = my_strdup(p) ;
	}

	/* additional keywords? */
	p = strchr(q, '(') ;
	if(!p) return EXIT_SUCCESS ;
	q = strchr(p, ')') ;
	p++ ;
	if(p != q) {
		if(q) { *q = '\0' ; q++ ; } 
		if(strcmp(p, "required") == 0) {
			o->required = TRUE ;
			o->set = TRUE ;
		}
	}

	return EXIT_SUCCESS ;
}


/* main function parsing the programrc file */
entry_s * parse_rc(FILE *in, int *line) {
	
	char bufor[BUFSIZ], **branches, *p, *q, *b ;
		/* the following three variables record the current state of the
		 * reading - that is, in which section of the rcfile we are now */
	int reading_options = FALSE, 
		reading_description = FALSE,

		section = 0,
		numofopts = 0,

		/* keeping track of start(program) and end(program) */
		program_found = FALSE,
		program_end_found = FALSE;
	entry_s *result ;

	result = NULL ;

	/* skipping to the first start(program) entry */
	while( !program_found && (fgets(bufor, BUFSIZ, in)) != NULL) {
		(*line)++ ;

		if( (p = strstr(bufor, "start(program)")) == bufor) {
			result = malloc(sizeof(*result)) ;

			p += strlen("start(program)") ;
			if(parse_program(p, result) == EXIT_FAILURE) {
				fprintf(stderr, "wrong entry in line %i\n", *line) ;
				free(result) ;
				return NULL ;
			}
			program_found = TRUE ;
		}
	}


	if(!program_found) return NULL ;

	while(!program_end_found && (fgets(bufor, BUFSIZ, in)) != NULL) {
		(*line)++ ;

		/* skipping comments and empty lines */
		if(*bufor == '\n' || *bufor == '#') continue ;
		b = bufor ;
		while(isspace(*b)) b++ ;

		section = 0 ;
		if(strstr(b, "start") == b) section = 1 ;
		if(strstr(b, "end") == b) section = 2 ;
		
		/* read one of the keywords -- 'start' or 'end' */
		if(section) {
			p = strchr(bufor, '(') ;
			q = strchr(p, ')') ;
			if(!p || !q) {
				fprintf(stderr, "error in configuration file line %i\n", *line) ;
				continue ;
			}
			p++ ;
			*q ='\0' ;
			q++ ;
		} 

		/* a 'normal line' */
		if(section == 0) {

			if(reading_options) 
				if(parse_option(bufor, result, *line)) 
					fprintf(stderr, "error parsing option in line %i\n", *line) ;

			if(reading_description) 
				if(parse_description(bufor, result)) 
					fprintf(stderr, "error parsing a description line %i\n", *line) ;
		}

		/* read a 'start(something)' section */
		if(section == 1) {

			/* section parameter is 'options' */
			if(strcmp(p, "options") == 0 && !reading_description) {
				if(reading_options) {
					fprintf(stderr, "repetition of start(options) on line %i\n", *line) ;
				}
				reading_options = TRUE ;
			}

			/* section parameter is 'description' */
			if(strcmp(p, "description") == 0 && !reading_description) {
				if(reading_options) {
					fprintf(stderr, "warn: start(options) unfinished on line %i\n", *line) ;
					reading_options = FALSE ;
				}
				reading_description = TRUE ;
			}
					
		} /* end of the 'if(section == 1)' statement '/

		/* bufor starts with an end() statement */
		if(section == 2) {

			/* end of the description */
			if(strcmp(p, "description") == 0) {
				if(!reading_description) {
					fprintf(stderr, 
						"warn: orphan end(description) section at line %i\n", *line) ;
				}
				reading_description = FALSE ;
			}

			/* end of a programs section */
			if(strcmp(p, "program") == 0 && !reading_description) {
				if(reading_options) {
					fprintf(stderr, 
						"warn: orphan start(options) section in line %i\n", *line) ;
				}
				reading_description = FALSE ;
				reading_options = FALSE ;
				program_end_found = TRUE ;
			}

			/* end of an options section */
			if(strcmp(p, "options") == 0 && !reading_description) {
				if(!reading_options)
					fprintf(stderr, 
						"warn: orphan end(options) section in line %i\n", *line) ;
				reading_options = FALSE ;
			}
			
		} /* end of the if(section == 2) statement */


	} /* end of the main 'while' loop */

	if(!program_end_found) 
		fprintf(stderr, "Warning: no end(program) statement found\n") ;

	return result ;
}
