/* ----------------------------------------------------------------------
 * Program for making ACC plots of sequences
 * Copyright (C) 2000 January Weiner III
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307,
 * USA.
 ---------------------------------------------------------------------- */

#include <stdio.h>
#include <ctype.h>
#include <stdlib.h>
#include <time.h>

#include "genpak.h"
#include "gp_getopt.h"

#define VERSION "0.1"
#define PROGNAME "gp_scan"

char *progname ;

typedef struct {
	FILE *in ;
	FILE *out ; 
	int z1 ;
	int z2 ;
	int win ;
	int ndesc ;
	int sequence ;
	int print_lag ;
	int lag ; } opt_s ;

	int ScanSeq(sekw* inseq, opt_s options, double** descriptor_t) ;
	double **ReadDescT(opt_s options) ;

/*
 *
 */

int main(int argc, char *argv[])
{
	extern int optind ;
	extern char *optarg ;
	int width = 70 ; /* width with which the sequence gets formatted */
	opt_s options ;
	double** descriptor_t ;
	sekw *inseq, *outseq ;

	int c;
	char message[100] ;
	int errflg = 0 ;

	options.z1= 1 ;
	options.z2= 1 ;
	options.ndesc = 3 ;
	options.lag = 1 ;
	options.win = 3 ;
	options.sequence = FALSE ;
	options.print_lag = FALSE ;

	progname = argv[0] ;

	while ((c = gp_getopt(argc, argv, "psl:w:z:Z:lqvdh")) != EOF)
		switch(c) {
		case 'w':
			if(sscanf(optarg,"%i",&options.win) != 1)
				gp_error("main: Could not parameter >%s< for option -w ",optarg) ;
			if(debug) gp_warn("main: window used is %i",options.win) ;
			break ;
		case 'l':
			if(sscanf(optarg,"%i",&options.lag) != 1)
				gp_error("main: Could not parameter >%s< for option -l ",optarg) ;
			if(debug) gp_warn("main: lag used is %i",options.lag) ;
			break ;
		case 'z':
			if(sscanf(optarg,"%i",&options.z1) != 1)
				gp_error("main: Could not parameter >%s< for option -z ",optarg) ;
			if(debug) gp_warn("main: first component is variable no.%i",options.z1) ;
			break ;
		case 'Z':
			if(sscanf(optarg,"%i",&options.z2) != 1)
				gp_error("main: Could not parameter >%s< for option -Z ",optarg) ;
			if(debug) gp_warn("main: second component is variable no.%i",options.z2) ;
			break ;
		case 'p':
			options.print_lag = TRUE ;
			if(debug) gp_warn("main: I will also print the lagged parameters") ;
			break ;
		case 's':
			options.sequence = TRUE ;
			if(debug) gp_warn("main: I will also print the sequence") ;
			break ;
		case 'q':
			quiet = TRUE ;
			break ;
		case 'v':
			fprintf(stderr,"%s version %s\n",progname,VERSION) ;
			exit(0) ;
			break ;
		case 'd':
			debug = TRUE ;
			gp_warn("Running in debug mode") ;
			break ;
		case 'h':
			Help() ;
			break ;
		default:
			errflg++ ;
			break;
		}


	if(errflg) gp_error("Type '%s -h' for help",progname) ;

/* open the file pointer to read the sequences 
 * from: standard input or a file provided? */
	if(optind >= argc) options.in = stdin ;
	else options.in = gp_file_open(argv[optind],"r") ;

/* opening the file pointer to write the output: 
 * standard output or file provided? */
	optind++ ;

	if(optind >= argc) options.out = stdout ;
	else options.out = gp_file_open(argv[optind],"wb") ;

	if(debug) 
		gp_warn("no. of z1 %i, no. of z2 %i, lag %i, window %i",
			options.z1, options.z2, options.lag, options.win) ;

	descriptor_t = ReadDescT(options) ;

	while( (inseq = gp_seq_read(options.in)) != NULL) {
		ScanSeq(inseq,options,descriptor_t) ;
	}

	
	if(html) gp_warn_print_all(options.out) ;
	fclose(options.out) ;
	fclose(options.in) ;
	return(0);
}


double **ReadDescT(opt_s o) {

	double **ret ;
	int i,j ;

	ret = calloc(129,sizeof(*ret)) ;
	for(i = 0;i<130;i++) {
		ret[i] = calloc(3,sizeof(*ret[i])) ;
		for(j = 0;j<3;j++) ret[i][j] = 0.0 ;
	}

	if(debug) gp_warn("ReadDescT: Zeroed and OK") ;

	ret['A'][0] = -1;
	ret['A'][1] = -1;
	ret['A'][2] = 1;

	ret['C'][0] = 1;
	ret['C'][1] = -1;
	ret['C'][2] = -1;

	ret['G'][0] = -1;
	ret['G'][1] = 1;
	ret['G'][2] = -1;

	ret['T'][0] = 1;
	ret['T'][1] = 1;
	ret['T'][2] = 1;

	ret['U'][0] = 1;
	ret['U'][1] = 1;
	ret['U'][2] = 1;

	
	return ret ;

	

}


int ScanSeq(sekw* s, opt_s o, double** d) {

	long i,j,k,l,c1,c2 ;
	double z1, z2, zz ;
	

	l = strlen(s->sequ) ;
	if(debug)
		gp_warn("Sequence %s, length %i, scanning from %i to %i",
			s->name, l, 1, (l - o.lag - o.win + 1)) ;

	if(o.sequence) {
		for(i = 0;i<(l);i++)
			fprintf(o.out,"%c\t",s->sequ[i]) ;
		fprintf(o.out,"\n") ;
	}


	for(i = 0;i<(l - o.lag - o.win + 1);i++) {
		zz = 0 ;
		for(j = 0;j<o.win;j++) {
			c1= toupper(s->sequ[i+j]) ;
			c2= toupper(s->sequ[i+j+o.lag]) ;
			z1= d[c1][o.z1-1] ;
			z2= d[c2][o.z2-1] ;
			zz += z1*z2 ;
			if(debug) gp_warn("%c-%1.0f %c-%1.0f",c1,z1,c2,z2) ;
		}
		zz /= o.win ;
		fprintf(o.out,"%.4f\t",zz) ;
		if(debug) gp_warn("zz %2.2f",zz) ;
	}

	fprintf(o.out,"\n") ;

	if(o.print_lag) {

		for(i = 0;i<o.lag;i++) fprintf(o.out,"----\t") ;
		for(i = 0;i<(l - o.lag - o.win + 1);i++) {
			zz = 0 ;
			for(j = 0;j<o.win;j++) {
				c1= toupper(s->sequ[i+j]) ;
				c2= toupper(s->sequ[i+j+o.lag]) ;
				z1= d[c1][o.z1-1] ;
				z2= d[c2][o.z2-1] ;
				zz += z1*z2 ;
				if(debug) gp_warn("%c-%1.0f %c-%1.0f",c1,z1,c2,z2) ;
			}
			zz /= o.win ;
			fprintf(o.out,"%.4f\t",zz) ;
			if(debug) gp_warn("zz %2.2f",zz) ;
		}
		fprintf(o.out,"\n") ;
	}


	return EXIT_SUCCESS ;

}

/* Standard mesage */

void Help()
{
printf("\n");
printf("%s, v. %s- computing auto-cross-correlation profiles",PROGNAME,VERSION);
printf("\n");
printf("  Usage:\n");
printf("     %s [options] [ input file ] [ output file ]\n",progname);
printf("\n");
printf("  Options:\n");
printf("     -l value : set lag to value\n");
printf("     -w value : set window width\n");
printf("     -s       : print also the sequence\n");
printf("     -p       : print also the lagged parameters\n");
printf("     -z value : value is the number of the first variable descriptor\n");
printf("     -Z value : value is the number of the second variable descriptor\n");
printf("     -v       : print version information & exit\n");
printf("     -h       : print this help screen & exit\n");
printf("     -q       : quiet, suppress error messages\n\n");
exit(0);
}


			
