/* ----------------------------------------------------------------------
 * "shifting" gene positions
 * Copyright (C) 2000 January Weiner III
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307,
 * USA.
 ---------------------------------------------------------------------- */

#include <stdio.h>
#include <ctype.h>
#include <stdlib.h>
#include <time.h>

#include "genpak.h"
#include "gp_getopt.h"

#define VERSION "0.2"
#define PROGNAME "gp_shift"

char *progname ;

typedef struct {
	gzFile *in ;
	FILE *out ; 
	long start, end ;
	int relative_to_end ;} opt_s ;

/*
 *
 */

int main(int argc, char *argv[])
{
	extern int optind ;
	extern char *optarg ;
	int width = 70 ; /* width with which the sequence gets formatted */
	opt_s opt ;
	sekw *inseq, *outseq ;
	int c;

	opt.start = 0 ;
	opt.end = 0 ;
	opt.relative_to_end = FALSE ;

	progname = argv[0] ;

	while ((c = gp_getopt(argc, argv, "3Hqdvh")) != EOF) {

		switch(c) {
		case '3':
			opt.relative_to_end = TRUE ;
			gp_warn(_("Printing positions relative to the 3' end of the sequence")) ;
			break ;
		case 'H':
			html = TRUE ;
			if(debug) gp_warn(_("Assuming HTML type output")) ;
			break ;
		case 'q':
			quiet = TRUE ;
			break ;
		case 'v':
			fprintf(stderr,_("%s version %s\n"),progname,VERSION) ;
			exit(0) ;
			break ;
		case 'd':
			debug = TRUE ;
			gp_warn(_("Running in debug mode")) ;
			break ;
		case 'h':
			Help() ;
			break ;
		default:
			gp_error("Type '%s -h' for help",progname) ;
			break;
		}
	}

	if(optind >= argc - 1)
		gp_error(_("You must specify two numeric arguments")) ;

	gp_warn("OK argc %i optind %i",argc,optind) ;

	if(sscanf(argv[optind],"%li",&opt.start) != 1)
		gp_error(_("Could not read shift start"));
	if(debug) gp_warn("start: %i", opt.start) ;

	optind++ ;

	if(sscanf(argv[optind],"%li",&opt.end) != 1)
		gp_error(_("Could not read shift end"));
	if(debug) gp_warn("end: %i", opt.end) ;

	optind++ ;

	/* open the file pointer to read the sequences 
 	* from: standard input or a file provided? */
	if(optind >= argc) opt.in = stdin ;
	else opt.in = gp_gzOpen(argv[optind],"r") ;

	/* opening the file pointer to write the output: 
 	* standard output or file provided? */
	optind++ ;

	if(optind >= argc) opt.out = stdout ;
	else opt.out = gp_file_open(argv[optind],"w") ;

	shift_positions(opt) ;
	
	if(html) gp_warn_print_all(opt.out) ;
	fclose(opt.out) ;
	gzclose(opt.in) ;
	return EXIT_SUCCESS;
}


/* shifting positions */
int shift_positions(opt_s o) {
	char bufor[BUFSIZ], name[BUFSIZ] ;
	long s, e, ss, ee ;
	int res = 0, line = 0;
	
	while(gzgets(o.in,bufor,BUFSIZ) != NULL) {
		line++ ;
		if(debug) gp_warn("reading line %i",line) ;
		*name = '\0' ;

		if(*bufor == '\n' || *bufor == '#') continue ;

		if(sscanf(bufor,"%li %li %s ", &s, &e, name) < 2) {
			gp_warn(_("Could not read line %i"), line) ;
			continue ;
		}

		if(s < e) {
			if(o.relative_to_end) {
				ss = e - o.start ;
				ee = e + o.end ;
			} else {
				ss = s - o.start ;
				ee = s + o.end ;
			}
		} else {
				if(o.relative_to_end) {
				ss = s + o.start ;
				ee = s - o.end ;
			} else {
				ss = e + o.start ;
				ee = e - o.end ;
			}
		}
		
		fprintf(o.out,"%i %i",ss,ee) ;

		if(*name != '\0') fprintf(o.out," %s\n",name) ;
		else fprintf(o.out,"\n") ;

	}

	return EXIT_SUCCESS ;
}

/* Standard mesage */
void Help()
{
	fprintf(stdout,_(""
	"\n"
	"%s, v. %s- printing shifted gene positions "
	"\n"
	"  Usage:\n"
	"     %s [options] shift_start shift_end [ input file ] [ output file ]\n"
	"\n"
	"  Options:\n"
	"     -3       : shift relative to sequence 3' end\n"
	"     -v       : print version information & exit\n"
	"     -h       : print this help screen & exit\n"
	"     -q       : quiet, suppress error messages\n\n"),
	PROGNAME,VERSION,progname);
	exit(EXIT_SUCCESS);
}


			

			
