# Mavric -- a module for manipulating and visualizing phylogenies

# Copyright (C) 2000 Rick Ree
# Email : rree@oeb.harvard.edu
# 	   
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation; either version 2 
# of the License, or (at your option) any later version.
#   
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details. 
# 
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

from toolkit import *
from constants import *

from Mavric import phylo, newick

class NodeAdornment:
    def __init__(self, type, value, position):
        self.type = type
        self.value = value
        self.position = position

class CanvasBranch:
    """\

    CanvasBranch is a class that represents a connection between a
    parent and child node.  It knows how to draw itself on its parent
    canvas (passed to constructor), and knows how to respond to
    events: clicks, drags, and so on.

    """

    def __init__(self, canvas, child, parent):
        self.canvas = canvas
        self.group = canvas.group
        self.prefs = canvas.prefs

        self.child = child
        self.parent = parent
        self.istip = child.istip

        self.child_adorn_info = []
        self.parent_adorn_info = []

        self.view_id = canvas.id
        vid = self.view_id
        self.cx, self.cy = child[vid]
        if parent != None:
            self.px, self.py = parent[vid]
        else:
            self.px = None; self.py = None

        self.line = None

        if self.child.istip:
            self.label_text = self.child.label
        elif self.child.back and self.child.back.istip:
            self.label_text = self.child.back.label

        self.label = None
        self.length_label = None

        self.render()

    def event_cb(self, canvas_line, event):
        """\
        callback method for events generated by the canvas line
        """
        if not self.canvas.editable:
            return FALSE
        
        event_type = event.type
        if event_type == GDK.ENTER_NOTIFY:
            return FALSE

        elif event_type == GDK.LEAVE_NOTIFY:
            return FALSE

        elif event_type == GDK.BUTTON_PRESS:
            self.line.lower_to_bottom()
            if self.label:
                self.label.lower_to_bottom()
            return FALSE

        elif event_type == GDK.BUTTON_RELEASE: # drop event
            if not self.canvas.drop_branch(self,
                                           event.x, event.y,
                                           event.state):
                self.render() # do nothing if not dropped on another branch

            self.canvas.set_dragged(None)
            return FALSE

        elif event_type == GDK._2BUTTON_PRESS:  # branch double-clicked
            return FALSE

        elif event_type == GDK.MOTION_NOTIFY:
            if event.state & GDK.BUTTON1_MASK: # branch dragged with button 1
                if self.parent != None:
                    self.drag_event(event.x, event.y)
            return FALSE
        else:
            pass

        return FALSE

    def set_highlight(self, flag):
        """\
        called on mouse-over events, to highlight the branch
        """
        prefs = self.prefs
        color = prefs['BranchColor']
        if flag: color = prefs['BranchHighlightColor']
        self.line.set(fill_color=color)

    def drag_event(self, x, y):
        self.canvas.dragged = self
        points = (self.cx, self.cy, x, y)
        self.line.set(points=points)

    def show(self):
        self.line.show()
        if self.label: self.label.show()

    def hide(self):
        self.line.hide()
        if self.label: self.label.hide()

    def destroy(self):
        self.line.destroy()
        if self.label: self.label.destroy()

    def update(self):
        cx, cy = self.child[self.view_id]
        if self.parent != None:
            px, py = self.parent[self.view_id]

        flag=0
        if cx != self.cx:
            self.cx = cx; flag=1
        if cy != self.cy:
            self.cy = cy; flag=1
        if self.parent != None:
            if px != self.px:
                self.px = px; flag=1
            if py != self.py:
                self.py = py; flag=1
        if self.child.istip:
            if self.label_text != self.child.label:
                self.label_text = self.child.label
                flag=1

        showbranchlengths = self.prefs['ShowBranchLengths']
        length_label = self.length_label
        if (self.length_label == None and showbranchlengths) or \
           (self.length_label != None and (not showbranchlengths)):
            flag=1

        if flag:
            self.render()

    def recreate_branch(self):
        self.line.destroy()
        self.line = self.new_line()
        self.render()

    def new_line(self, points):
        line = self.group.add('line',
                              points=points,
                              cap_style=GDK.CAP_ROUND,
                              join_style=GDK.JOIN_ROUND,
                              width_pixels=self.prefs['BranchWidth'],
                              fill_color=self.prefs['BranchColor'])
        line.set_data('self', self)
        line.set_data('type', 'branch')
        line.set_data('child', self.child)
        line.set_data('parent', self.parent)
        line.connect('event', self.event_cb)

        return line

    def new_label(self, x=0, y=0, anchor=GTK.ANCHOR_WEST):
        prefs = self.prefs
        label_font = prefs['LeafLabelFont']
        ct = self.group.add('text',
                            text=self.label_text,
                            x=x, y=y,
                            font=label_font,
                            fill_color=prefs['LeafLabelColor'],
                            anchor=anchor)
        ct.set_data('type', 'label')
        ct.connect('event', self.label_event_cb)
        return ct

    def new_lengthlabel(self):
        child = self.child
        prefs = self.prefs
        label_font = prefs['LengthLabelFont']
        label_color = prefs['LengthLabelColor']
        label_text = str(child.length)
        if child.length == None: label_text = 'Unset'
        x_offset = -3; y_offset = -3
        anchor = GTK.ANCHOR_SE
        if self.cx < self.px and self.px != None:
            x_offset = -x_offset
            anchor = GTK.ANCHOR_SW
        label = self.group.add('text',
                               text=label_text,
                               x=self.cx+x_offset, y=self.cy+y_offset,
                               font=label_font,
                               fill_color=label_color,
                               anchor=anchor)
        label.set_data('type', 'lengthlabel')
        #label.connect('event', self.lengthlabel_event_cb)
        return label

    def label_event_cb(self, canvastext, event):
        # callback code for label events
        event_type = event.type

        if event_type == GDK._2BUTTON_PRESS:  # label double-clicked
            self.canvas.editable = 0
            x1,y1, x2,y2 = canvastext.get_bounds()

            e = GtkEntry()
            e.set_text(self.child.label)
            e.select_region(0,-1)

            def canvaswidget_destroy(widget, event, entry=e, self=self):
                if event.type == GDK.BUTTON_PRESS:
                    entry.destroy()
                    widget.connect('event', self.canvas.event_cb)
                    self.canvas.editable = 1

            ce = self.canvas.group.add('widget',
                                       widget=e,
                                       x=x1, y=y1)
            ce.set_data('entry', e)
            ce.show()

            e.connect('event', self.label_entry_event_cb, ce)
            #self.canvas.connect('event', canvaswidget_destroy)

            e.show()
            e.grab_focus()

        return FALSE

    def label_entry_event_cb(self, entry, event, canvaswidget):
        # callback code for entry events
        event_type = event.type
        canvas = self.canvas

        if event_type == GDK.KEY_PRESS:

            keyval = event.keyval

            if keyval == GDK.Return:
                self.child.label = entry.get_text()
                self.label.set(text=self.child.label)
                self.label_text = self.child.label
                canvaswidget.destroy()
                self.canvas.editable = 1

            elif keyval == GDK.Escape:
                self.canvas.editable = 1
                canvaswidget.destroy()
                
        return FALSE

    def render(self):
        # calculate the point coordinates
        cx = self.cx; cy = self.cy
        prefs = self.prefs
        child = self.child
        if self.parent != None:
            px = self.px; py = self.py
        else:
            if self.child.back == None:
                if prefs['TreeOrientation'] == EAST:
                    px = cx - prefs['UnitHorizontal']
                elif prefs['TreeOrientation'] == WEST:
                    px = cx + prefs['UnitHorizontal']
            else:
                px = child.back[self.view_id][0]
            py = cy
                
        style = prefs['Style']
        if style == SQUARED:
            points = (cx,cy, px,cy, px,py)
        elif style == SLANTED:
            points = (cx,cy, px,py)
        else:
            raise 'CanvasNodeError', "unknown style '%s'in render()" % style

        if self.line: self.line.destroy()
        self.line = self.new_line(points=points)

        if self.label:
            self.label.destroy()

        if self.length_label:
            self.length_label.destroy()
            #self.length_label = None
        
        if prefs['ShowBranchLengths'] and (not self.child.istip):
            x_offset = -3; y_offset = -3
            anchor = GTK.ANCHOR_SE
            if cx < px and px != None:
                x_offset = -x_offset
                anchor = GTK.ANCHOR_SW
            self.length_label = self.new_lengthlabel()

        if self.child.__dict__.get('collapsed'):
            w = self.prefs['BranchWidth']
            self.canvas.group.add('ellipse',
                                  x1=cx-w, y1=cy-w, x2=cx+w, y2 = cy+w,
                                  fill_color = 'red')

        if self.child.istip:
            if cx > px:
                self.label = self.new_label(cx+10, cy, GTK.ANCHOR_WEST)
            else:
                self.label = self.new_label(cx-10, cy, GTK.ANCHOR_EAST)
        elif self.child.back and self.child.back.istip:
            if self.prefs['TreeOrientation'] == EAST:
                self.label = self.new_label(px-10, py, GTK.ANCHOR_EAST)
            else:
                self.label = self.new_label(px+10, py, GTK.ANCHOR_WEST)

        if self.__dict__.get('child_adorned'):
            self.child_adorned.destroy()
            self.adorn_child(self.child_adorned_color)

    def adorn_child(self, color):
        self.child_adorned_color = color
        cx = self.cx; cy = self.cy
        w = self.prefs['BranchWidth']
        self.child_adorned = self.canvas.group.add('rect',
                                      x1=cx-w, y1=cy-w, x2=cx+w, y2 = cy+w,
                                      fill_color = color)
        self.child_adorned.raise_to_top()
