# Mavric -- a module for manipulating and visualizing phylogenies

# Copyright (C) 2000 Rick Ree
# Email : rree@oeb.harvard.edu
# 	   
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation; either version 2 
# of the License, or (at your option) any later version.
#   
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details. 
# 
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

from toolkit import *
from constants import *

import os, sys, string

from Mavric import phylo, newick, VERSION

from treeview import TreeView
from baseapp import BaseApp

class SimpleViewer(BaseApp):
    app_name = 'Mavric'
    app_title = 'Mavric'

    def __init__(self, root=None):
        BaseApp.__init__(self, appname=self.app_name, title=self.app_title)

        new_menuinfo = [
            UIINFO_ITEM('Pectinate tree...',
                        cb=self.menuitem_new_pectinate_tree_cb),
            UIINFO_ITEM('Balanced tree...',
                        cb=self.menuitem_new_balanced_tree_cb)
            ]

        self.file_menuinfo = [
            UIINFO_SUBTREE('New...', new_menuinfo),
            UIINFO_MENU_OPEN_ITEM(cb = self.menuitem_open_cb),
            UIINFO_MENU_SAVE_AS_ITEM(cb = self.menuitem_save_cb),
            UIINFO_ITEM('Save as PostScript...',
                        cb = self.menuitem_ps_convert_cb),
            UIINFO_SEPARATOR,
            UIINFO_MENU_CLOSE_ITEM(cb = self.menuitem_close_cb),
            UIINFO_SEPARATOR,
            UIINFO_MENU_EXIT_ITEM(cb = self.menuitem_exit_cb)
        ]

        self.main_menuinfo[0] = UIINFO_SUBTREE('File', self.file_menuinfo)

        self.view_menuinfo.insert(0,
                               UIINFO_ITEM('Toggle rooted',
                                           cb=self.menuitem_toggle_rooted_cb))
        self.view_menuinfo.insert(0,
                               UIINFO_ITEM('Toggle branchlengths',
                                     cb=self.menuitem_toggle_branchlengths_cb))
        self.view_menuinfo.insert(0,
                               UIINFO_ITEM('Toggle scaled',
                                           cb=self.menuitem_toggle_scaled_cb))
        self.create_menus(self.main_menuinfo)

        self.view = None

        if root:
            self.set_tree(root)

        self.show_all()

        self.connect('event', self.event_cb)
        self.toplevel_menus = \
                       self.get_dock().children()[0].children()[0].get_child()

    def event_cb(self, widget=None, event=None):
        #if event.type == GDK.MOTION_NOTIFY:
        #    self.appbar.set_status(self.view.canvas.status)
        #    tmpl = 'gnapp: %d, %d; canvas: %d, %d'
        #    self.appbar.set_status(tmpl %(event.x, event.y,
        #                                   self.view.canvas.x,
        #                                   self.view.canvas.y))

        event_type = event.type
        if event_type == GDK.KEY_PRESS:
            self.view.keypress_cb(widget, event)
        return FALSE

    def set_tree_from_file(self, filename):
        try:
            tree = newick.parse_from_file(filename)
            self.set_tree(tree)
        except IOError:
            self.error("Can't open %s!" % filename)

    def set_tree(self, root):
        if self.view: self.view.destroy()
        
        x1,y1, x2,y2 = self.get_allocation()
        width = x2-x1; height = y2-y1
        
        self.view = TreeView(root, 0.9*width, 0.9*height)
        self.set_contents(self.view)
        #self.connect('key_press_event', self.view.keypress_cb)
        self.show_all()

    def set_status(self, status):
        self.appbar.set_status(status)

#    def new_tree(self, menuitem=None, shape='pectinate'):
    def new_tree(self, shape):
        def f(s, self=self, shape=shape):
            v = int(s)
            if shape == 'pectinate':
                t = phylo.pectinate_tree(v)
            elif shape == 'balanced':
                t = phylo.balanced_tree(v)
            i = 1
            for lf in t.leaves():
                lf.label = 'Taxon_%d' % i; i = i+1
            self.set_tree(t)

        p = ''
        if shape == 'pectinate':
            p = 'Create pectinate tree\n\nEnter number of taxa:'
        elif shape == 'balanced':
            p = 'Create balanced tree\n\n'
            p = p+'Enter depth (tree will have 2**N species!):'
        self.request_string(p, f)

    def menuitem_new_pectinate_tree_cb(self, menuitem):
        self.new_tree('pectinate')

    def menuitem_new_balanced_tree_cb(self, menuitem):
        self.new_tree('balanced')

    def menuitem_open_cb(self, menuitem):
        self.open_file_selection.set_title('Open treefile...')
        self.open_file_selection.openfunc = self.open_file
        self.open_file_selection.show()

    def menuitem_save_cb(self, menuitem):
        self.save_file_selection.show()

    def menuitem_toggle_rooted_cb(self, menuitem):
        self.view.toggle_rooted()

    def menuitem_toggle_branchlengths_cb(self, menuitem):
        self.view.toggle_branchlengths()

    def menuitem_toggle_scaled_cb(self, menuitem):
        self.view.toggle_scaled()

    def menuitem_close_cb(self, menuitem):
        self.confirm_close()
        self.view.destroy()
        self.view = None

    def menuitem_ps_convert_cb(self, menuitem):
        self.save_file_selection.set_title('Save as PostScript')
        self.save_file_selection.savefunc = self.view.canvas.ps_convert
        self.save_file_selection.show()

    def menuitem_about_cb(self, menuitem):
        about = GnomeAbout('Mavric', VERSION, 'Copyright 2000',
                       ['Rick Ree'], 'A phylogeny editor for GNOME')
        about.show()

    def save_file(self, filename, mode):
        treestr = newick.to_string(self.view.canvas.node)+';\n'
        file = open(filename, mode)
        file.write(treestr)
        file.close()

    def open_file(self, filename):
        file = open(filename, 'r')
        content = string.strip(file.read())
        treedescs = string.split(content, ';')
        tree = newick.parse(treedescs[0])
        self.set_tree(tree)
        file.close()

    def confirm_close(self): pass
