# Mavric -- a module for manipulating and visualizing phylogenies

# Copyright (C) 2000 Rick Ree
# Email : rree@oeb.harvard.edu
# 	   
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation; either version 2 
# of the License, or (at your option) any later version.
#   
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details. 
# 
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

import sys
from cStringIO import StringIO
from toolkit import *
from constants import *
import layout
layout_node = layout.layout_node
from canvasbranch import CanvasBranch
from Mavric import phylo, newick
from fileselection import OpenFileSelection
import string, math

#try:
#    import piddlePS
#except ImportError:
#    piddlePS = None

import PSDraw

class TreeCanvas(GnomeCanvas):
    """\
    Widget derived from GnomeCanvas upon which trees are rendered.
    """
    def __init__(self, view, rootnode, width, height, prefs):
        GnomeCanvas.__init__(self)
        self.view = view
        self.prefs = prefs
        self.id = id(self)
        self.node = rootnode
        self.width = width
        self.height = height
        self.unit_hor = 0.0
        self.unit_ver = 0.0
        self.branches = []
        self.layout(width, height)

        self.node2cbranch = {}

        self.group = self.root().add('group')
        self.render()

        # save currrent state as newick string
        self.view.set_current_state(newick.to_string(rootnode))


        self.group.connect('event', self.group_event_cb)
        self.connect('event', self.event_cb)

        self.nodefuncs = []
        for f in (phylo.Fnode.make_pectinate,
                  #phylo.Fnode.rotate,
                  phylo.Fnode.swivel_180_degrees,
                  phylo.Fnode.swivel_180_recursive,
                  phylo.Fnode.collapse,
                  phylo.Fnode.make_polytomy,
                  self.graft_tree_from_file,
                  #self.adjust_branchlength,
                  self.toggle_collapsed_view,
                  self.cut_branch,
                  self.copy_branch,
                  self.paste_branch,
                  self.reroot,
                  phylo.Fnode.print_leaves):
            self.register_nodefunc(f)

        self.editable = 1
        self.clipboard = None

        for n in rootnode.descendants():
            if n.length == 0.0:
                n.length = 0.00001

    def __del__(self):
        for node in self.node.descendants():
            del node[self.id]
        if self.node.back:
            for node in self.node.back.descendants():
                del node[self.id]

    def event_cb(self, widget, event):
        if not self.editable: return FALSE
        # insert callback code for right-clicks, etc
        event_type = event.type

        if event_type == GDK.BUTTON_PRESS:
            b = event.button
            x = event.x; y = event.y
            item = self.get_item_at(x, y)
            #if item: pass
            #else: self.set_dragged(None)

        return FALSE

    def group_event_cb(self, widget, event):
        if not self.editable: return FALSE
        event_type = event.type

        if event_type != GDK.KEY_PRESS:  # highlight branch on mouseover
            item = self.get_item_at(event.x, event.y)
            if item:
                self.set_highlighted(item.get_data('self'))

        if event_type == GDK.LEAVE_NOTIFY:
            self.set_highlighted(None)

        if event_type == GDK._2BUTTON_PRESS:  # group double-clicked
            x = event.x; y = event.y
            item = self.get_item_at(x, y)
            if item and item.get_data('type') == 'branch':
                self.selected_branch = item
                node = item.get_data('child')
                self.popup_branchmenu(event.button, event.time, node)

        elif event_type == GDK.BUTTON_PRESS:
            b = event.button
            x = event.x; y = event.y
            item = self.get_item_at(x, y)

            if b == 3:
                if item and item.get_data('type') == 'branch':
                    self.selected_branch = item
                    node = item.get_data('child')
                    self.popup_branchmenu(b, event.time, node)
            return FALSE
            
        return TRUE

    def register_nodefunc(self, func, label=None):
        if label == None:
            label = string.replace(func.__name__, '_', ' ')
        self.nodefuncs.append((func, label))

    def toggle_rooted(self):
        root = self.node
        rooted = 1
        if root.back != None:
            rooted = 0

        if rooted:
            if root.next.next.next == root:  # basal dichotomy
                newroot = root.next.back
                newroot.isroot = 1
                phylo.connect(newroot, root.next.next.back)
                self.node = newroot
                for n in root.fnodes(): n.unlink()
            else:
                p = root.fnodes()[-2]
                q = p.next.back
                p.next.unlink()
                p.next = root
                phylo.connect(root, q)

        else:  # convert to rooted
            if root.next != None: # root is an internal node
                p = phylo.Fnode(data=root.data)
                q = root.fnodes()[-1]
                q.next = p
                p.next = root
                phylo.connect(root.back, p)
                root.back = None
            else:
                root.isroot = 0
                self.node = root.back
                self.node.isroot = 1
                self.toggle_rooted()
                

    def reroot(self, newroot):
        toggle_rooted_flag = 0
        if self.node.back == None:
            self.toggle_rooted()
            toggle_rooted_flag = 1
        self.node.isroot = 0
        newroot.isroot = 1
        self.node = newroot
        if toggle_rooted_flag:
            self.toggle_rooted()
        #self.node = phylo.reroot(self.node, newroot)

    def popup_branchmenu(self, button, time, node):
        menu = self.__dict__.get('branch_menu')
        if menu == None:
            self.create_nodefunc_menu()
            menu = self.branch_menu
        for m in menu.children():
            m.set_data('node', node)
        menu.popup(None, None, None, button, time)

    def create_nodefunc_menu(self):
        infolist = []
        r = 0
        for func, label in self.nodefuncs:
            info = ('/%s' % label, None, self.branchmenu_cb, r, '')
            infolist.append(info)
            r = r+1
        itemfactory = GtkItemFactory(type=GtkMenu,
                                     path="<main>",
                                     accel_group=GtkAccelGroup())
        itemfactory.create_items(infolist)
        for a in range(r):
            w = itemfactory.get_widget_by_action(a)
            w.set_data('func', self.nodefuncs[a][0])
        menu = itemfactory.get_widget('<main>')
        self.branch_menu = menu

    def branchmenu_cb(self, action, menuitem):
        node = menuitem.get_data('node')
        func = menuitem.get_data('func')
        func(node)
        self.redraw()

    def drop_branch(self, dropped, x, y, state):
        #dropped is a CanvasBranch instance
        item = self.get_item_at(x, y)
        if item: target = item.get_data('self')
        else: target = None
        if target and (target != dropped):
            if dropped.child.has_descendant(target.child) or \
               (dropped.parent.next.next.next == dropped.parent and
                dropped.parent.data == target.parent.data):
                return 0

            if state & GDK.SHIFT_MASK:  # branch swap
                tc = target.child; dc = dropped.child
                tcb = tc.back; dcb = dc.back
                phylo.connect(tc, dcb)
                phylo.connect(dc, tcb)

            else:  # prune-and-graft branches
                if dropped.parent == self.node and \
                   self.node.next.next.next == self.node:
                    oldroot = self.node
                    newroot = self.node.next
                    if newroot == dropped.child.back:
                        newroot = newroot.next
                    newroot = newroot.back
                    if oldroot.back != None:
                        phylo.connect(oldroot.back, newroot)
                    else:
                        newroot.back = None
                        oldroot.back = None
                        oldroot.unlink()
                    self.node = newroot
                    #for n in oldroot.fnodes(): n.unlink()
                else:
                    n = dropped.child.back.next
                    dropped.child.back.prune()
                    if n.next.next == n:  # collapse this node
                        n.back.back = n.next.back
                        n.next.back.back = n.back
                        n.next.unlink(); n.unlink()

                target.child.bisect()
                target.child.back.insert_fnode(dropped.child.back)

            self.redraw()
            return 1
        return 0

    def redraw(self):
        """Destroy all CanvasBranches and recreate them"""
        self.group.destroy()
        self.group = self.root().add('group', x=0,y=0)
        self.group.connect('event', self.group_event_cb)
        del self.branches; self.branches = []
        self.layout(self.width, self.height)
        self.render()
        self.update_scroll_region()

        # save currrent state as newick string
        self.view.set_current_state(newick.to_string(self.node))

    def update(self, width=None, height=None):
        """Make all CanvasBranches update themselves"""
        if width: self.width = width
        if height: self.height = height
        self.layout(self.width, self.height)
        for branch in self.branches: branch.update()
        self.update_scroll_region()

    def layout(self, width, height):
        prefs = self.prefs
        direction = self.prefs['TreeOrientation']
        font = load_font(prefs['LeafLabelFont'])

        if not self.__dict__.get('associated'):
            self.unit_hor, self.unit_ver, self.scale \
                          = layout_node(self.id, self.node, 0,0, width,height,
                                        font.width, prefs['ScaledBranches'],
                                        prefs['UnitHorizontal'],
                                        prefs['UnitVertical'], direction)


    def render(self, node=None, parent=None,
               renderfunc=None, appendfunc=None):
        if not node:
            appendfunc = self.branches.append
            appendfunc(CanvasBranch(self, self.node, None))
            renderfunc = self.render
            for child in self.node.children():
                renderfunc(child, self.node, renderfunc, appendfunc)
            back = self.node.back
            if back:
                if back.istip:
                    pass
                else:
                    for child in self.node.back.children():
                        renderfunc(child, self.node.back,
                                   renderfunc, appendfunc)

        else:
            cb = CanvasBranch(self, node, parent)
            appendfunc(cb)
            self.node2cbranch[node] = cb
            if not node.istip:
                for child in node.children():
                    #appendfunc(CanvasBranch(self, child, node))
                    renderfunc(child, node, renderfunc, appendfunc)

    def update_scroll_region(self):
        apply(self.set_scroll_region, self.root().get_bounds())

    def set_highlighted(self, item):
        highlighted = self.__dict__.get('highlighted')
        dragged = self.__dict__.get('dragged')
        if highlighted:
            if highlighted != dragged:
                try: highlighted.set_highlight(0)
                except AttributeError: pass
        if item:
            try: item.set_highlight(1)
            except AttributeError: pass
        self.highlighted = item

    def set_dragged(self, item):
        dragged = self.__dict__.get('dragged')
        if dragged:
            try: dragged.set_highlight(0)
            except AttributeError: pass
        if item:
            try: item.set_highlight(1)
            except AttributeError: pass
        self.dragged = item
        
    def graft_tree_from_file(self, node):
        assert not (node == None)
        def graft(filename, node=node, self=self,
                  parse_from_file=newick.parse_from_file):
            tree = parse_from_file(filename)
            tree.isroot = 0
            if tree.back != None:
                n = phylo.Fnode()
                tree.insert_fnode(n)
                tree = n
            newnode = node.bisect()
            newnode.add_child(tree)
            self.redraw()
        fs = OpenFileSelection(graft)
        fs.show()

    def adjust_branchlength(self, node):
        self.editable = 0

        initlen = node.length
        if initlen == None: initlen = 1.0
        print 'initlen:', initlen

        exponent = math.floor(math.log10(initlen))
        sci_not_flag = 0
        if exponent < -4.0: sci_not_flag = 1
        print 'exponent:', exponent

        if sci_not_flag and initlen != 0.0:
            initlen = initlen*(eval('1e%d' % exponent))

        a = GtkAdjustment(initlen, 0.0, initlen*10, initlen/10, initlen/10)
        sb = GtkSpinButton(a, digits=5)
        #hs = GtkHScale(a)
        #hs.show()
        d = GnomeDialog(title='Set branchlength',
                        b1='Ok', b2='Cancel')

        comp = GtkCheckButton(label='Adjust sibling lengths')
        comp.set_active(FALSE)
        comp.show()

        def set_len(sb, self=self, node=node, comp=comp,
                    initlen=initlen, exponent=exponent):
            #currentlen = node.length or node.back.length or 1.0
            value = sb.get_value_as_float()
            diff = value - initlen
            node.length=value
            if comp.get_active():
                parent = self.selected_branch.get_data('parent')
                if parent != None:
                    sibs = parent.children()
                    assert node in sibs
                    sibs.remove(node)
                    for s in sibs:
                        if s.length == None: s.length = 1.0
                        s.length = s.length-(diff/len(sibs))
            self.update()

        def reset_len(b, sb=sb, self=self, node=node, comp=comp,
                      initlen=initlen, exponent=exponent):
            value = sb.get_value_as_float()
            diff = value - initlen
            sb.set_value(initlen)
            if comp.get_active():
                parent = self.selected_branch.get_data('parent')
                if parent != None:
                    sibs = parent.children()
                    assert node in sibs
                    sibs.remove(node)
                    for s in sibs:
                        if s.length == None: s.length = 1.0
                        s.length = s.length+(diff/len(sibs))
            self.editable = 1

        def ok_cb(b, c=self, d=d):
            c.editable = 1

        sb.set_value(initlen)
        sb.show()
        sb.connect('changed', set_len)

        d.button_connect(0, ok_cb)
        d.button_connect(1, reset_len)
        #d.vbox.pack_start(hs)
        d.vbox.pack_start(sb)
        d.vbox.add(comp)

        d.run_and_close()
        self.editable = 1

    def cut_branch(self, node):
        nb = node.back
        n = nb.next
        nb.prune()
        if n.next.next == n:  # collapse this node
            n.back.back = n.next.back
            n.next.back.back = n.back
            n.next.unlink(); n.unlink()

        node.back = None
        self.clipboard = node

    def copy_branch(self, node):
        s = newick.to_string(node)
        self.clipboard = newick.parse(s)
        self.clipboard.isroot = 0
        print s
        

    def paste_branch(self, node):
        if not self.clipboard: return
        newnode = node.bisect()
        newnode.add_child(self.clipboard)
        self.clipboard = None
        
    def toggle_collapsed_view(self, node):
        collapsed = node.__dict__.get('collapsed')
        if collapsed == 1:
            node.istip = 0
            node.collapsed = 0
        elif not node.istip:
            node.istip = 1
            node.collapsed = 1
            node.label = '[Collapsed]'

    def ps_convert(self, filename, mode="w",
                   psfont='Helvetica', fontsize=12):
        x1, y1, x2, y2 = self.group.get_bounds()
        PG_HEIGHT_PTS = x2-x1
        PG_WIDTH_PTS = y2-y1

        PG_TOP = 792.0

        XOFF = 40; YOFF = -40

        if filename == "-":
            file = StringIO()
        else:
            file = open(filename, mode)

        psd = PSDraw.PSDraw(file)
        psd.begin_document()

        psd.setfont(psfont, fontsize)

        if self.scale:
            ltt = max(phylo.length_to_tips(self.node))
            scalebar_len_px = (ltt/4.0)*self.scale
            psd.line((100, 100), (100+scalebar_len_px, 100))
            psd.text((100, 120), "%02f" % (ltt/4.0))

        for item in self.group.children():
            it = item.get_type
            if it == GnomeCanvasLine.get_type:
                points = item['points'][:]
                i = 0
                while i <= len(points) - 3:
                    p1 = points[i:i+2]; p2 = points[i+2:i+4]
                    p1 = (p1[0]+XOFF, PG_TOP-p1[1]+YOFF)
                    p2 = (p2[0]+XOFF, PG_TOP-p2[1]+YOFF)

                    apply(psd.line, (p1,p2))
                    i = i+2

            elif it == GnomeCanvasText.get_type:
                s = item['text']
                x = item['x']; y = item['y']
                anchor = item['anchor']

                if anchor in (GTK.ANCHOR_W, GTK.ANCHOR_E):
                    y = y + float(fontsize)/2.0 - 2.0
                    #off = load_font(self.prefs['LeafLabelFont']).height(s)/2
                    #y = y + off
                elif anchor in (GTK.ANCHOR_N, GTK.ANCHOR_NE, GTK.ANCHOR_NW):
                    y = y + fontsize
                if anchor in (GTK.ANCHOR_E, GTK.ANCHOR_NE, GTK.ANCHOR_SE):
                    x = x - (fontsize/2.0)
                    #x = x - load_font(self.prefs['LeafLabelFont']).width(s)
                    #x = x - (item.get_bounds()[2]-item.get_bounds()[0])

                x = x+XOFF; y = PG_TOP-y+YOFF
                psd.text((x,y), s)

        psd.end_document()
        if filename == "-":
            print file.getvalue()
        else:
            file.close()

    def piddle_ps_convert(self, filename):
        assert piddlePS, 'Piddle module required for PS output'

        x1, y1, x2, y2 = self.group.get_bounds()
        PG_HEIGHT_PTS = x2-x1
        PG_WIDTH_PTS = y2-y1

        # letter paper:
        #PG_HEIGHT_PTS = 792.0
        #PG_WIDTH_PTS = 612.0

        COLOR = piddlePS.black

        pc = piddlePS.PSCanvas(size=(PG_WIDTH_PTS, PG_HEIGHT_PTS),
                               name=filename)
        pc.defaultLineColor = COLOR

        for item in self.group.children():
            it = item.get_type
            if it == GnomeCanvasLine.get_type:
                print item.__data__
                points = item['points'][:]
                i = 0
                while i <= len(points) - 3:
                    apply(pc.drawLine, points[i:i+4])
                    i = i+2

            elif it == GnomeCanvasText.get_type:
                s = item['text']
                x = item['x']; y = item['y']
                anchor = item['anchor']

                if anchor in (GTK.ANCHOR_W, GTK.ANCHOR_E):
                    y = y + pc.fontHeight()/2
                elif anchor in (GTK.ANCHOR_N, GTK.ANCHOR_NE, GTK.ANCHOR_NW):
                    y = y + pc.fontHeight()
                if anchor in (GTK.ANCHOR_E, GTK.ANCHOR_NE, GTK.ANCHOR_SE):
                    x = x - ps.stringWidth(s)

                pc.drawString(s, x, y)

        pc.flush()
