# Mavric -- a module for manipulating and visualizing phylogenies

# Copyright (C) 2000 Rick Ree
# Email : rree@oeb.harvard.edu
# 	   
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation; either version 2 
# of the License, or (at your option) any later version.
#   
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details. 
# 
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

from toolkit import *
from constants import *
from treecanvas import TreeCanvas

from Mavric import newick, phylo

class TreeView(GtkEventBox):
    def __init__(self, root, width=100, height=100):
        GtkEventBox.__init__(self)
        self.scrwin = GtkScrolledWindow()
        self.scrwin.set_policy(GTK.POLICY_AUTOMATIC, GTK.POLICY_AUTOMATIC)
        self.prefs = {
            'Style':                  SQUARED,
            'BranchColor':            'darkblue',
            'BranchHighlightColor':   'lightblue',
            'NodeColor':              'darkred',
            'ShowBranchLengths':      0,
            'ScaledBranches':         0,
            'AutoFit':                1,
            'LeafLabelColor':         'darkblue',
            'LeafLabelFont':          'fixed',
            'LengthLabelColor':       'darkred',
            'LengthLabelFont':        'fixed',
            'BranchWidth':            5,
            'TreeOrientation':        EAST,
            'UnitHorizontal':         20,
            'UnitVertical':           20 }

        self.canvas = None
        self.state_stack = []
        self.current_state = ''

        if root:
            self.canvas = TreeCanvas(self, root, width, height, self.prefs)
            self.scrwin.add(self.canvas)
            self.canvas.connect('key_press_event', self.keypress_cb)

        self.add(self.scrwin)
        self.connect('size_allocate', self.size_allocate_cb)

        self.set_usize(100,100)

    def set_tree(self, root):
        if self.canvas: self.canvas.destroy()
        x1,y1, x2,y2 = self.get_allocation()
        w = (x2-x1)*0.9; h = (y2-y1)*0.9
        self.canvas = TreeCanvas(self, root, w, h, self.prefs)
        self.scrwin.add(self.canvas)

    def size_allocate_cb(self, widget, event):
        self.update(self.prefs['AutoFit'])

    def event_cb(self, widget, event):
        return FALSE

    def keypress_cb(self, widget, event):
        v = self.scrwin.get_vadjustment()
        h = self.scrwin.get_hadjustment()
        if event.keyval == GDK.Page_Up:
            v.clamp_page(v.value-v.page_size+v.step_increment, v.value)

        elif event.keyval == GDK.Page_Down:
            v.clamp_page(v.value+v.page_size-v.step_increment,
                         v.value+2*v.page_size-v.step_increment)
            
        elif event.keyval == GDK.Up:
            v.clamp_page(v.value-(0.1*v.page_size)+v.step_increment, v.value)

        elif event.keyval == GDK.Down:
            v.clamp_page(v.value+(0.1*v.page_size)-v.step_increment,
                         v.value+2*v.page_size-v.step_increment)

        elif event.keyval == GDK.Right:
            h.clamp_page(h.value+(0.1*h.page_size)-h.step_increment,
                         h.value+2*h.page_size-h.step_increment)

        elif event.keyval == GDK.Left:
            h.clamp_page(h.value-(0.1*h.page_size)+h.step_increment, h.value)

        return FALSE

    def update(self, widget=None, event=None, autofit=1):
        #autofit = self.prefs['AutoFit']
        if autofit:
            x1,y1, x2,y2 = self.get_allocation()
            w = (x2-x1)*0.9; h = (y2-y1)*0.9
        else:
            w = None; h = None
        if self.canvas:
            self.canvas.update(w, h)

    def destroy(self):
        if self.canvas: self.canvas.destroy()
        self.scrwin.destroy()
        GtkEventBox.destroy(self)

    def get_viewable_rect(self):
        height = self.scrwin.get_vadjustment().page_size
        width = self.scrwin.get_hadjustment().page_size
        return width, height

    def toggle_branchlengths(self):
        self.prefs['ShowBranchLengths'] = not self.prefs['ShowBranchLengths']
        self.update()

    def toggle_scaled(self):
        self.prefs['ScaledBranches'] = not self.prefs['ScaledBranches']
        self.update()

    def toggle_rooted(self):
        self.canvas.toggle_rooted()
        self.canvas.redraw()

    def set_current_state(self, newickstring):
        self.state_stack.insert(0, self.current_state)
        self.current_state = newickstring
