VERSION 1.0 CLASS
BEGIN
  MultiUse = -1  'True
  Persistable = 0  'NotPersistable
  DataBindingBehavior = 0  'vbNone
  DataSourceBehavior  = 0  'vbNone
  MTSTransactionMode  = 0  'NotAnMTSObject
END
Attribute VB_Name = "Servo6K"
Attribute VB_GlobalNameSpace = False
Attribute VB_Creatable = True
Attribute VB_PredeclaredId = False
Attribute VB_Exposed = False
'SERVO CONTROL 6K4 CLASS MODULE
'This class requires use of the ErrorMessage and Stopwatch classes.

'   Copyright (C) 2004  The Institute for Systems Biology
'   For license information, read Copying.txt
'

'WinAPI declarations.
Private Declare Sub CopyMemory Lib "Kernel32" Alias "RtlMoveMemory" _
    (destination As Any, Source As Any, ByVal numbytes As Long)
Private Declare Sub Sleep Lib "Kernel32" (ByVal dwMilliseconds As Long)

'Fast Status: Do not change the ordering of elements.
Private Type FastStatusInfo
    UpdateID As Integer               ' Reserved for internal use
    Counter As Integer                ' time frame counter (2ms per count)
    MotorPos(1 To 8) As Long          ' commanded position (counts)
    EncoderPos(1 To 8) As Long        ' actual position (counts)
    MotorVel(1 To 8) As Long          ' commanded velocity (counts/sec)
    AxisStatus(1 To 8) As Long        ' axis status (TAS)
    SysStatus As Long                 ' system status (TSS)
    ErrorStatus As Long               ' user status (TER)
    UserStatus As Long                ' user status (TUS)
    Timer As Long                     ' timer value (TIM - milliseconds)
    Limits As Long                    ' limit status (TLIM)
    ProgIn(0 To 3) As Long            ' programmable input status (TIN)
    ProgOut(0 To 3) As Long           ' programmable output status (TOUT)
    Triggers As Long                  ' trigger interrupt status (TTRIG)
    Analog(1 To 2) As Integer         ' lo-res analog input voltage (TANV)
    VarB(1 To 10) As Long             ' VARB1 - VARB10
    VarI(1 To 10) As Long             ' VARI1 - VARI10
    Reserved As Long                  ' Reserved for internal use
    CmdCount As Long                  ' Command Count (from communications port)
End Type

Private Const PARKER_IP As String = "192.168.10.30"
Private Const VELO As String = "V60,60,10"
Private Const ACCE As String = "A100,100,10"
Private Const DECE As String = "AD100,100,10"
Private Const XMINLIM As Long = -250000
Private Const XMAXLIM As Long = 250000
Private Const YMINLIM As Long = -250000
Private Const YMAXLIM As Long = 250000
Private Const ZMINLIM As Long = -85000      'This one is actually important.
Private Const ZMAXLIM As Long = 250000
Private Const CCLOSE As Long = 10 'Don't wait on motion if we're this close.

' Fast Status packet size in bytes
Private Const FS_PKT_SIZE As Long = 280

' Send Variable bit masks for the SendVariable and SendVariablePacket Com6srvr Methods
Private Const VARI1 As Long = 1
Private Const VARI2 As Long = 2
Private Const VARI3 As Long = 4
Private Const VARI4 As Long = 8
Private Const VARI5 As Long = 16
Private Const VARI6 As Long = 32
Private Const VARI7 As Long = 64
Private Const VARI8 As Long = 128
Private Const VARI9 As Long = 256
Private Const VARI10 As Long = 512
Private Const VARI11 As Long = 1024
Private Const VARI12 As Long = 2048

Private Const VAR1 As Long = 4096
Private Const VAR2 As Long = 8192
Private Const VAR3 As Long = 16384
Private Const VAR4 As Long = 32768
Private Const VAR5 As Long = 65536
Private Const VAR6 As Long = 131072
Private Const VAR7 As Long = 262144
Private Const VAR8 As Long = 524288
Private Const VAR9 As Long = 1048576
Private Const VAR10 As Long = 2097152
Private Const VAR11 As Long = 4194304
Private Const VAR12 As Long = 8388608

Private Const VARB1 As Long = 16777216
Private Const VARB2 As Long = 33554432
Private Const VARB3 As Long = 67108864
Private Const VARB4 As Long = 134217728
Private Const VARB5 As Long = 268435456
Private Const VARB6 As Long = 536870912
Private Const VARB7 As Long = 1073741824
Private Const VARB8 As Long = &H80000000

' Definition of the SendVariable packet strucure
' Do not change the ordering of these data elements in the structure
Private Type SendVariableStructure
  Mask As Long
  Reserved1 As Long
  Reserved2 As Long
  Reserved3 As Long
  VarI(1 To 12) As Long
  VarR(1 To 12) As Double
  VarB(1 To 8) As Long
End Type

' SendVariable packet size in bytes
Private Const SV_PKT_SIZE As Long = 192
Private Const MOTIONDUR As Double = 30

'Variables global for this class.
Private c6k As Object               'Comm server object (Use Ethernet).
Private fsinfo As FastStatusInfo    'Fast status information.
Private gConnected As Boolean       'Flag to indicate connection state.
Private sw As Stopwatch
Private istat As Integer
Private lstat As Long


'Object constructor.
Private Sub Class_Initialize()
    Dim stat As Long, lstat As Long
    Set sw = New Stopwatch
    Set c6k = CreateObject("COM6SRVR.NET")
    stat = c6k.Connect(PARKER_IP)
    c6k.FSUpdateRate = 100 'Status update every 100 ms.  Was 25ms.
    c6k.FSEnabled = True
    'Set general servo parameters.
    lstat = c6k.Write("DRIVE0000" & vbLf)        'Turn drives off.
    lstat = c6k.Write("COMEXC0" & vbLf)          'disable continuous command execution.
    lstat = c6k.Write("T2" & vbLf)               'Pause 2 sec.
    lstat = c6k.Write("MA111" & vbLf)            'Absolute positioning mode.
    lstat = c6k.Write("SFB1" & vbLf)             'Select feedback.
    'Set servo gain parameters.
    lstat = c6k.Write("SGP12,12,10" & vbLf)       'Set proportional gain.
    lstat = c6k.Write("SGI4,4,0" & vbLf)         'Set integral gain.
    lstat = c6k.Write("SGI4,4,0" & vbLf)         'Set integral gain.
    lstat = c6k.Write("SGILIM4,0,0" & vbLf)      'Set ILIM gain.
    lstat = c6k.Write("SGV4,4,0" & vbLf)         'Set velocity gain.
    lstat = c6k.Write("SGAF4,4,0" & vbLf)        'Set AF gain.
    lstat = c6k.Write("SGVF4,4,0" & vbLf)        'Set VF gain.
    'Set target zone for end-of-motion.
    lstat = c6k.Write("LH3,3,3,3" & vbLf)            'Enable checking of end-of-travel limits.
    lstat = c6k.Write("STRGTD5,5,250" & vbLf)        'Increase target distance zone for Z axis.
    lstat = c6k.Write("STRGTD5,5,100" & vbLf)        'Increase target distance zone for Z axis.
    lstat = c6k.Write("STRGTV.01,.01,.01" & vbLf)    'Sets the velocity target zone to <= 0.01 units/sec.
    lstat = c6k.Write("STRGTT100,100,100" & vbLf)    'Sets the timeout period to 100 milliseconds on all axes.
    lstat = c6k.Write("STRGTE110" & vbLf)            'Enables the target zone criterion for XY axes.
    'Set default motion parameters.
    lstat = c6k.Write("D0,0,0" & vbLf)           'Set distances to 0.
    lstat = c6k.Write(VELO & vbLf)               'Set velocities to 10.
    lstat = c6k.Write(ACCE & vbLf)               'Set accelerations to 50.
    lstat = c6k.Write(DECE & vbLf)        'Set decelerations.
    'Setup a precise homing profile.
    lstat = c6k.Write("HOMA10,10,5" & vbLf)      'Set homing acceleration.
    lstat = c6k.Write("HOMAA10,10,5" & vbLf)     'Set homing ave accel.
    lstat = c6k.Write("HOMAD10,10,1" & vbLf)     'Set homing deceleration.
    lstat = c6k.Write("HOMADA10,10,1" & vbLf)    'Set homing ave decel.
    lstat = c6k.Write("HOMBAC111" & vbLf)        'Enable backup to home switch
    lstat = c6k.Write("HOMEDG000" & vbLf)        'Positive edge of home
    lstat = c6k.Write("HOMDF1,1,1" & vbLf)       'Set final homing direction
    lstat = c6k.Write("HOMV10,10,5" & vbLf)      'Set homing velocity a little faster.
    lstat = c6k.Write("HOMVF.5,.5,.5" & vbLf)    'Set final homing velocity
    lstat = c6k.Write("DRIVE1110" & vbLf)        'Turn drives on.
    'Outputs.
    lstat = c6k.Write("OUTFNC1-A" & vbLf)        'Set output #1 to default, programmable.
    'Error reporting.
    If (stat <> 1) Then
        ErrorMessage.Display "Error opening 6K4 Ethernet communication.", 0
    End If
End Sub


'Object destructor.
Private Sub Class_Terminate()
    lstat = c6k.Write("2%K" & vbLf)          'Be sure to kill SetTriggers program.
    lstat = c6k.Write("DRIVE0000" & vbLf)    'Turn drives off.
    lstat = c6k.Write("2%COMEXC0" & vbLf)    'disable continuous command execution.
    lstat = c6k.Write("COMEXC0" & vbLf)      'disable continuous command execution.
    lstat = c6k.Write("OUTFNC1-A" & vbLf)    'Set output #1 back to default, programmable.
    c6k.FSEnabled = False
    sw.Reset
    Do: Loop Until sw.Elapsed(1)
    Set c6k = Nothing
    Set sw = Nothing
End Sub


Private Function IsBitSet(TestValue As Long, TestBit As Long) As Boolean
    'TestBit in the Range 1..32 (6K Convention).
    Dim Pwr As Long
    ' return false if out of range
    If TestBit < 1 Or TestBit > 32 Then
        IsBitSet = False
        Exit Function
    End If
    ' If MSB set return True
    If TestBit = 32 Then
        If TestValue < 0 Then
            IsBitSet = True
        Else
            IsBitSet = False
        End If
        Exit Function
    End If
    'Comparison.
    Pwr = 2 ^ (TestBit - 1)
    IsBitSet = (TestValue And Pwr)
End Function


Private Function UnSignedLong(InData As Integer) As Long
    ' Converts a 16 bit signed integer into a 32 bit unsigned long
    If InData < 0 Then
        UnSignedLong = CLng(InData) + 65536
    Else
        UnSignedLong = CLng(InData)
    End If
End Function


Private Function SignedInt(InData As Long) As Integer
    ' Converts a 32 bit unsigned long into a 16 bit signed integer
    Dim ConversionValue As Long
    If InData > 32767 Then
        ConversionValue = InData - 65536
        SignedInt = CInt(ConversionValue)
    Else
        SignedInt = CInt(InData)
    End If
End Function


'Important properties.
'*********************

Property Get absX() As Long
    absX = c6k.MotorPos(1)
End Property

Property Get absY() As Long
    absY = c6k.MotorPos(2)
End Property

Property Get absZ() As Long
    absZ = c6k.MotorPos(3)
End Property

Property Get xHomed() As Boolean
    xHomed = IsBitSet(c6k.AxisStatus(1), 5)
End Property

Property Get yHomed() As Boolean
    yHomed = IsBitSet(c6k.AxisStatus(2), 5)
End Property

Property Get zHomed() As Boolean
    zHomed = IsBitSet(c6k.AxisStatus(3), 5)
End Property


'Ordinary subroutines and functions.
'***********************************

Public Sub WaitForMotionStop()
    sw.Reset
    While (IsBitSet(c6k.AxisStatus(1), 1)) Or (IsBitSet(c6k.AxisStatus(2), 1)) _
        Or (IsBitSet(c6k.AxisStatus(3), 1))
        'Wait for motion to stop.
        If (sw.Elapsed(MOTIONDUR)) Then
            ErrorMessage.Display "Error stopping relative Y motion", 15
            Exit Sub
        End If
    Wend
End Sub


Public Sub MoveXNoWait(ByVal x As Long)
    'Here we take relative coordinates, implement with absolute ones.
    'This is a special MoveX for Fire on the Fly.  It is nonmodal.
    'Error checking.
    If ((absX + x) > XMAXLIM Or (absX + x) < XMINLIM) And (xHomed) Then
        MsgBox "Error: absolute X coordinate out of range."
        Exit Sub
    End If
    'Cause motion in axis 1.  May end sub even before motion stops.
    If (x = 0) Then Exit Sub
    x = x + absX
    lstat = c6k.Write("D" + CStr(x) & vbLf)          'Set distance.
    lstat = c6k.Write("GO1" & vbLf)                  'Command motion.
End Sub


Public Sub MoveX(ByVal x As Long)
    'Here we take relative coordinates, implement with absolute ones.
    'Error checking.
    If ((absX + x) > XMAXLIM Or (absX + x) < XMINLIM) And (xHomed) Then
        MsgBox "Error: absolute X coordinate out of range."
        Exit Sub
    End If
    
    'Cause motion in axis 1.
    If (x = 0) Then Exit Sub
    x = x + absX
    lstat = c6k.Write("D" + CStr(x) & vbLf)          'Set distance.
    lstat = c6k.Write("GO1" & vbLf)                  'Command motion.
    sw.Reset
    While (Abs(x) > CCLOSE) And Not (IsBitSet(c6k.AxisStatus(1), 1))
        'Wait for motion to start.
        If (sw.Elapsed(MOTIONDUR)) Then
            ErrorMessage.Display "Error starting relative X motion", 15
            Exit Sub
        End If
    Wend
    sw.Reset
    While (IsBitSet(c6k.AxisStatus(1), 1))
        'Wait for motion to stop.
        If (sw.Elapsed(MOTIONDUR)) Then
            ErrorMessage.Display "Error stopping relative X motion", 15
            Exit Sub
        End If
    Wend
End Sub

Public Sub MoveY(ByVal y As Long)
    'Here we take relative coordinates, implement with absolute ones.
    'Error checking.
    If ((absY + y) > YMAXLIM Or (absY + y) < YMINLIM) And (yHomed) Then
        MsgBox "Error: absolute Y coordinate out of range."
        Exit Sub
    End If
    'Cause motion in axis 2.
    If (y = 0) Then Exit Sub
    y = y + absY
    lstat = c6k.Write("D," + CStr(y) & vbLf)          'Set distance.
    lstat = c6k.Write("GO01" & vbLf)                  'Command motion.
    sw.Reset
    While (Abs(y) > CCLOSE) And Not (IsBitSet(c6k.AxisStatus(2), 1))
        'Wait for motion to start.
        If (sw.Elapsed(MOTIONDUR)) Then
            ErrorMessage.Display "Error starting relative Y motion", 15
            Exit Sub
        End If
    Wend
    sw.Reset
    While (IsBitSet(c6k.AxisStatus(2), 1))
        'Wait for motion to stop.
        If (sw.Elapsed(MOTIONDUR)) Then
            ErrorMessage.Display "Error stopping relative Y motion", 15
            Exit Sub
        End If
    Wend
End Sub

Public Sub MoveZ(ByVal z As Long)
    'Here we take relative coordinates, implement with absolute ones.
    'Error checking.
    If ((absZ + z) > ZMAXLIM Or (absZ + z) < ZMINLIM) And (zHomed) Then
        MsgBox "Error: absolute Z coordinate out of range."
        Exit Sub
    End If
    'Cause motion in axis 3.
    If (z = 0) Then Exit Sub
    z = z + absZ
    lstat = c6k.Write("D,," + CStr(z) & vbLf)        'Set distance.
    lstat = c6k.Write("GO001" & vbLf)                'Command motion.
    sw.Reset
    While (Abs(z) > CCLOSE) And Not (IsBitSet(c6k.AxisStatus(3), 1))
        'Wait for motion to start.
        If (sw.Elapsed(MOTIONDUR)) Then
            ErrorMessage.Display "Error starting relative Z motion", 15
            Exit Sub
        End If
    Wend
    sw.Reset
    While (IsBitSet(c6k.AxisStatus(3), 1))
        'Wait for motion to stop.
        If (sw.Elapsed(MOTIONDUR)) Then
            ErrorMessage.Display "Error stopping relative Z motion", 15
            Exit Sub
        End If
    Wend
End Sub


Public Sub HomeX()
    Dim txt As String
    If xHomed Then
        txt = "The X axis has already been homed.  Re-home?"
        If (MsgBox(txt, vbYesNo) = vbNo) Then Exit Sub
    End If
    Screen.MousePointer = vbHourglass
    lstat = c6k.Write("HOM0XXX" & vbLf)              'Home axis 1 positive.
    While (IsBitSet(c6k.AxisStatus(1), 5)): Wend        'Wait for not home success.
    While Not (IsBitSet(c6k.AxisStatus(1), 5)): Wend    'Wait for home success.
    Screen.MousePointer = vbDefault
End Sub

Public Sub HomeY()
    Dim txt As String
    If yHomed Then
        txt = "The Y axis has already been homed.  Re-home?"
        If (MsgBox(txt, vbYesNo) = vbNo) Then Exit Sub
    End If
    Screen.MousePointer = vbHourglass
    lstat = c6k.Write("HOMX0XX" & vbLf)              'Home axis 2 positive.
    While (IsBitSet(c6k.AxisStatus(2), 5)): Wend        'Wait for not home success.
    While Not (IsBitSet(c6k.AxisStatus(2), 5)): Wend    'Wait for home success.
    Screen.MousePointer = vbDefault
End Sub

Public Sub HomeZ()
    Dim txt As String
    If zHomed Then
        txt = "The Z axis has already been homed.  Re-home?"
        If (MsgBox(txt, vbYesNo) = vbNo) Then Exit Sub
    End If
    Screen.MousePointer = vbHourglass
    lstat = c6k.Write("HOMXX0X" & vbLf)              'Home axis 3 positive (up).
    While (IsBitSet(c6k.AxisStatus(3), 5)): Wend        'Wait for not home success.
    While Not (IsBitSet(c6k.AxisStatus(3), 5)): Wend    'Wait for home success.
    Screen.MousePointer = vbDefault
End Sub


Public Sub MoveAbsX(x As Long)
    'The 6K4 is set for absolute coordinates with MA111.
    
    'Error checking.
    If (xHomed = False) Then
        MsgBox "You can't use this MoveAbsX until X has been homed."
        Exit Sub
    End If
    If (x > XMAXLIM Or x < XMINLIM) Then
        MsgBox "Error: absolute X coordinate out of range."
        Exit Sub
    End If
    
    'Move.
    If (x = absX) Then Exit Sub
    sw.Reset
    lstat = c6k.Write("D" + CStr(x) & vbLf)        'Set distance.
    lstat = c6k.Write("GO1" & vbLf)                'Command motion.
    While (Abs(x - absX) > CCLOSE) And Not (IsBitSet(c6k.AxisStatus(1), 1))
        'Wait for motion to start.
        If (sw.Elapsed(MOTIONDUR)) Then
            ErrorMessage.Display "Error starting X motion", 15
            Exit Sub
        End If
    Wend
    sw.Reset
    While (IsBitSet(c6k.AxisStatus(1), 1))
        'Wait for motion to stop.
        If (sw.Elapsed(MOTIONDUR)) Then
            ErrorMessage.Display "Error stopping X motion", 15
            Exit Sub
        End If
    Wend
End Sub


Public Sub MoveAbsY(y As Long)
    'The 6K4 is set for absolute coordinates with MA111.
    Dim sTime As Long
    
    'Error checking.
    If (yHomed = False) Then
        MsgBox "You can't use this MoveAbsY until Y has been homed."
        Exit Sub
    End If
    If (y > YMAXLIM Or y < YMINLIM) Then
        MsgBox "Error: absolute Y coordinate out of range."
        Exit Sub
    End If
    
    'Move.
    If (y = absY) Then Exit Sub
    sw.Reset
    lstat = c6k.Write("D," + CStr(y) & vbLf)        'Set distance.
    lstat = c6k.Write("GO01" & vbLf)                'Command motion.
    While (Abs(y - absY) > CCLOSE) And Not (IsBitSet(c6k.AxisStatus(2), 1))
        'Wait for motion to start.
        If (sw.Elapsed(MOTIONDUR)) Then
            ErrorMessage.Display "Error starting Y motion", 15
            Exit Sub
        End If
    Wend
    sw.Reset
    While (IsBitSet(c6k.AxisStatus(2), 1))
        'Wait for motion to stop.
        If (sw.Elapsed(MOTIONDUR)) Then
            ErrorMessage.Display "Error stopping Y motion", 15
            Exit Sub
        End If
    Wend
End Sub


Public Sub MoveAbsZ(z As Long)
    'The 6K4 is set for absolute coordinates with MA111.
    Dim sTime As Long
    
    'Error checking.
    If (zHomed = False) Then
        MsgBox "You can't use this MoveAbsZ until Z has been homed."
        Exit Sub
    End If
    If (z > ZMAXLIM Or z < ZMINLIM) Then
        MsgBox "Error: absolute Z coordinate out of range."
        Exit Sub
    End If
    
    'Move.
    If (z = absZ) Then Exit Sub
    sw.Reset
    lstat = c6k.Write("D,," + CStr(z) & vbLf)        'Set distance.
    lstat = c6k.Write("GO001" & vbLf)                'Command motion.
    While (Abs(z - absZ) > CCLOSE) And Not (IsBitSet(c6k.AxisStatus(3), 1))
        'Wait for motion to start.
        If (sw.Elapsed(MOTIONDUR)) Then
            ErrorMessage.Display "Error starting Z motion", 15
            Exit Sub
        End If
    Wend
    sw.Reset
    While (IsBitSet(c6k.AxisStatus(3), 1))
        'Wait for motion to stop.
        If (sw.Elapsed(MOTIONDUR)) Then
            ErrorMessage.Display "Error stopping Z motion", 15
            Exit Sub
        End If
    Wend
End Sub


Public Sub SetVelocity(sglVelocity As Single)
    'Set velocities in three axes.  Default was 10.
    'The parameter sglVelocity will reflect the Y axis speed.
    'Make X axis 2x faster.
    'Make Z axis 2x slower.
    Dim txt As String
    'Check for error or reset.
    If (sglVelocity > 100) Then Exit Sub     'Invalid value, might be too fast.
    If (sglVelocity <= 0) Then      'Reset with a default value.
        lstat = c6k.Write(VELO & vbLf)
        Exit Sub
    End If
    'Normal execution.
    txt = Format(sglVelocity * 2, "0.00") + "," + Format(sglVelocity, "0.00") _
        + "," + Format(sglVelocity * 0.5, "0.00")
    lstat = c6k.Write("V" + txt & vbLf)  'Set velocities.
End Sub


Public Function GetPos(ByRef x As Long, ByRef y As Long, ByRef z As Long) As Integer
    'Return actual encoder position rather than commanded position.
    x = c6k.EncoderPos(1)
    y = c6k.EncoderPos(2)
    z = c6k.EncoderPos(3)
    GetPos = 0 'Return no error message.
End Function


Public Sub SetTrigger(ByVal pos As Long)
    '!This function is Depracated CGL 030918!
    'Set trigger to activate when position exceeds pos.
    Rem lstat = c6k.Write("AXSDEF1" & vblf)   'Define axis 1 as servo.
    Rem lstat = c6k.Write("SFB1" & vblf)      'Select encoder feedback for axis 1.
    lstat = c6k.Write("OUT.1-0" & vbLf)       'Reset trigger line low.
    lstat = c6k.Write("OUTFNC1-H" & vbLf)     'Set onboard output #1 as an "output on position" output.
    lstat = c6k.Write("OUTPA1,0," & pos & ",50" & vbLf)   'Program trigger.  High for 50 ms.
End Sub


Public Function SetTriggers(ByVal lFirstXPos As Long, ByVal lSpacing As Long, ByVal lCount As Long) As Integer
    'Set lCount triggers of 2ms using output bit #1.
    'Spacing is lSPacing encoder counts.
    'Turn on continuous command execution during program "STRIG".
    'Use some 6K4 variables.
    'VAR2: holds the trigger position in abs units of encoder counts.
    'VARI1: 1=Keep going, 0=exit now.
    'VARI4: number of triggers activated.
    'VARI5: 1=moved past trigger position already error, 0=no error.
    'Updated CGL 030918.

    'Parameter error checking here.
    SetTriggers = 0
    If (lFirstXPos < XMINLIM) Or ((lCount * lSpacing + FirstXPos) > XMAXLIM) Then
        ErrorMessage.Display "Error in Servo6k->SetTriggers: Triggers out of range.", 0
        SetTriggers = 1
        Exit Function
    End If

    'Transmit and start the program.
    lstat = c6k.Write("COMEXC0" & vbLf)             'Disable continuous command execution.
    lstat = c6k.Write("DEL STRIG" & vbLf)           'Delete any old STRIG program.
    lstat = c6k.Write("DEF STRIG" & vbLf)           'Start the STRIG program.
    lstat = c6k.Write("VAR2=" & lFirstXPos & vbLf)  'VAR2 holds trigger position.
    lstat = c6k.Write("VARI1=1" & vbLf)             'VARI1 flag to kill the loop.
    lstat = c6k.Write("VARI4=0" & vbLf)             'VARI4 as a counter since the 6K4 array pointer wraps around the end of the array to the beginning
    lstat = c6k.Write("VARI5=0" & vbLf)             'VARI5 flag warns if printhead too fast for 6K4.
    lstat = c6k.Write("OUT.1-0" & vbLf)             'Reset trigger line low.
    lstat = c6k.Write("OUTFNC1-A" & vbLf)
    'lstat = c6k.Write("OUTFNC1-H" & vbLf)           'Set output #1 as an "output on position" output.
    lstat = c6k.Write("L" & lCount & vbLf)          'Loop lCount times.
    'lstat = c6k.Write(" OUTPA1,0,VAR2,50" & vbLf)    ''Output#1 when >VAR2 for 2ms.
    lstat = c6k.Write(" IF(1PE>VAR2)" & vbLf)       ''Check IF we're already past trigger.
    lstat = c6k.Write("  VARI5=1" & vbLf)           '''Set VARI5 flag as errror.
    lstat = c6k.Write("  LX" & vbLf)                '''Exit loop.
    lstat = c6k.Write(" NIF" & vbLf)                ''End IF.
    lstat = c6k.Write(" WAIT(1PE>VAR2)" & vbLf)     ''Wait for printhead to pass trigger position.
    lstat = c6k.Write("OUT.1-1" & vbLf)
    lstat = c6k.Write("OUT.1-0" & vbLf)
    lstat = c6k.Write(" VAR2=VAR2+" & lSpacing & vbLf)  ''Increment trigger position.
    lstat = c6k.Write(" VARI4=VARI4+1" & vbLf)      ''Increment counter.
    lstat = c6k.Write(" IF(VARI1=0)" & vbLf)        ''Check IF we should kill loop.
    lstat = c6k.Write("  LX" & vbLf)                '''Exit loop.
    lstat = c6k.Write(" NIF" & vbLf)                ''End if.
    lstat = c6k.Write("LN" & vbLf)                  'End Loop.
    lstat = c6k.Write("COMEXC0" & vbLf)             'Disable continuous command execution.
    lstat = c6k.Write("END" & vbLf)                 'end STRIG program.
    lstat = c6k.Write("2%COMEXC1" & vbLf)           'Enable continuous command execution in task 2.
    lstat = c6k.Write("2%STRIG" & vbLf)             'Run the program as task 2 so only it is in COMEXC1.
End Function

Public Sub KillTriggersLoop()
    'Set VARI1 = 0, wait 250ms, then shut off continuous command execution.
    lstat = c6k.SendVariable(VARI1, 0)
    lstat = c6k.Write("T.250" & vbLf)
    lstat = c6k.Write("2%COMEXC0" & vbLf)
End Sub

Public Property Get TriggersTripped() As Long
    'Get the contents of VARI4 and reset it = 0.
    'This requires that fast-status is enabled (and ethernet is used). See 6K addendum p25.
    TriggersTripped = c6k.VarI(4)
    lstat = c6k.SendVariable(VARI4, 0)
End Property

Public Property Get TriggersMissed() As Long
    'Get the contents of VARI5 and reset it = 0.
    'This requires that fast-status is enabled (and ethernet is used). See 6K addendum p25.
    TriggersMissed = c6k.VarI(5)
    lstat = c6k.SendVariable(VARI5, 0)
End Property

Rem Public Property Let FastStatusEnable(ByVal bStatus As Boolean)
Rem     c6k.FSEnabled = bStatus
Rem End Property


