=head1 NAME

new_contig - module for creating contig entries in database

=head1 SYNOPSIS

pass new_contig a hash consisting of the following entries

=over

=item dbh

Give a valid database handle

=item organism

Give the organism id

=item chr

Give the recognized chromosome number

=item version

Give the version of the database you are working on

=item name

=item length

=item seq

The sequence

=item orientation

=item centromere

y or n

=item left_nbr

contig name

=item right_nbr

contig name

=item left_overlap

an integer

=item right_overlap

an integer

=back

=head1 DESCRIPTION

Takes that hash and puts the appropriate values into the database.

=cut

package new_contig;
use DBI;
use Exporter;
use vars qw(@ISA @EXPORT);
@ISA=qw(Exporter);
@EXPORT = qw(new_contig);


#new contig could be an interface new_contigI that calls an adaptor to actually
#fill assembly, fill contig, etc.  Depending on the adaptor, the contig would be created
#in das, across a corba bridge, or in a local database (flat file, etc)

sub new_contig {
    my (%args)=@_;
    fill_assembly(\%args);
    fill_contig(\%args);
    fill_contig_assembly(\%args);
    fill_tiling_path(\%args);
    fill_sequence(\%args);
    return $args{contig_id};
}

sub fill_assembly {
    my $args=shift;
    my $orgid=$args->{organism};
    my $chr=$args->{chr};
    my $version=$args->{version};
    my $assembly=_check($args->{dbh},'Assembly',[qw(organism chr_id version)],[$orgid,$chr,$version]);
    $args->{assembly}=$assembly;
}

sub fill_contig {
    my $args=shift;
    my $list=['name','centromere'];
    my $vals=[$args->{name},$args->{centromere}];
    my $id=_check($args->{dbh},'Contig',
		  $list,
		  $vals
		 );
    die "Failed insert into Contig" unless $id;
    $args->{cid}=$id;
}

sub fill_contig_assembly {
    my $args=shift;
    if ($args->{cid}) {
	my $sth=$args->{dbh}->prepare("SELECT id FROM ContigAssembly WHERE ".
				      "contig_id=? and version=? and assembly=?");
	$sth->execute($args->{cid},
		      $args->{version},
		      $args->{assembly},
		     );
	my ($test)=$sth->fetchrow_array;
	if ($test) {
	    $args->{contig_id}=$test;
	    return;
	}
    }
    $sth=$args->{dbh}->prepare("INSERT INTO ContigAssembly (contig_id,version,assembly) ".
			       "VALUES (".
			       $args->{cid}.",".
			       $args->{version}.",".
			       $args->{assembly}.")");
    $sth->execute;
    my $id=$args->{dbh}->{mysql_insertid};
    die "Failed insert into ContigAssembly" unless $id;
    $args->{contig_id}=$id;
    $args->{dbh}->commit;
}

sub fill_tiling_path {
    my $args=shift;
    my $sth=$args->{dbh}->prepare("SELECT contig_id FROM Tiling_Path ".
				  "WHERE contig_id=?");
    $sth->execute($args->{contig_id});
    my ($test)=$sth->fetchrow_array;
    if ($test) {
	return;
    }
    $sth=$args->{dbh}->prepare("INSERT INTO Tiling_Path ".
			       "(contig_id,length,threeprime,overlap3,fiveprime,overlap5) ".
			       "VALUES (?,?,?,?,?,?)");
    $sth->execute($args->{contig_id},
		  $args->{length},
		  $args->{right_nbr},
		  $args->{right_overlap},
		  $args->{left_nbr},
		  $args->{left_overlap},
		 );
    $args->{dbh}->commit;
}

sub fill_sequence {
    my $args=shift;
    my $sth=$args->{dbh}->prepare("SELECT contig_id FROM Sequence ".
				  "WHERE contig_id=?");
    $sth->execute($args->{contig_id});
    my ($test)=$sth->fetchrow_array;
    if ($test) {
	return;
    }
    if ($args->{orientation} eq '-') {
	$args->{seq}=scalar reverse $args->{seq};
	$args->{seq}=~tr/ATCGatcg/TAGCtagc/;
    }
    $sth=$args->{dbh}->prepare("INSERT INTO Sequence (contig_id,seq) ".
			       "VALUES (".$args->{contig_id}.",?)");
    $sth->execute($args->{seq});
    $args->{dbh}->commit;
}

sub _check {
    my ($dbh,$table,$cols,$vals)=@_;
    my $where = '';
    foreach (@$cols) {
	$where .= "$_=? AND ";
    }
    for (1..5) {  #remove ' AND ' at end
	chop $where;
    }
    my $sth=$dbh->prepare("SELECT id FROM $table WHERE $where");
    $sth->execute(@$vals) or die "$where $DBI::errstr";
    my ($id)=$sth->fetchrow_array;
    return $id if $id;
    #Okay, we have to enter the stuff
    my $list = join ',', @$cols;
    my $valqs= '?,' x @$cols;
    chop $valqs;                  #remove final comma
    my $sth=$dbh->prepare("INSERT INTO $table ($list) ".
			  "VALUES ($valqs)");
    $sth->execute(@$vals);
    $id=$dbh->{mysql_insertid};
    $context->commit;
    return $id;
}








