# $Id: ContextI.pm,v 1.5 2002/03/06 23:28:52 dblock Exp $
#
# Genquire module for GQ::Server::ContextI
#
# Cared for by David Block <dblock@gene.pbi.nrc.ca>
#
# Copyright NRC
#

# POD documentation - main docs before the code

=head1 NAME

GQ::Server::ContextI - Interface for data source module

=head1 SYNOPSIS

This object is not meant to be invoked directly.  It serves as a template for the creation
of context objects for various data sources.

=head1 DESCRIPTION

The methods listed here enable much of the rich functionality of Genquire.  The data 'context'
is the data environment in which everything else in Genquire exists.  It must be able to 
give out information about itself so that the features stay in their correct context.

One of the context's most important jobs is assigning the correct adaptor to each feature.
The adaptors contain the data-source-specific code for loading, adding, editing, and
deleting objects from the persistent store (database, series of flat files, etc.).

The context encompasses the source of the data, the organism and data release version,
and the user's username.  It must take care of the allowed tags available for tag/value
annotation, the locking system to prevent concurrent annotation on the same region, and
the flagging system for marking features as interesting.

The context is also the place where such data-source-specific events as storing blast hits,
finding common blast homologies, and retrieving EST data occur.

The context is the place Genquire looks to find out which contigs are available, and what
their location is on each assembly/chromosome.  The context knows which chromosomes are being
served, and the assemblies available on each chromosome, and those assemblies' contigs.  It
also keeps track of the organism that is being examined.

The context object should be the second object created in Genquire, following only the
actual data connection object (dbobj), which is the context's sole parameter.  There should
be a different Context implementation for each kind of data source.  Each context
implementation should have associated with it a complete set of adaptors for the Genquire
GQ::Server::XXX objects.  See the reference implementations of database adaptors for
documentation of the adaptor layer.

=head1 FEEDBACK

like it or lump it, let me know at dblock@gene.pbi.nrc.ca

=head1 AUTHOR - David Block

Email dblock@gene.pbi.nrc.ca

=head1 CONTRIBUTORS

This was developed as part of Genquire with Mark Wilkinson, mwilkinson@gene.pbi.nrc.ca

=head1 APPENDIX

The rest of the documentation details each of the object methods.

=cut


# Let the code begin...


package GQ::Server::ContextI;
use vars qw(@ISA);
use strict;
use vars qw( @ISA $AUTOLOAD );  #Keep 'use strict' happy
use Carp;

# Object preamble - inherits from Bio::Root::Root

use Bio::Root::Root;


@ISA = qw(Bio::Root::Root);

=head2 new

 Title   : new
 Function: return a GQ::Server::ContextI compliant object
 Returns : the object
 Args    : Whatever is needed to make the data connection
           (GQ::Server::DB::Context takes a DbObj)

=cut

sub new {
  my($class,@args) = @_;

  my $self = $class->SUPER::new(@args);

}

sub _abstract_death {
    my ($self,$msg)=@_;
    $self->throw("GQ::Server::ContextI is an abstract class, which ".
		 "does not implement this method: $msg\n");
}

=head2 commit

 Title   : commit
 Usage   : $context->commit
 Function: sends a commit signal for sql database changes - should
           know whether the data source can handle the commit, and
           not send it if it's not required
 Args    : none


=cut

sub commit {
    my ($self)=@_;
    $self->_abstract_death('commit');
}

=head2 rollback

 Title   : rollback
 Usage   : $context->rollback
 Function: sends a rollback signal for sql database changes - should
           know whether the data source can handle the rollback, and
           not send it if it's not required
 Args    : none


=cut

sub rollback {
    my ($self)=@_;
    $self->_abstract_death('rollback');
}

=head2 _type

 Title   : _type
 Usage   : my $type=$self->context->_type($self->type);
 Function: return the Genquire class that is supposed to handle biological
           objects of the given type
 Example : $self->context->_type('Gene') returns 'GQ::Server::Gene'
 Returns : the string name of the Genquire class that instantiates that type
           of object, or a DEFAULT type if the requested type is unknown
 Args    : one of a short list of approved types, stored in a class hash variable
           (in a closure)
 Note    : This allows the SequenceAdaptor to load features from the persistent
           store, and turn them into the appropriate object without knowing
           beforehand what that type of object would be.  See
           GQ::Server::DB::Adaptor::SequenceAdaptor::load for an example of use.
           The particular Context instance should have a DEFAULT type which
           is returned when the desired type is unknown eg "GQ::Server::GenericFeature"

=cut

sub _type{
   my ($self,@args) = @_;
   $self->_abstract_death('_type');
}

=head2 adaptor

 Title   : adaptor
 Usage   : my $adaptor=$self->context->adaptor($self);
 Function: provide the data adaptor for the given object
 Example : from within a Gene object, this call will return
           your implementation of a GeneAdaptor, which will
           take care of all the data source-specific functions
           of the Gene object
 Returns : the appropriate adaptor object
 Args    : the object that needs the adaptor


=cut

sub adaptor {
    my ($self)=@_;
    $self->_abstract_death('adaptor');
}

=head2 dbobj

 Title   : dbobj
 Usage   : my $dbobj=$context->dbobj;
 Function: get the object responsible for connecting to the datasource
 Returns : the dbobj (database object)
 Args    : none


=cut

sub dbobj { return $_[0]->_abstract_death('dbobj') }

=head2 dbh

 Title   : dbh
 Usage   : my $dbh=$context->dbh
 Function: get the database handle - should only be called if appropriate
 Returns : a valid DBI database handle
 Args    : none


=cut

sub dbh { return $_[0]->_abstract_death('dbh')}

=head2 user

 Title   : user
 Usage   : my $user=$context->user
 Function: return the username used to contact the data source
 Returns : a string
 Args    : none


=cut

sub user { return $_[0]->_abstract_death('user') }

=head2 version

 Title   : version
 Usage   : my $version=$context->version
 Function: return the version of the data that is in context
 Example : if a data source has two different data releases available,
           there needs to be some way of distinguishing between them.
           The version variable should hold the key that lets the data
           source know which data release is relevant
 Returns : the primary key of the relevant data release
 Args    : none


=cut

sub version {return $_[0]->_abstract_death('version')}

=head2 DEFAULT_ID, DEFAULT_VERSION

 Title   : DEFAULT_ID and DEFAULT_VERSION
 Usage   : my $id=$given_id||$context->DEFAULT_ID;
 Function: return the default organism id and version
 Example : if most of the time, a user will use the same
           data release from a source, that information can
           go here to eliminate some typing/clicking
 Returns : default values (probably strings or numbers)
 Args    : none


=cut

sub DEFAULT_ID {return;}
sub DEFAULT_VERSION {return;}

=head2 organism

 Title   : organism
 Usage   : $obj->organism($newval)
 Function: set the relevant organism for this context.  If
           there is no newval, and there is no current organism,
           use the default organism and version variables
 Returns : value of organism
 Args    : newvalue (optional), which could be a GQ::Server::Organism object, or the string 'default'


=cut

sub organism {
    require GQ::Server::Organism;
    my ($self,$newval)=@_;
    $self->_abstract_death('organism');
}

=head2 all_orgs_by_id

 Title   : all_orgs_by_id
 Usage   : my @orgs=$context->all_orgs_by_id;
 Function: return a list of all the organisms available,
           sorted by their unique ids
 Returns : a list of GQ::Server::Organism objects
 Args    : none


=cut

sub all_orgs_by_id {
    require GQ::Server::Organism;
    my $self=shift;
    $self->_abstract_death('all_orgs_by_id');
}


=head2 add_org

 Title   : add_org
 Usage   : $context->add_org(
			     common => $common_name,
			     latin  => $latin_name,
			     code   => $two_letter_code,
                            );
 Function: adds an organism to the data source, and creates a new GQ::Server::Organism object
 Returns : the newly created GQ::Server::Organism object
 Args    : a hash containing
            common => $common_name,
            latin  => $latin_name,
            code   => $two_letter_code
            for the organism


=cut

sub add_org {
    require GQ::Server::Organism;
    my ($self,$common,$latin,$code)=@_;
    $self->_abstract_death('add_org');
}

=head2 get_contigs_by_name

 Title   : get_contigs_by_name
 Usage   : my @contignames=$context->get_contigs_by_name
 Returns : a list of the names of the contigs available for
           the current organism/version
 Args    : none


=cut

sub get_contigs_by_name {
    my ($self)=@_;
    $self->_abstract_death('get_contigs_by_name');
}

=head2 get_chrs_by_id

 Title   : get_chrs_by_id
 Usage   : my @chrids=$context->get_chrs_by_id
 Returns : a list of chromosome ids for the current organism/version
 Args    : none


=cut

sub get_chrs_by_id {
    my ($self)=@_;
    $self->_abstract_death('get_chrs_by_id');
}
=head2 get_ordered_assemblies_by_chr

 Title   : get_ordered_assemblies_by_chr
 Usage   : my $assemblies=$context->get_ordered_assemblies_by_chr
 Returns : a reference to a hash, keyed by the assembly id, with
           values being lists of pairs- the pairs being the name and
           length of the contigs in that assembly.  The contigs are sorted
           by their chromosomal position
 Args    : the chromosome id you want the assemblies from


=cut

sub get_ordered_assemblies_by_chr {
    my ($self)=@_;
    $self->_abstract_death('get_ordered_assemblies_by_chr');
}

=head2 get_next

 Title   : get_next
 Usage   : my $next_contig_name=$context->get_next($current_contig, "next");
        or: my $next_contig_name=$context->get_next($current_contig);
 Function: find the next contig in the ordered list of contigs, either to the
           left/up/lower abs_start, or the right/down/greater abs_start.  In the
           latter case, send as a second parameter the string 'next'.
 Returns : the next contig name, a string
 Args    : the current contig, and optionally the string 'next'


=cut

sub get_next {
    my ($self,$present,$np)=@_;
    my ($self)=@_;
    $self->_abstract_death('get_next');
}

=head2 contig

 Title   : contig
 Usage   : my $contigobj=$context->contig($contigname);
 Returns : a GQ::Server::Contig object corresponding to the current organism/version,
           with the given name
 Args    : the contig name


=cut

sub contig {
    my ($self)=@_;
    $self->_abstract_death('contig');
}

=head2 contig_by_id

 Title   : contig_by_id
 Usage   : my $contigobj=$context->contig_by_id($contig_id);
 Returns : a GQ::Server::Contig object corresponding to the current organism/version,
           with the given id
 Args    : the contig id


=cut

sub contig_by_id{
   my ($self,@args) = @_;
   $self->_abstract_death('contig_by_id');
}

=head2 get_contig_sequence

 Title   : get_contig_sequence
 Usage   : my $sequence=$context->get_contig_sequence($contig_name)
 Returns : the nucleotide sequence as a string from the contig
 Args    : the contig name
 Note    : this functionality is now available within the contig object,
           so this call can simply find the appropriate contig object and ask it
           for its sequence.


=cut

sub get_contig_sequence {
    my ($self)=@_;
    $self->_abstract_death('get_contig_sequence');
}

=head2 get_contig_length

 Title   : get_contig_length
 Usage   : my $length=$context->get_contig_length($contig_name)
 Returns : the length of the sequence from the contig
 Args    : the contig name
 Note    : this functionality is now available within the contig object,
           so this call can simply find the appropriate contig object and ask it
           for its length.


=cut

sub get_contig_length {
    my ($self)=@_;
    $self->_abstract_death('get_contig_length');
}


=head2 get_contig_id

 Title   : get_contig_id
 Usage   : my $id=$context->get_contig_id($contig_name)
 Returns : the id of the contig
 Args    : the contig name
 Note    : this functionality is now available within the contig object,
           so this call can simply find the appropriate contig object and ask it
           for its id.


=cut
sub get_contig_id {
    my ($self)=@_;
    $self->_abstract_death('get_contig_id');
}

=head2 get_boundaries

 Title   : get_boundaries
 Usage   : my $boundaries=$context->get_boundaries($contig_name)
 Returns : the pair of numbers corresponding to the start and stop of the contig,
           usually (1,contig length).
 Args    : the contig name
 Note    : this functionality is now available within the contig object,
           so this call can simply find the appropriate contig object and ask it
           for its boundaries.


=cut

sub get_boundaries {
    my ($self)=@_;
    $self->_abstract_death('get_boundaries');
}

=head2 find_assembly

 Title   : find_assembly
 Usage   : my $assembly_id=$context->find_assembly($contig_name)
 Returns : the assembly id containing that contig
 Args    : the contig name
 Note    : this functionality is now available within the contig object,
           so this call can simply find the appropriate contig object and ask it
           for its assembly.


=cut

sub find_assembly {
    my ($self)=@_;
    $self->_abstract_death('find_assembly');
}

=head2 _get_contig_start_stop

 Title   : _get_contig_start_stop
 Usage   : my ($assembly,$start,stop)=$context->_get_contig_start_stop
 Function: find the contig's assembly, and its start and stop in assembly
           (usually chromosomal) coordinates
 Returns : a list with three elements, all numbers
 Args    : the contig name
 Note    : this function can be replaced by three calls to the appropriate
           contig object, but it's here for convenience


=cut

sub _get_contig_start_stop {
    my ($self)=@_;
    $self->_abstract_death('_get_contig_start_stop');
}

=head2 get_contig_info_by_name

 Title   : get_contig_info_by_name
 Usage   : my $hash=$context->get_contig_info_by_name
 Function: find all of the contigs available in the organism, and return their
           names as keys to a hashref, the values being the two-element list containing
           their contig ids and their start in assembly (chromosomal) coordinates
 Example : $has->{$name}=[$contig_id,$abs_start];
 Returns : a reference to a hash of lists
 Args    : none


=cut

sub get_contig_info_by_name {
    my ($self)=@_;
    $self->_abstract_death('get_contig_info_by_name');
}

=head2 get_EST_sources

 Title   : get_EST_sources
 Usage   : my $list=$context->get_EST_sources
 Returns : a reference to a list of unique sources of ESTs
 Args    : none


=cut

sub get_EST_sources {
    my ($self)=@_;
    $self->_abstract_death('get_EST_sources');
}

=head2 get_EST_sequence

 Title   : get_EST_sequence
 Usage   : my $seq=$context->get_EST_sequence($EST_gb_acc);
 Returns : the sequence of the EST
 Args    : the genbank accession id of the EST


=cut

sub get_EST_sequence {
    my ($self)=@_;
    $self->_abstract_death('get_EST_sequence');
}

=head2 get_EST_hits

 Title   : get_EST_hits
 Usage   : my @list=$context->get_EST_hits(%args);
 Function: find all the ESTs that overlap a contig region
 Returns : a list, each element being a list containing
           the alignment id
           the EST start
           the EST end
           the subject strand
           the strand
           the EST source
 Args    : a hash containing
             (contig=>$contig_name,
              start =>$contig_start,
              stop  =>$contig_stop)
           the start and stop elements are optional, and would be replaced by
           the boundaries of the contig if not provided


=cut

sub get_EST_hits {
    my ($self)=@_;
    $self->_abstract_death('get_EST_hits');
}

=head2 get_common_exons

 Title   : get_common_exons
 Usage   : my @commons=$context->get_common_exons($exon_id)
 Function: find the features which share blast homology to the same accession
 Returns : a list of Feature ids
 Args    : the id of the reference feature


=cut

sub get_common_exons {
    my ($self)=@_;
    $self->_abstract_death('get_common_exons');
}

=head2 get_tags

 Title   : get_tags
 Usage   : my @tags=$context->get_tags;
 Returns : a list of all of the valid tag/value tags available
 Args    : none


=cut

sub get_tags {
    my ($self)=@_;
    $self->_abstract_death('get_tags');
}
=head2 parse_Blast_to_db

 Title   : parse_Blast_to_db
 Usage   : $context->parse_Blast_to_db($feature_id,$BlastObj);
 Function: given a feature which has been blasted, store the results, including
           looking up all features which have been blasted and have homology to the
           same accession, and storing that relationship (for get_common_exons).
 Returns : nothing
 Args    : the feature that was blasted, and a Bio::Tools::BPLite Blast Object


=cut

sub parse_Blast_to_db {
    my ($self)=@_;
    $self->_abstract_death('parse_Blast_to_db');
}

=head2 create_tag

 Title   : create_tag
 Usage   : $context->create_tag($newtag)
 Function: make a new tag available for use, only if it does not exist already
 Args    : a new tag that can be used as a valid tag/value key


=cut

sub create_tag {
    my ($self)=@_;
    $self->_abstract_death('create_tag');
}

=head2 checklock

 Title   : checklock
 Usage   : see example
 Function: checks to see if the user has a lock on the region he is attempting 
           to annotate
 Example : die "Must Acquire Lock On This Region To Proceed.\n"
             if $self->context->checklock(lockid       => $self->lockid,
		             		  contig_id    => $self->contig_id,
				          contig_start => $self->contig_start,
				          contig_stop  => $self->contig_stop);

 Returns : 1 if the user does not have a lock,
           undef if the user does have a lock
 Args    : hash with the feature's lockid, contig_id, contig_start, contig_stop


=cut

sub checklock {
    my ($self)=@_;
    $self->_abstract_death('checklock');
}

=head2 lock

 Title   : lock
 Usage   : my $lockid=$self->context->lock(add=>[$contig,$start,$stop]);
           my $lockid=$self->context->lock(contig=>$contig);

           $self->context->lock(release=>$self->lockid);

           if ($abs_start<= $self->lock(stop=>$args{lockid}) &&
	       $abs_stop >= $self->lock(start=>$args{lockid})) {return}

 Function: place a lock on a region, release a lock, examine the details of the lock
 Returns : with arg add or contig, returns a lockid after creating the lock
           with arg release, returns nothing and clears the lock
           with arg start and a lockid, returns the start of the lock region in assembly
                coordinates
           with arg stop and a lockid, returns the stop of the lock region in assembly
                coordinates
           with arg both and a lockid, returns the start and stop of the lock region
                in assembly coordinates as an anonymous list


=cut

sub lock{
    my ($self)=@_;
    $self->_abstract_death('lock');
}

=head2 get_flagged_features

 Title   : get_flagged_features
 Usage   : my @features=$context->get_flagged_features('dumpy');
 Function: find all the features that are flagged with a certain flag
 Returns : a list of feature ids that are flagged
 Args    : the flag text


=cut

sub get_flagged_features {
    my ($self)=@_;
    $self->_abstract_death('get_flagged_features');
}

=head2 get_flagged_contigs

 Title   : get_flagged_contigs
 Usage   : my @contigs=$context->get_flagged_contigs('dumpy');
 Function: find all the contigs that contain features that are flagged with a certain flag
 Returns : a list of contig ids that contain flagged features
 Args    : the flag text


=cut

sub get_flagged_contigs {
    my ($self)=@_;
    $self->_abstract_death('get_flagged_contigs');
}

=head2 remove_flag

 Title   : remove_flag
 Usage   : $context->remove_flag('dumpy');
 Function: removes all instances of the flag with the given text
 Returns : nothing
 Args    : the flag text


=cut

sub remove_flag {
    my ($self)=@_;
    $self->_abstract_death('remove_flag');
}

=head2 add_flag

 Title   : add_flag
 Usage   : $context->flag_text(%args);
 Function: add the flag given ($flag_text) to the given feature_id on the given
           contig_id.  If the flag id is known, it can be provided as an efficiency step
 Returns : the id of the instance of the flag
 Args    : a hash containing:
           (flag=     >$flag_text,
            feature_id=>$feature_id,
            contig_id =>$contig_id,
            flag_id   =>$flag_id (optional)
           );


=cut

sub add_flag {
    my ($self)=@_;
    $self->_abstract_death('add_flag');
}

=head2 flag_text

 Title   : flag_text
 Usage   : $context->flag_text(%args);
 Function: adds flags to all features that match a given text list
 Returns : nothing
 Args    : a hash containing:
           (flag    =>$flag,              #the text of the flag to be applied
            textlist=>$textlist,          #a ref to a list of strings to be flagged
            andor   =>'and'               #and/or allows boolean searches
           );


=cut

sub flag_text {
    my ($self)=@_;
    $self->_abstract_death('flag_text');
}


=head2 rename_flag

 Title   : rename_flag
 Usage   : $context->rename_flag($old_flag,$new_flag);
 Function: change the flag of all features associated with a flag to a new flag (huh?)
 Example : $context->rename_flag('hypothetical','confirmed');  #time to break out champagne
 Returns : the new flag's id, or undef on failure
 Args    : old flag text, new flag text


=cut

sub rename_flag{
   my ($self,@args) = @_;
   $self->_abstract_death('rename_flag');
}

=head2 all_flags

 Title   : all_flags
 Usage   : my @flags=$context->all_flags;
 Returns : a list of the available flags
 Args    : none


=cut

sub all_flags{
   my ($self,@args) = @_;
   $self->_abstract_death('all_flags');

}


1;







