=head1 GenomeMapper.pm

=head2 NAME

GenomeMapper.pm - Graphical display of Genomes/chromosome (Genquire assembly) objects.

=head2 AUTHORS

Mark Wilkinson (mwilkinson@gene.pbi.nrc.ca)
Plant Biotechnology Institute, National Research Council of Canada.
Copyright (c) National Research Council of Canada, October, 2000.

=head2 DISCLAIMER

Anyone who intends to use and uses this software and code acknowledges and
agrees to the following: The National Research Council of Canada (herein "NRC")
disclaims any warranties, expressed, implied, or statutory, of any kind or
nature with respect to the software, including without limitation any warranty
or merchantability or fitness for a particular purpose.  NRC shall not be liable
in any event for any damages, whether direct or indirect,
consequential or incidental, arising from the use of the software.

=head2 SYNOPSIS

 # To create a Genome map and return a handle to the map object:

 use Tk;
 use
 Begin();
 MainLoop;

 sub Begin {

   # set up the Tk Windows
   my $GM = GenomeMapper->new (-title => "Map Of Chromosome 3");

   # Context comes from ContextPicker->new
   my @chrs = $self->Context->get_chrs_by_id;
	
   foreach my $chr_id(@chrs){
     my $assembled=$self->Context->get_ordered_assemblies_by_chr($chr_id);
     $GenomeMap->addChromosome($chr_id, $assembled);
   }

 }


=head2 DESCRIPTION and ACKNOWLEDGEMENTS

Creates an interactive map of contigs in a tiling path, breaks between non-assembled units
and spaces between tiling path sections which belong to different chromosomes.

=head2 CONTACT

Mark Wilkinson (mwilkinson@gene.pbi.nrc.ca)

=head2 Object Tags:

Each map-widget has one "reliable" tag attached to it:  "contig $contig_id"

e.g.  the tag would be "contig A34TG32"

Additional tags can be added to individual contigs or groups of contigs using
the lightAndAddTag method.  Similarly, they can be removed with the
dimAndRemoveTag method.  Using these methods allows you to 'light up' and tag
various contigs in various colors to display genomic features of interest.

=cut



package GQ::Client::GenomeMapper;
use strict;
use Carp;
use Tk::widgets qw(Dialog);
use Tk;
use GQ::Client::GenomeMapperContig;
use GQ::Client::Object;
use vars qw( @ISA $AUTOLOAD );

Tk::Widget->Construct('GenomeMapper');
@ISA = qw(GQ::Client::Object Tk::MainWindow);

BEGIN {
{	# encapsulated object data
    my %_attr_data = #     DEFAULT    ACCESSIBILITY
      (	-background     	=> ["white", "read/write"],
	-foreground     	=> ["#000000", "read/write"],
	-height         	=> [300,        "read/write"],
	#-width         	=> [10,        "read/write"],
	-scrollbars     	=> ['se', 		"read/write"],
	CurrentFlag			=> [undef, 		'read/write'],
	chrList         	=> [undef, 		"read/write"], # ref to list of chromosomes
	contigList      	=> [undef, 		"read/write"], # array of currently mapped contigs
	contigIndex			=> [undef, 		"read/write"], # hash associating the Canvas WidgetID of the contig with the contig name
	#assemblies      	=> [undef, 		"read/write"], # ref to hash of assemblies
	#current_contig  	=> [undef, 		"read/write"], # what has just been clicked on
	#mouseover_contig	=> [undef, 		"read/write"], # what has just been clicked on
	#ChrMap          	=> [undef, 		"read/write"], # the "scrolled" widget canvas on which we are drawing
	#ChromosomeMap   	=> [undef, 		"read/write"], # the real canvas on which we are drawing
	#ChromosomeWindow	=> [undef, 		"read/write"], # the window for external binding of events
	#Context         	=> [undef, 		"read/write"], # this is the value of Context from QueryScreen when this genome map was created
	_chr_right_offset	=> ["1", 		"read/write"],
	FlagMenu			=> [undef, 		"read/write"],

      );
    
    #my $chr_right_offset;
    
    # Is a specified object attribute accessible in a given mode
    sub _accessible  {
    	my ($self, $attr, $mode) = @_;
    	return $_attr_data{$attr}[1] =~ /$mode/
    }
    
    # Classwide default value for a specified object attribute
    sub _default_for {
    	my ($self, $attr) = @_;
    	return $_attr_data{$attr}[0]
    }
    
    # List of names of all specified object attributes
    # A hash so names are not repeated
    sub _standard_keys { keys %_attr_data }
    
    sub _next_chr_location {
		my ($self) = @_;
		my $chr_right_offset = $self->_chr_right_offset;
		$self->_chr_right_offset($chr_right_offset+ 70);
		return $self->_chr_right_offset;
    }
    
    use subs (keys %_attr_data)
}
} #end of BEGIN block



=head2 new

 $GM = GenomeMapper->new(%args)
 creates a new genome map.  %args are any valid argument for a
 Tk::MainWindow.

=cut


sub new {
    my ($caller, $QueryScreen, %args) = @_;  # called with GenomeMapper->new($QueryScreen)

    my $caller_is_obj = ref($caller);
    my $class = $caller_is_obj || $caller;

    my $self=bless $QueryScreen->Toplevel(%args),$class;

    foreach my $attrname ( $self->_standard_keys() ) {
    	if (exists $args{$attrname}) {
    	    $self->{$attrname} = $args{$attrname} }
    	elsif ($caller_is_obj) {
    	    $self->{$attrname} = $caller->{$attrname} }
    	else {
    	    $self->{$attrname} = $self->_default_for($attrname) }
    }
    #return 0 if (!$self->chrList);
    #return 0 if (!$self->assemblies);

    foreach ($self->_standard_keys) {
	next if (!/^-/);       			# filter out all args that must not be passed through to the Tk parent
		$args{$_}=$self->{$_};      # set the defaults for Tk parent
    }


    ######################################################
    ###############  THIS WILL CHANGE SOON!! #############
    $self->Context($QueryScreen->context);  # grab the Context object from QueryScreen...
    ######################################################
    ######################################################
	
    # object is now initialized

    #orig
    #$self->{Frame} = $self->Frame(-relief => 'ridge',  -borderwidth => '2')->pack(-side => 'left', -fill => "both", -expand => 1);
    #my $ChrFrame = $self->{Frame};

    my $ChrFrame = $self->Frame(-relief => 'ridge',  -borderwidth => '2')->pack(-side => 'left', -fill => "both", -expand => 1);


    $self->ChrMap($ChrFrame->Scrolled('Canvas',        # create the chromosome map canvas itself (a Tk::Scrolled object at first)
				      -scrollbars => "sw",
				      -background => $self->{-background},
				      #-width => $self->{-width},
				      -height => $self->{-height},
				     )->pack(
					     -side => 'left',
					     -fill => 'both',
					     -expand => 1));

	$self->ChrMap->configure(-scrollregion => $self->ChrMap->bbox("all"));   # set the scrollbars to include everything
	$self->ChromosomeMap($self->ChrMap->Subwidget("canvas"));                # get a reference to the actual canvas object
		                                                                     # this will be used for all interactions, rather than $self->ChrMap, which is a Scrolled object
	
	
	# this menu uses a referene to QueryScreen.  This is quite ugly!
	# it is necessary because the routines that are called require
	# both the default directory structure ($QueryScreen), and the specific
	# information about this genome ($QueryScreen->Context).
	# This needs to be changed one day...
	my $menu=$self->Menu(-tearoff => 0,
			-menuitems => [
			['command' => 'Refresh',	-command => sub {$self->refreshFlags} ],
			undef,
			['command' => 'Query Annotations',	-command => sub {$self->queryAnnotations} ],
			['command' => 'Remove Current Flags',	-command => sub {$self->removeAllFlags} ],
			['command' => 'Re-name Current Flags', -command => sub{$self->renameFlag} ],
			undef,
			]);
	my $flags = $menu->cascade(-label => '~View Flags of type', -tearoff => 0);
    
    foreach my $type ($self->Context->all_flags) {
		$flags->command(
             -label => "$type",
             -underline => 14,
	         -command => sub {
					$self->removeAllFlags;
					my $ContigListRef = $self->Context->get_flagged_contigs($type);
					$self->lightAndAddTag("red", "flagged", $ContigListRef);
					$self->CurrentFlag($type);
							},
        );
	}
	$self->FlagMenu($flags); # this will need to be updated at various times, so keep a ref to it

	$self->ChromosomeMap->Tk::bind("<Button-3>" => sub {$menu->Popup(-popover => 'cursor',
						        -popanchor => 'nw')});

	$self->update;
	$self->CurrentFlag($self->Context->user);
	return $self;
	
}


sub renameFlag {
	my ($self) = @_;
	my $flags = $self->FlagMenu;

	my $newflag;
	my $AskFlag = $self->DialogBox(-title => 'New Flag Name:',
		   -default_button=> 'OK',
		   -buttons => [qw/OK Cancel/]);
	$AskFlag->add("LabEntry",
		-textvariable=>\$newflag,
		-label=>'Enter the new flag name',
		)->pack;
	my $answer=$AskFlag->Show;
	return if $answer eq 'Cancel';

	my $newid = $self->Context->rename_flag($self->CurrentFlag, $newflag);
	return if (scalar (grep {$_ eq $newflag} $self->Context->all_flags) > 1);
	$flags->command(
		 -label => "$newflag",
		 -underline => 14,
		 -command => sub {
				$self->removeAllFlags;
				my $ContigListRef = $self->Context->get_flagged_contigs($newflag);
				$self->lightAndAddTag("red", "flagged", $ContigListRef);
				$self->CurrentFlag($newflag);
						},
	);
}


=head2 addChromosome

 $GM->addChromosome($chr_id, $assembly);
  $chr_id comes from ContextPicker->get_chrs_by_id
  $assembly comes from ContextPicker->get_ordered_assemblies_by_chr($chr_id)

=cut

sub addChromosome{
    my ($self, $chr, $assembly_hash) = @_;
    $self->_pushAssemblyList([$chr, $assembly_hash]);
    $self->_pushChromosomeList($chr);
    
    $self->_widen_by_one($chr);
    
    my $ChrMap = $self->ChrMap;
    
    # foreach assembled unit, figure out which is the root contig, then follow the path along, taking the length to build the box
    # I dont' know if this is better done using Tiling_Path.abs_start, or Contig.threeprime... but it can be done with one query using Abs_start
    my $cur_offset = 20;  # the starting offset from the top of the viewing frame
    my $total_length = 0;  # will hold the total length of all assembled subunits
    
    foreach my $assembly(keys %$assembly_hash){
	$cur_offset += 10; # offset from the last assembled unit
	foreach my $con(@{$assembly_hash->{$assembly}}) {
	    my ($name,$len)=@$con;
	    my $CONTIG = GQ::Client::GenomeMapperContig->new(  # create new contig object
							     context => $self->Context,
							     length => $len,
							     name => $name,
							     centromere => 0,
							     GenomeMap => $self,
							    );
	    $CONTIG->offset($cur_offset);  # give it all it needs to know about itself
	    $CONTIG->y_pos($self->getWidth); # " " " "
	    my $widgetID = $CONTIG->draw;  # then ask it to draw itself
	    $self->_addContigToArray($name, $CONTIG);  # and add it to the hash of drawn objects with name as key and object as value
		$self->_addContigToIndex($name, $widgetID);  # also associate the WidgetID with the contig name for easy referencing
	    $cur_offset += ($len/100000) + 1;
	    $total_length += $len
	}  #end of foreach my $con
    } # end of foreach $assembly
    
    my $mb_len = sprintf "%0.2f", ($total_length/1000000);
    $ChrMap->createText(($self->getWidth -65),15,-text => "$mb_len Mb", -anchor => "nw");
    
    $ChrMap->configure(-scrollregion => [$ChrMap->bbox("all")]);
    $self->update;
}


=head2 getSelectedContig

 $GM->getSelectedContig
 returns the contig id of the currently selected contig

=cut

sub getSelectedContig {
	my ($self) = @_;
	return $self->current_contig;
}

=head2 setSelectedContig

 $GM->setSelectedContig($contig_id)
 sets the contig as selected (same as clicking it)

=cut

sub setSelectedContig {
	my ($self, $ctg) = @_;
	$self->current_contig($ctg);
}

=head2 getMouseoverContig

 $GM->getMouseoverContig()
 gets the contig id of the current mouse-over

=cut


sub getMouseoverContig {
	my ($self) = @_;
	return $self->mouseover_contig;
}

=head2 setMouseoverContig

 $GM->setMouseoverContig($contig_id)
 sets the contig id of the current mouse-over
 (I doubt this will ever be used...)

=cut

sub setMouseoverContig {
	my ($self, $ctg) = @_;
	$self->mouseover_contig($ctg);
}


sub _widen_by_one {
	# this routine is almost completely non-functional
	# because a scrolled Canvas has infinite size and can not be re-sized except by a mouse.
	# One day i may change it such that I code the scrollbars
	# manually, and thus have control over the canvas width at run-time
	# but at the moment this is not a priority...
	my ($self, $chr_id) = @_;
	my $ChrMap = $self->ChrMap;
	
	my $width = $self->_next_chr_location; 	# increment width for this chromosome to a max of 4 chromosomes
    #if ($width < $self->screenwidth){
    	$self->setWidth($width);
    #}

	#$ChrMap->configure(-width => $width); 	# widen map by this increment up to a max of 280 (4 chromosomes)
	$ChrMap->createText(($width-65),5,-text => "chr $chr_id", -anchor => "nw");

}
	
=head2 getWidth

 $GM->getWidth()
 returns the width of the chromosome map in pixels

=cut

sub getWidth{
	my ($self) = @_;
	return $self->{width}
}

=head2 setWidth

 $GM->setWidth()
 set the width of the chromosome map in pixels
 THIS IS NOT FUNCTIONAL AT THE MOMENT w.r.t. changing the size of the actual canvas

=cut

sub setWidth {
	my ($self, $width) = @_;
	$self->{width} = $width;
}

=head2 lightContig

 $GM->lightContig($contig_id)

=cut


sub lightContig {
	my ($self, $contig) = @_;
	return unless $contig;
	my $CONTIG = $self->contigList->{$contig};
	return unless $CONTIG;
	$CONTIG->light;
}

=head2 dimContig

 $GM->dimContig($contig_id)

=cut

sub dimContig {
	my ($self, $contig) = @_;
	return unless $contig;
	my $CONTIG = $self->contigList->{$contig};
	return unless $CONTIG;
	$CONTIG->dim;
}

=head2 dimAllContigs

 $GM->dimAllContigs
 dims all contigs lit up by lightContig

=cut

sub dimAllContigs {
	my ($self) = @_;
	my @contigIDs = $self->ChromosomeMap->find("withtag", "lit");
	foreach my $contigID(@contigIDs){  # these are the numerical WidgetID's, which are associated in a hash to the contig name
		my $contig = $self->contigIndex->{$contigID};
		my $CONTIG = $self->contigList->{$contig};
		next unless $CONTIG;
		$CONTIG->dim;
	}
}		

=head2 getLitIDs

 $GM->getLitIDs
 gets the contig ID of all contigs lit up by lightContig

=cut

sub getLitIDs {
	my ($self) = @_;
	my @result;  # this will contain contig names
	my @contigIDs = $self->ChromosomeMap->find("withtag", "lit");
	foreach my $contigID(@contigIDs){  # these are the numerical WidgetID's, which are associated in a hash to the contig name
		my $contig = $self->contigIndex->{$contigID};
		my $CONTIG = $self->contigList->{$contig};
		next unless $CONTIG;
		push @result, $CONTIG->name;
	}	
	return \@result;
}

=head2 lightAndAddTag

 $GM->lightAndAddTag($color, $tag, \@contigs)
  $color is any valid hex or color string
  $tag is the tag which you want to add
  @contigs is a list of contig id's to which these should be applied

=cut

sub lightAndAddTag {
	my ($self, $color, $tag, $ctg_ids) = @_;
	return 0 if (!$ctg_ids);
	foreach my $contig(@{$ctg_ids}){
		my $CONTIG = $self->contigList->{$contig};
		next unless $CONTIG;
		$CONTIG->addtag($tag);
		$CONTIG->highlight($color);
	}
}	

=head2 dimAndRemoveTag

 $GM->dimAndRemoveTag($tag)
 "dims" (restores to normal lighting) any contig with the tag $tag
 and removes that tag from the contig.

=cut


sub dimAndRemoveTag {
	my ($self, $tag) = @_;
	my @contigIDs = $self->ChromosomeMap->find("withtag", $tag);
	foreach my $contigID(@contigIDs){  # these are the numerical WidgetID's, which are associated in a hash to the contig name
		my $contig = $self->contigIndex->{$contigID};
		my $CONTIG = $self->contigList->{$contig};
		next unless $CONTIG;
		$CONTIG->dim("default");
		$CONTIG->removetag($tag);
	}
}

=head2 removeAllFlags

 $GM->removeAllFlags($flag)
 "dims" (restores to normal lighting) any contig flagged with the $flag
 and removes that flag.  Flags are set by queries.


=cut


sub removeAllFlags {

	my ($self) = @_;
	$self->Context->remove_flag($self->CurrentFlag);
	$self->dimAndRemoveTag("flagged");
	$self->dimAllContigs;

}

=head2 queryAnnotations

 $GM->queryAnnotations()
 brings up a query dialog box to obtain keywords from the user.
 keywords are then used in a query against the database.
 'hits' are flagged with the users MySQL username and the hit contigs are lit.


=cut

sub queryAnnotations {

	my ($self) = @_;
	my $user = $self->Context->user;
	
	my $keywords; my $andor;
	my $KeywordsBox = $self->DialogBox(-title => 'Enter Keywords',
				   -default_button=> 'Find',
				   -buttons => [qw/Find Cancel/]);
    $KeywordsBox->add("LabEntry",
		  -textvariable=>\$keywords,
		  -label=>'Enter keywords below, also select the AND versus OR joining of keywords',
		 )->pack;
    $KeywordsBox->add("Radiobutton", -text => "AND", -variable => \$andor, -value => "AND")->pack;
    $KeywordsBox->add("Radiobutton", -text => "OR", -variable => \$andor,  -value => "OR")->pack;
    $andor = "AND";
    my $Keywords=$KeywordsBox->Show;
    return if $Keywords eq 'Cancel';
	
	$self->configure (-cursor => "watch");
	$self->update;
	my @keywords = split /\s+/, $keywords;
    $self->Context->flag_text(flag=>$user, textlist=>\@keywords, andor=>$andor);
    $self->dimAndRemoveTag("flagged");
    my $ContigListRef = $self->Context->get_flagged_contigs($self->Context->user);
	$self->lightAndAddTag("blue", "flagged", $ContigListRef);
	$self->configure (-cursor => "left_ptr");
	$self->update;
	$self->CurrentFlag($self->Context->user);
		
}

sub refreshFlags {
	my ($self) = @_;
	my @flagged = $self->ChromosomeMap->find("withtag", "flagged");
	my @contigs;
	foreach my $flagged(@flagged){
		my @tags = $self->ChrMap->gettags($flagged);
		foreach my $tag(@tags){
			if ($tag =~ /contig\s(.+)/){push @contigs, $1; last}
		}
	}
	$self->lightAndAddTag("blue", "flagged", \@contigs);

}


#  internal
sub _pushChromosomeList {
	my ($self, $chr) = @_;
	unless ($self->{chrList}){$self->{chrList} = []}
	push @{$self->{chrList}}, $chr;
}
sub _pushAssemblyList {
	my ($self, $ass) = @_;
	unless ($self->{assemblies}){$self->{assemblies} = []}
	push @{$self->{assemblies}}, $ass;
}
sub _addContigToArray {
	my ($self, $name, $CONTIG) = @_;
	return unless $name;
	return unless $CONTIG;
	unless ($self->contigList){$self->{contigList} = {}}
	${$self->contigList}{$name} = $CONTIG;
	return;
}

sub _addContigToIndex {
	my ($self, $name, $widgetID) = @_;
	return unless $name;
	return unless $widgetID;
	unless ($self->contigIndex){$self->{contigIndex} = {}}
	${$self->contigIndex}{$widgetID} = $name;
	return;
}



sub ChromosomeMap {
	my ($self, $map) = @_;
	if ($map){$self->{ChromosomeMap} = $map}
	return $self->{ChromosomeMap}
}		
sub ChromosomeWindow {
	my ($self, $window) = @_;
	if ($window){$self->{ChromosomeWindow} = $window}
	return $self->{ChromosomeWindow}
}		
sub Context {
	my ($self, $context) = @_;
	if ($context){$self->{Context} = $context}
	return $self->{Context}
}		

sub ChrMap {
	my ($self, $map) = @_;
	if ($map){$self->{ChrMap} = $map}
	return $self->{ChrMap}
}

sub chrList {
	my ($self, $list) = @_;
	if ($list){$self->{chrList} = $list}
	return $self->{chrList}
}

sub current_contig {
	my ($self, $ctg) = @_;
	if ($ctg){$self->{current_contig} = $ctg}
	return $self->{current_contig}
}

sub mouseover_contig {
	my ($self, $ctg) = @_;
	if ($ctg){$self->{mouseover_contig} = $ctg}
	return $self->{mouseover_contig}
}

sub assemblies {
	my ($self, $ass) = @_;
	if ($ass){$self->{assemblies} = $ass}
	return $self->{assemblies}
}

#sub contigList {
#	my ($self, $list) = @_;
#	if ($list){$self->{contigList} = $list}
#	return $self->{contigList}
#}#

#sub contigIndex {
#	my ($self, $idx) = @_;
#	if ($idx){$self->{contigIndex} = $idx}
#	return $self->{contigIndex}
#}


1;
