package GQ::Server::DB::Context;
use strict;
use DBI;
use vars qw( @ISA $AUTOLOAD );  #Keep 'use strict' happy
use Carp;

{
    my %type_hash=(
		#   feature_name           appropriate GQ object     (one of the given three, or roll your own if you are brave!)
			Gene				=>'GQ::Server::Gene',
			Exon				=>'GQ::Server::Feature',
			UTR					=>'GQ::Server::Feature',
			'TRNA Gene'			=>'GQ::Server::Gene',
			'RNA Exon'			=>'GQ::Server::Feature',
			Promoter			=>'GQ::Server::Feature',
			Intron				=>'GQ::Server::Feature',
			Transcript			=>'GQ::Server::Transcript',
			Poly_A_site			=>'GQ::Server::Feature',
			Misc_Feature		=>'GQ::Server::Feature',
			DEFAULT				=>'GQ::Server::Feature',
		  );

    sub _type {
	my ($self,$type)=@_;
	if (defined $type_hash{$type}){return $type_hash{$type}}
	else {return $type_hash{'DEFAULT'}}
    }
    
    sub _get_types {
		my ($self)=@_;
		return %type_hash
	}

}

sub new {
    my ($caller, $dbobj)=@_;
    my $class=ref($caller)||$caller;
    my $self= {};
    bless $self, $class;
    $self->{dbobj}=$dbobj;
    $self->{autocommit}=$dbobj->{autocommit};
    return $self;
}

sub adaptor {
    my ($self, $object)=@_;
    my $adaptor_class=$self->_adaptor_for($object);
    unless ($self->{adaptor}{$adaptor_class}) {
	eval "require $adaptor_class;";
	if ($@) {
	    die "$@";
	}
	$self->{adaptor}{$adaptor_class}=$adaptor_class->new(context=>$self,
							     dbh    =>$self->dbh);
    }
    return $self->{adaptor}{$adaptor_class};
}

sub _adaptor_for {
    my ($self, $thing)=@_;
    #this is a catalogue of adaptors that matches the object with the correct adaptor
    #for this context, which in this case is a
    #mysql database environment
    my $adaptor='GQ::Server::DB::Adaptor::';
    my $object=ref($thing)||$thing;
    #most specific first
    if ($object eq 'GQ::Server::Feature::inMemory') {
	$adaptor .= 'Feature::inMemoryAdaptor';
    } elsif ($object eq 'GQ::Server::Feature::mod') {
	$adaptor .= 'FeatureAdaptor';
    } elsif ($object eq 'GQ::Server::Feature') {
	$adaptor .= 'FeatureAdaptor';
    } elsif ($object eq 'GQ::Server::Gene') {
	$adaptor .= 'GeneAdaptor';
    } elsif ($object eq 'GQ::Server::Transcript') {
	$adaptor .= 'TranscriptAdaptor';
    } elsif ($object eq 'GQ::Server::Sequence') {
	$adaptor .= 'SequenceAdaptor';
    } elsif ($object eq 'GQ::Client::Chat') {
	$adaptor .= 'ChatAdaptor';
    } elsif ($object eq 'GQ::Server::Organism') {
	$adaptor .= 'OrganismAdaptor';
    } elsif ($object eq 'GQ::Server::Contig') {
	$adaptor .= 'ContigAdaptor';
    } else {
	$adaptor .= 'ObjectAdaptor';
    }

    return $adaptor;
}

sub all_orgs_by_id {
    require GQ::Server::Organism;
    my ($self)=@_;
    my (@organisms);
    my $sth=$self->dbh->prepare('SELECT common,latin,code,Organism.id,Version.id FROM Organism,Version '.
				'WHERE Organism.id=Version.organism ORDER BY Organism.id');
    $sth->execute or return;
    while (my ($common,$latin,$code,$id,$version)=$sth->fetchrow_array) {
	push @organisms, GQ::Server::Organism->new(context=> $self,
						   common => $common,
						   latin  => $latin,
						   code   => $code,
						   id     => $id,
						   version=> $version,
						  );
    }
    return @organisms;
}

=head2 add_org

 Title   : add_org
 Usage   : $context->add_org(
			     common => $common_name,
			     latin  => $latin_name,
			     code   => $two_letter_code,
                            );
 Function: adds an organism to the database, and creates a new GQ::Server::Organism object
 Returns : the newly created GQ::Server::Organism object
 Args    : a hash containing
            common => $common_name,
            latin  => $latin_name,
            code   => $two_letter_code
            for the organism


=cut

sub add_org {
    require GQ::Server::Organism;
    my ($self, %args,)=@_;
    my $dbh=$self->dbh;
    my $sth=$dbh->prepare("INSERT INTO Organism (common,latin,code) ".
			  "VALUES ('".$args{common}."','".$args{latin}."','".$args{code}."')") or return;
    $sth->execute or return;
    my $newid=$dbh->{mysql_insertid};
    $sth=$dbh->prepare("INSERT INTO Version (id,organism) VALUES (?,?)");
    $sth->execute(1,$newid);
    $self->commit;
    my $neworg=GQ::Server::Organism->new(context=> $self,
					 common => $args{common},
					 latin  => $args{latin},
					 code   => $args{code},
					 id     => $newid,
					);
    return $neworg;
}

sub commit {
    my ($self)=@_;
    unless ($self->{autocommit}) {
	$self->dbh->commit;
    }
}

sub rollback {
    my ($self)=@_;
    unless ($self->{autocommit}) {
	$self->dbh->rollback;
    }

}

sub COMMIT {
	return 1;
}

sub dbobj { return $_[0]->{dbobj} }

sub dbh { return $_[0]->dbobj->dbh }

sub user { return $_[0]->dbobj->user }

sub DEFAULT_ID {1}
sub DEFAULT_VERSION {1}

=head2 organism

 Title   : organism
 Usage   : $context->organism($newval)
 Function: gets/sets the current organism, creating a new object if needed
 Returns : the current GQ::Server::Organism object for this context
 Args    : newvalue (optional), which could be a GQ::Server::Organism object, or the string 'default'


=cut

sub organism {
    require GQ::Server::Organism;
    my ($self, $newval)=@_;
    if ($newval) {
	if ($newval eq 'default') {
	    my $id=$self->DEFAULT_ID;
	    my $sth=$self->dbh->prepare('SELECT common,latin,code FROM Organism WHERE id=?');
	    $sth->execute($id) or return;
	    while (my ($common,$latin,$code)=$sth->fetchrow_array) {
		$newval=GQ::Server::Organism->new(context=> $self,
						  common => $common,
						  latin  => $latin,
						  code   => $code,
						  id     => $id,
						  version=> $self->DEFAULT_VERSION,
						 );
	    }
	}
	$self->{organism}=$newval;
    }
    return $self->{organism};
}

sub version {return $_[0]->organism->version}

sub get_contigs_by_name {
    my ($self)=@_;
    my (@list);
    my $sth=$self->dbh->prepare("SELECT distinct name FROM Contig,ContigAssembly,Assembly WHERE Assembly.version=? ".
				"AND Contig.id=contig_id and assembly=Assembly.id and Assembly.organism=? ORDER BY name DESC");
    $sth->execute($self->version,$self->organism->id);
    while (my ($name)=$sth->fetchrow_array) {
	push @list, $name;
    }
    return @list;
}

sub get_chrs_by_id {
    my ($self)=@_;
    my @list;
    my $chr_query = $self->dbh->prepare("SELECT DISTINCT chr_id FROM Assembly WHERE version=? and organism=? order by chr_id asc");
    $chr_query->execute($self->version,$self->organism->id);
    while (my ($chr_id)=$chr_query->fetchrow_array) {
	push @list,$chr_id;
    }
    return @list;
}

sub get_ordered_assemblies_by_chr {
    my ($self, $chr_id)=@_;
    my %struct;

    my $order_query = $self->dbh->prepare("SELECT Assembly.id, name, VC_length FROM Tiling_Path, Contig, ContigAssembly, Assembly ".
					  "WHERE Tiling_Path.contig_id=ContigAssembly.id AND Contig.id=ContigAssembly.contig_id ".
					  "AND assembly=Assembly.id AND chr_id=? AND Assembly.version=? ORDER BY Assembly.id,abs_start ASC");
    $order_query->execute($chr_id,$self->version);
    while (my ($id,$name,$len) = $order_query->fetchrow_array) {
        push @{$struct{$id}},[$name,$len];
    }
    return \%struct;
}

sub get_next {
    my ($self,$present,$np)=@_;
    my ($sth,$than,$dir);
    if ($np eq "next") {
	$than='>';
	$dir='ASC';
    } else {
	$than='<';
	$dir='DESC';
    }
    $sth=$self->dbh->prepare(	"SELECT ConA.name FROM Assembly, " .
				"Tiling_Path as A, Contig as ConA, ContigAssembly as CAA, ".
				"Tiling_Path as B, Contig as ConB, ContigAssembly as CAB " .
				"WHERE A.contig_id=CAA.id and B.contig_id=CAB.id " .
				"AND ConA.id=CAA.contig_id and ConB.id=CAB.contig_id ".
				"AND CAA.assembly = CAB.assembly and CAB.assembly=Assembly.id " .
				"AND Assembly.organism=? " .
				"AND ConB.name = ? " .
				"AND A.abs_start $than B.abs_start " .
				"ORDER BY A.abs_start $dir " .
				"LIMIT 1");

    $sth->execute($self->organism->id,$present);
    my ($contig) = $sth->fetchrow_array;
    return $contig;
}

sub contig {
    require GQ::Server::Contig;
    my ($self, $contig)=@_;
    unless ($self->{contiglist}{$contig}) {
	my $contigobj=GQ::Server::Contig->new(name   =>$contig,
					      context=>$self,
					     );
	$self->{contiglist}{$contig}=$contigobj;
	$self->{contig_by_id}{$contigobj->id}=$contigobj;
    }
    return $self->{contiglist}{$contig};
}

sub contig_by_id {
    require GQ::Server::Contig;
    my ($self, $contig_id)=@_;
    unless ($self->{contig_by_id}{$contig_id}) {
	my $sth=$self->dbh->prepare("SELECT name FROM Contig,ContigAssembly ".
				    "WHERE ContigAssembly.id=? and contig_id=Contig.id");
	$sth->execute($contig_id);
	my ($contig)=$sth->fetchrow_array;
	my $contigobj=GQ::Server::Contig->new(name   =>$contig,
					      context=>$self,
					     );
	$self->{contig_by_id}{$contig_id}=$contigobj;
	$self->{contiglist}{$contig}=$contigobj;
    }
    return $self->{contig_by_id}{$contig_id};
}

sub get_contig_sequence {
    my ($self, $contig)=@_;
    my $contigobj=$self->contig($contig);
    return $contigobj->sequence;
}

sub get_contig_length {
    my ($self, $contig)=@_;
    my $contigobj=$self->contig($contig);
    return $contigobj->length;
}

sub get_contig_id {
    my ($self, $contig)=@_;
    my $contigobj=$self->contig($contig);
    return $contigobj->id;
}

sub get_boundaries {
    my ($self, $contig)=@_;
    my $contigobj=$self->contig($contig);
    return $contigobj->boundaries;
}

sub _get_contig_start_stop {
    my ($self, $contig)=@_;
    my $contigobj=$self->contig($contig);
    my $assembly=$contigobj->assembly;
    my $start=$contigobj->abs_start;
    my $len=$contigobj->length;
    my $stop=$start+$len-1;
    return ($assembly,$start,$stop);
}

sub find_assembly {
    my ($self, $contig)=@_;
    my $contigobj=$self->contig($contig);
    return $contigobj->assembly;
}

#This is a bulk load of contig info, organized by name.  I'll leave it for now, but it may be rationalized soon.
sub get_contig_info_by_name {
    my ($self)=@_;
    my %seq;
    my $sth=$self->dbh->prepare('SELECT name,Tiling_Path.contig_id,abs_start '.
				'FROM Contig,Tiling_Path,ContigAssembly '.
				'WHERE ContigAssembly.id=Tiling_Path.contig_id AND Contig.id=ContigAssembly.contig_id '.
				"AND version=?");
    $sth->execute($self->version);
    while (my ($name,$cid,$abs)=$sth->fetchrow_array) {
	$seq{$name}=[$cid,$abs];
    }
    return \%seq;
}

#the following three methods are schema-dependent, and should only be implemented where they make sense
sub insert_contig {
    my ($self, %args)=@_;
    my $centromere = $args{centromere} || 'n';
    my $sth=$self->dbh->prepare("INSERT INTO Contig (name,centromere) VALUES (?,?)");
    $sth->execute($args{contig_name},$centromere);
    my $contig_id=$self->dbh->{mysql_insertid};
    $self->commit;
    return $contig_id;
}

sub insert_contigAssembly {
    my ($self, %args)=@_;
    my $sth=$self->dbh->prepare("INSERT INTO ContigAssembly (contig_id,version,assembly) ".
				"VALUES (?,?,?)");
    $sth->execute($args{contig_id},$args{version},$args{assembly});
    my $contig=$self->dbh->{mysql_insertid};
    $self->commit;
    return $contig;
}

sub insert_Assembly {
    my ($self, %args)=@_;
    my $sth=$self->dbh->prepare("SELECT id FROM Assembly WHERE ".
			        "organism=? AND ".
			        "chr_id=? AND ".
			        "version=?");
    $sth->execute($self->organism->id,$args{chr},$self->version);
    my ($id)=$sth->fetchrow_array;
    return $id if $id;
    $sth=$self->dbh->prepare("INSERT INTO Assembly (organism,chr_id,version) ".
			     "VALUES (?,?,?)");
    $sth->execute($self->organism->id,$args{chr},$self->version);
    my $Assembly_id=$self->dbh->{mysql_insertid};
    $self->commit;
    return $Assembly_id;
}

sub get_EST_sources {
    my ($self)=@_;
    my @list;
    my $sth=$self->dbh->prepare("SELECT DISTINCT source FROM EST");
    $sth->execute;
    while (my ($src)=$sth->fetchrow_array) {
	push @list,$src;
    }
    return \@list;
}

sub get_EST_sequence {
    my ($self, $EST_name)=@_;
    my $sth = $self->dbh->prepare("SELECT seq FROM EST WHERE gb = ?");
    $sth->execute($EST_name);
    my ($seq)=$sth->fetchrow_array;
    return $seq;
}



sub get_EST_hits {
    my ($self, %args)=@_;
    my ($start,$stop,$contig)=@args{'start','stop','contig'};
    unless ($stop) {
	($start,$stop)=$self->get_boundaries($contig);
    }
    my  $sth = $self->dbh->prepare("SELECT Alignment, (B.abs_start + hsp_subj_start - (A.abs_start+?-1)) AS hit_start, ".
				   "(B.abs_start + hsp_subj_end - (A.abs_start+?-1)) AS hit_end, hsp_subj_strand, hsp_query_strand, Blast_vs_EST.source ".
				   "FROM Blast_vs_EST, Tiling_Path A, Tiling_Path B, Contig, ContigAssembly CAA, ContigAssembly CAB ".
				   "WHERE A.contig_id=CAA.id and CAA.contig_id=Contig.id and Contig.name=? ".
				   "AND (B.abs_start+hsp_subj_start-(A.abs_start+?-1)) between ? and ? AND ".
				   "B.contig_id=Blast_vs_EST.contig_id AND ".
				   "B.contig_id=CAB.id AND A.contig_id=CAA.id AND ".
				   "CAB.version=CAA.version AND CAB.version=?");
    $sth->execute($start,$start,$contig,$start,$start,$stop,$self->version);
    my @list;
    while (my ($id, $start, $end, $subj_str, $str, $source)=$sth->fetchrow_array) {
	push @list, [$id, $start, $end, $subj_str, $str, $source];
    }
    return @list;
}

sub get_common_exons {
    my ($self, $fid)=@_;
    my @list;
    my $sth = $self->dbh->prepare("SELECT id2 FROM BlastLookUp WHERE id1 = $fid");
    $sth->execute;
    while (my ($id2)=$sth->fetchrow_array) {
	push @list,$id2;
    }
    return @list;
}

sub get_tags {
    my ($self)=@_;
    my @list;
    my $sth = $self->dbh->prepare("SELECT tag FROM Tags");
    $sth->execute;
    while (my ($tag)=$sth->fetchrow_array) {
	push @list,$tag;
    }
    return @list;
}


sub _prepare_handles {
    my ($self)=@_;
    my $sth_FindShared = $self->dbh->prepare("SELECT DISTINCT Feature1.id FROM Feature AS Feature1, Feature AS Feature2, BlastAcc AS Blast,  BlastAcc AS Blast1 ". # select other Feature IDs where
					     "WHERE Feature2.id=? ".       # this FeatureID is the exon of interest
					     "AND Blast.gi=Blast1.gi ".    # and the features share a blast homology
					     "AND Blast.exon_id=Feature1.id AND Blast1.exon_id=Feature2.id ".  # (join Blast results and feature table)
					     "AND Feature1.id <> Feature2.id");   # but they are not the same feature

    my $sth_UpdateBlastAcc = $self->dbh->prepare("INSERT IGNORE BlastAcc (exon_id, gi, db, rawscore, probability, accession, description) ".
						 "VALUES (?, ?, ?, ?, ?, ?, ?)");

    my $sth_LookUp = $self->dbh->prepare("INSERT IGNORE INTO BlastLookUp VALUES (?,?)");
    $ContextPicker::sth_list=[$sth_FindShared,$sth_UpdateBlastAcc,$sth_LookUp];
}


sub parse_Blast_to_db {
    my ($self, $DB_ID, $BlastResultObj)=@_;
    
    my $class=ref($self);

    unless ($class::sth_list) {
	$self->_prepare_handles;
    }
	$BlastResultObj->{'_hitindex'} = 0;  # reset the blast report
	
    #while (my $result = $BlastObj->next_result){
		while (my $hit = $BlastResultObj->next_hit){
			my (@elements) = split /\|/, $hit->name;
			my $gi = $elements[1];                         # so break it on the pipe and take individual elements
			my $db = $elements[2];
			my $dbacc = $elements[3];
			my $desc = $hit->description;

			my $hsp = $hit->next_hsp;  # just take the first one
			my $raw = $hsp->score;
			my $prob = $hsp->evalue;
			
			$ContextPicker::sth_list->[1]->execute($DB_ID, $gi, $db, $raw, $prob, $dbacc, $desc);
			$self->commit;
		}
	#}
	
    my $sth=$ContextPicker::sth_list->[0];
    $sth->execute($DB_ID);

    while (my ($id2)=$sth->fetchrow_array) {
	my $sth_LookUp=$ContextPicker::sth_list->[2];
	$sth_LookUp->execute($DB_ID,$id2);
	$sth_LookUp->execute($id2,$DB_ID);
	$self->commit;
    }
}

sub create_tag {
    my ($self, $tag) = @_;
    foreach ($self->get_tags) {
	return if $_ eq $tag;
    }
    my $sth=$self->dbh->prepare("INSERT INTO Tags (tag) VALUES (?)");
    $sth->execute($tag) or die "Couldn't insert $tag: $DBI::errstr";
    $self->commit;
}

sub checklock {
    my ($self, %args) = @_;
	
	# this code was not functioning properly, so I removed it.
	# I don't know if it is this code, or the call to this code
	# unless you are viewing a contig from position 1, this will fail.
	
    if ($args{lockid} eq 'admin' || !$args{contig_stop}) {
	return;
    }
    my ($abs_start,$abs_stop)=$self->get_location(@args{'contig_id','contig_start','contig_stop'});
    if ($abs_start<= $self->lock(stop=>$args{lockid}) &&
	$abs_stop >= $self->lock(start=>$args{lockid})) {
	return;
    } else {
	return 1;
    }
}

sub get_location {
    my ($self, $contig_id, $contig_start, $contig_stop) = @_;
     my $contigobj=$self->contig_by_id($contig_id);
    $contigobj || return;
    my $abs_start=$contigobj->abs_start;
    my $start=$abs_start+$contig_start-1;
    my $stop= $abs_start+$contig_stop -1;
    return ($start,$stop);
}

sub lock {
    my ($self,$parm,$new)=@_;
    if ($parm eq 'start') {
	return $self->{lockstart}{$new};
    } elsif ($parm eq 'stop') {
	return $self->{lockstop}{$new};
    } elsif ($parm eq 'both') {
	return ([$self->{lockstart}{$new},$self->{lockstop}{$new}]);
    } elsif ($parm eq 'add') {
	return unless ((my $contig = $new->[0]) &&
		       (my $newstart=$new->[1]) &&
		       (my $newstop =$new->[2]));
	my ($assembly,$absstart,$absstop)=$self->_get_contig_start_stop($contig);
	$newstart=$absstart+$newstart-1;
	$newstop= $absstart+$newstop -1;
	return $self->_look_lock($assembly,$newstart,$newstop);
    } elsif ($parm eq 'contig') {
	my ($assembly,$newstart,$newstop)=$self->_get_contig_start_stop($new);
	return $self->_look_lock($assembly,$newstart,$newstop);
    } elsif ($parm eq 'release') {
	my $sth=$self->dbh->prepare("DELETE FROM AnnotationLock ".
				    "WHERE id = ?");
	$sth->execute($new) or return;
	$self->commit;
	return 1;
    }
}

sub _look_lock {
    my ($self,$assembly,$newstart,$newstop)=@_;
    my $sth=$self->dbh->prepare("SELECT user,start,stop ".  #only select start/stop from same
				"FROM AnnotationLock ".     #reference points
				"WHERE version=? ".
				"AND assembly=?");
    $sth->execute($self->version,$assembly);
    while (my ($user,$start,$stop)=$sth->fetchrow_array) {
	next if $user eq $self->user;  #this means it will be okay regardless

	return if ($start<=$newstop &&     #this means there is overlap with another user- bad!
		   $stop >=$newstart);

    }
    $sth=$self->dbh->prepare("INSERT INTO AnnotationLock (user,version,assembly,start,stop) ".
			     "VALUES (?,?,?,?,?)");
    $sth->execute($self->user,$self->version,$assembly,$newstart,$newstop) or return;
    my $lockid=$self->dbh->{mysql_insertid};
    $self->commit;

    $self->{lockstart}{$lockid}=$newstart;  #now update in memory
    $self->{lockstop}{$lockid}=$newstop;  #now update in memory	
    return $lockid;
}

sub get_flagged_features {
    my ($self,$flag)=@_;
    my @list;
    my $flid=$self->_check_flag($flag);
    my $sth=$self->dbh->prepare("SELECT feature_id FROM Flag,ContigAssembly WHERE flag=? ".
			        "AND Flag.contig_id=ContigAssembly.contig_id and version=?");
    $sth->execute($flid,$self->version);
    while (my ($fid)=$sth->fetchrow_array) {
	push @list,$fid;
    }
    return \@list;
}

sub get_flagged_contigs {
    my ($self,$flag)=@_;
    my @list;
    my $flid=$self->_check_flag($flag);
    my $sth=$self->dbh->prepare("SELECT name FROM Flag,Contig,ContigAssembly WHERE flag=? ".
			        "AND Flag.contig_id=ContigAssembly.id and ContigAssembly.contig_id=Contig.id ".
			        "AND version=?");
    $sth->execute($flid,$self->version);
    while (my ($contig)=$sth->fetchrow_array) {
	push @list,$contig;
    }
    return \@list;
}

sub remove_flag {
    my ($self,$flag)=@_;
    my $flid=$self->_check_flag($flag);
    my $sth=$self->dbh->prepare("DELETE FROM Flag WHERE flag=?");
    $sth->execute($flid) or die "Flag removal unsuccessful: $DBI::errstr";
    $sth=$self->dbh->prepare("DELETE FROM Flags WHERE id=?");
    $sth->execute($flid) or die "Flag removal unsuccessful: $DBI::errstr";
    $self->commit;
}

sub _check_flag {
    my ($self,$flag)=@_;
    my $sth=$self->dbh->prepare('SELECT id FROM Flags WHERE name LIKE ?');
    $sth->execute($flag);
    while (my ($flid)=$sth->fetchrow_array) {
	return $flid;
    }
    #we're here, so there is no such flag.  Insert it.
    $sth=$self->dbh->prepare("INSERT INTO Flags (name) VALUES (?)");
    $sth->execute($flag) or die "Couldn't insert $flag into Flags table: $DBI::errstr";
    my $flid=$self->dbh->{mysql_insertid};
    $self->commit;
    return $flid;
}

sub add_flag {
    my ($self, %args)=@_;
    my ($flag,$feature,$contig,$flid)=@args{'flag','feature_id','contig_id','flag_id'};
    $flid ||= $self->_check_flag($flag);
    my $sth=$self->dbh->prepare("INSERT INTO Flag (feature_id,contig_id,flag) ".
			        "VALUES (?,?,?)");
    $sth->execute($feature,$contig,$flid) or die "Couldn't insert $flag onto $feature: $DBI::errstr";
    my $id=$self->dbh->{mysql_insertid};
    $self->commit;
    return $id;
}

sub rename_flag {
    my ($self,$old_flag,$new_flag)=@_;
    my $old_id=$self->_check_flag($old_flag);
    my $new_id=$self->_check_flag($new_flag);
    my $sth=$self->dbh->prepare("UPDATE Flag SET flag=? WHERE flag=?");
    $sth->execute($new_id,$old_id);
    $self->commit;
    return $new_id;
}

sub all_flags {
    my ($self)=@_;
    my @flags;
    my $sth=$self->dbh->prepare("SELECT name FROM Flags");
    $sth->execute;
    while (my ($flag)=$sth->fetchrow_array) {
	push @flags,$flag;
    }
    return @flags;
}

sub flag_text {
    my ($self,%args)=@_;
    my ($flag,$textlist,$andor)=@args{'flag','textlist','andor'};
    my $flid=$self->_check_flag($flag);
    my $sth=$self->dbh->prepare("SELECT fid FROM WordBase,Word WHERE Word.word=? AND Word.id=WordBase.word");
    my $sth2=$self->dbh->prepare("SELECT feature_id,contig_id FROM fid WHERE id=?");
    my (%options,%left);
    foreach (@$textlist) {   # for each keyword
		$options{$_}=();
		$sth->execute($_);	# search for the feature id(s) that match
		
		while (my ($fid)=$sth->fetchrow_array) {  # for each matching feature id
			if ($andor !~ /and/i) {					# if it is an 'or' statement
				$sth2->execute($fid);				# then get the feature_id and contig_id
				my ($feature, $contig)=$sth2->fetchrow_array;# and put it into the array ref 'coords'
				next unless ($feature && $contig);
				$self->add_flag(flag => $flag, feature_id => $feature, contig_id => $contig, flag_id => $flid); # then flag that feature
			} else {
				$options{$_}{$fid}=1;  # But if it is an AND just push it onto a hash of hashes to double-check later
			}
		}
    }
    if ($andor =~ /and/i) {  # okay, now deal with the 'AND' case
		my $addedfirst=0;
		foreach my $opt (keys %options) { # for each existing keyword
			unless ($addedfirst) {		# distinguish between the first time this routine is run, and later times
				foreach my $fid(keys %{$options{$opt}}) {  # check the 'hit' feature_ids from the given keyword
					$left{$fid}=1;						   # and flag them as being valid
				}
				$addedfirst++;
			}
			foreach my $left(keys %left) {					# now remove anything that does not also come up in later searches (AND)
				unless (exists $options{$opt}{$left}) {
					delete $left{$left};
				}
			}
		}
		foreach my $fid (keys %left) {  # for anything left, get teh coords (contig id and feature id)
			$sth2->execute($fid);
			my ($feature, $contig)=$sth2->fetchrow_array;# and put it into the array ref 'coords'
			next unless ($feature && $contig);
			$self->add_flag(flag => $flag, feature_id => $feature, contig_id => $contig, flag_id => $flid); # then flag that feature
		}
    }
}


1;







