# Gene module for Genquire

# by David Block <dblock@gene.pbir.nrc.ca>

# Copyright David Block and NRC-PBI

# POD Documentation - main docs before the code

=head1 NAME

Gene - Gene Object for Genquire

=head1 SYNOPSIS

=head2 Loading Genes from the database

$par_pos is the chromosomal position of the start of the sequence
region you are interested in (often the same as $start).  It is used to
define the relative location of the gene within that sequence.

=head2 Creating a new Gene:

=head2 Deleting a Gene:

$gene->delete_from_db;

=head2 Finding out about a Gene:

$start=$gene->start;

returns the start of the gene relative to its parent sequence ($par_pos).

$end=$gene->end;

returns the end of the gene relative to its parent sequence ($par_pos).

$absstart=$gene->absstart;

returns the gene's start position on its assembly.

$absstop=$gene->absstop;

returns the gene's end position on its assembly.

$strand=$gene->strand;

returns the gene's strand (+1,-1,0)

$gff_strand=$gene->gff_strand;

returns the gene's strand (+,-,.)

$length=$gene->length;

returns the gene's length.

=head2 Finding out about a Gene's Features:

The List of features in a hash:

$hash_of_features=$gene->features;

An individual feature:

$feature=$gene->feature($feature_id);

Adding a feature:

$gene->features($feature);

=head1 DESCRIPTION

Gene is a collection of transcripts which contain features assembled
by an annotator at some point.  It lives in the database in the Feature
table, with its elements available via the Container table.  It is a
GenericFeature, which means it implements Bio::SeqFeatureI as well as
all the methods common to Feature, Gene, and Feature::mod.

=head1 FEEDBACK

Like it or lump it, report to dblock@gene.pbi.nrc.ca.  Feel free to add
to the docs available at
http://bioinfo.pbi.nrc.ca/wiki

=head2 Reporting Bugs

Email the author with any bug reports.

=head1 AUTHOR - David Block

email dblock@gene.pbi.nrc.ca

=cut

package GQ::Server::Gene;
$VERSION = 1.00;
use strict;

use vars qw( $AUTOLOAD @ISA);  #Keep 'use strict' happy

use Carp;
use DBI;

use GQ::Server::Container;
use GQ::Root;
use Bio::SeqFeature::Gene::GeneStructure;
@ISA = qw(GQ::Server::Container Bio::SeqFeature::Gene::GeneStructure);

{
    #Encapsulated class data
    #replace with your column names, and sensible defaults

                                 #       DEFAULT      ACCESSIBILITY
    GQ::Root->create(
		transcripts   =>  [undef,        'read/write'],
	    gene_name     =>  ['',           'read/write'],
		    );

    my $_context;
    my $_par_pos;

    sub _store_context {
	my ($self,$context,$par_pos)=@_;
	$_context=$context;
	$_par_pos=$par_pos;
    }

    sub _context { return $_context }

    sub _par_pos { return $_par_pos }
}


sub _db_keys {
    my ($self)=@_;
    my @list;
    foreach ($self->SUPER::_db_keys) {
	push @list,$_ unless /^id$|transcripts|gene_name/;
    }

    push @list,'id';  #ensure that id is last
    return @list;
}

### write in any custom subroutines here

sub _initialize {
    my ($self)=@_;
#Go get the correct adaptor from self's context
    $self->adaptor($self->context->adaptor($self));

###  This section inserts the object's data into the database
###  only if the new method was called by the class, i.e.
###  not as a copy of a previous object,  OR
###  if the caller does not have an id at present
###  which it would if it were loaded from the database (see sub load)

    if (not $self->{id} ) {
	$self->get_initial_details;

	die "Must Acquire Lock On This Region To Proceed.\n"
	  if $self->context->checklock(lockid       => $self->lockid,
				       contig_id    => $self->contig_id,
				       contig_start => $self->contig_start,
				       contig_stop  => $self->contig_stop);

	unless ($self->name) {
	    $self->{name}=$self->_generate_name;
	}

	$self->{id}=$self->adaptor->insert_new(keys  =>[$self->_db_keys],
					       values=>[$self->_db_values],
				  );
	$self->last_modified;   #put the db last_modified timestamp value into the object
    }

    $self->flesh_out_details if $self->transcripts;
    $self->insert_tag_values if defined $self->tagvalues;
    $self->gene_name($self->name);
    return $self;
}

sub _generate_name {
    my ($self)=shift;
    return $self->adaptor->generate_name($self);
}

=head2 gene_name

 Title   : gene_name
 Usage   : $obj->gene_name($newval)
 Function: gets the gene_name attribute
 Returns : value of gene_name
 Args    : none


=cut

sub gene_name{
   my ($self) = @_;
   if ( not $self->{gene_name}) {
      $self->{gene_name} = $self->name;
    }
    return $self->{gene_name};
}

sub remove_feature {
    my ($self,$feature)=@_;
    return $self->remove_transcript($feature) if $feature->type eq 'Transcript';
    return $self if $feature->access eq 'ro';
    $self->entire_seq->add_SeqFeature($feature);  #the feature is now independent - don't remove it from the db
    foreach ($self->transcripts) {
	$self->{transcripts}{$_->id}=$_->remove_feature($feature); #remove it from participating transcripts
    }
    unless ($self->transcripts) {  #i.e. if that was the only remaining exon
	$self->delete_from_db;     #I have no reason to exist
	undef $self;               #I no longer exist
	return;                    #I return nothing
    }
    return unless $self->feature($feature);
    delete $self->{features}{$feature};
    $self->flesh_out_details;
    return $self;
}

sub all_tags {
    my ($self)=@_;
    my @list=$self->SUPER::all_tags;
    return (@list,"gene_name");
}

sub sub_SeqFeature {
    my ($self)=@_;
    my @list;
    foreach ($self->transcripts) {
	next unless $_;
	push @list, $_;
	push @list, $_->sub_SeqFeature;
    }
    push @list, values %{$self->{features}};  #just in case we have any non-transcript features
    return @list;
}

sub get_initial_details {
    my ($self,$featurelist)=@_;
    return unless $featurelist;
    my $first=1;
    foreach my $feature (sort {$a->absstart <=> $b->absstart} @$featurelist) {
	if ($first) {
	    $self->{contig_id}   =$feature->contig_id;
	    $self->{contig_start}=$feature->contig_start;
	    $self->{contig_stop} =$feature->contig_stop;
	    $self->{strand}      =$feature->strand;
	    $first=0;
	} else {
	    if ($feature->absstart < $self->absstart) {
		$self->{contig_start}=$feature->contig_start;
		$self->{contig_id}=$feature->contig_id;
	    }
	    if ($feature->absstop > $self->absstop) {
		if ($feature->contig_id eq $self->contig_id) {
		    $self->{contig_stop}=$feature->contig_stop;
		} else {
		    my $difference=$self->adaptor->diff($self,$feature);
		    $self->{contig_stop}=$feature->contig_stop + $difference;
		}
	    }
	}
    }
    $self->{length}=$self->end - $self->start + 1;
}

sub transcripts_hash {
    require GQ::Server::Transcript;
    my ($self,$arg)=@_;
    if ($arg) {
	my @arg=(ref($arg) eq 'LIST')?@$arg:($arg);
	foreach my $this (@arg) {
	    my $new;
	    if ($this->isa('GQ::Server::Transcript')) {
		$self->{transcripts}{$this->id}=$this;   #form the in-memory connection top-down
		if ($this->name ne $self->name) {
		    $this->name($self->name);         #form the long-term bottom-up connection
		}
		$new=$this;
	    } else {
		$new=GQ::Server::Transcript->new(%$this,
						 context =>$self->context,
						 lockid  =>$self->lockid,
						 name    =>$self->name);  #long-term bottom-up
		$self->{transcripts}{$new->id}=$new if $new;          #short-term top-down
	    }
	    $new->attach_seq($self->entire_seq);  # parent sequence object
	    $self->adaptor->add_transcript($self,$new);
	    $self->flesh_out_details;
	}
    }
    if ($self->{transcripts}) {
	return $self->{transcripts};
    } else {
	return {};
    }
}

sub transcript { return $_[0]->{transcripts}{$_[1]} }

sub remove_transcript {
    my ($self,$transcript)=@_;
    return $self if $transcript->access eq 'ro';
    foreach ($transcript->features) {
	$transcript->remove_feature($_);
    }
    delete $self->{transcripts}{$transcript->id};
    if (not $self->transcripts) {
	$self->delete_from_db;
	undef $self;
	return;
    }
    $self->flesh_out_details;
    return $self;
}

########### IMPLEMENTATION OF Bio::SeqFeature::Gene::GeneStructure #####

sub transcripts {
    my ($self,$arg)=@_;
    my $hash = $self->transcripts_hash($arg);
    if (defined($hash)) { return values %$hash; }
}

sub add_transcript {
    my ($self,$fea)=@_;
    die "Must be a Bio::SeqFeature::Gene::TranscriptI object" unless
      $fea->isa('Bio::SeqFeature::Gene::TranscriptI');

    if ($fea->isa('GQ::Server::Transcript')) {
	return $self->transcripts_hash($fea);
    } else {
	my @list=$fea->sub_SeqFeature;
	return $self->add_transcript_as_features(@list);
    }
}

sub flush_transcripts {
    my ($self)=@_;
    if (exists $self->{transcripts}) {
	delete($self->{transcripts});
    }
}

sub add_transcript_as_features {
    require GQ::Server::Transcript;
    #require GQ::Server::Feature::inMemory;
    my ($self,@features)=@_;
	
    my $trans=GQ::Server::Transcript->new(
				context  =>$self->context,
				lockid   =>$self->entire_seq->lockid,
				par_pos  =>$self->par_pos,
				contig_id=>$self->contig_id,
				name     =>$self->name,
				primary_tag=>'Transcript',
				source_tag=>$self->context->user,
				);
    foreach my $exon(@features) {
		my $newfea;
		if ($exon->isa('GQ::Server::GenericFeature') && $exon->id) {  # if it hs no id then we need to assume it is an inMemory feature
			$newfea=$exon;
		} else {
			
			$newfea=GQ::Server::Feature->new(
				context=>$self->context,
				lockid=>$self->entire_seq->lockid,
				par_pos => $exon->par_pos,
				contig_id=>$self->contig_id,
				source_tag=>$exon->source_tag,
				primary_tag=>$exon->primary_tag,
				contig_start=>$exon->contig_start,
				contig_stop=>$exon->contig_stop,
				score=>$exon->score,
				strand=>$exon->strand,
				length=>$exon->length,
				frame=>$exon->frame,
				name => $self->name,
				type => $exon->type,
				);
		}
		$trans->features($newfea);
    }
    $trans->flesh_out_details;
    $self->add_transcript($trans);
}

=head2 features

 Title   : features
 Usage   : my @gene_features=$gene->features;
 Function: return all of the non-transcript features in the gene
 Returns : a list of GQ::Server::GenericFeature objects
 Args    : none


=cut

sub features {
    my ($self,$new)=@_;
    if ($new) {
	$self->SUPER::features($new);
    }
    my @list;

    foreach ($self->transcripts()) {
	push @list, $_->features();
    }
    return @list;
}

sub is_feature_orphan {
    my ($self,$transcript,$feature)=@_;
    my @copies;
    foreach my $tr ($self->transcripts) {       #foreach transcript
	next if ($tr eq $transcript);            #except the original
	push @copies, grep {$_ eq $feature} $tr->features; #find all other copies of the feature
    }
    return 1 unless @copies;  #the feature is an orphan if there are no other copies
    return;
}

1;
