package GQ::Client::GenomeMapperContig;

$VERSION = 1.00;
use strict;
use vars qw( @ISA $AUTOLOAD );  #Keep 'use strict' happy
use Carp;

use GQ::Client::Object;
@ISA=qw(GQ::Client::Object);

BEGIN {
    {
	#Encapsulated class data

	my %_attr_data =	#    DEFAULT         ACCESSIBILITY
	  (	length 		=> 	[1, 	'read/write'], # the length of the contig
		name		=>  [1, 	'read/write'], # the contig id (designation)
		centromere	=>  [0, 	'read/write'], # is it a centromere (boolean)
		GenomeMap	=>  [undef, 'read/write'], # the map object onto which I should draw myself
		def_color	=>  ['#bbbbbb', 'read/write'],   # default pink color
		color		=>	["#bbbbbb", 'read/write'],  # current color
		selected 	=>  [0, 	'read/write'], # boolean selected or not
		offset		=>  [undef, 'read/write'], # offset from the top of the map
		y_pos		=>  [undef, 'read/write'], # the Y coordinate of my center point __
		tags		=>  [[], 	'read/write'], # tags associated with this contig
		outline 	=>  [undef, 'read/write'], # outline color if any
		widgetID	=>  [undef, 'read/write'], # the ID of the canvas widget
		
	  );

	#Class methods, to operate on encapsulated class data

	# Does a specified attribute exist?
	sub _exists {
	    my ($self, $attr) = @_;
	    if (exists $_attr_data{$attr}) {
		return 1;
	    } else {
		return $self->SUPER::_exists($attr);
	    }
	}

	# Is a specified object attribute accessible in a given mode
	sub _accessible  {
	    my ($self, $attr, $mode) = @_;
	    return $_attr_data{$attr}[1] =~ /$mode/ if exists $_attr_data{$attr};
	    return $self->SUPER::_accessible($attr,$mode);
	}

	# Classwide default value for a specified object attribute
	sub _default_for {
	    my ($self, $attr) = @_;
	    return $_attr_data{$attr}[0] if exists $_attr_data{$attr};
	    return $self->SUPER::_default_for($attr);
	}

	# List of names of all specified object attributes
	# A hash so names are not repeated
	sub _standard_keys {
	    my ($self)=@_;
	    my %list;
	    foreach ($self->SUPER::_standard_keys(), keys %_attr_data) {
		$list{$_}++;
	    }
	    return keys %list;
	}
	
	use subs (keys %_attr_data)

    }
} #end of BEGIN block

=head2 new

    Title	:	new
    Usage	:	GenomeMapperContig->new(context => $Context,
										length => $len,
										name => $name,
										centromere => [1 | 0],
										GenomeMap => $map)
    Function:	Create a new Contig object to be mapped onto GenomeMapper
    Returns	: 	$Contig
    Args	: Context object, length, name, is_centromere (1|0)

=cut

sub new {
    my ($caller, %args) = @_;
    my $caller_is_obj = ref($caller);
    my $class = $caller_is_obj || $caller;

    my $self=$class->SUPER::new(%args);
    
    foreach my $attrname ( $self->_standard_keys() ) {
    	if (exists $args{$attrname}) {
    	    $self->{$attrname} = $args{$attrname} }
    	elsif ($caller_is_obj) {
    	    $self->{$attrname} = $caller->{$attrname} }
    	else {
    	    $self->{$attrname} = $self->_default_for($attrname) }
    }
	return $self;

}

sub draw {
	# draws itself on the appropriate canvas
	# returns the canvas widget ID so that the map can
	# be made aware of the Widget and its associated contig ID
	my ($self) = @_;
	my $Canvas = $self->GenomeMap->ChromosomeMap;
	my $Genome = $self->GenomeMap;	
	my $len = $self->length;
	my $contig = $self->name;
	my $contig_tag = "contig " . ($self->name);
	my $offset = $self->offset;
	my $y = $self->y_pos;
	
	my $widgetID = $Canvas->createRectangle(($y-40), $offset, ($y - 30), ($offset+($len/100000)), -outline => ($self->outline), -fill => ($self->color),  -tags => [$contig_tag]);
	$self->widgetID($widgetID);
	$Canvas->bind($widgetID, "<Enter>" =>
			sub {$Genome->setMouseoverContig($contig);
				 });
	$Canvas->bind($widgetID, "<Leave>" =>
			sub {$Genome->setMouseoverContig(undef);
				 });

	$Canvas->bind($widgetID, "<Button-1>" =>
			sub{
				$Genome->dimContig($Genome->getSelectedContig);
				$Genome->setSelectedContig($contig);
				$Genome->lightContig($contig);
				});
	$Canvas->bind($widgetID, "<Double-Button-1>" =>
			sub{$Genome->setSelectedContig($contig);
				$Genome->lightContig($contig);
				});
	return $widgetID;
}

sub light {
	my ($self) = @_;
	my $id = $self->widgetID;
	my $Canvas = $self->GenomeMap->ChromosomeMap;
	$Canvas->addtag("lit", "withtag", $id);
	$Canvas->itemconfigure($id, -fill => $self->color);
	my ($x1, $y1, $x2, $y2) = $Canvas->coords($id); # get the coords of the last one
	my $x = (($x2-$x1)/2) + $x1;  #/
	
	$Canvas->coords($id,$x-20, $y1, $x+20, $y2);  # increase its size
	
}

sub highlight {
	my ($self, $color) = @_;
	my $id = $self->widgetID;
	my $Canvas = $self->GenomeMap->ChromosomeMap;
	$Canvas->addtag("lit", "withtag", $id);
	$Canvas->itemconfigure($id, -fill => $color);
	$self->color($color);  # this is now color of self
	my ($x1, $y1, $x2, $y2) = $Canvas->coords($id); # get the coords of the last one
	my $x = (($x2-$x1)/2) + $x1;  #/
	
	$Canvas->coords($id,$x-20, $y1, $x+20, $y2);  # increase its size
	
}

sub dim {
	my ($self, $unhighlight) = @_;
	# if called with a second parameter then it will not only reduce size,
	# but also set back to default color
	
	my $id = $self->widgetID;
	my $Canvas = $self->GenomeMap->ChromosomeMap;
	$Canvas->dtag($id, "lit");
	my $color;
	if ($unhighlight){$color = $self->def_color}
	else {$color = $self->color}
	$Canvas->itemconfigure($id, -fill => $color);	
	$self->color($self->def_color);
	my ($x1, $y1, $x2, $y2) = $Canvas->coords($id); # get the coords of the last one
	my $x = (($x2-$x1)/2) + $x1;  #/
	$Canvas->coords($id,$x-5, $y1, $x+5, $y2);  # reduce its size
}

sub addtag {
	my ($self, $tag) = @_;
	$self->GenomeMap->ChromosomeMap->addtag($tag, "withtag", $self->widgetID);
	push @{$self->tags}, $tag;
}

sub removetag {
	my ($self, $tag) = @_;
	$self->GenomeMap->ChromosomeMap->dtag($self->widgetID, $tag);
	my @newlist;
	foreach my $oldtag(@{$self->tags}){
		unless ($oldtag eq $tag){push @newlist}
	}
	$self->tags(\@newlist);
}	

1;




