{***************************************************************************
 *
 * Author:     Alberto Pascual Montano (pascual@fis.ucm.es)
 *             http://www.dacya.ucm.es/apascual

 * Complutense University of Madrid (UCM). Madrid, Spain
 * The KEY Institute for Brain-Mind Research, Zurich, Switzerland
 * National Center for Biotechnology (CNB). Madrid, Spain
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 *  All comments concerning this program package may be sent to the
 *  e-mail address 'pascual@fis.ucm.es'
 ***************************************************************************}

unit microarray;

interface

uses
  Windows, Messages, SysUtils, Variants, Classes, Graphics, Controls, Forms,
  Dialogs, Grids,uMatToolsDyn, StdCtrls, Buttons, ComCtrls, ExtCtrls;

type
  str255 = string[255];
   TInternalStringVector=array[1..1] of str255;
   stringvector=^TInternalStringVector;
   TInternalStringMatrix=array[1..1] of stringVector;
   stringMatrix=^TInternalStringMatrix;
   TInternalBoolVector=array[1..1] of boolean;
   boolvector=^TInternalBoolVector;
   TInternalBoolMatrix=array[1..1] of boolvector;
   boolmatrix=^TInternalBoolMatrix;
   TMatRecord = record
     name: string[255];
     evar, min, max: real;
     AIndex, BIndex: intvector;
   end;
   TFactRecord = record
     name:  string[255];
     evar, min, max: real;
     weight: real;
     amin, amax, bmin, bmax: real;
     Avec, Bvec: vector;
     AIndex, BIndex: intvector;
   end;
   TInternalMatRecordVector=array[1..1] of TMatRecord;
   matvector=^TInternalMatRecordVector;
   TInternalFactRecordVector=array[1..1] of TFactRecord;
   factvector=^TInternalFactRecordVector;

  Tmicroform = class(TForm)
    StringGrid1: TStringGrid;
    DrawGrid1: TDrawGrid;
    Label1: TLabel;
    Label2: TLabel;
    CheckBox1: TCheckBox;
    TrackBar1: TTrackBar;
    TrackBar2: TTrackBar;
    Label5: TLabel;
    Label6: TLabel;
    UpLabel: TLabel;
    DownLabel: TLabel;
    ComboBox1: TComboBox;
    evlabel: TLabel;
    CheckBox2: TCheckBox;
    Button1: TButton;
    minLabel: TLabel;
    MaxLabel: TLabel;
    Image1: TImage;
    Button2: TButton;
    SaveDialog1: TSaveDialog;
    minColorLabel: TLabel;
    maxcolorlabel: TLabel;
    Label3: TLabel;
    Label4: TLabel;
    Button3: TButton;
    SaveDialog2: TSaveDialog;
    RowsLabel: TLabel;
    ColumnsLabel: TLabel;
    StatusBar1: TStatusBar;
    procedure DrawGrid1DrawCell(Sender: TObject; ACol, ARow: Integer;
      Rect: TRect; State: TGridDrawState);
    procedure DrawGrid1SelectCell(Sender: TObject; ACol, ARow: Integer;
      var CanSelect: Boolean);
    procedure CheckBox1Click(Sender: TObject);
    procedure FormShow(Sender: TObject);
    procedure TrackBar2Change(Sender: TObject);
    procedure TrackBar1Change(Sender: TObject);
    procedure ComboBox1Change(Sender: TObject);
    procedure FormClose(Sender: TObject; var Action: TCloseAction);
    procedure CheckBox2Click(Sender: TObject);
    procedure Button1Click(Sender: TObject);
    procedure StringGrid1SelectCell(Sender: TObject; ACol, ARow: Integer;
      var CanSelect: Boolean);
    procedure StringGrid1DrawCell(Sender: TObject; ACol, ARow: Integer;
      Rect: TRect; State: TGridDrawState);
    procedure Button2Click(Sender: TObject);
    procedure FormCreate(Sender: TObject);
    procedure Button3Click(Sender: TObject);
  private
    { Private declarations }
    Up, Down: extended;
    maxUp, maxDown: extended;
    slopeUp, slopeDown: extended;
    gridselflag, stringselflag: boolean;
    mat: matrix;
    procedure MouseToCells(yc, xc: integer; var X, Y: Integer);
    procedure MouseLeftButtonUp(var Msg: TMsg; var Handled: Boolean);
  public
    { Public declarations }
    n, p, q: integer;
    rowlabels, varlabels: stringvector;
    numberofrowheaders: integer;
    matvec: matvector;
    factvec: factvector;
    mousedown:boolean;
    dimmatvec, dimfactvec: integer;
    importancemodelstr :string;
    orgmat: pmatrix;
    procedure DrawStringGrid;
    procedure repaintbitmap;
  end;

procedure CreateDestroy(Create:boolean; n:integer; var v:stringvector); overload;
procedure CreateDestroy(Create:boolean; nr, nc:integer; var m:stringmatrix); overload;
procedure CreateDestroy(Create:boolean; n:integer; var v:boolvector); overload;
procedure CreateDestroy(Create:boolean; nr, nc:integer; var m:boolmatrix); overload;
procedure CreateDestroy(Create:boolean; n:integer; var v:matvector); overload;
procedure CreateDestroy(Create:boolean; n:integer; var v:factvector); overload;

function detectFormat(fn:string): integer;

function getEngeneDataDimensions(fn:string;
                   var n,p, numberofrowheaders, numberofvarheaders:integer
                   ): boolean;

function ReadEngeneData(fn:string;
                   var n,p, numberofrowheaders, numberofvarheaders:integer;
                   var x:matrix;
                   var rowlabels, varlabels: stringvector;
                   var mean, min, max: real;
                   var dataneg: boolean): boolean;

function getPureDataDimensions(fn:string;
                   var n,p, numberofrowheaders, numberofvarheaders:integer
                   ): boolean;


function ReadPureData(fn:string;
                    nobjs, nvars:integer;
                    var x:matrix;
                    var mean, min, max: real;
                    var dataneg: boolean):boolean;

function getLabeledDataDimensions(fn:string;
                   var n,p, numberofrowheaders, numberofvarheaders:integer
                   ): boolean;

function ReadLabeledData(fn:string;
                   var n,p, numberofrowheaders, numberofvarheaders:integer;
                   var x:matrix;
                   var rowlabels, varlabels: stringvector;
                   var mean, min, max: real;
                   var dataneg: boolean): boolean;

function TransposeData(var x:matrix;
                       var n,p, numberofrowheaders, numberofvarheaders:integer;
                       var rowlabels, varlabels: stringvector
                       ): boolean;
                       
function normalizeData(var x:matrix; normMethod: integer; var negflag: boolean; var min, max, mean: real): boolean;
function ApplyLogTransformation(var x:matrix): boolean;


var
  microform: Tmicroform;

implementation

uses factors, strutils, math;

{$R *.dfm}
const
   min = 0;

procedure Tmicroform.DrawGrid1DrawCell(Sender: TObject; ACol,
  ARow: Integer; Rect: TRect; State: TGridDrawState);
var
  c: TColor;
  auxE: extended;
  auxUp, auxDown, newrow, newcol,i: integer;
begin
   // Paint the color of the cell, if visible
    if ((DrawGrid1.RowHeights[Arow] = 0) or (DrawGrid1.colwidths[Acol] = 0)) then begin
       exit;
    end;
    if ((ACol >0) and (ARow >0)) then begin
        newrow := Arow;
        newcol := Acol;
        if ((ComboBox1.ItemIndex = 0) {and (factorsform.visible)}) then begin
            for i:= 1 to p do
                if (factvec[factorsform.internalMatrixIndex].BIndex.v[i] = Acol) then begin
                    newcol := i;
                    break;
                end;
            for i:= 1 to n do
                if (factvec[factorsform.internalMatrixIndex].AIndex.v[i] = Arow) then begin
                   newrow := i;
                   break;
                end;
        end;
        if (factorsform.itemmask[newrow] and factorsform.varmask[newcol]) then begin
            auxE := mat.m[Arow,Acol];
            if (auxE > 0) then begin
                if (auxE > Up) then auxE := Up;
                auxUp:= round(slopeUp*(auxE-min));
                c:=rgb(auxUp, 0, 0);
                DrawGrid1.Canvas.Brush.Color := c
            end else begin
                auxE := abs(auxE);
                if (auxE > Down) then auxE := Down;
                auxDown:= round(slopeDown*(auxE-min));
                c:=rgb(0, auxDown, 0);
                DrawGrid1.Canvas.Brush.Color := c
            end;
        end else begin
            c:=rgb(100, 100, 100);
            DrawGrid1.Canvas.Brush.Color := c
        end;
        DrawGrid1.Canvas.FillRect(Rect);
    end;
end;

procedure Tmicroform.DrawGrid1SelectCell(Sender: TObject; ACol,
  ARow: Integer; var CanSelect: Boolean);
begin
    // Focus the cell in String grid
   if (not stringselflag) then begin
        gridselflag := true;
        StringGrid1.col := ACol;
        StringGrid1.row := ARow;
   end else
        stringselflag := false;
end;

procedure Tmicroform.CheckBox1Click(Sender: TObject);
begin
   if (checkbox1.checked) then begin
      DrawGrid1.GridLineWidth := 1;
   end else begin
      DrawGrid1.GridLineWidth := 0;
   end;
   DrawGrid1.LeftCol := 0;
   DrawGrid1.TopRow := 0;
end;

procedure Tmicroform.FormShow(Sender: TObject);
var
  i,j,k, he, wi: integer;

begin
  mousedown:= false;
  microform.StringGrid1.ColCount := p+1;
  microform.StringGrid1.RowCount := n+1;
  microform.DrawGrid1.ColCount := p+1;
  microform.DrawGrid1.RowCount := n+1;
  microform.DrawGrid1.RowHeights[0] := 0;
  microform.DrawGrid1.colwidths[0] := 0;
  wi := 10;
  he := 10;
  DrawGrid1.scrollBars := ssBoth;
  Image1.Height := DrawGrid1.Height;
  Image1.Width := DrawGrid1.Width;
  Image1.top := DrawGrid1.top;
  Image1.left := DrawGrid1.left;

  createdestroy(true,n,factorsform.itemmask);
  createdestroy(true,p,factorsform.varmask);

  for i:= 1 to dimmatvec do begin
    ComboBox1.AddItem(matvec[i].name, Sender);
  end;
  ComboBox1.ItemIndex := 0;

  Label1.Caption := 'Expression data : ' + ComboBox1.Items[ComboBox1.ItemIndex];
  microform.StringGrid1.Cells[0, 0] := '';
  for i:=1 to numberofrowheaders-1 do
     StringGrid1.Cells[0, 0] := StringGrid1.Cells[0, 0] + varlabels[i] + #9;
  if (numberofrowheaders >0) then
      StringGrid1.Cells[0, 0] := StringGrid1.Cells[0, 0] + varlabels[numberofrowheaders];

  k:=ComboBox1.ItemIndex+1;
  for i:=1 to n do begin
     microform.StringGrid1.Cells[0, i] := rowlabels[matvec[k].Aindex.v[i]];
     microform.DrawGrid1.RowHeights[i] := he;
     factorsform.itemmask[i] := true;
  end;
  for j:=1 to p do begin
     microform.StringGrid1.Cells[j, 0] := varlabels[matvec[k].Bindex.v[j]+numberofrowheaders];
     microform.DrawGrid1.colwidths[j] := wi;
     factorsform.varmask[j] := true;
  end;

  create1(n,p,mat);
  matvec[k].min:= 656000; matvec[k].max := -656000;
  for i:=1 to n do begin
    for j:=1 to p do begin
      mat.m[i,j] := orgmat[matvec[k].Aindex.v[i],matvec[k].Bindex.v[j]];
      if (matvec[k].min > mat.m[i,j]) then matvec[k].min := mat.m[i,j];
      if (matvec[k].max < mat.m[i,j]) then matvec[k].max := mat.m[i,j];
      microform.StringGrid1.Cells[j, i] := floattostr(mat.m[i,j]);
    end;
  end;
  if (ComboBox1.ItemIndex = 0) then
    evLabel.Caption := '';

   maxLabel.Caption :=  'Maximum value: ' + FloatToStrF(matvec[k].max, ffFixed, 4, 1);
   minLabel.Caption :=  'Minimum value: ' + FloatToStrF(matvec[k].min, ffFixed, 4, 1);
   RowsLabel.Caption :=  'Rows (items): ' + IntToStr(n);
   ColumnsLabel.Caption :=  'Columns (variables): ' + IntToStr(p);

   StringGrid1.Hint := 'Item label: ' + StringGrid1.Cells[0, 1] + #13 + 'Var Label: ' +  StringGrid1.Cells[1, 0];
   DrawGrid1.Hint := StringGrid1.Hint;

   if (matvec[k].max <= 0) then begin
      maxUp:= 0;
      trackbar2.position := 0;
      trackbar2.Enabled := false;
      maxcolorLabel.Caption :=  '0.0';
   end else begin
      MaxUp := abs(matvec[k].max);
      trackbar2.Enabled := true;
      maxcolorLabel.Caption :=  FloatToStrF(matvec[k].max, ffFixed, 4, 2);
      Up := MaxUp/2;
      if (MaxUp <> 0) then
          trackbar2.position := round(Up*trackbar2.Max/MaxUp)
      else
          trackbar2.position := trackbar2.min;
   end;
   if (matvec[k].min >= 0) then begin
      maxDown := 0;
      trackbar1.position := trackbar1.max;
      trackbar1.Enabled := false;
      mincolorLabel.Caption :=  '0.0';
   end else begin
      MaxDown := abs(matvec[k].min);
      Down := MaxDown/2;
      trackbar1.Enabled := true;
      mincolorLabel.Caption :=  FloatToStrF(matvec[k].min, ffFixed, 4, 2);
      if (MaxDown <> 0) then
          trackbar1.position := trackbar1.Max - round(Down*trackbar1.Max/MaxDown)
      else
          trackbar1.position := trackbar1.max;
   end;

   if (Up <> min) then
      slopeUp := (255)/(Up-min)
   else slopeUp := 0;
   if (Down <> min) then
      slopeDown := (255)/(Down-min)
   else slopeDown := 0;

   CheckBox2.checked := false;
   drawgrid1.Repaint;


   StringGrid1.Options := StringGrid1.Options + [goDrawFocusSelected] + [goRowSizing] + [goColSizing];
   factorsform.factvec  := @microform.factvec;
   factorsform.p  := @microform.p;
   factorsform.n  := @microform.n;
   factorsform.q  := @microform.q;
   create1(n,p,factorsform.mat);
   factorsform.rowlabels  := @microform.rowlabels;
   factorsform.varlabels  := @microform.varlabels;
   factorsform.numberofrowheaders := @microform.numberofrowheaders;
   factorsform.showfirsttime := true;
   microform.CheckBox2.checked := false;
   factorsform.show;

end;

procedure Tmicroform.TrackBar2Change(Sender: TObject);
begin
   // Positive (up regulation)
   if (trackbar2.Max <> 0) then
      Up := trackbar2.position*MaxUp/trackbar2.Max
   else
      Up := 0;   
   UpLabel.caption := FloatToStrF(Up, ffFixed, 4, 2);
   if (not mousedown) then begin
       if (Up <> min) then
          slopeUp := (255)/(Up-min)
       else slopeUp := 0;
      if CheckBox2.Checked then
          repaintbitmap;
       DrawGrid1.repaint;
   end;
end;

procedure Tmicroform.TrackBar1Change(Sender: TObject);
begin
  // Negative (down regulation)
   if (trackbar1.Max <> 0) then
      Down := (trackbar1.Max+1 - trackbar1.position)*MaxDown/(trackbar1.Max+1)
   else
      Down := 0;   
   DownLabel.caption := '-' + FloatToStrF(Down, ffFixed, 4, 2);
   if (not mousedown) then begin
       if (Down <> min) then
          slopeDown := (255)/(Down-min)
       else slopeDown := 0;
      if CheckBox2.Checked then
         repaintbitmap;
       DrawGrid1.repaint;
   end;
end;

procedure Tmicroform.ComboBox1Change(Sender: TObject);
var
  Save_Cursor:TCursor;
  i,j, k: integer;
begin
  Save_Cursor := Screen.Cursor;
  Screen.Cursor := crHourGlass;    { Show hourglass cursor }
  k:=ComboBox1.ItemIndex+1;
  // Change in comboBox
  Label1.Caption := 'Expression data : ' + ComboBox1.Items[ComboBox1.ItemIndex];
  if (ComboBox1.ItemIndex <> 0) then
    evLabel.Caption := importancemodelstr + FloatToStrF(matvec[k].evar, ffFixed, 4, 2) +'%'
  else
    evLabel.Caption := '';

  /// Now, load the data and properly sort it
  matvec[k].min:= 656000; matvec[k].max := -656000;
  for i:=1 to n do
      for j:=1 to p do begin
          mat.m[i,j] := orgmat[matvec[k].Aindex.v[i],matvec[k].Bindex.v[j]];
          if (matvec[k].min > mat.m[i,j]) then matvec[k].min := mat.m[i,j];
          if (matvec[k].max < mat.m[i,j]) then matvec[k].max := mat.m[i,j];
      end;

  // Change also the combo in the factors visualization form
  if (Sender <> factorsform) then begin
    if ((microform.ComboBox1.ItemIndex <> 0) and (microform.ComboBox1.ItemIndex <> factorsform.internalMatrixIndex)) then begin
          //if factorsform.visible then begin
          factorsform.ComboBox1.ItemIndex := factorsform.fromMComboToFCombo(ComboBox1.ItemIndex);
          factorsform.ComboBox1Change(microform);
      //end;
    end;
  end;
  if CheckBox2.Checked then
     repaintbitmap;
  DrawStringGrid;
  DrawGrid1.repaint;
  Screen.Cursor := Save_Cursor;  { Always restore to normal }
end;

procedure Tmicroform.DrawStringGrid;
var
  i, j, k: integer;
begin
  k:=ComboBox1.ItemIndex+1;
  for i:=1 to n do begin
     microform.StringGrid1.Cells[0, i] := rowlabels[matvec[k].Aindex.v[i]];
  end;
  for j:=1 to p do begin
     microform.StringGrid1.Cells[j, 0] := varlabels[matvec[k].Bindex.v[j] + numberofrowheaders];
  end;
  for i:=1 to n do begin
    for j:=1 to p do begin
      microform.StringGrid1.Cells[j, i] := floattostr(mat.m[i,j]);
    end;
  end;
end;

procedure Tmicroform.FormClose(Sender: TObject; var Action: TCloseAction);
var
  i: integer;
begin
  // Closes the factors visualizer form
  createdestroy(false,n,factorsform.itemmask);
  createdestroy(false,p,factorsform.varmask);
  destroy1(factorsform.mat);
  destroy1(mat);
  factorsform.close;
  // Release memory
  for i:= 0 to ComboBox1.items.count do begin
    ComboBox1.items.delete(i);
  end;
  ComboBox1.Clear;
  // Release memory in factors form
  for i:= 0 to factorsform.ComboBox1.items.count do begin
    factorsform.ComboBox1.items.delete(i);
  end;
  factorsform.ComboBox1.Clear;
 
end;

procedure Tmicroform.repaintbitmap;
type
     TpIntArray=array[1..maxword] of tcolor;
var
  wi, he, i, j, restwi, resthe, nn, pp, ap, an, counter, ir, jr: integer;
  auxup, auxdown: integer;
  ah, bh, aw, bw, auxe: extended;
  c: TColor;
  ttt:^TpIntArray;
  ii, jj, kk, counteri, counterj:integer;
  MyBitmap: TBitmap;
  iv, jv, icv, jcv : intvector;

begin
      an := 0;
      ap := 0;
      create1(n,iv);
      create1(p,jv);
      create1(n,icv);
      create1(p,jcv);

      counter := 0;
      for i:=1 to n do begin
        if ((ComboBox1.ItemIndex = 0){ and (factorsform.visible)}) then begin
            for j:= 1 to n do
                if (factvec[factorsform.internalMatrixIndex].AIndex.v[j] = i) then begin
                       kk := j;
                       break;
                end;
        end else kk:= i;
        icv.v[i] := kk;
        if (not factorsform.showdelcells) then begin
          if not factorsform.itemmask[icv.v[i]] then an := an + 1
          else begin
             counter := counter + 1;
             iv.v[counter] := i;
          end;
        end else iv.v[i] := i;
      end;  
      counter := 0;
      for j:=1 to p do begin
        if ((ComboBox1.ItemIndex = 0){ and (factorsform.visible)}) then begin
            for i:= 1 to p do
                if (factvec[factorsform.internalMatrixIndex].BIndex.v[i] = j) then begin
                       kk := i;
                       break;
                end;
        end else kk:= j;
        jcv.v[j] := kk;
        if (not factorsform.showdelcells) then begin
          if not factorsform.varmask[jcv.v[j]] then ap := ap + 1
          else begin
             counter := counter + 1;
             jv.v[counter] := j;
          end;
        end else jv.v[j] := j;
      end;

      if (not CheckBox2.Checked) then begin
          an := 0;
          ap := 0;
      end;

      pp := p - ap;
      nn := n - an;

      MyBitmap := TBitmap.Create;
      MyBitmap.Width := DrawGrid1.ClientWidth;
      MyBitmap.Height := DrawGrid1.ClientHeight;
      MyBitmap.PixelFormat:=pf32bit;
      Image1.Picture.Graphic := MyBitmap;
      MyBitmap.Free;

      wi :=  DrawGrid1.ClientWidth div pp;
      restwi := DrawGrid1.ClientWidth mod pp;
      he :=  DrawGrid1.ClientHeight div nn;
      resthe := DrawGrid1.ClientHeight mod nn;

      if (he = 0) then begin
         // Do downsampling by rows
        bh := (nn-1)/(DrawGrid1.ClientHeight-1);
        ah := 1-bh;
      end;
      if (wi = 0) then begin
         // Do downsampling by columns
        bw := (pp-1)/(DrawGrid1.ClientWidth-1);
        aw := 1-bw;
      end;

      // Now paint Bitmap
      counteri:= 0; i:=1;
      for ii:= 1 to DrawGrid1.ClientHeight-1 do begin
        ttt:=image1.Picture.Bitmap.scanline[ii-1];
        if (he = 0) then begin
            i := round(ah + bh*ii);
        end else begin
            counteri := counteri +1;
            if (i <= resthe) then  begin
                if (counteri = (he+1)) then begin
                    i:=i+1;
                    counteri := 0;
                end;
            end else begin
                if (counteri = he) then begin
                    i:=i+1;
                    counteri := 0;
                end;
            end;
        end;
        ir := iv.v[i];
        j := 1; counterj := 0;
        for jj:=1 to DrawGrid1.ClientWidth-1 do begin
            if (wi = 0) then begin
                j := round(aw + bw*jj);
            end else begin
               counterj := counterj +1;
                if (j <= restwi) then  begin
                    if (counterj = (wi+1)) then begin
                        j:=j+1;
                        counterj := 0;
                    end;
                end else begin
                    if (counterj = wi) then begin
                        j:=j+1;
                        counterj := 0;
                    end;
                end;
            end;
            jr := jv.v[j];
            if (factorsform.itemmask[icv.v[ir]] and factorsform.varmask[jcv.v[jr]]) then begin
                auxE := mat.m[ir,jr];
                if (auxE > 0) then begin
                    if (auxE > Up) then auxE := Up;
                    auxUp:= round(slopeUp*(auxE-min));
                    c:=rgb(0, 0, auxUp);
                end else begin
                    auxE := abs(auxE);
                    if (auxE > Down) then auxE := Down;
                    auxDown:= round(slopeDown*(auxE-min));
                    c:=rgb(0, auxDown, 0);
                end;
                ttt^[jj]:=c;
            end else begin
                if (factorsform.showdelcells) then begin
                    c:=rgb(100, 100, 100);
                    ttt^[jj]:=c;
                end;
            end;
        end;
    end;
    destroy1(iv);
    destroy1(jv);
    destroy1(icv);
    destroy1(jcv);

end;

procedure Tmicroform.CheckBox2Click(Sender: TObject);
type
     TpIntArray=array[1..maxword] of tcolor;
var
  wi, he, i, j, restwi, resthe, nn, pp, ap, an, counter, ir, jr: integer;
  auxup, auxdown: integer;
  ah, bh, aw, bw, auxe: extended;
  c: TColor;
  ttt:^TpIntArray;
  ii, jj, kk, counteri, counterj:integer;
  MyBitmap: TBitmap;
  iv, jv, icv, jcv : intvector;

begin

  // Fit array to window. Uses downsampling if necessary

  an := 0;
  ap := 0;
  create1(n,iv);
  create1(p,jv);
  create1(n,icv);
  create1(p,jcv);
  counter := 0;
  for i:=1 to n do begin
    if ((ComboBox1.ItemIndex = 0)) then begin
        for j:= 1 to n do
            if (factvec[factorsform.internalMatrixIndex].AIndex.v[j] = i) then begin
                   kk := j;
                   break;
            end;
    end else kk:= i;
    icv.v[i] := kk;
    if (not factorsform.showdelcells) then begin
      if not factorsform.itemmask[icv.v[i]] then an := an + 1
      else begin
         counter := counter + 1;
         iv.v[counter] := i;
      end;
    end else iv.v[i] := i;
  end;
  counter := 0;
  for j:=1 to p do begin
    if ((ComboBox1.ItemIndex = 0)) then begin
        for i:= 1 to p do
            if (factvec[factorsform.internalMatrixIndex].BIndex.v[i] = j) then begin
                   kk := i;
                   break;
            end;
    end else kk:= j;
    jcv.v[j] := kk;
    if (not factorsform.showdelcells) then begin
      if not factorsform.varmask[jcv.v[j]] then ap := ap + 1
      else begin
         counter := counter + 1;
         jv.v[counter] := j;
      end;
    end else jv.v[j] := j;
  end;

  if (not CheckBox2.Checked) then begin
      an := 0;
      ap := 0;
  end;

  pp := p - ap;
  nn := n - an;

  if (checkbox2.checked) then begin
    DrawGrid1.Visible := false;
    checkbox1.checked := false;
    checkbox1.enabled := false;
    DrawGrid1.scrollBars := ssNone;
  end;
   DrawGrid1.LeftCol := 0;
   DrawGrid1.TopRow := 0;

  if (not checkbox2.checked) then begin
      he := 10; wi := 10;
      for i:=1 to nn do
          DrawGrid1.RowHeights[i] := he;
      for j:=1 to pp do
          DrawGrid1.colwidths[j] := wi;
      // Shows control
      Image1.Visible := false;
      DrawGrid1.Visible := true;
  end else begin

      MyBitmap := TBitmap.Create;
      MyBitmap.Width := DrawGrid1.ClientWidth;
      MyBitmap.Height := DrawGrid1.ClientHeight;
      MyBitmap.PixelFormat:=pf32bit;
      Image1.Picture.Graphic := MyBitmap;
      MyBitmap.Free;

      wi :=  DrawGrid1.ClientWidth div pp;
      restwi := DrawGrid1.ClientWidth mod pp;
      he :=  DrawGrid1.ClientHeight div nn;
      resthe := DrawGrid1.ClientHeight mod nn;

      if (he = 0) then begin
         // Do downsampling by rows
        bh := (nn-1)/(DrawGrid1.ClientHeight-1);
        ah := 1-bh;
      end else begin
          for i:=1 to nn do begin
              if (i <= resthe) then
                  DrawGrid1.RowHeights[i] := he+1
              else
                  DrawGrid1.RowHeights[i] := he;
          end;
      end;
      if (wi = 0) then begin
         // Do downsampling by columns
        bw := (pp-1)/(DrawGrid1.ClientWidth-1);
        aw := 1-bw;
      end else begin
          for j:=1 to pp do begin
              if (j <= restwi) then
                  DrawGrid1.colwidths[j] := wi + 1
              else
                  DrawGrid1.colwidths[j] := wi;
          end;
      end;

      // Now paint Bitmap
    counteri:= 0; i:=1;
    for ii:= 1 to DrawGrid1.ClientHeight-1 do begin
        ttt:=image1.Picture.Bitmap.scanline[ii-1];
        if (he = 0) then begin
            i := round(ah + bh*ii);
        end else begin
            counteri := counteri +1;
            if (i <= resthe) then  begin
                if (counteri = (he+1)) then begin
                    i:=i+1;
                    counteri := 0;
                end;
            end else begin
                if (counteri = he) then begin
                    i:=i+1;
                    counteri := 0;
                end;
            end;
        end;
        ir := iv.v[i];

        j := 1; counterj := 0;
        for jj:=1 to DrawGrid1.ClientWidth-1 do begin
            if (wi = 0) then begin
                j := round(aw + bw*jj);
            end else begin
               counterj := counterj +1;
                if (j <= restwi) then  begin
                    if (counterj = (wi+1)) then begin
                        j:=j+1;
                        counterj := 0;
                    end;
                end else begin
                    if (counterj = wi) then begin
                        j:=j+1;
                        counterj := 0;
                    end;
                end;
            end;
            jr := jv.v[j];

            if (factorsform.itemmask[icv.v[ir]] and factorsform.varmask[jcv.v[jr]]) then begin
                auxE := mat.m[ir,jr];
                if (auxE > 0) then begin
                    if (auxE > Up) then auxE := Up;
                    auxUp:= round(slopeUp*(auxE-min));
                    c:=rgb(0, 0, auxUp);
                end else begin
                    auxE := abs(auxE);
                    if (auxE > Down) then auxE := Down;
                    auxDown:= round(slopeDown*(auxE-min));
                    c:=rgb(0, auxDown, 0);
                end;
                ttt^[jj]:=c;
            end else begin
                if (factorsform.showdelcells) then begin
                    c:=rgb(100, 100, 100);
                    ttt^[jj]:=c;
                end;
            end;
        end;
    end;
      // Hide control and show bitmap
      Image1.Visible := true;
  end;
  destroy1(iv);
  destroy1(jv);
  destroy1(icv);
  destroy1(jcv);

  if (not checkbox2.checked) then begin
    checkbox1.enabled := true;
    checkbox1.checked := true;
    DrawGrid1.scrollBars := ssBoth;
  end;
  
end;

procedure Tmicroform.Button1Click(Sender: TObject);
begin
   factorsform.show;
end;

procedure Tmicroform.StringGrid1SelectCell(Sender: TObject; ACol,
  ARow: Integer; var CanSelect: Boolean);
var
   i, newArow, newAcol: integer;  
begin
    // Focus the cell in String grid
   if (not gridselflag) then begin
        stringselflag := true;
        DrawGrid1.col := ACol;
        DrawGrid1.row := ARow;
   end else
        gridselflag := false;

   // Change also the cell in visualization form

    if (not factorsform.factorselflag) then begin
        factorsform.microselflag := true;
        if (ComboBox1.ItemIndex >0) then begin
            if (factorsform.internalMatrixSubIndex = 0) then begin
                // One to one correspondence
                factorsform.StringGrid1.Col := Acol;
                factorsform.StringGrid1.Row := Arow;
            end else if (factorsform.internalMatrixSubIndex = 1) then begin
                // row correspondence
                factorsform.StringGrid1.Col := 1;
                factorsform.StringGrid1.Row := Arow;
            end else begin
                // column correspondence
                factorsform.StringGrid1.Col := Acol;
                factorsform.StringGrid1.Row := 1;
            end;
        end else begin
            if (factorsform.internalMatrixSubIndex = 0) then begin
                // One to one correspondence
                for i:= 1 to p do
                  if (factvec[factorsform.internalMatrixIndex].BIndex.v[i] = Acol) then newAcol := i;
                for i:= 1 to n do
                  if (factvec[factorsform.internalMatrixIndex].AIndex.v[i] = Arow) then newArow := i;
            end else if (factorsform.internalMatrixSubIndex = 1) then begin
                // row correspondence
                newAcol := 1;
                for i:= 1 to n do
                  if (factvec[factorsform.internalMatrixIndex].AIndex.v[i] = Arow) then newArow := i;
            end else begin
                // column correspondence
                for i:= 1 to p do
                  if (factvec[factorsform.internalMatrixIndex].BIndex.v[i] = Acol) then newAcol := i;
                newArow := 1;
            end;
            factorsform.StringGrid1.Col := newAcol;
            factorsform.StringGrid1.Row := newArow;
        end;
    end else
        factorsform.factorselflag := false;
   // Show column and row name in the hint label
   StringGrid1.Hint := 'Item label: ' + StringGrid1.Cells[0, Arow] + #13 + 'Var Label: ' +  StringGrid1.Cells[Acol, 0];
   DrawGrid1.Hint := StringGrid1.Hint;              
end;

procedure Tmicroform.StringGrid1DrawCell(Sender: TObject; ACol,
  ARow: Integer; Rect: TRect; State: TGridDrawState);
var
   c: TColor;
   strtemp: string;
   i,newcol, newrow:integer;
const
  SelectedColor = Clblue;
begin
  StringGrid1.Options := StringGrid1.Options + [goDrawFocusSelected] + [goRowSizing] + [goColSizing];
  // Change the font color according to mask
  if ((Acol = 0) or (Arow = 0)) then exit;
  if ((gdFocused in state) or (gdSelected in state)) then begin
      StringGrid1.Canvas.Brush.Color := SelectedColor;
      StringGrid1.Canvas.FillRect(Rect);
      StringGrid1.Canvas.TextRect(Rect, Rect.Left + 2, Rect.Top + 2, StringGrid1.Cells[aCol, aRow]);
      exit;
  end;
  strTemp := StringGrid1.Cells[ACol,ARow];
  newrow:= Arow; newcol:= Acol;
  if ((ComboBox1.ItemIndex = 0){ and (factorsform.visible)}) then begin
      for i:= 1 to p do
          if (factvec[factorsform.internalMatrixIndex].BIndex.v[i] = Acol) then begin
              newcol := i;
              break;
          end;
      for i:= 1 to n do
          if (factvec[factorsform.internalMatrixIndex].AIndex.v[i] = Arow) then begin
             newrow := i;
             break;
          end;   
  end;
  if ((not factorsform.itemmask[newrow]) or (not factorsform.varmask[newcol])) then
      c:=rgb(100, 100, 100)
  else begin
      c:=rgb(255, 255, 255);
      StringGrid1.Font.Color := clDefault;
  end;
  StringGrid1.Canvas.Brush.Color := c;
  StringGrid1.Canvas.FillRect(Rect);
  DrawText(StringGrid1.Canvas.Handle,PChar(strTemp),-1,Rect,0);
end;

procedure Tmicroform.MouseToCells(yc, xc: integer; var X, Y: Integer);
var
  counteri, counterj, ii, jj: integer;
  wi, he, restwi, resthe, nn, pp, ap, an, counter, i, j: integer;
  ah, bh, aw, bw: extended;
  iv, jv : intvector;
  
begin
      an := 0;
      ap := 0;
      create1(n,iv);
      create1(p,jv);
      counter := 0;
      for i:=1 to n do
        if (not factorsform.showdelcells) then begin
          if not factorsform.itemmask[i] then an := an + 1
          else begin
             counter := counter + 1;
             iv.v[counter] := i;
          end;
        end else iv.v[i] := i;
      counter := 0;
      for j:=1 to p do
        if (not factorsform.showdelcells) then begin
          if not factorsform.varmask[j] then ap := ap + 1
          else begin
             counter := counter + 1;
             jv.v[counter] := j;
          end;
        end else jv.v[j] := j;

      pp := p - ap;
      nn := n - an;

      wi :=  DrawGrid1.ClientWidth div pp;
      restwi := DrawGrid1.ClientWidth mod pp;
      he :=  DrawGrid1.ClientHeight div nn;
      resthe := DrawGrid1.ClientHeight mod nn;

      if (he = 0) then begin
         // Do downsampling by rows
        bh := (nn-1)/(DrawGrid1.ClientHeight-1);
        ah := 1-bh;
      end;
      if (wi = 0) then begin
         // Do downsampling by columns
        bw := (pp-1)/(DrawGrid1.ClientWidth-1);
        aw := 1-bw;
      end;

      if (he = 0) then 
          x := round(ah + bh*yc)
      else begin
          counteri:= 0; x:=1;
          for ii:= 1 to yc do begin
                counteri := counteri +1;
                if (x <= resthe) then  begin
                    if (counteri = (he+1)) then begin
                        x:=x+1;
                        counteri := 0;
                    end;
                end else begin
                    if (counteri = he) then begin
                        x:=x+1;
                        counteri := 0;
                    end;
                end;
          end;      
      end;
      x := iv.v[x];

      if (wi = 0) then
          y := round(aw + bw*xc)
      else begin
        y := 1; counterj := 0;
        for jj:=1 to xc do begin
            counterj := counterj +1;
            if (y <= restwi) then  begin
                if (counterj = (wi+1)) then begin
                    y:=y+1;
                    counterj := 0;
                end;
            end else begin
                if (counterj = wi) then begin
                    y:=y+1;
                    counterj := 0;
                end;
            end;
        end;
      end;
      y := jv.v[y];
      if (x <= 0) then x:=1;
      if (y <= 0) then y:=1;
      if (x > n) then x:=n;
      if (y > p) then y:=p;

end;

procedure Tmicroform.Button2Click(Sender: TObject);
var
  S: string;
  FileExt :string[4];
  Save_Cursor:TCursor;
begin
  // Save bitmap
  if savedialog1.Execute then begin
    S := savedialog1.FileName;
    FileExt := AnsiUpperCase(ExtractFileExt(S));
    if (FileExt <> '.BMP') then S := S + '.bmp';
    if FileExists(S) then
      if MessageDlg(S+' Exists. Overwrite?',mtConfirmation,[mbYes, mbNo],0) = mrNo then exit;
    Save_Cursor := Screen.Cursor;
    Screen.Cursor := crHourGlass;    { Show hourglass cursor }
    repaintbitmap;
    Image1.Picture.Bitmap.SavetoFile(S);
    Screen.Cursor := Save_Cursor;  { Always restore to normal }
  end;
end;

procedure Tmicroform.FormCreate(Sender: TObject);
begin
  Application.OnMessage := MouseLeftButtonUp;
end;

procedure Tmicroform.MouseLeftButtonUp(var Msg: TMsg; var Handled: Boolean);
begin
  handled:=false;
  if msg.message= WM_LButtonUp then begin
      mousedown:= false;
  end else if msg.message= WM_LButtonDown then begin
      mousedown:= true;
  end;
end;

procedure Tmicroform.Button3Click(Sender: TObject);
var
  S: string;
  FileExt :string[4];
  txt:textfile; i,j, k, newrow, newcol:integer;
  Save_Cursor:TCursor;
  flag: boolean;

begin
  if savedialog2.Execute then begin
    S := savedialog2.FileName;
    FileExt := AnsiUpperCase(ExtractFileExt(S));
    if (FileExt <> '.TXT') then S := S + '.txt';
    if FileExists(S) then
      if MessageDlg(S+' Exists. Overwrite?',mtConfirmation,[mbYes, mbNo],0) = mrNo then exit;

    k := 0;
    for i:=1 to n do
      if factorsform.itemmask[i] then k:= k+1;
    if (k=0) then begin
        showmessage('The matrix has no active items. File can not be saved.');
        exit;
    end;
    k := 0;
    for j:=1 to p do
      if factorsform.varmask[j] then k:= k+1;
    if (k=0) then begin
        showmessage('The matrix has no active variables. File can not be saved.');
        exit;
    end;

    // Now save the file
    Save_Cursor := Screen.Cursor;
    Screen.Cursor := crHourGlass;    { Show hourglass cursor }
    assignfile(txt,S);
    filemode:=1;
    rewrite(txt);
    for i:= 0 to StringGrid1.RowCount-1 do begin
        newrow := i;
        if ((ComboBox1.ItemIndex = 0) and (i <> 0)) then begin
            for k:= 1 to n do
                if (factvec[factorsform.internalMatrixIndex].AIndex.v[k] = i) then begin
                   newrow := k;
                   break;
                end;
        end;
        if ((i <> 0) and (not factorsform.itemmask[newrow])) then continue;
        flag:= true;
        for j:= 0 to StringGrid1.ColCount-1 do begin
           newcol := j;
           if ((ComboBox1.ItemIndex = 0) and (j <> 0)) then begin
                for k:= 1 to p do
                    if (factvec[factorsform.internalMatrixIndex].BIndex.v[k] = j) then begin
                        newcol := k;
                        break;
                    end;
           end;
           if ((j <> 0) and (not factorsform.varmask[newcol])) then continue;
            if ((i=0) and (j=0) and (numberofrowheaders = 0)) then
              write(txt,'ItemsHeader',#9)
            else begin
                 if (not flag) then
                   write(txt,#9, StringGrid1.Cells[j,i])
                 else
                   write(txt, StringGrid1.Cells[j,i]);
                 flag := false;
            end;
        end;
        writeln(txt);
    end;
    closefile(txt);
    Screen.Cursor := Save_Cursor;  { Always restore to normal }
  end;
end;

function detectFormat(fn:string): integer;
var txt:textfile; 
    S: string;
    size, code: integer;
    aux: extended;
    ch: char;

begin
    detectFormat := -1; // not supported format
    assignfile(txt,fn);
    filemode:=0;
    reset(txt);
    assignfile(txt, fn);
    reset(txt);
    size := FileSize(txt);
    if (size = 0) then begin
      showmessage('The input file is empty');
      exit;
    end;

    while not(eof(txt)) do begin
        // Read one line
            S := '';
            ch := ' ';
            repeat
              S := S + ch;
              Read(txt, ch);
            until ((ch = #9) or (ch = #13)  or (ch = #10) or (ch = #26));
            if (ch = #26) then break; //unexpected end of file
            if (S = ' ') then begin
                detectFormat := 2; // Engene format
            end else begin
                val(S, aux, code);
                if (code <> 0) then
                    detectFormat := 1 // Columns contains header
                else
                    detectFormat := 0; // Pure data file
            end;
            break;
    end;
   closefile(txt);
end;

function getEngeneDataDimensions(fn:string;
                   var n,p, numberofrowheaders, numberofvarheaders:integer
                   ): boolean;
var txt:textfile; i:integer;
    S, S2: string;
    index, empty, size,
    numberofvars, addon: integer;
    ch: char;
    gofaster: boolean;

begin
    getEngeneDataDimensions:= true;
    assignfile(txt,fn);
    filemode:=0;
    reset(txt);
    //Read the number of rows and colums in the matrix
    assignfile(txt, fn);
    reset(txt);
    size := FileSize(txt);
    if (size = 0) then begin
      getEngeneDataDimensions:= false;
      exit;
    end;

    i:=0; empty:=0; index:=0; numberofvarheaders := 0; addon := 1;
    numberofvars := 0; numberofrowheaders := 0;
    gofaster := false;

    while not(eof(txt)) do begin
        // Read one line
        if (not gofaster) then begin
            i:=i+1;
            S := '';
            ch := ' ';
            repeat
              S := S + ch;
              Read(txt, ch);
            until ((ch = #9) or (ch = #13)  or (ch = #10) or (ch = #26));
            if (ch = #26) then break; //unexpected end of file
            if (S = ' ') then begin
                empty := empty+addon;
            end else
                addon := 0;

            if ((ch = #13) or (ch = #10) ) then begin
               if (ch <> #10) then readln(txt, S2);
               if (index = 0) then begin
                  numberofvars := i - empty - 1;
                  numberofrowheaders := empty;
               end;
               if (empty > 0)  then numberofvarheaders := numberofvarheaders + 1
               else gofaster := true;
               empty := 0;
               addon := 1;
               i:=0;
               index := index+1;
            end;
        end else begin
            readln(txt, S2);
            if (S2 <> '') then index := index+1;
        end;

    end;

   n := index - numberofvarheaders - 1;
   p := numberofvars;
   closefile(txt);
end;


function ReadEngeneData(fn:string;
                   var n,p, numberofrowheaders, numberofvarheaders:integer;
                   var x:matrix;
                   var rowlabels, varlabels: stringvector;
                   var mean, min, max: real;
                   var dataneg: boolean): boolean;
var txt:textfile; i,j:integer;
    S, S2: string;
    index, code: integer;
    ch: char;
    tmplabels: stringvector;
    aux: extended;
begin
   ReadEngeneData:=true;
   assignfile(txt,fn);
   filemode:=0;
   reset(txt);
   mean:=0; min:= 656000; max := -656000;

   createdestroy(true,p+numberofrowheaders+1,tmplabels);

  //Read the headers and data points in the matrix
    reset(txt);
    i:=0; index:=0;
    while not(eof(txt)) do begin
        // Read one line
        i:=i+1;
        S := '';
        ch := ' ';
        repeat
          S := S + ch;
          Read(txt, ch);
        until ((ch = #9) or (ch = #13)  or (ch = #10) or (ch = #26));
        if (ch = #26) then break; //unexpected end of file
        tmplabels[i] := leftstr(S, 255);

        // Now check whose line is this

        if ((ch = #13) or (ch = #10)) then begin
           if (ch <> #10) then readln(txt, S2);
           index := index+1;
           if (index <= numberofvarheaders) then begin
               for j:= 1 to p do begin
                  if (index = 1) then
                    varlabels[numberofrowheaders+j] := tmplabels[numberofrowheaders+j+1]
                  else
                    varlabels[numberofrowheaders+j] := varlabels[numberofrowheaders+j] + '-' + tmplabels[numberofrowheaders+j+1];
               end;   
           end else if (index = (numberofvarheaders + 1)) then begin
               for j:= 1 to numberofrowheaders do
                    varlabels[j] := tmplabels[j]
           end else if (index > (numberofvarheaders + 1)) then begin  // now read the data
               for j:= 1 to p + numberofrowheaders + 1 do begin
                  if (j <= numberofrowheaders) then begin
                    if (j = 1) then
                      rowlabels[index-numberofvarheaders-1] := tmplabels[j]
                    else
                      rowlabels[index-numberofvarheaders-1] := rowlabels[index-numberofvarheaders-1] + #9 + tmplabels[j];
                  end else if (j > numberofrowheaders+1) then begin
                    // Check if it is OK
                    val(tmplabels[j], aux, code);
                    if (code <> 0) then begin
                        ReadEngeneData := false;
                        createdestroy(false,p+numberofrowheaders+1,tmplabels);
                        exit;
                    end;
                    x.m[index-numberofvarheaders-1, j-numberofrowheaders-1] := aux;
                    mean:=mean+aux;
                    if (aux < 0) then dataneg := true;
                    if (aux < min) then
                        min := aux;
                    if (aux > max) then
                        max := aux;
                  end;
               end;
           end;
           i:=0;
        end;
    end;

    mean:=mean/(n*p);
    closefile(txt);
    createdestroy(false,p+numberofrowheaders+1,tmplabels);
end;

function getPureDataDimensions(fn:string;
                   var n,p, numberofrowheaders, numberofvarheaders:integer
                   ): boolean;
var txt:textfile;
    S: string;
    index, size,
    numberofvars, code: integer;
    ch: char;
    gofaster: boolean;
    aux: extended;

begin
    getPureDataDimensions:= true;
    assignfile(txt,fn);
    filemode:=0;
    reset(txt);
    //Read the number of rows and colums in the matrix
    assignfile(txt, fn);
    reset(txt);
    size := FileSize(txt);
    if (size = 0) then begin
      getPureDataDimensions:= false;
      exit;
    end;

    index:=0; numberofvarheaders := 0; numberofrowheaders := 0;
    numberofvars := 0;
    n := -1; p := -1;
    gofaster := false;

    while not(eof(txt)) do begin
        // Read one line
        if (not gofaster) then begin
            S := '';
            ch := ' ';
            repeat
              S := S + ch;
              Read(txt, ch);
            until ((ch = #9) or (ch = #13)  or (ch = #10) or (ch = #26));
            if (ch = #26) then break; //unexpected end of file
            if (S <> ' ') then begin
                val(S, aux, code);
                if (code <> 0) then begin
                    getPureDataDimensions:= false;
                    exit  // Wrong format
                end else
                    numberofvars := numberofvars + 1; // Pure data file
            end;

            if ((ch = #13) or (ch = #10) ) then begin
               if (ch <> #10) then readln(txt, S);
               gofaster := true;
               index := index+1;
            end;
        end else begin
            readln(txt, S);
            if (S <> '') then index := index+1;
        end;

    end;

   n := index;
   p := numberofvars;
   closefile(txt);
end;


function ReadPureData(fn:string;
                    nobjs, nvars:integer;
                    var x:matrix;
                    var mean, min, max: real;
                    var dataneg: boolean):boolean;
var txt:textfile; i,j:integer; s:string;
    aux: real;
begin
  assignfile(txt,fn);
  filemode:=0;
  reset(txt);
    ReadPureData:=true;
    dataneg := false;
    mean:=0; min:= 656000; max := -656000;
    for i:=1 to Nobjs do begin
      for j:=1 to Nvars do begin
        read(txt,aux);
        mean := mean + aux;
        x.m[i,j] := aux;
        if (aux < 0) then dataneg := true;
        if (aux < min) then
            min := aux;
        if (aux > max) then
            max := aux;
      end;
      readln(txt,s);
      s:=trim(s);
    end;
    mean:=mean/(nvars*nobjs);
    closefile(txt);
end;


function getLabeledDataDimensions(fn:string;
                   var n,p, numberofrowheaders, numberofvarheaders:integer
                   ): boolean;
var txt:textfile;
    S: string;
    index, size,
    numberofvars, code: integer;
    ch: char;
    gofaster: boolean;
    aux: extended;

begin
    getLabeledDataDimensions := true;
    assignfile(txt,fn);
    filemode:=0;
    reset(txt);
  //Read the number of rows and colums in the matrix
    assignfile(txt, fn);
    reset(txt);
    size := FileSize(txt);
    if (size = 0) then begin
      getLabeledDataDimensions := false;
      exit;
    end;

    index:=0;
    numberofvars := 0; numberofrowheaders := 0;
    gofaster := false;

    while not(eof(txt)) do begin
        // Read one line
        if (not gofaster) then begin
            S := '';
            ch := ' ';
            repeat
              S := S + ch;
              Read(txt, ch);
            until ((ch = #9) or (ch = #13)  or (ch = #10) or (ch = #26));
            if (ch = #26) then break; //unexpected end of file
            if (index = 1) then begin
                val(S, aux, code);
                if (code <> 0) then
                        numberofrowheaders := numberofrowheaders+1
                    else
                        numberofvars := numberofvars+1;
            end;
            if ((ch = #13) or (ch = #10) ) then begin
               if (ch <> #10) then readln(txt, S);
               if (index = 1) then gofaster:= true;
               index := index+1;
            end;
        end else begin
            readln(txt, S);
            if (S <> '') then index := index+1;
        end;

    end;

   n := index - 1;
   p := numberofvars;
   numberofvarheaders:= 1;
   closefile(txt);
end;

function ReadLabeledData(fn:string;
                   var n,p, numberofrowheaders, numberofvarheaders:integer;
                   var x:matrix;
                   var rowlabels, varlabels: stringvector;
                   var mean, min, max: real;
                   var dataneg: boolean): boolean;
var txt:textfile; i,j:integer;
    S, S2: string;
    index, code: integer;
    ch: char;
    tmplabels: stringvector;
    aux: extended;
begin
   ReadLabeledData := true;
   assignfile(txt,fn);
   filemode:=0;
   reset(txt);
   mean:=0; min:= 656000; max := -656000;

   createdestroy(true,p+numberofrowheaders,tmplabels);

  //Read the headers and data points in the matrix
    reset(txt);
    i:=0; index:=0;
    while not(eof(txt)) do begin
        // Read one line
        i:=i+1;
        S := '';
        ch := ' ';
        repeat
          S := S + ch;
          Read(txt, ch);
        until ((ch = #9) or (ch = #13)  or (ch = #10) or (ch = #26));
        if (ch = #26) then break; //unexpected end of file
        tmplabels[i] := leftstr(S, 255);

        // Now check whose line is this

        if ((ch = #13) or (ch = #10)) then begin
           if (ch <> #10) then readln(txt, S2);
           index := index+1;
           if (index <= numberofvarheaders) then begin
               for j:= 1 to numberofrowheaders + p do
                    varlabels[j] := tmplabels[j];
           end else begin  // now read the data
               for j:= 1 to p + numberofrowheaders do begin
                  if (j <= numberofrowheaders) then begin
                    if (j = 1) then
                      rowlabels[index-numberofvarheaders] := tmplabels[j]
                    else
                      rowlabels[index-numberofvarheaders] := rowlabels[index-numberofvarheaders] + #9 + tmplabels[j];
                  end else if (j > numberofrowheaders) then begin
                    // Check if it is OK
                    val(tmplabels[j], aux, code);
                    if (code <> 0) then begin
                        ReadLabeledData := false;
                        createdestroy(false,p+numberofrowheaders,tmplabels);
                        exit;
                    end;
                    x.m[index-numberofvarheaders, j-numberofrowheaders] := aux;
                    mean:=mean+aux;
                    if (aux < 0) then dataneg := true;
                    if (aux < min) then
                        min := aux;
                    if (aux > max) then
                        max := aux;
                 end;       
               end;
           end;
           i:=0;
        end;
    end;

    mean:=mean/(n*p);
    closefile(txt);
    createdestroy(false,p+numberofrowheaders,tmplabels);
end;

procedure CreateDestroy(Create:boolean; n:integer; var v:stringvector);
begin
  if create then
    getmem(v,n*sizeof(str255))
  else
    freemem(v,n*sizeof(str255))
end;

procedure CreateDestroy(Create:boolean; nr, nc:integer; var m:stringmatrix);
var i:integer;
begin
  if create then begin
    getmem(m,nr*sizeof(pointer));
    for i:=1 to nr do
      getmem(m[i],nc*sizeof(matrix))
  end else begin
    for i:=nr downto 1 do
      freemem(m[i],nc*sizeof(matrix));
    freemem(m,nr*sizeof(pointer));
  end;
end;

procedure CreateDestroy(Create:boolean; n:integer; var v:boolvector);
begin
  if create then
    getmem(v,n*sizeof(boolean))
  else
    freemem(v,n*sizeof(boolean))
end;

procedure CreateDestroy(Create:boolean; n:integer; var v:matvector);
begin
  if create then
    getmem(v,n*sizeof(TMatRecord))
  else
    freemem(v,n*sizeof(TMatRecord))
end;

procedure CreateDestroy(Create:boolean; n:integer; var v:factvector);
begin
  if create then
    getmem(v,n*sizeof(TFactRecord))
  else
    freemem(v,n*sizeof(TFactRecord))
end;

procedure CreateDestroy(Create:boolean; nr,nc:integer; var m:boolmatrix);
var i:integer;
begin
  if create then begin
    getmem(m,nr*sizeof(pointer));
    for i:=1 to nr do
      getmem(m[i],nc*sizeof(boolean))
  end else begin
    for i:=nr downto 1 do
      freemem(m[i],nc*sizeof(boolean));
    freemem(m,nr*sizeof(pointer));
  end;
end;

function normalizeData(var x:matrix; normMethod: integer; var negflag: boolean; var min, max, mean: real): boolean;
var
  k,i,j,p,n: integer;
  cm, cs:vector;
begin
  normalizeData := true;
  negflag := false;
  min := 65000; max:= -65000;
  p:=x.nc;
  n:=x.nr;
  if (normMethod = 1) then begin // Subtract grand mean
    mean := 0;
    for i:=1 to n do
      for j:=1 to p do
         mean := mean + x.m[i,j];
    mean := mean/(p*n);
    for i:=1 to n do
      for j:=1 to p do begin
          x.m[i,j]:=x.m[i,j]-mean;
          if (x.m[i,j] < 0) then negflag := true;
      end;
  end else if (normMethod = 2) then begin // Getz Normalization
     // calculate column mean
     create1(p,cm);
     negflag:= false;
     for j:= 1 to p do begin
        cm.v[j] := 0;
        for i:=1 to n do begin
           if (x.m[i,j] < 0) then negflag:= true;
           cm.v[j] := cm.v[j] + x.m[i,j];
        end;
        cm.v[j] := cm.v[j]/n;
     end;
     // Divide each column by its mean (or subtract it in case data is in log-scale
     for j:= 1 to p do begin
        for i:=1 to n do begin
            if (negflag) then
               x.m[i,j] := x.m[i,j]-cm.v[j]
            else begin
                if (cm.v[j] <> 0) then
                    x.m[i,j] := x.m[i,j]/cm.v[j]
                else begin
                    normalizeData := false;
                    exit;
                end;
            end;
        end;
     end;
     destroy1(cm);

     // calculate row mean
     create1(n,cm);
     for i:= 1 to n do begin
        cm.v[i] := 0;
        for j:=1 to p do begin
           cm.v[i] := cm.v[i] + x.m[i,j];
        end;
        cm.v[i] := cm.v[i]/p;
     end;
     // calculate row sd
     create1(n,cs);
     for i:= 1 to n do begin
        cs.v[i] := 0;
        for j:=1 to p do begin
           cs.v[i] := cs.v[i] + sqr((x.m[i,j]-cm.v[i]));
        end;
        cs.v[i] := sqrt(cs.v[i]);
     end;

     // Subtract the mean to each row and divide by its sd
     for i:= 1 to n do begin
        for j:=1 to p do begin
            if (cs.v[i] = 0) then cs.v[i] := 1;
            x.m[i,j] := (x.m[i,j]-cm.v[i])/cs.v[i];
            if (x.m[i,j] < 0) then negflag := true;
        end;
     end;
     destroy1(cs);
     destroy1(cm);
  end else if (normMethod = 3) then begin // Mean = 0, SD = 1 by rows
     create1(n,cs);
     create1(n,cm);
     // calculate row mean
     for i:= 1 to n do begin
        cm.v[i] := 0;
        for j:=1 to p do begin
           cm.v[i] := cm.v[i] + x.m[i,j];
        end;
        cm.v[i] := cm.v[i]/p;
     end;
     // calculate row sd
     for i:= 1 to n do begin
        cs.v[i] := 0;
        for j:=1 to p do begin
           cs.v[i] := cs.v[i] + sqr((x.m[i,j]-cm.v[i]));
        end;
        cs.v[i] := sqrt(cs.v[i]);
     end;
     // Subtract the mean to each row and divide by its sd
     for i:= 1 to n do begin
        for j:=1 to p do begin
            if (cs.v[i] = 0) then cs.v[i] := 1;
            x.m[i,j] := (x.m[i,j]-cm.v[i])/cs.v[i];
            if (x.m[i,j] < 0) then negflag := true;
        end;
     end;
     destroy1(cs);
     destroy1(cm);
  end else if (normMethod = 4) then begin // Mean = 0, SD = 1 by columns
     create1(p,cs);
     create1(p,cm);
     // Calculate column mean
     for j:= 1 to p do begin
        cm.v[j] := 0;
        for i:=1 to n do begin
           cm.v[j] := cm.v[j] + x.m[i,j];
        end;
        cm.v[j] := cm.v[j]/n;
     end;
     // calculate columns sd
     for j:= 1 to p do begin
        cs.v[j] := 0;
        for i:=1 to n do begin
           cs.v[j] := cs.v[j] + sqr((x.m[i,j]-cm.v[j]));
        end;
        cs.v[j] := sqrt(cs.v[j]);
     end;
     // Subtract the mean to each column and divide it by its sd
     for j:= 1 to p do begin
        for i:=1 to n do begin
            if (cs.v[j] = 0) then cs.v[j] := 1;
            x.m[i,j] := (x.m[i,j]-cm.v[j])/cs.v[j];
            if (x.m[i,j] < 0) then negflag := true;
        end;
     end;
     destroy1(cs);
     destroy1(cm);
  end else if (normMethod = 5) then begin // Subtract mean by rows
     // calculate row mean
     create1(n,cm);
     for i:= 1 to n do begin
        cm.v[i] := 0;
        for j:=1 to p do begin
           cm.v[i] := cm.v[i] + x.m[i,j];
        end;
        cm.v[i] := cm.v[i]/p;
     end;
     // Subtract the mean to each row
     for i:= 1 to n do begin
        for j:=1 to p do begin
            x.m[i,j] := (x.m[i,j]-cm.v[i]);
            if (x.m[i,j] < 0) then negflag := true;
        end;
     end;
     destroy1(cm);
  end else if (normMethod = 6) then begin // Subtract mean by columns
     // calculate column mean
     create1(p,cm);
     for j:= 1 to p do begin
        cm.v[j] := 0;
        for i:=1 to n do begin
           cm.v[j] := cm.v[j] + x.m[i,j];
        end;
        cm.v[j] := cm.v[j]/n;
     end;
     // Subtract the mean to each column
     for j:= 1 to p do begin
        for i:=1 to n do begin
            x.m[i,j] := x.m[i,j]-cm.v[j];
            if (x.m[i,j] < 0) then negflag := true;
        end;
     end;
     destroy1(cm);
  end else if (normMethod = 7) then begin // Subtract mean by rows and columns
     // calculate row mean
     create1(n,cm);
     for i:= 1 to n do begin
        cm.v[i] := 0;
        for j:=1 to p do begin
           cm.v[i] := cm.v[i] + x.m[i,j];
        end;
        cm.v[i] := cm.v[i]/p;
     end;
     // Subtract the mean to each row
     for i:= 1 to n do begin
        for j:=1 to p do begin
            x.m[i,j] := (x.m[i,j]-cm.v[i]);
        end;
     end;
     destroy1(cm);
     // calculate column mean
     create1(p,cm);
     for j:= 1 to p do begin
        cm.v[j] := 0;
        for i:=1 to n do begin
           cm.v[j] := cm.v[j] + x.m[i,j];
        end;
        cm.v[j] := cm.v[j]/n;
     end;
     // Subtract the mean to each column
     for j:= 1 to p do begin
        for i:=1 to n do begin
            x.m[i,j] := x.m[i,j]-cm.v[j];
            if (x.m[i,j] < 0) then negflag := true;
        end;
     end;
     destroy1(cm);
  end else if (normMethod = 8) then begin // Iterative row and column normalization
     // Apply logarithmic transformation
     ApplyLogTransformation(x);
     for k:= 1 to 1 do begin
         // calculate row mean
         create1(n,cm);
         for i:= 1 to n do begin
            cm.v[i] := 0;
            for j:=1 to p do begin
               cm.v[i] := cm.v[i] + x.m[i,j];
            end;
            cm.v[i] := cm.v[i]/p;
         end;
         // Subtract the mean to each row
         for i:= 1 to n do begin
            for j:=1 to p do begin
                x.m[i,j] := (x.m[i,j]-cm.v[i]);
            end;
         end;
         destroy1(cm);
         // calculate column mean
         create1(p,cm);
         for j:= 1 to p do begin
            cm.v[j] := 0;
            for i:=1 to n do begin
               cm.v[j] := cm.v[j] + x.m[i,j];
            end;
            cm.v[j] := cm.v[j]/n;
         end;
         // Subtract the mean to each column
         for j:= 1 to p do begin
            for i:=1 to n do begin
                x.m[i,j] := x.m[i,j]-cm.v[j];
            end;
         end;
         destroy1(cm);
     end; // for k
     for k:= 1 to 10 do begin
         negflag := false;
         create1(n,cs);
         create1(n,cm);
         // calculate row mean
         for i:= 1 to n do begin
            cm.v[i] := 0;
            for j:=1 to p do begin
               cm.v[i] := cm.v[i] + x.m[i,j];
            end;
            cm.v[i] := cm.v[i]/p;
         end;
         // calculate row sd
         for i:= 1 to n do begin
            cs.v[i] := 0;
            for j:=1 to p do begin
               cs.v[i] := cs.v[i] + sqr((x.m[i,j]-cm.v[i]));
            end;
            cs.v[i] := sqrt(cs.v[i]);
         end;
         // Subtract the mean to each row and divide by its sd
         for i:= 1 to n do begin
            for j:=1 to p do begin
                if (cs.v[i] = 0) then cs.v[i] := 1;
                x.m[i,j] := (x.m[i,j]-cm.v[i])/cs.v[i];
            end;
         end;
         destroy1(cs);
         destroy1(cm);
         create1(p,cs);
         create1(p,cm);
         // Calculate column mean
         for j:= 1 to p do begin
            cm.v[j] := 0;
            for i:=1 to n do begin
               cm.v[j] := cm.v[j] + x.m[i,j];
            end;
            cm.v[j] := cm.v[j]/n;
         end;
         // calculate columns sd
         for j:= 1 to p do begin
            cs.v[j] := 0;
            for i:=1 to n do begin
               cs.v[j] := cs.v[j] + sqr((x.m[i,j]-cm.v[j]));
            end;
            cs.v[j] := sqrt(cs.v[j]);
         end;
         // Subtract the mean to each column and divide it by its sd
         for j:= 1 to p do begin
            for i:=1 to n do begin
                if (cs.v[j] = 0) then cs.v[j] := 1;
                x.m[i,j] := (x.m[i,j]-cm.v[j])/cs.v[j];
                if (x.m[i,j] < 0) then negflag := true;
            end;
         end;
         destroy1(cs);
         destroy1(cm);
     end; // for k
  end else if (normMethod = 9) then begin // Log-Interactions normalization
     // Apply logarithmic transformation
     negflag := false;
     ApplyLogTransformation(x);

     // calculate row mean
     mean := 0;
     create1(n,cm);
     for i:= 1 to n do begin
        cm.v[i] := 0;
        for j:=1 to p do begin
           cm.v[i] := cm.v[i] + x.m[i,j];
           mean := mean + x.m[i,j];
        end;
        cm.v[i] := cm.v[i]/p;
     end;
     mean := mean/(n*p);
     // calculate column mean
     create1(p,cs);
     for j:= 1 to p do begin
        cs.v[j] := 0;
        for i:=1 to n do begin
           cs.v[j] := cs.v[j] + x.m[i,j];
        end;
        cs.v[j] := cs.v[j]/n;
     end;
     // Creates the interaction matrix
     for i:= 1 to n do begin
        for j:=1 to p do begin
            x.m[i,j] := x.m[i,j]-cm.v[i]-cs.v[j] + mean;
            if (x.m[i,j] < 0) then negflag := true;
        end;
     end;
     destroy1(cm);
     destroy1(cs);
  end;

  // calculate min and max
  mean := 0;
  for i:= 1 to n do begin
     for j:=1 to p do begin
        mean := mean + x.m[i,j];
        if (x.m[i,j] < min) then
           min := x.m[i,j];
        if (x.m[i,j] > max) then
           max := x.m[i,j];
     end;
  end;
  if (p*n <> 0) then mean := mean / (p*n) else mean := 0;
end;

function ApplyLogTransformation(var x:matrix): boolean;
var
  i,j: integer;
begin
    ApplyLogTransformation := true;
    for i:=1 to x.nr do
      for j:=1 to x.nc do begin
        if (x.m[i,j] <= 0) then begin
            ApplyLogTransformation := false;
            exit;
        end;
      end;
    for i:=1 to x.nr do
      for j:=1 to x.nc do
            x.m[i,j]:= log2(x.m[i,j]);
end;

function TransposeData(var x:matrix;
                       var n,p, numberofrowheaders, numberofvarheaders:integer;
                       var rowlabels, varlabels: stringvector
                       ): boolean;
var
  xt: matrix;
  rowlabelst, varlabelst: stringvector;
  i,j: integer;
  Save_Cursor:TCursor;
begin
   Save_Cursor := Screen.Cursor;
   Screen.Cursor := crHourGlass;    { Show hourglass cursor }
   TransposeData := true;
   create1(p,n,xt);
   TRANSP(xt,x);
   destroy1(x);
   n:=xt.nr; p:=xt.nc;
   x:=xt;
   createdestroy(true,p,varlabelst);
   createdestroy(true,n, rowlabelst);

   for i:=1 to n do begin
      rowlabelst[i] := varlabels[i+numberofrowheaders];
   end;
   for i:=1 to p do begin
      varlabelst[i] := rowlabels[i];
      for j:=1 to 255 do begin
         if varlabelst[i,j] = #9 then varlabelst[i,j] := '-';
      end;
   end;

   createdestroy(false,n + numberofrowheaders,varlabels);
   createdestroy(false,p, rowlabels);

   numberofrowheaders := 0;
   numberofvarheaders := 0;

   varlabels:=varlabelst;
   rowlabels:=rowlabelst;

   Screen.Cursor := Save_Cursor;  { Always restore to normal }
end;

end.


