/*
    Copyright (C) 2006  Laurent Poirrier

    This file is part of YGL2.

    YGL2 is free software; you can redistribute it and/or
    modify it under the terms of the GNU Lesser General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    YGL2 is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Lesser General Public License for more details.

    You should have received a copy of the GNU Lesser General Public
    License along with this library; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA

*/

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>
#include "draw.h"
#include "fonts.h"



char YtkPrintfBuffer[256];


// ******************************************************
//
// ******************************************************
struct ytk_font *YtkFontInit(unsigned char *data, int w, int h)
{
	struct ytk_font *r;
	struct ytk_font_char *c;
	int cx, cy, x, y;
	int lx, rx, uy, dy;
	int a, x0, x1, y0, y1;
	
	r = (struct ytk_font *)malloc(sizeof(struct ytk_font));
	if (!r) return((struct ytk_font *)0);

	// While global image is standard top-to-bottom, left-to-right,
	// character rasters are ordered bottom-to-top, then left-to-right
	// this is for historical reason (original OpenGL coordinate system).
	// The image is 16xN: char_width = char_height = image_width/16
	r->bytes_per_line = w;
	r->height = r->width = w / 16;
	r->data = data;
	r->nchars = (h / r->height) * 16;
	r->chars = (struct ytk_font_char *)
		malloc(sizeof(struct ytk_font_char)*r->nchars);
	if (!r->chars) { free(r); return((struct ytk_font *)0); }
	r->iso_width = 0;
	lx = r->width-1; rx = 0;
	uy = r->height-1; dy = 0;
	c = r->chars;
	
	for (cy=h-r->height; cy>=0; cy-=r->height)
	for (cx=0; (cx+r->width)<=w; cx+=r->width) {
		x0 = x1 = y0 = y1 = 0;
		for (x=0; x<r->width; x++) {
			a = 0;
			for (y=0; y<r->height; y++)
				a += data[(cy+y)*w+cx+x];
			if (a==0) { if (x0==x) x0++; }
			else x1=x;
		}
		for (y=0; y<r->height; y++) {
			a = 0;
			for (x=0; x<r->width; x++)
				a += data[(cy+y)*w+cx+x];
			if (a==0) { if (y0==y) y0++; }
			else y1=y;
		}
		if (x0<lx) lx = x0;
		if (x1>rx) rx = x1;
		if (y0<uy) uy = y0;
		if (y1>dy) dy = y1;
		//c->ptr = data+(cy*w)+cx;
		c->x = cx; c->y = cy;
		if (x0>x1) { c->var_x = cx; c->var_w = r->width; }
		else {
			c->var_x = cx+x0;
			c->var_w = x1-x0+1;
			if (c->var_w > r->iso_width)
				r->iso_width = c->var_w;
		}
		c++;
	}
	if ((lx>rx) || (uy>dy))
		{ free(r->chars); free(r); return((struct ytk_font *)0); }
	r->cst_dx = lx; r->cst_dy = uy;
	r->cst_w = rx-lx+1; r->cst_h = dy-uy+1;

	// Center offset for isometric fonts
	c = r->chars;
	for (y=0; y<r->nchars; y++) {
		x0 = (c->var_x - c->x) - ((r->iso_width - c->var_w + 1) >> 1);
		if (x0 < 0) x0 = 0;
		if (x0 + r->iso_width > r->width) x0 = r->width - r->iso_width;
		c->iso_x = c->x + x0;
		c++;
	}
	return(r);
}


// ******************************************************
//
// ******************************************************
void YtkFontFree(struct ytk_font *fnt)
{
	free(fnt->chars);
	free(fnt);
}

// ******************************************************
//
// ******************************************************
int YtkFontWidth(struct ytk_font *fnt, int mode, int spacing, char *str)
{
	int l, r, c;
	
	l = strlen(str);
	if (mode==FONT_DIRECT) return((fnt->width+spacing) * l);
	if (mode==FONT_CONSTANT) return((fnt->cst_w+spacing) * l);
	if (mode==FONT_ISOMETRIC) return((fnt->iso_width+spacing) * l);
	
	r = 0;
	while(l>0) {
		c = *str;
		if (c < fnt->nchars)
			r += fnt->chars[c].var_w + spacing;
		str++; l--;
	}
	return(r);
}

// ******************************************************
//
// ******************************************************
void YtkPrint(struct ytk_ws *ws, int dx, int dy, unsigned int color,
	struct ytk_font *fnt, int mode, int spacing, char *text)
{
	struct ytk_font_char *c;
	int cx, cy, cw, ch;
	unsigned char *ptr;
	unsigned int dst;
	int v, sr, sg, sb, dr, dg, db;
	int x, y;
	
	YtkColorToRGB(&sr, &sg, &sb, color);
	dy += fnt->cst_dy;
	ch = fnt->cst_h;
	
	while (*text != 0) {
		// Size
		c = &fnt->chars[*((unsigned char *)text)];
		cy = c->y + fnt->cst_dy;
		if (mode == FONT_CONSTANT)
			{ cx = c->x+fnt->cst_dx; cw = fnt->cst_w; }
		else if (mode == FONT_ISOMETRIC)
			{ cx = c->iso_x; cw = fnt->iso_width; }
		else if (mode == FONT_CLOSEST)
			{ cx = c->var_x; cw = c->var_w; }
		else //if (mode == FONT_DIRECT)
			{ cx = c->x; cw = fnt->width; }
		
		// Draw the char
		ptr = fnt->data + (cy * fnt->bytes_per_line) + cx;
		for (y=0; y<ch; y++) {
			for (x=0; x<cw; x++) if (ptr[x]) {
				v = ptr[x] + 1;
				dst = YtkGetPixel(ws, dx+x, dy+y);
				YtkColorToRGB(&dr, &dg, &db, dst);
				dr = ((sr*v)+(dr*(256-v)))>>8;
				dg = ((sg*v)+(dg*(256-v)))>>8;
				db = ((sb*v)+(db*(256-v)))>>8;
				dst = YtkRGBToColor(dr, dg, db);
				YtkPutPixel(ws, dx+x, dy+y, dst);
			}
			ptr += fnt->bytes_per_line;
		}
		
		// Next
		text++;
		dx += cw;
		dx += spacing;
	}
}
