#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <errno.h>
#include <string.h>
#include <time.h>
#include <sys/time.h>
#include "video_device.h"
#include "hm12.h"
#include "YGL2/ytk.h"

#define PE(s...)	fprintf(stderr, s)
#define PO(s...)	printf(s)

int pixel_threshold = 32;
float image_threshold = 0.001;

void frame_handler(void *data, void *data_back, void *img_data,
	int width, int height, int csp)
{
	int x, y, v;
	unsigned char *d1, *d0, *iy, *iu, *iv;
	int count;
	
	struct tm *hms;
	struct timeval curtime;
	
	d1 = (unsigned char *)data;
	d0 = (unsigned char *)data_back;
	if (csp == VIDEO_YU12) {
		iy = (unsigned char *)img_data;
		iu = (unsigned char *)img_data + (width * height);
		iv = (unsigned char *)img_data + (width * height * 5 / 4);
	} else if (csp == VIDEO_YV12) {
		iy = (unsigned char *)img_data;
		iv = (unsigned char *)img_data + (width * height);
		iu = (unsigned char *)img_data + (width * height * 5 / 4);
	} else if (csp == VIDEO_YUYV) {
		iy = (unsigned char *)img_data;
	}
	
	count = 0;
	
	if (csp == VIDEO_YUYV) {
		for (y = 0; y < height; y++) {
			for (x = 0; x < width; x++) {
				v = *d1 - *d0;
				if ((v < -pixel_threshold)
				||  (v > pixel_threshold)) {
					count++;
					iy[(x<<1) + 1] = 0;
				}
				d1 += 2;
				d0 += 2;
			}
			iy += width << 1;
		}
	} else {
		for (y = 0; y < height; y++) {
			for (x = 0; x < width; x++) {
				v = *d1 - *d0;
				if ((v < -pixel_threshold)
				||  (v > pixel_threshold)) {
					count++;
					iu[x >> 1] = 0;
					iv[x >> 1] = 0;
				}
				d1++;
				d0++;
			}
			if (y & 1) {
				iu += width >> 1;
				iv += width >> 1;
			}
		}
	}
	
	gettimeofday(&curtime, NULL);
	hms = localtime(&curtime.tv_sec);
	PO("%04d.%02d.%02d %02d:%02d:%02d.%06d : %d\n",
		hms->tm_year + 1900, hms->tm_mon, hms->tm_mday,
		hms->tm_hour, hms->tm_min, hms->tm_sec,
		(int)curtime.tv_usec, count);
	
	if ((float)count > image_threshold * width * height) {
		PO("*** MOVE ***\n");
	}
}


int main(int argc, char **argv)
{
	char *devname;
	video_device dev;
	Workspace img;
	Window win;
	void *capture, *data, *data_back;
	int win_width, win_height;
	
	if (argc != 2) {
		PE("Usage: gemvid <video-device>\n\n"
			"<video-device> is most often /dev/video\n");
		return(0);
	}
	
	devname = argv[1];
	
	if (video_init(&dev, 0, 0, VIDEO_YUYV, devname))
		return(1);
	
	if ((dev.fourcc != VIDEO_YV12) && (dev.fourcc != VIDEO_HM12)
	&&  (dev.fourcc != VIDEO_YU12) && (dev.fourcc != VIDEO_YUYV)) {
		PE("%s: Current pixel format (0x%x) not supported. "
			"Try changing it.\n", dev.device, dev.fourcc);
		video_shutdown(&dev);
		return(1);
	}
	
	if (YtkInit()) {
		video_shutdown(&dev);
		return(1);
	}
	
	win = YCreateWindow(0, 0, dev.width, dev.height, 0,
		YGL_FULL_EVENT_MASK);
	if (win == None) {
		YtkClose();
		video_shutdown(&dev);
		return(1);
	}
	YSetName(win, "gembla", "gembla");
	
	// YGL2 supports X11 <-> v4l2 colorspace naming conversions
	if (YtkCreateWorkspace(&img, YTK_CSPIMAGE, dev.width, dev.height,
	(dev.fourcc == VIDEO_HM12) ? YTK_YV12 : dev.fourcc)) {
		YDestroyWindow(win);
		YtkClose();
		video_shutdown(&dev);
		return(1);
	}
	
	if (dev.fourcc == VIDEO_HM12) {
		data = malloc(dev.size * 2);
		if (!data) {
			PE("malloc(2x%d): %s\n", dev.size, strerror(errno));
			return(1);
		}
		data_back = data + dev.size;
	} else {
		data = NULL;
		data_back = malloc(dev.size);
		if (!data_back) {
			PE("malloc(%d): %s\n", dev.size, strerror(errno));
			return(1);
		}
	}
	
	while(1) {
		capture = video_capture(&dev);
		if (!capture) return(1);
		
		if (dev.fourcc == VIDEO_HM12) {
			hm12_to_yv12(data, capture, dev.width, dev.height);
		} else {
			data = capture;
		}
		
		memcpy(img.data, data, dev.size);
		
		frame_handler(data, data_back, img.data,
			dev.width, dev.height,
			(dev.fourcc == VIDEO_HM12) ? VIDEO_YV12 : dev.fourcc);
		
		memcpy(data_back, data, dev.size);
		
		YGetSize(win, &win_width, &win_height);
		YtkPutExtWorkspace(&img, win, 0, 0, win_width, win_height,
			0, 0, dev.width, dev.height);
		YSync();
	}
	
	if (dev.fourcc == VIDEO_HM12) free(data);
	free(data_back);
	YtkClose();
	video_shutdown(&dev);
}



