#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <string.h>
#include <errno.h>
#include <sys/ioctl.h>
#include <sys/mman.h>
#include <linux/videodev2.h>
#include "video_device.h"


#define PE(s...)	fprintf(stderr, s)

#define QUERY_DELAY	4000
#define CONTROL_DELAY	4000

#ifndef V4L2_CID_PRIVATE_LAST
#define V4L2_CID_PRIVATE_LAST   (V4L2_CID_PRIVATE_BASE + 20)
#endif


#define V4L2_CTRL_CLASS_CAMERA 0x009A0000       /* Camera class controls */

#ifndef V4L2_CID_CAMERA_CLASS_BASE
#define V4L2_CID_CAMERA_CLASS_BASE	(V4L2_CTRL_CLASS_CAMERA | 0x900)
#endif

#ifndef V4L2_CID_CAMERA_CLASS_LAST
#define V4L2_CID_CAMERA_CLASS_LAST      (V4L2_CID_CAMERA_CLASS_BASE + 20)
#endif

#ifndef V4L2_CID_LAST
//#define V4L2_CID_LAST (V4L2_CID_LASTP1 - 1)
#define V4L2_CID_LAST   (V4L2_CID_BASE + 48)
#endif

// ***********************************************************
// Dump info and value of a specific control
// ***********************************************************
static void video_query_control(int fd, int index)
{
	struct v4l2_queryctrl qctrl;
	struct v4l2_control ctrl;
	int relid;
	char idbase;
	struct v4l2_querymenu menu;

	// make debugging quieter
	memset(&qctrl, 0, sizeof(qctrl));
	memset(&ctrl, 0, sizeof(ctrl));
	memset(&menu, 0, sizeof(menu));
	
	qctrl.id = index;
	if (ioctl(fd, VIDIOC_QUERYCTRL, &qctrl) != 0) return;

	if (qctrl.flags & V4L2_CTRL_FLAG_DISABLED) return;

	if (qctrl.id >= V4L2_CID_PRIVATE_BASE) {
		relid = qctrl.id - V4L2_CID_PRIVATE_BASE;
		idbase = 'P';
	} else if (qctrl.id >= V4L2_CID_CAMERA_CLASS_BASE) {
		relid = qctrl.id - V4L2_CID_CAMERA_CLASS_BASE;
		idbase = 'C';
	} else if (qctrl.id >= V4L2_CID_BASE) {
		relid = qctrl.id - V4L2_CID_BASE;
		idbase = 'V';
	} else {
		relid = qctrl.id;
		idbase = '0';
	}
	
	PE("%c+%2d: %s %31s ",
		idbase, relid,
		(qctrl.type == V4L2_CTRL_TYPE_INTEGER) ? "int " :
		(qctrl.type == V4L2_CTRL_TYPE_BOOLEAN) ? "bool" :
		(qctrl.type == V4L2_CTRL_TYPE_MENU) ? "menu" :
		(qctrl.type == V4L2_CTRL_TYPE_BUTTON) ? "btn " : "??  ",
		qctrl.name);
	if (qctrl.type == V4L2_CTRL_TYPE_BOOLEAN) {
		PE("(    0;    1)* 1 [%5d]=", qctrl.default_value);
	} else {
		PE("(%5d;%5d)*%2d [%5d]=",
			qctrl.minimum, qctrl.maximum, qctrl.step,
			qctrl.default_value);
	}
	
	ctrl.id = qctrl.id;
	if (ioctl(fd, VIDIOC_G_CTRL, &ctrl))
		PE(" (?)");
	else	PE("%4d", ctrl.value);

	PE(" %s\n", (qctrl.flags&V4L2_CTRL_FLAG_GRABBED)?"g/ ":"");

	if (qctrl.type == V4L2_CTRL_TYPE_MENU) {
		menu.id = index;
		for (menu.index = qctrl.minimum; menu.index <= qctrl.maximum;
		menu.index++) {
			if (ioctl(fd, VIDIOC_QUERYMENU, &menu))
				PE("\t> %d: ??? (failed)\n", menu.index);
			else
				PE("\t> %d: %s\n", menu.index, menu.name);
		}
	}
}


// ***********************************************************
// Change a control value
// ***********************************************************
static int video_rcontrol(video_device *vi, int id, int relative, int val)
{
	struct v4l2_queryctrl qctrl;
	struct v4l2_control ctrl;
	int has_info;

	if ((relative) && (val == 0)) return(0);

	// make debugging quieter
	memset(&qctrl, 0, sizeof(qctrl));
	memset(&ctrl, 0, sizeof(ctrl));

	// Query name & range
	qctrl.id = id;
	has_info = (ioctl(vi->fd, VIDIOC_QUERYCTRL, &qctrl) == 0);
	if (has_info) PE("%s: '%s' [%d %d]: ",
		vi->device, qctrl.name, qctrl.minimum, qctrl.maximum);
	else PE("%s: id 0x%x: ", vi->device, id);

	// Query value
	ctrl.id = id;
	if (ioctl(vi->fd, VIDIOC_G_CTRL, &ctrl)) {
		PE("  (?)");
		if (relative) { PE("ABORTED!\n"); return(1); }
	} else PE("%5d", ctrl.value);

	// Update
	if (relative) ctrl.value += val;
	else ctrl.value = val;
	
	PE(" -> %4d : ", ctrl.value);
	if ((has_info)
	&&  (qctrl.type != V4L2_CTRL_TYPE_BOOLEAN)) {
	// &&  (id != V4L2_CID_EXPOSURE_AUTO))  // :-/  !
		if (ctrl.value < qctrl.minimum) ctrl.value = qctrl.minimum;
		if (ctrl.value > qctrl.maximum) ctrl.value = qctrl.maximum;
	}

	PE(" -> checked %4d : ", ctrl.value);
	if (ioctl(vi->fd, VIDIOC_S_CTRL, &ctrl)) {
		PE("FAILED!\n");
		return(1);
	}
	PE("OK\n");
	return(0);
}

int video_control(video_device *vi, int id, int relative, int val)
{
	int retry = 0;

	while (retry <= 3) {
		if (retry) {
			usleep(CONTROL_DELAY);
			PE("retry %d:\n", retry);
		}
		if (video_rcontrol(vi,id,relative,val) == 0)
			return(0);
		retry++;
	}
	return(1);
}

// ***********************************************************
// Initialize video stream acquire
// ***********************************************************
#define VIDEO_STREAM_MAX_BUF	2
#define VIDEO_STREAM_REQ_BUF	2
struct video_stream_info {
	int nbufs;
	int started;
	struct v4l2_buffer buf_active;
	struct video_buffer_info {
		void *start;
		size_t length;
	} buffers[VIDEO_STREAM_MAX_BUF];
};

static int video_init_streaming(video_device *vi)
{
	struct v4l2_requestbuffers reqbuf;
	struct v4l2_buffer buffer;
	struct video_stream_info *s;
	int i;

	// make debugging quieter
	memset(&reqbuf, 0, sizeof(reqbuf));
	memset(&buffer, 0, sizeof(buffer));

	vi->iomethod = IOMETHOD_STREAMING;

	reqbuf.type = V4L2_BUF_TYPE_VIDEO_CAPTURE;
	reqbuf.memory = V4L2_MEMORY_MMAP;
	reqbuf.count = VIDEO_STREAM_REQ_BUF;

	if (ioctl(vi->fd, VIDIOC_REQBUFS, &reqbuf) == -1) {
		PE("\tVideo buffers request failed.\n");
		return(1);
	}
	
	if (reqbuf.count != VIDEO_STREAM_REQ_BUF)
		PE("\tioctl(VIDIOC_REQBUFS): got %d buffers instead of %d\n",
			reqbuf.count, VIDEO_STREAM_REQ_BUF);
	if (reqbuf.count > VIDEO_STREAM_MAX_BUF)
		return(1);
	
	vi->m_data = malloc(sizeof(struct video_stream_info));
	if (!vi->m_data) {
		PE("\tmalloc() failed!\n");
		return(-1);
	}
	s = (struct video_stream_info *)vi->m_data;
	s->nbufs = reqbuf.count;
	s->started = 0;

	for (i=0; i<s->nbufs; i++) {
		memset(&buffer, 0, sizeof(buffer));
		buffer.type = V4L2_BUF_TYPE_VIDEO_CAPTURE;
		buffer.memory = V4L2_MEMORY_MMAP;
		buffer.index = i;

		if (ioctl(vi->fd, VIDIOC_QUERYBUF, &buffer)) {
			PE("\tquerybuf(%d) failed!\n", i);
			for (i--; i>=0; i--)
				munmap(s->buffers[i].start,
					s->buffers[i].length);
			free(vi->m_data);
			return(1);
		}

		s->buffers[i].length = buffer.length;
		s->buffers[i].start = mmap(NULL, buffer.length,
			PROT_READ, MAP_SHARED,
			vi->fd, buffer.m.offset);
		if (s->buffers[i].start == MAP_FAILED) {
			PE("\t%d: %d\n", buffer.m.offset, buffer.length);
			PE("\tmmap(buffer %d) failed!\n", i);
			for (i--; i>=0; i--) munmap(s->buffers[i].start,
					s->buffers[i].length);
			free(vi->m_data);
			return(1);
		}

		if (ioctl(vi->fd, VIDIOC_QBUF, &buffer)) {
			PE("\tqueue_buf(%d) failed!\n", i);
			for (i--; i>=0; i--) munmap(s->buffers[i].start,
					s->buffers[i].length);
			free(vi->m_data);
			return(1);
		}
	}

	PE("\tVideo stream: %d buffers enqueued\n", reqbuf.count);
	return(0);
}

// ***********************************************************
// Initialize std read/write io acquire
// ***********************************************************
static int video_init_readwrite(video_device *vi)
{
	vi->iomethod = IOMETHOD_READWRITE;
	vi->m_data = malloc(vi->size);
	if (!vi->m_data) {
		PE("\tmalloc(%d) failed\n", vi->size);
		return(-1);
	}
	return(0);
}


// ***********************************************************
// Initialize video device
// ***********************************************************
#define PCAP(s)		if (cap.capabilities & V4L2_CAP_ ## s) PE(#s " ");
int video_init(video_device *vi, int tw, int th, int fcc, char *device)
{
	struct v4l2_capability cap;
	struct v4l2_input vin;
	struct v4l2_fmtdesc fmtd;
	struct v4l2_format fmt;
	int fd, index, r, fccs[2];

	if (!vi) { PE("video_init(): BUG: null ptr passed\n"); return(-1); }

	PE("%s:\n", device);

	// ************** Make debugging quieter
	memset(&cap, 0, sizeof(cap));
	memset(&vin, 0, sizeof(vin));
	memset(&fmtd, 0, sizeof(fmtd));
	memset(&fmt, 0, sizeof(fmt));

	// ************** Open
	fd = open(device, O_RDONLY);
	if (fd < 0) {
		PE("\topen(): %s\n", strerror(errno));
		return(-1);
	}

	// ************** Caps
	if (ioctl(fd, VIDIOC_QUERYCAP, &cap)) {
		PE("\tCapabilities query failed: %s\n", strerror(errno));
		close(fd);
		return(-1);
	}
	PE("\t%s (driver %s)\n\t", cap.card, cap.driver);
	PCAP(VIDEO_CAPTURE); PCAP(VIDEO_OUTPUT); PCAP(VIDEO_OVERLAY);
	PCAP(VBI_CAPTURE); PCAP(VBI_OUTPUT);
	//PCAP(SLICED_VBI_CAPTURE); PCAP(SLICED_VBI_OUTPUT);
	PCAP(RDS_CAPTURE); PCAP(TUNER); PCAP(AUDIO); PCAP(RADIO);
	PCAP(READWRITE); PCAP(ASYNCIO); PCAP(STREAMING);
	PE("\n");

	// ************** Inputs
	index = -1;
	do {
		index++;
		vin.index = index;
		r = ioctl(fd, VIDIOC_ENUMINPUT, &vin);
		if (!r) {
			PE("\t Input %d: %s, %s\n", vin.index, vin.name,
				(vin.type==V4L2_INPUT_TYPE_CAMERA)?
				"camera" : "tuner/other");
		}
	} while(r == 0);
	if (ioctl(fd, VIDIOC_G_INPUT, &index))
		PE("\tWarning: input query failed: %s\n", strerror(errno));
	PE("\tTotal %d inputs (current #%d)\n", vin.index, index);

	// ************** (Optional) Available formats
	index = -1;
	do {
		index++;
		fmtd.type = V4L2_BUF_TYPE_VIDEO_CAPTURE;
		fmtd.index = index;
		r = ioctl(fd, VIDIOC_ENUM_FMT, &fmtd);
		if (r == 0) {
			fccs[0] = fmtd.pixelformat;
			fccs[1] = 0;
			PE("\t Format %d: %s %s (0x%x == %s)\n",
				fmtd.index,
				(fmtd.flags&V4L2_FMT_FLAG_COMPRESSED) ?
				"compressed" : "raw", fmtd.description,
				fmtd.pixelformat, (char *)fccs);
		}
	} while (r == 0);
	PE("\tTotal %d pixel formats\n", fmtd.index);

	// ************** Controls
	// Querying controls *BEFORE* changing capture format
	// ensures no Input/Output failure when setting capture format.
	// :-( :-( :-( !
	PE("\tControls:\n");
	
	index = -1;
	for (index = V4L2_CID_BASE; index <= V4L2_CID_LAST; index++)
		video_query_control(fd, index);
	for (index = V4L2_CID_PRIVATE_BASE; index <= V4L2_CID_PRIVATE_LAST;
	index++) {
		//usleep(QUERY_DELAY);
		video_query_control(fd, index);
	}
	for (index = V4L2_CID_CAMERA_CLASS_BASE;
	index <= V4L2_CID_CAMERA_CLASS_LAST; index++) {
		usleep(QUERY_DELAY);
		video_query_control(fd, index);
	}

	PE("\n");

	// ************** Current format
	fmt.type = V4L2_BUF_TYPE_VIDEO_CAPTURE;
	if (ioctl(fd, VIDIOC_G_FMT, &fmt)) {
		PE("Cannot get current capture format: %s\n",
			strerror(errno));
		close(fd);
		return(-1);
	}
	fccs[0] = fmt.fmt.pix.pixelformat; fccs[1] = 0;
	PE("\tPrevious format: %dx%d, %s, (%dBpl, %dB total), colorspace %d\n",
		fmt.fmt.pix.width, fmt.fmt.pix.height, (char *)fccs,
		fmt.fmt.pix.bytesperline, fmt.fmt.pix.sizeimage,
		fmt.fmt.pix.colorspace);

	// ************** Set format
	if (tw <= 0) tw = fmt.fmt.pix.width;
	if (th <= 0) th = fmt.fmt.pix.height;
	if ((fcc == 0) || (fcc == -1)) fcc = fmt.fmt.pix.pixelformat;
	
	fmt.type = V4L2_BUF_TYPE_VIDEO_CAPTURE;
	fmt.fmt.pix.width = tw;
	fmt.fmt.pix.height = th;
	fmt.fmt.pix.pixelformat = fcc;
	fmt.fmt.pix.bytesperline = 0;
	fmt.fmt.pix.sizeimage = 0;
	index = 0;
	do {
		usleep(QUERY_DELAY);
		r = ioctl(fd, VIDIOC_S_FMT, &fmt);
		if (r) {
			PE("\tCannot set capture format: %s\n",
				strerror(errno));
			close(fd);
			usleep(QUERY_DELAY);
			fd = open(device, O_RDONLY);
			if (fd < 0) {
				PE("\topen(): %s\n", strerror(errno));
				return(-1);
			}
		}
		index++;
	} while ((r) && (index < 3));
	if (r) {
		PE("\tGiving up\n");
		close(fd);
		return(-1);
	}
	
	// ************** Obtained format
	fmt.type = V4L2_BUF_TYPE_VIDEO_CAPTURE;
	if (ioctl(fd, VIDIOC_G_FMT, &fmt)) {
		PE("Cannot get current capture format: %s\n",
			strerror(errno));
		close(fd);
		return(-1);
	}
	fccs[0] = fmt.fmt.pix.pixelformat; fccs[1] = 0;
	PE("\tCurrent  format: %dx%d, %s, (%dBpl, %dB total), colorspace %d\n",
		fmt.fmt.pix.width, fmt.fmt.pix.height, (char *)fccs,
		fmt.fmt.pix.bytesperline, fmt.fmt.pix.sizeimage,
		fmt.fmt.pix.colorspace);

	// ************** Output info
	vi->width = fmt.fmt.pix.width;
	vi->height = fmt.fmt.pix.height;
	vi->fd = fd;
	vi->size = fmt.fmt.pix.sizeimage;
	vi->fourcc = fmt.fmt.pix.pixelformat;
	vi->device = device;

	// We prefer streaming over readwrite, if available
	if (cap.capabilities & V4L2_CAP_STREAMING) {
		if (video_init_streaming(vi) == 0) return(0);
	}
	if (cap.capabilities & V4L2_CAP_READWRITE) {
		if (video_init_readwrite(vi) == 0) return(0);
	}
	close(fd);
	return(-1);
}

// ***********************************************************
// Close video device
// ***********************************************************
void video_shutdown(video_device *vi)
{
	struct video_stream_info *s = (struct video_stream_info *)vi->m_data;
	int i;

	switch(vi->iomethod) {
	case IOMETHOD_READWRITE:
		free(vi->m_data);
		break;
	case IOMETHOD_STREAMING:
		for (i=0; i<s->nbufs; i++)
			munmap(s->buffers[i].start, s->buffers[i].length);
		i = V4L2_BUF_TYPE_VIDEO_CAPTURE;
		if (s->started) {
			if (ioctl(vi->fd, VIDIOC_STREAMOFF, &i))
				PE("%s: STREAMOFF FAILED!\n", vi->device);
		}
		free(s);
		break;
	default:
		break;
	}

	close(vi->fd);
	PE("%s: closed\n", vi->device);
}

// ***********************************************************
// Grab one frame
// ***********************************************************
void *video_capture(video_device *vi)
{
	struct video_stream_info *s = (struct video_stream_info *)vi->m_data;
	int n;

	if (vi->iomethod == IOMETHOD_READWRITE) {
		n = read(vi->fd, vi->m_data, vi->size);
		if (n != vi->size) {
			PE("%s: read(): %d/%d -- %s\n", vi->device,
				n, vi->size, strerror(errno));
			return(NULL);
		}
		return(vi->m_data);
	}

	// stream
	if (!s->started) {
		n = V4L2_BUF_TYPE_VIDEO_CAPTURE;
		if (ioctl(vi->fd, VIDIOC_STREAMON, &n)) {
			PE("stream_on(): %s\n", strerror(errno));
			return(NULL);
		}
	} else {
		if (ioctl(vi->fd, VIDIOC_QBUF, &s->buf_active)) {
			PE("queue_buf(): %s\n", strerror(errno));
			return(NULL);
		}
	}
	
	// Make debugging quieter (a quick check on docs shows this is useless)
	memset(&s->buf_active, 0, sizeof(s->buf_active));
	
	s->buf_active.type = V4L2_BUF_TYPE_VIDEO_CAPTURE;
	s->buf_active.memory = V4L2_MEMORY_MMAP;
	if (ioctl(vi->fd, VIDIOC_DQBUF, &s->buf_active)) {
		PE("dequeue_buf(): %s\n", strerror(errno));
		return(NULL);
	}
	s->started = 1;
	return(s->buffers[s->buf_active.index].start);
}

// ***********************************************************
// Video file source
// ***********************************************************
int   videofile_init(video_device *vi, int w, int h, int fourcc,
	int size, char *file)
{
	int fccs[2];
	
	vi->fd = open(file, O_RDONLY);
	if (vi->fd < 0) {
		PE("%s: %s\n", file, strerror(errno));
	}

	vi->width = w;
	vi->height = h;
	vi->size = size;
	vi->fourcc = fourcc;

	fccs[0] = fourcc;
	fccs[1] = 0;
	PE("%s: %d x %d, %s, %d bytes per frame\n",
		file, w, h, (char *)fccs, size);

	if (video_init_readwrite(vi)) return(0);
	
	close(vi->fd);
	return(-1);
}

// ***********************************************************
/*
int video_move(video_device *vi, int pan, int tilt)
{
	int r = 0;

	if (pan) r |= video_control(vi, V4L2_CID_PAN_RELATIVE, 0, pan);
	if (tilt) r |= video_control(vi, V4L2_CID_TILT_RELATIVE, 0, tilt);

	return(r);
}

int video_position_reset(video_device *vi)
{
	return(video_control(vi, V4L2_CID_PANTILT_RESET, 0, 1));
}
*/


