#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "text.h"

#define TXT_FILE	0
#define TXT_FD		1
#define TXT_BUFFER	2

#define TXT_HISTORY	16
#define TXT_COMMENTS	4

struct txt_input {
	int type;
	char *name;
	union {
		char *buff;
		int fd;
	} src;
	char *buff;
	int buffsize;
	int pos, line, eof;
	char history[TXT_HISTORY];
	int hislen;
	struct {
		char start[TXT_HISTORY];
		char stop[TXT_HISTORY];
		int startlen, stoplen;
	} comments[TXT_COMMENTS];
	int nocomment;
};

// **********************************
// Generic input creator
// **********************************
static struct txt_input *txt_input(int type, int fd, char *s)
{
	struct txt_input *i;
	int t;
	
	if (type == TXT_FILE) {
		fd = open(s, O_RDONLY);
		if (fd < 0) return(NULL);
	}
	
	i = (struct txt_input *)malloc(sizeof(struct txt_input));
	if (i==NULL) return(NULL);

	i->type = type;
	if (type == TXT_FILE) {
		i->name = s;
		i->src.fd = fd;
	} else if (type == TXT_FD) {
		i->name = "?";
		i->src.fd = fd;
	} else {
		i->name = "?";
		i->src.buff = s;
	}
	for (t=0; t<TXT_COMMENTS; t++) i->comments[t].startlen = 0;
	
	i->buffsize = 1024;
	i->buff = (char *)malloc(i->buffsize);
	if (i->buff == NULL) { free(i); return(NULL); }
	i->pos = 0;
	txt_reset(i);
	
	return(i);
}

int txt_reset(struct txt_input *i)
{
	int t;

	if ((i->type != TXT_BUFFER) && (i->pos != 0)) {
		if (lseek(i->src.fd, 0, SEEK_SET) == -1) return(-1);
	}
	for (t=0; t<TXT_HISTORY; t++) i->history[t] = 0;
	i->pos = i->eof = i->hislen = 0;
	i->line = 1;
	i->nocomment = 0;
	return(0);
}

// **********************************
// Specific input creators
// **********************************
struct txt_input *txt_input_file(char *name)
{ return(txt_input(TXT_FILE, -1, name)); }
struct txt_input *txt_input_fd(int fd)
{ return(txt_input(TXT_FD, fd, NULL)); }
struct txt_input *txt_input_buffer(char *s)
{ return(txt_input(TXT_FILE, -1, s)); }

// **********************************
// Input closer
// **********************************
int txt_close(struct txt_input *i)
{
	int type, fd;
	
	type = i->type;
	fd = i->src.fd;
	free(i->buff);
	free(i);
	if (type == TXT_FD) return(fd);
	if (type == TXT_FILE) close(fd);
	return(0);
}


// **********************************
// Input setup: Buffer size
// **********************************
int txt_buffsize(struct txt_input *i, int sz)
{
	char *b;
	
	if (sz < 16) sz = 16;

	b = (char *)malloc(sz);
	if (b == NULL) return(-1);

	free(i->buff);
	i->buff = b;
	i->buffsize = sz;
	return(0);
}

// **********************************
// Input setup: Comments
// **********************************
int txt_comment(struct txt_input *i, int num, char *start, char *stop)
{
	int l, t;
	
	if ((num < 0) || (num >= TXT_COMMENTS)) return(-1);
	
	if (start == NULL) l = 0; else l = strlen(start);
	if (l > TXT_HISTORY) return(-1);
	i->comments[num].startlen = l;
	for (t=0; t<l; t++) i->comments[num].start[t] = start[t];
	
	if (stop == NULL) l = 0; else l = strlen(stop);
	if (l > TXT_HISTORY) return(-1);
	i->comments[num].stoplen = l;
	for (t=0; t<l; t++) i->comments[num].stop[t] = stop[t];

	return(0);
}

// **********************************
// Input info
// **********************************
int txt_curline(struct txt_input *i)
{
	return(i->line);
}

int txt_eof(struct txt_input *i)
{
	return(i->eof && (i->hislen == 0));
}

// **********************************
// Input setup: File name
// **********************************
char *txt_name(struct txt_input *i, char *name)
{
	char *r = i->name;
	if (name) i->name = name;
	return(r);
}

// **********************************
// By char...
// **********************************
static void txt_fillhis(struct txt_input *i)
{
	int n;
	char r;

	while ((!i->eof) && (i->hislen < TXT_HISTORY)) {
		if (i->type == TXT_BUFFER) {
			r = i->src.buff[i->pos];
			if (r==0) i->eof = 1;
		} else {
			n = read(i->src.fd, &r, 1);
			if (n != 1) i->eof = 1;
		}
		if (!i->eof) {
			i->history[i->hislen] = r;
			i->hislen++;
			i->pos++;
		}
	}
}
static char txt_pop(struct txt_input *i)
{
	int n;
	char r;
	
	if (i->hislen == 0) return(0);
	r = i->history[0];
	for (n=0; n<TXT_HISTORY-1; n++) i->history[n] = i->history[n+1];
	i->history[TXT_HISTORY-1] = 0;
	i->hislen--;
	
	if (r == '\n') i->line++;
	return(r);
}

static void txt_unpop(struct txt_input *i, char r)
{
	int n;
	
	if (i->hislen == TXT_HISTORY) return; // HHuuuhhggh!
	for (n=TXT_HISTORY-1; n>0; n--) i->history[n] = i->history[n-1];
	i->history[0] = r;
	i->hislen++;
	if (r == '\n') i->line--;
}

// **********************************
// Read a char
// **********************************
char txt_char(struct txt_input *i)
{
	int t, gotstart, gotend;

	if (i->nocomment) {
		txt_fillhis(i);
		return(txt_pop(i));
	}
	
	gotstart = -1;
	txt_fillhis(i);
	for (t=0; t<TXT_COMMENTS; t++) if (i->comments[t].startlen) {
		if (strncmp(i->history, i->comments[t].start,
		i->comments[t].startlen)==0) gotstart = t;
	}
	
	if (gotstart == -1) return(txt_pop(i));
	
	for (t=0; t<i->comments[gotstart].startlen; t++) txt_pop(i);
	gotend = 0;
	while (!gotend) {
		txt_fillhis(i);
		if (strncmp(i->history,
		i->comments[gotstart].stop,
		i->comments[gotstart].stoplen)==0) gotend = 1;
		else if (txt_pop(i)==0) return(0);
	}
	for (t=0; t<i->comments[gotstart].stoplen; t++) txt_pop(i);
	return(' ');
}

// **********************************
// Read a line
// **********************************
char *txt_line(struct txt_input *i)
{
	int p, l;
	char r;

	l = i->line;
	p = 0;

	while(1) {
		r = txt_char(i);
		if ((r==0)||(i->line != l)) {
			i->buff[p] = 0;
			return(i->buff);
		}
		i->buff[p] = r;
		if (p < i->buffsize - 1) p++;
	}
}

// **********************************
// Read a word
// **********************************
char *txt_word(struct txt_input *i)
{
	int p;
	char c, b;

	i->buff[0] = 0;
	c = txt_char(i);
	if (c==0) return(i->buff);
	while ((c==' ')||(c=='\t')||(c=='\n')||(c=='\r')) c = txt_char(i);

	if (c==0) return(i->buff);
	if ((c=='"')||(c=='\'')) {
		b = c;
		p = 0;
		i->nocomment = 1;
		while(1) {
			c = txt_char(i);
			if ((c==b)||(c==0)) {
				i->buff[p] = 0;
				i->nocomment = 0;
				return(i->buff);
			}
			i->buff[p] = c;
			if (p < i->buffsize - 1) p++;
		}
	}
	
	if ( ((c>='A')&&(c<='Z')) || ((c>='a')&&(c<='z'))
	|| (c=='_') || (c<0) ) {
		p = 0;
		while(1) {
			if ( ((c>='A')&&(c<='Z')) || ((c>='a')&&(c<='z'))
			|| (c=='_') || (c<0) || ((c>='0')&&(c<='9'))
			|| (c=='.') ) i->buff[p] = c;
			else {
				if (c != 0) txt_unpop(i, c);
				i->buff[p] = 0;
				return(i->buff);
			}
			if (p < i->buffsize - 1) p++;
			c = txt_char(i);
		}
	}

	if ( ((c>='0')&&(c<='9')) || (c=='.') ) {
		p = 0;
		while(1) {
			if ( ((c>='0')&&(c<='9')) || (c=='.')
			|| ((c>='A')&&(c<='F')) || ((c>='a')&&(c<='f'))
			|| (c=='x') || (c=='X') || (c=='h') || (c=='H') )
				i->buff[p] = c;
			else {
				if (c != 0) txt_unpop(i, c);
				i->buff[p] = 0;
				return(i->buff);
			}
			if (p < i->buffsize - 1) p++;
			c = txt_char(i);
		}
	}

	if ((c=='<')||(c=='>')||(c=='=')||(c=='&')||(c=='|')||(c=='^')
	||(c=='+')||(c=='-')) {
		b = txt_char(i);
		if (b == c) {
			i->buff[0] = i->buff[1] = c;
			i->buff[2] = 0;
			return(i->buff);
		} else txt_unpop(i, b);
	}
	
	if ((c=='+')||(c=='-')||(c=='*')||(c=='/')||(c=='%')||(c=='!')) {
		b = txt_char(i);
		if (b == '=') {
			i->buff[0] = c;
			i->buff[1] = b;
			i->buff[2] = 0;
			return(i->buff);
		} else txt_unpop(i, b);
	}

	i->buff[0] = c;
	i->buff[1] = 0;
	return(i->buff);
}

