
(* Mathematica Package *)

(* COPYRIGHT (C) 2010 Wynand Verwoerd 
under the terms of the GNU General Public License V3
This program is distributed in the hope that it will be useful,but WITHOUT ANY WARRANTY; 
without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  
See the GNU General Public License for more details.
*)

BeginPackage["FileReadWrite`",{"NetSplitter`"}]
(* Exported symbols added here with SymbolName::usage *)  
Nonzeroes::usage="Nonzeroes[y] gives the number of non-zero entries in vector y"
Zerows::usage="Zerows[A] gives a list of zero rows in matrix A"
Nonzerows::usage="Nonzerows[A] gives a list of non-zero rows in matrix A"
Trim::usage="Utility to trim initial and final blanks from a string"
LastWord::usage="Utility to extract the last whitespace separated word from a string"
GetInputS::usage="GetInputS calls readers for input files and digests the results"
GetExternals::usage="GetExternals[Exfile] reads external compounds"
GetTargets::usage="GetTargets[Targetfile] reads target compounds"
SubExport::usage="SubEport creates individual subnets and controls their storage in separate files"
TSVreader::usage="Reads text format stoichiometry input files" 
SBMLreader::usage="Reads stoichiometry input files in Sys Bio Markup Lang format"
TSVExport::usage="Writes text file specification for a single subnet"
SBMLExport::usage="Writes SBML file specification for a single subnet"

(* Variables only shared with the Printout package, defined here to avoid cluttering
   the list of generally shared "semi-global" variables defined in the Netsplitter` context *)
Fluxmess="";Externalmess;Targetmess;ModelNote;
RemovedRXN;RemovedComp;DropRXN;
Reversecount;Extras;

Begin["`Private`"] (* Begin Private Context *) 

Nonzeroes[y_]:=Count[y,x_/;x!= 0]

Zerows[A_]:=Flatten[Position[Map[Nonzeroes,A],x_/;x==0]]

Nonzerows[A_]:=Flatten[Position[Map[Nonzeroes,A],x_/;x!= 0]]

Trim[text_] := StringReplace[text,
	(StartOfString~~Whitespace) | (Whitespace~~EndOfString) :> ""]
      
LastWord[stringarr_]:=Map[Reverse,StringReverse[StringSplit[StringReverse[stringarr],Whitespace,2]],{Depth[stringarr]-1}]

GetInputS[]:=Module[{Extens,duplis,duplicates,duplisat={},NewIDs,Extrarows,
	MetabCompartlist,Expandedlist,ncomp,nreact,nrows,ncols,Fluxes,falsereverse,nabsent,ReadOK},Off[Remove::"remal"];
	SplitDone=False;Status="Processing the input files";StartTime=TimeUsed[];Fluxes={};Fluxmess="";
	Extens=If[ToUpperCase[Infile]=="DEMO",Infile=ToFileName[NotebookDirectory[],"Demo.tsv"];"TSV",ToUpperCase@StringDrop[Infile,{1,Last[StringPosition[Infile,"."]][[1]]}]];
	ReadOK=Switch[Extens,"TSV",TSVreader[Infile],"SBML",SBMLreader[Infile],"XML",SBMLreader[Infile],
		_,EmitSound[Crash];
			DialogInput[DialogNotebook[{TextCell["Unable to read input file \nOnly .SBML, .XML or .TSV files can be processed \nUse Browse button to find valid input."],
			Button["OK",DialogReturn[1]]}]];False];
	If[ReadOK,DialogInput[DialogNotebook[{TextCell["Successfully read ID: "<>ModelID<>
		"\nModel name: "<>ModelName<>"\n\nNotes on the model: \n"<>ModelNote,PageWidth -> 400],
			Button["OK",DialogReturn[1]]},WindowTitle->"Input Model"]];,Return[False]];

(* Compound/Reaction numbers converted to strings, embedded quotes removed *)
	Complist=ToString/@Complist;Complist=Map[StringReplace[#,"\""->""]&,Complist];
	StoichioExternals=ToString/@StoichioExternals;StoichioExternals=Map[StringReplace[#,"\""->""]&,StoichioExternals];
	CompNames=Map[StringReplace[#,"\""->""]&,CompNames];
	Reactlist=ToString/@Reactlist;Reactlist=Map[StringReplace[#,"\""->""]&,Reactlist];
	Reverselist=ToString/@Reverselist;Reverselist=Map[StringReplace[#,"\""->""]&,Reverselist];
(* Truncate long names on the compound list *)
	Complist=Replace[Complist,x_/;StringLength[x]>50:> StringTake[x,49]<>" #"(*<>ToString[n++]*),1];

(* Do a few data integrity checks *)
	ncomp=Length[Complist]; nreact=Length[Reactlist]; nrows=Dimensions[S][[1]];ncols=Dimensions[S][[2]];
	duplis=Length[Complist]-Length@DeleteDuplicates[Complist];
	If[ncomp==0||nreact==0||nrows==0||ncols==0,EmitSound[Crash];DialogInput[DialogNotebook[{TextCell[
		"Invalid data : Input file\n"<>Infile<>"\ndid not yield a metabolite list, reaction list and stoichiometry matrix."],
		Button["OK",DialogReturn[1]]}]];Return[False]];
	If[duplis > 0, duplicates = Map[#[[1]] &, Select[Tally[Complist], #[[2]] > 1 &]];
		If[Total[StringCount[duplicates, " #"]] > 1,
		(*	Revert duplicates created by truncation to the full ID *)
			duplisat = Flatten@Map[Position[Complist, #] &, duplicates];
			Complist[[duplisat]] = CompIDs[[duplisat]];
			StoichioExternals = DeleteCases[StoichioExternals, x_ /; StringMatchQ[x, CompIDs[[duplisat]]~~" "~~___]];
		(* Remove troublemakers before truncating names in the s. externals, 
			then below just CompIDS are added back for them  *)
			duplis = Length[Complist] - Length@DeleteDuplicates[Complist];]
	];
(* Truncate long names for externals as well, but avoid any duplicate found above *)
	StoichioExternals=Replace[StoichioExternals,x_/;StringLength[x]>50:>StringTake[x,49]<>" #"(*<>ToString[n++]*),1];
	StoichioExternals=Join[StoichioExternals,CompIDs[[duplisat]]];
	(* Now repeat duplicates test to check if reversion solved the problem *)
	If[duplis>0,EmitSound[Crash];
		DialogInput[DialogNotebook[{TextCell["Invalid data - Duplicate metabolite IDs: 
			\n"<>ToString@Take[Commonest[Complist],Min[3,duplis]]<>" 
			\netc. were found. Check that metabolites listed as external 
			\nin the stoichiometry file are not listed under internals as well!"],
			Button["OK",DialogReturn[1]]}]];Return[False]];
	duplis=Length[Reactlist]-Length@DeleteDuplicates[Reactlist];
	If[duplis>0,EmitSound[Crash];DialogInput[DialogNotebook[{TextCell["Invalid data - Duplicate reaction IDs: 
		\n"<>ToString@Take[Commonest[Reactlist],Min[3,duplis]]<>" 
		\netc. were found."],Button["OK",DialogReturn[1]]}]];Return[False]];
	If[ncomp!= nrows,EmitSound[Crash];DialogInput[DialogNotebook[{TextCell[
		"Invalid data : "<>ToString[ncomp]<>" metabolite names listed but "<>ToString[nrows]<>" rows in S matrix "],
		Button["OK",DialogReturn[1]]}]];Return[False]];
	If[nreact!= ncols,EmitSound[Crash];DialogInput[DialogNotebook[{TextCell[
		"Invalid data : "<>ToString[nreact]<>" reaction names listed but "<>ToString[ncols]<>" columns in S matrix "],
		Button["OK",DialogReturn[1]]}]];Return[False]];
	falsereverse=Complement[Reverselist,Reactlist];nabsent=Length[falsereverse];
	If[nabsent>0,EmitSound[Crash];DialogInput[DialogNotebook[{TextCell[
		"Invalid data - listed reversibles "<>ToString@Take[falsereverse,Min[3,nabsent]]<>" etc. not found on the full reaction list "],
		Button["OK",DialogReturn[1]]}]];Return[False]];


	Newlist={};Refusals={};Stepcount=0;RepeatMerge=False;OldMerge={};
	Exfile=If[StringLength[Exfile]== 0," ",Exfile];
	Externalmess=If[ToString[FileType[Exfile]]== "File",
		GetExternals[Exfile];
		"\n"<>ToString[Length[Newlist]]<>" external metabolites read from input file: "<>Exfile,"\nNo external metabolites input file was read."];

	RemovedRXN={};RemovedComp={};DropComps={};
	DropRXN=Position[Reactlist,x_/;StringMatchQ[ToString[x],Map[___~~#~~___&,StringSplit[Dropstring,"|"]]],{1}];
	If[DropRXN!= {},
		RemovedRXN=Reactlist[[Flatten[DropRXN]]];
		Reactlist=Delete[Reactlist,DropRXN];
		If[ReactionXML!={},ReactionXML=Delete[ReactionXML,DropRXN]];
		S=Transpose@Delete[Transpose[S],DropRXN];
		DropComps=Split@Zerows[S];
		S=Delete[S,DropComps];
		RemovedComp=Complist[[Flatten[DropComps]]];
		Complist=Delete[Complist,DropComps];
		CompIDs=Delete[CompIDs,DropComps];
		CompNames=Delete[CompNames,DropComps];
		Compartlist=Delete[Compartlist,DropComps];
		StoichioExternals=StoichioExternals\[Intersection]Complist;
		Reverselist=Reverselist\[Intersection]Reactlist];
	Reversecount=Length[Reverselist];

	Reversibles=Flatten@Map[Position[Reactlist,#]&,Reverselist];
	(*Keep a copy to use when saving subnets
	OriginalReversibles=Reversibles;OriginalReverselist=Reverselist;*)
	Fluxfile=If[StringLength[Fluxfile]== 0," ",Fluxfile];
	DoFlux=If[ToString[FileType[Fluxfile]]== "File",If[StringMatchQ[Infile,___~~"Block"~~NumberString~~"."~~__],
		EmitSound[Crash];
		ChoiceDialog[Column[{TextCell["ARE YOU SURE THE FLUX FILE SHOULD BE APPLIED?"],
			TextCell["\nThe input file name "<> Infile<>" corresponds to a subnet file created by Netsplitter in a previous run."],
			TextCell["\nIf so, flux information used in that run is already incorporated and repeating it will cause chaos."]}],
		{"Ignore flux file"->False,"Load flux file"->True},
		WindowFloating->True,ButtonBoxOptions->{ImageMargins->20}],True],False];
	Fluxmess=Fluxmess<>If[DoFlux,Fluxes=Which[
		StringCount[ToUpperCase[Fluxfile],".XLS"]== 1,Import[Fluxfile][[1]],
		StringCount[ToUpperCase[Fluxfile],".DIF"]== 1,Import[Fluxfile],
		StringCount[ToUpperCase[Fluxfile],".CSV"]== 1,Import[Fluxfile],
		StringCount[ToUpperCase[Fluxfile],".TXT"]== 1,Import[Fluxfile,"Table"],
		StringCount[ToUpperCase[Fluxfile],".TSV"]== 1,Import[Fluxfile,"Table"],
		True,{{"Invalid ","format"}}][[All,1;;2]];
	  flip=0;noflip=0;
	  If[Fluxes== {{"Invalid ","format"}},"\nWARNING - unable to process flux file format !",
	  Map[(If[MemberQ[Reactlist,#[[1]]],rno=Position[Reactlist,#[[1]]][[1,1]];
		Which[#[[2]]<0,S[[All,rno]]=-S[[All,rno]];Reversibles=DeleteCases[Reversibles,rno];flip++;,
			#[[2]]>0,Reversibles=DeleteCases[Reversibles,rno];noflip++;]])&,Fluxes];
	  Reverselist=Reactlist[[Reversibles]];
	  "\nFlux values were read from file "<>Fluxfile<>
	  "\nAs a result, "<>ToString[noflip]<>" reaction directions were confirmed and "<>
	  ToString[flip]<>" were reversed, leaving "<>
	  ToString[Length[Reversibles]]<>" as undetermined reversibles."],
	"\nNo flux values input file was read."];

CNodes=Dimensions[S][[1]];RNodes=Dimensions[S][[2]];
Externals=StoichioExternals;
Externalrows=Flatten[Table[Position[Complist,Externals[[i]]],{i,1,Length[Externals]}]];StExrows=Externalrows;

If[Convert,InputExternalrows={};OutputExternalrows={};
Inpath=DirectoryName[Infile];CurDir=Directory[];SetDirectory[Inpath];
FileTitle={ModelID<>"_Converted",ModelName<>" Converted from "<>Extens<>" by Netsplitter."};
Switch[Extens,"TSV",SBMLExport[FileTitle,Complement[Complist,StoichioExternals]],
	"SBML",TSVExport[FileTitle,Complement[Complist,StoichioExternals]]];
SetDirectory[CurDir];Return[True]];

CR=Abs[Sign[S]]-Sign[S];
RC=Transpose[Abs[Sign[S]]+Sign[S]];
(*Append duplicate rows/cols for reversible reactions *) If[ReverseThem,
RC=Join[RC,Transpose@CR[[All,Reversibles]]];CR=Transpose@Join[Transpose[CR],RC[[Reversibles]]]];
InternalComplist=Complist;InternalCNodes=CNodes;
InputExternalrows=Flatten[Position[Total[RC],0]]; OutputExternalrows=Flatten[Position[Total[Transpose[CR]],0]];
StructuralExternalrows=Union[InputExternalrows,OutputExternalrows];Externalrows=Union[Externalrows,StructuralExternalrows];

Connex=Total[Transpose[Abs[Sign[S]]]];ConnectionExternalrows=Ordering[Connex,Count[Connex,x_/;x>ConnexMax],Greater];
If[Refusals!= {},Refusedrows=Flatten[Table[Position[Complist,Refusals[[i]]],{i,1,Length[Refusals]}]];
	ConnectionExternalrows=DeleteCases[ConnectionExternalrows,x_/;MemberQ[Refusedrows,x]]];
BasicExternals=Complist[[Union[StructuralExternalrows,ConnectionExternalrows]]];
Externalrows=Union[Externalrows,ConnectionExternalrows];
Removedrows=Externalrows;Externals=Complist[[Externalrows]];

(* External compound spec (ID Compartment) from file is expanded to a list 
	by regular expression matching to the full list of Id's and compartments *)
MetabCompartlist = MapThread[StringJoin,{CompIDs,ConstantArray[" ",Length[CompIDs]],Compartlist}];
Expandedlist=Cases[MetabCompartlist, x_ /; StringMatchQ[x, RegularExpression/@Newlist]];
NoMatch=Complement[Newlist,Expandedlist,  SameTest->(StringMatchQ[#2, RegularExpression[#1]] &)];
Newlist=Expandedlist;
Expandedlist=Cases[MetabCompartlist, x_ /; StringMatchQ[x, RegularExpression/@Refusals]];
NoMatch=Join[NoMatch,Complement[Refusals,Expandedlist,  SameTest->(StringMatchQ[#2, RegularExpression[#1]] &)]];
Refusals=Expandedlist;
Newlist = Complement[Newlist, Refusals];
Extras = {}; Extrarows = {};
If[Newlist != {}, NewIDs = StringSplit[Newlist, Whitespace, 2][[All, 1]];
  Extrarows = Flatten[Table[Position[CompIDs, NewIDs[[i]]], {i, 1, Length[NewIDs]}]];
  Extrarows = Complement[Extrarows, Externalrows];
  Externalrows = Join[Externalrows, Extrarows];
  Removedrows = Externalrows; Externals = Complist[[Externalrows]]];
Extras=Complist[[Extrarows]];

(* The following code can be uncommented to force metabolites read from the externals file to
be taken as external and stop them from being reincorporated (for debugging purposes.)
This is not easily done in an existing Smatrix.TSV input file because it would require reordering the 
matrix so the newly classified externals come last. For SBML input there is no problem, simply add
"BoundaryCondition=True" to any metabolites that are forced to be external  *)

(*StoichioExternals=Union[StoichioExternals,Extras]; Extras={};*)


TargetCompounds={};Targetfile=If[StringLength[Targetfile]== 0," ",Targetfile];
Targetmess=If[ToString[FileType[Targetfile]]== "File",GetTargets[Targetfile],"\nNo target metabolites input file was read."];

Return[True]
]

GetExternals[Exfile_]:=Module[{fulltext,Nlines,ReadConfig,Configline,configno,config,
	ReadSteps,RSteps,ReadMerge,Exclusion,takeit,i},
	fulltext=Import[Exfile,"Lines"];Nlines=Length[fulltext];
(*Read the configuration from Externals file*)
	ReadConfig=False;
	Do[If[StringMatchQ[fulltext[[i]],"%%%%%%%%\tConfiguration\t%%%%%%%%"],ReadConfig=True];
		If[ReadConfig&&StringMatchQ[fulltext[[i]],"% "~~NumberString~~__],
			Configline=StringReplace[fulltext[[i]],"\t%"~~__->""];
			Configline=StringSplit[Configline,Whitespace,3];
			configno=ToExpression[Configline[[2]]];
			config=If[Length[Configline]== 3,Configline[[3]],""];
			Switch[configno,
				1,ReverseThem=ToExpression[config];,
				2,Dropstring=config;,
				3,DistChoice=ToExpression[config],
				4,LinkChoice=ToExpression[config],
				5,ConnexMax=ToExpression[config],
				6,Fluxfile=config]];
		If[StringMatchQ[fulltext[[i]],"%%%%%%%%\tChoices\t%%%%%%%%"],
	Break[]],{i,1,Nlines}];

(*Read the merge history from Externals file*)
	ReadMerge=False;
	Do[If[StringMatchQ[fulltext[[i]],"%%%%%%%%\tMerges\t%%%%%%%%"],ReadMerge=True];
		If[ReadMerge&&StringMatchQ[fulltext[[i]],"% "~~NumberString~~__],
			AppendTo[OldMerge,ToExpression[StringSplit[fulltext[[i]],"\t"][[2;;3]]]];
			If[StringMatchQ[fulltext[[i]],StartOfString~~Except["%"]~~___],Break[]]
	],{i,1,Nlines}];
	OldMerge=Map[{If[Length[#[[1]]]>0,#[[1]],{}],If[Length[#[[2]]]>0,#[[2]],{}]}&,OldMerge];
	If[OldMerge!= {},ReadSteps=Length[OldMerge];RSteps = Null;
		EmitSound[Attention];
		RepeatMerge=ChoiceDialog[Column[{
			TextCell["A merge history of " <> ToString[ReadSteps] <> " steps was read from the externals file."], 
   			TextCell["\nSteps up to the following number are reapplied to this run."], 
			InputField[Dynamic[RSteps], Number, FieldSize -> 12, 
    			FieldHint -> ToString[ReadSteps] <> "       (Click to change)", 
    			FieldHintStyle -> {Red, Opacity[1], Plain}], 
			TextCell["\nChoosing None allows externals selection to be continued.
					\nChoosing OK but with merge steps = 0 skips selection to perform a new merging cycle."]
    	  }],{"\tOK\t" -> True, "\tNone\t" -> False}, 
    	WindowFloating -> True, ButtonBoxOptions -> {ImageMargins -> 20}];
	If[ToString[RSteps] != "Null", ReadSteps = Max[0, Min[RSteps, ReadSteps]]];
	If[RepeatMerge,OldMerge=OldMerge[[1;;ReadSteps]]]
	];

(*Read the externals and refusals from Externals file.
  Keep only first 2 fields, and put .* to match any compartment if the 2nd is missing.*)
	Exclusion=False;
	For[i=1,i<= Nlines,i++,
		takeit=StringDrop[fulltext[[i]],Flatten[StringPosition[fulltext[[i]],"%"~~___,Overlaps->False],1]];
		takeit=Trim[takeit];
		If[takeit!="",takeit=StringJoin[Riffle[PadRight[StringSplit[takeit, Whitespace], 2, ".*"][[{1, 2}]], " "]]];
		If[ToUpperCase[takeit]== "<REFUSALS> .*",Exclusion=True;takeit=""];
		If[StringLength[takeit]>0,If[Exclusion,AppendTo[Refusals,takeit],AppendTo[Newlist,takeit]]]];

]

GetTargets[Targetfile_]:=Module[{fulltext,takeit,Newlist={},MetabCompartlist,Targetrows={},i},
	fulltext=Import[Targetfile,"Lines"];
For[i=1,i<= Dimensions[fulltext][[1]],i++,takeit=StringDrop[fulltext[[i]],Flatten[StringPosition[fulltext[[i]],"%"~~___,Overlaps->False],1]];
	If[takeit!="",takeit=StringJoin[Riffle[PadRight[StringSplit[takeit, Whitespace], 2, ".*"][[{1, 2}]], " "]]];
	takeit=Trim[takeit];
	If[StringLength[takeit]>0,AppendTo[Newlist,takeit]]];
	TargetIDs={};TargetCompounds={};
	If[Newlist!= {},
		MetabCompartlist = MapThread[StringJoin,{CompIDs,ConstantArray[" ",Length[CompIDs]],Compartlist}];
		Newlist=Cases[MetabCompartlist, x_ /; StringMatchQ[x, RegularExpression/@Newlist]];
		Targetrows=Flatten[Table[Position[MetabCompartlist,Newlist[[i]]],{i,1,Length[Newlist]}]]];
	TargetCompounds=Complist[[Targetrows]];TargetIDs=CompIDs[[Targetrows]];
Return["\n"<>ToString[Length[TargetCompounds]]<>" target metabolites read from input file: "<>Targetfile]
]

SubExport[]:=Module[{CurDir,IDlist,rowlist,Extlisting,Refuselisting,Configuration,DecisionHistory,MergeHistory},Status="Exporting subnetwork files";Off[DeleteDirectory::"nodir"];Off[DeleteDirectory::"dirne"];
Inpath=DirectoryName[Infile];CurDir=Directory[];SetDirectory[Inpath];
DeleteDirectory["Subnetworks",DeleteContents->True];
CreateDirectory["Subnetworks"];SetDirectory["Subnetworks"];
PreMergeExternals=Complement[PreMergeExternals,BasicExternals];
IDlist = StringSplit[PreMergeExternals, Whitespace, 2][[All, 1]];
rowlist=Flatten[Table[Position[CompIDs, IDlist[[i]]],{i,1,Length[IDlist]}]];
Extlisting =  MapThread[StringJoin, 
	{IDlist, ConstantArray[" ",Length[IDlist]], Compartlist[[rowlist]],
	 ConstantArray["\t\t% ", Length[IDlist]], CompNames[[rowlist]]}];
IDlist = StringSplit[Refusals, Whitespace, 2][[All, 1]];
rowlist=Flatten[Table[Position[CompIDs, IDlist[[i]]],{i,1,Length[IDlist]}]];
Refuselisting =  MapThread[StringJoin, 
	{Refusals, ConstantArray["\t\t% ", Length[rowlist]], CompNames[[rowlist]]}];	 
Configuration={StringJoin[{"%%%%%%%%\tConfiguration\t%%%%%%%%\n",
							"% 1 "<> ToString[ReverseThem] <>"\t%(Reversible reactions duplicated)\n",
							"% 2 "<> Dropstring<>"\t%(Reaction omission)\n",
							"% 3 "<>ToString[DistChoice]<>"\t%(Distance function)\n",
							"% 4 "<>ToString[LinkChoice]<>"\t%(Intercluster distance)\n",
							"% 5 "<>ToString[ConnexMax]<>"\t%(Maximal internal connectivity)\n",
							"% 6 "<>Fluxfile<>"\t%(Flux values path)"}]};
DecisionHistory={StringJoin[Prepend[Riffle[Map[Riffle[#,"\t"]&,
	Map[ToString,DecisionTrace,{2}]],"\n% "],"%%%%%%%%\tChoices\t%%%%%%%%\n% "]]};
MergeHistory={StringJoin[Prepend[Riffle[Map[Riffle[#,"\t"]&,
	Map[ToString,MergeTrace,{2}]],"\n% "],"%%%%%%%%\tMerges\t%%%%%%%%\n% "]]};
Export[Outfile,Join[{"%%% List of external compounds to recreate the subnetworks found from",
	"%%% Input file "<>Infile<>" calculated at "<>DateString[]},
	Configuration,DecisionHistory,MergeHistory,Extlisting,
	{"<Refusals>"},Refuselisting],{"Text","Lines"}];
If[FinalBlockCount>1,
	Blocktitle=Table[{ModelID<>"_Block_"<>ToString[i],"Subnet of "<>ModelName},{i,1,FinalBlockCount}];
	Blocktitle[[-1]]={ModelID<>"_Orphan_Block","Minor subnets of "<>ModelName};
	dest=ChoiceDialog["Which output file format?",{"SBML"->"sbml","Text file"->"tsv","Cancel"->"exit"},WindowFloating->True,ButtonBoxOptions->{ImageMargins->6}];
	Which[dest== "exit",Null,
		  dest== "tsv",MapThread[TSVExport,{Blocktitle,FinalBlocks}],
		  dest== "sbml",MapThread[SBMLExport,{Blocktitle,FinalBlocks}]];];
SetDirectory[CurDir];
]
Trim[text_] := StringReplace[text,
	(StartOfString~~Whitespace) | (Whitespace~~EndOfString) :> ""]
      
LastWord[stringarr_]:=Map[Reverse,StringReverse[StringSplit[StringReverse[stringarr],Whitespace,2]],{Depth[stringarr]-1}]

TSVreader[filepath_]:=Module[{fulltext,Splitlist,InputName,fullname},
	fulltext=StringSplit[Import[filepath,"String"],"<"~~LetterCharacter..~~">"~~Whitespace];
	InputName=ImportString[fulltext[[1]],"TSV"][[1,1]];
	InputName=Trim[InputName]; If[InputName=="",InputName="NoName"];
	fullname = StringSplit[InputName, Whitespace, 2]; 
	{ModelID, ModelName} = If[Length[fullname] == 1, {InputName, InputName}, fullname];
	ModelNote="TSV input file - no notes.";
	FuncDefs = {};UnitDefs = {};CompartTypes = {};SpecieTypes = {};
	Parameters = {};InitAss = {};Rules = {};Constraints = {};Events = {};
	ReactionXML = {}; SpeciesXML={};
	Reactlist=Flatten[ImportString[fulltext[[2]],"TSV"]];
	Reverselist=Flatten[ImportString[fulltext[[3]],"TSV"]];
	ReactNames=ConstantArray["NoName",Length[Reactlist]];
	Complist=Flatten[ImportString[fulltext[[4]],"TSV"]];
	Complist=Map[StringReplace[#, {(StartOfString ~~Whitespace) | (Whitespace ~~ EndOfString) :>  "",Whitespace->" "}]&,Complist];
	StoichioExternals=Flatten[ImportString[fulltext[[5]],"TSV"]];
	StoichioExternals=Map[StringReplace[#, {(StartOfString ~~Whitespace) | (Whitespace ~~ EndOfString) :>  "",Whitespace->" "}]&,StoichioExternals];
	Complist=Join[Complist,StoichioExternals];

(*Splitlist=Transpose@LastWord[Complist];*)
	Splitlist=Transpose@StringSplit[Complist,Whitespace,2];
	{CompIDs,Compartlist}=If[Length[Splitlist]== 1,{Splitlist[[1]],ConstantArray["SYSTEM",Length[Splitlist[[1]]]]},Splitlist];
	CompartIDs=Union[Compartlist];
	CompartOuters=ConstantArray["Missing",Length[CompartIDs]];
	CompartSizes=ConstantArray["1.0",Length[CompartIDs]];
	CompartUnits=ConstantArray["volume",Length[CompartIDs]];
	CompartConstant=ConstantArray["Missing",Length[CompartIDs]];
	CompartDims=ConstantArray["3",Length[CompartIDs]];
	If[Length[CompartIDs]<Length[Compartlist]/2,CompNames=ConstantArray["NoName",Length[CompIDs]],
		CompNames=Compartlist;Compartlist=ConstantArray["SYSTEM",Length[CompIDs]];CompartIDs={"SYSTEM"};CompartOuters={"Missing"};];

	S=ImportString[fulltext[[6]],"MTX"];
	Remove[fulltext];True
	]

TabsepnBlock[Namelist_]:=StringJoin[Riffle[StringJoin[Riffle[#,"\t"]]&/@Partition[Namelist,5,5,1,{}],"\n"]]

Reconcile[AllItems_,WithAttributes_,Default_:None]:=Module[{flag=1,attribute},
	Map[(attribute=Default;
		Which[flag>Length[WithAttributes],
			Null,MatchQ[WithAttributes[[flag]],{#,_}],
			attribute=WithAttributes[[flag,2]];flag++;Exit;,
			True,Null];attribute)&,AllItems]
	]

TSVExport[Title_,Blocklisting_]:=Module[{BlockIntRows,BlockIntCols,BlockRevCols,BlockRows,BlockExtRows,
	BlockStoi,Inflows,Outflows,Crossflows,Reactlisting,Reverselisting,Intlisting,Extlisting,
	Smatlisting,Outfile,i},
If[Length[Blocklisting]>0,
	BlockIntRows=Flatten[Table[Position[Complist,Blocklisting[[i]]],{i,1,Length[Blocklisting]}]];
	BlockIntCols=Nonzerows[Normal[Transpose[S[[BlockIntRows]]]]];
	BlockRevCols=BlockIntCols\[Intersection]Reversibles;
	BlockRows=Nonzerows[Normal[S[[All,BlockIntCols]]]];
	BlockExtRows=Complement[BlockRows,BlockIntRows];
	Inflows=BlockExtRows\[Intersection]InputExternalrows;
	Outflows=BlockExtRows\[Intersection]OutputExternalrows;
	Crossflows=Complement[BlockExtRows,Inflows\[Union]Outflows];
	BlockExtRows=Join[Inflows,Crossflows,Outflows];
	BlockRows=Join[BlockIntRows,BlockExtRows];
	BlockStoi=S[[BlockRows,BlockIntCols]];
	Reactlisting=TabsepnBlock[Reactlist[[BlockIntCols]]];
	Reverselisting=TabsepnBlock[Reactlist[[BlockRevCols]]];
	Intlisting=TabsepnBlock[Complist[[BlockIntRows]]];
	Extlisting=TabsepnBlock[Complist[[BlockExtRows]]];
	Smatlisting=ExportString[SparseArray[BlockStoi],"MTX"]; 
	Outfile=Title[[1]]<>".tsv";
	Export[Outfile,{"<Title>",Title[[1]]<>" : "<>Title[[2]],"<Reactions>",Reactlisting,"<ReversibleReactions>",Reverselisting,
		"<InternalCompounds>",Intlisting,"<ExternalCompounds>",Extlisting,
		"<Stoichiometry>",Smatlisting},"List"];
]]

SBMLreader[filepath_]:=Module[{InputXML,NoteXML,ExIDs,ExNames,Srules,SBMLerror,
	FuncDefXML,UnitDefXML,CompartTypeXML,SpecieTypeXML,ParameterXML,
	InitAssXML,RulesXML,ConstraintsXML,EventsXML,Fluxes,LowLims,UpLims,BoundFluxes,rno,flip=0,noflip=0},
	
	InputXML=Import[filepath,"XML"];
	{{ModelID, ModelName}} = Cases[InputXML,XMLElement["model", {___,"id" -> modid_, ___,"name" -> modname_, ___}, _] :> {modid, modname}, Infinity];
	NoteXML = Cases[InputXML, XMLElement["model",_,{XMLElement["notes",{},{note_}],__}]:>note, Infinity];
	NoteXML = If[NoteXML == {}, "No notes read from SBML", NoteXML[[1]]];
	ModelNote = If[StringQ[NoteXML], NoteXML, StringJoin@Riffle[Flatten@Cases[NoteXML, XMLElement["p", _, parag_] :> parag, Infinity],"\n"]];
(* Note: global definitions appear at nesting level 5 in InputXML. 
   Using Infinity will pick up local Parameter declarations in reactions (level 11) as well.*)
	FuncDefXML = Cases[InputXML, XMLElement["listOfFunctionDefinitions", _, _], 5];
	FuncDefs = XMLtoText[FuncDefXML];
	UnitDefXML = Cases[InputXML, XMLElement["listOfUnitDefinitions", _, _], 5];
	UnitDefs = XMLtoText[UnitDefXML];
	CompartTypeXML = Cases[InputXML, XMLElement["listOfCompartmentTypes", _, _], 5];
	CompartTypes = XMLtoText[CompartTypeXML];
	SpecieTypeXML = Cases[InputXML, XMLElement["listOfSpeciesTypes", _, _], 5];
	SpecieTypes = XMLtoText[SpecieTypeXML];
	ParameterXML = Cases[InputXML, XMLElement["listOfParameters", _, _], 5];
	Parameters = XMLtoText[ParameterXML];
	InitAssXML = Cases[InputXML, XMLElement["listOfInitialAssignments", _, _], 5];
	InitAss = XMLtoText[InitAssXML];
	RulesXML = Cases[InputXML, XMLElement["listOfRules", _, _], 5];
	Rules = XMLtoText[RulesXML];
	ConstraintsXML = Cases[InputXML, XMLElement["listOfConstraints", _, _], 5];
	Constraints = XMLtoText[ConstraintsXML];
	EventsXML = Cases[InputXML, XMLElement["listOfEvents", _, _], 5];
	Events = XMLtoText[EventsXML];

	CompartIDs=Cases[InputXML,XMLElement["compartment",{___,"id"->compartid_,___},{}]:> compartid,Infinity];
	CompartOuters=Cases[InputXML,XMLElement["compartment",{___,"id"->compartid_,___,"outside"->outid_,___},{}]:> {compartid,outid},Infinity];
	CompartOuters=Reconcile[CompartIDs,CompartOuters,"Missing"];
	CompartDims=Cases[InputXML,XMLElement["compartment",{___,"id"->compartid_,___,"spatialDimensions"->outid_,___},{}]:> {compartid,outid},Infinity];
	CompartDims=Reconcile[CompartIDs,CompartDims,"3"];
	CompartSizes=Cases[InputXML,XMLElement["compartment",{___,"id"->compartid_,___,"size"->outid_,___},{}]:> {compartid,outid},Infinity];
	CompartSizes=Reconcile[CompartIDs,CompartSizes,"Missing"];
	CompartUnits=Cases[InputXML,XMLElement["compartment",{___,"id"->compartid_,___,"units"->outid_,___},{}]:> {compartid,outid},Infinity];
	CompartUnits=Reconcile[CompartIDs,CompartUnits,"Missing"];
	CompartConstant=Cases[InputXML,XMLElement["compartment",{___,"id"->compartid_,___,"constant"->outid_,___},{}]:> {compartid,outid},Infinity];
	CompartConstant=Reconcile[CompartIDs,CompartConstant,"Missing"];

	ReactionXML = Cases[InputXML, XMLElement["reaction", _, _], Infinity];
	Reactlist=Cases[ReactionXML,XMLElement["reaction",{___,"id"->reactid_,___},{___}]:>reactid,Infinity];
	Reverselist=Complement[Reactlist,Cases[ReactionXML,XMLElement["reaction",{___,"id"->reactid_,___,"reversible"->"false",___},{___}]:>reactid,Infinity]];
	ReactNames=Cases[ReactionXML,XMLElement["reaction",{___,"id"->reactid_,___,"name"->reactname_,___},{}]:> {reactid,reactname},Infinity];
	ReactNames=Reconcile[Reactlist,ReactNames,"NoName"];

    Fluxes = Cases[ReactionXML,
		XMLElement["reaction", {___, "id" -> reactid_, ___}, {__, 
     		XMLElement["kineticLaw", {}, {___, 
				XMLElement["listOfParameters", {}, {___, 
					XMLElement["parameter", {___,"id"->"FLUX"|"FLUX_VALUE"|"FLUXVALUE",___,"value"->fluxval_,___},{}],
		 ___}],___}],___}] :> {reactid,ToExpression[fluxval]}, 1];
	LowLims = Cases[ReactionXML,
		XMLElement["reaction",{___,"id"->reactid_,___}, {__, 
     		XMLElement["kineticLaw",{},{___, 
				XMLElement["listOfParameters",{},{___, 
					XMLElement["parameter",{___,"id"->"LOWERBOUND"|"LOWER_BOUND",___,"value"->fluxval_,___},{}],
		___}],___}],___}] :> {reactid,ToExpression[fluxval]},1];
	UpLims = Cases[ReactionXML,
		XMLElement["reaction",{___,"id"->reactid_,___}, {__, 
     		XMLElement["kineticLaw",{},{___, 
				XMLElement["listOfParameters",{},{___, 
					XMLElement["parameter",{___,"id"->"UPPERBOUND"|"UPPER_BOUND",___,"value"->fluxval_,___},{}],
		___}],___}],___}] :> {reactid,ToExpression[fluxval]},1];
	Fluxes = Map[MapAt[Sign, #, 2] &, DeleteCases[Fluxes, {_, 0}]];
	BoundFluxes = If[Length[UpLims] == Length[LowLims], 
		MapThread[{#1[[1]], Sign[Sign[#1[[2]]] + Sign[#2[[2]]]]} &, {UpLims,LowLims}], {}];
	BoundFluxes = DeleteCases[BoundFluxes, {_, 0}];
	Fluxes = Union[Fluxes, BoundFluxes];
	
	SpeciesXML = Cases[InputXML, XMLElement["species", _, _], Infinity];
	CompIDs=Cases[SpeciesXML,XMLElement["species",{___,"id"->compid_,___},{}]:> compid,Infinity];
	CompNames = Cases[SpeciesXML, XMLElement["species", {___,"id"->compid_,___, "name" -> compname_, ___}, {}] :> {compid,compname}, Infinity];
	CompNames=Reconcile[CompIDs,CompNames,"NoName"];
	Compartlist = Cases[SpeciesXML, XMLElement["species", {___,"id"->compid_,___, "compartment" -> compname_, ___}, {}] :> {compid,compname}, Infinity];
	If[Compartlist!={},Compartlist=Reconcile[CompIDs,Compartlist,"SYSTEM"]];

	If[CompIDs[[1]] == "DUMMY",
		SpeciesXML = Rest[SpeciesXML]; CompIDs = Rest[CompIDs]; CompNames = Rest[CompNames]; 
  		If[Compartlist != {}, Compartlist = Rest[Compartlist];]];
  	
	SBMLerror=Catch[Srules=Flatten[Map[ReactionParse,ReactionXML],1];];
	If[SBMLerror,Return[False]];
	S=SparseArray[Srules];
	If[Fluxes != {},Reversibles = Flatten@Map[Position[Reactlist, #] &, Reverselist]; 
 		Map[(If[MemberQ[Reactlist, #[[1]]], rno = Position[Reactlist, #[[1]]][[1, 1]];
		Which[#[[2]] < 0, S[[All, rno]] = -S[[All, rno]]; Reversibles = DeleteCases[Reversibles, rno]; flip++;,
			  #[[2]] > 0, Reversibles = DeleteCases[Reversibles, rno]; noflip++;]])&, Fluxes];
		Reverselist = Reactlist[[Reversibles]];
	Fluxmess = 	"\nFluxes and/or Bounds were read from the SBML input file." <> 
				"\nAs a result, " <> ToString[noflip] <> " reaction directions were confirmed and " <>
				 ToString[flip] <> " were reversed, leaving " <> ToString[Length[Reversibles]] <> 
   				" as undetermined reversibles."];
	
	ExIDs=Cases[InputXML,XMLElement["species",{___,"id"->compid_,___,"boundaryCondition"->"true",___},{}]:> compid,Infinity];
	ExNames = Cases[InputXML, 
		XMLElement["species",{___,"id"->compid_,___,"name"->compname_,___,"boundaryCondition"->"true",___},{}] | 
     	XMLElement["species",{___,"id"->compid_,___,"boundaryCondition"->"true",___,"name"->compname_,___},{}] :>{compid, compname}, Infinity];
	ExNames = Reconcile[ExIDs, ExNames, "NoName"];
	If[CompIDs!= CompNames,
		StoichioExternals=MapThread[StringJoin,{ExIDs,ConstantArray[" ",Length[ExIDs]],ExNames}];
		Complist=MapThread[StringJoin,{CompIDs,ConstantArray[" ",Length[CompIDs]],CompNames}],
		StoichioExternals=ExIDs; Complist=CompIDs];
	Complist=Map[StringReplace[#, {(StartOfString ~~Whitespace) | (Whitespace ~~ EndOfString) :>  "",Whitespace->" "}]&,Complist];
	StoichioExternals=Map[StringReplace[#, {(StartOfString ~~Whitespace) | (Whitespace ~~ EndOfString) :>  "",Whitespace->" "}]&,StoichioExternals];
Remove[InputXML];True]


XMLtoText[XML_] := 
 If[XML == {}, {}, 
  Flatten@Map[StringReplace[StringSplit[StringReplace[ExportString[#, "XML"], 
        ">\r\n" -> ">\[DownArrow]"], "\[DownArrow]"], "\r\n" -> ""] &,
     XML]]
     
ReactionParse[RxnSBML_]:=Module[{Scol,SrowIn,SrowOut,reaction},
	Scol=RxnSBML/.XMLElement["reaction",{___,"id"->rname_,___},_]:>Position[Reactlist,rname,1][[1,1]];
	reaction=Reactlist[[Scol]];
	SrowIn=Cases[Cases[RxnSBML,XMLElement["listOfReactants",_,_],{2}],
		XMLElement["speciesReference", {___,"species" -> spec_,___,"stoichiometry" -> stoich_, ___}, {}] | 
 		XMLElement["speciesReference", {___, "species" -> spec_, ___}, {}]:>
			{Quiet@Check[Position[CompIDs,spec,1][[1,1]],Squeal[spec,reaction];Throw[True],Part::partw],
				N@If[stoich== "",1,ToExpression[stoich]]},{3}];
	SrowOut=Cases[Cases[RxnSBML,XMLElement["listOfProducts",_,_],{2}],
		XMLElement["speciesReference", {___,"species" -> spec_,___,"stoichiometry" -> stoich_, ___}, {}] | 
 		XMLElement["speciesReference", {___, "species" -> spec_, ___}, {}]:>
			 {Quiet@Check[Position[CompIDs,spec,1][[1,1]],Squeal[spec,reaction];Throw[True],Part::partw],N@If[stoich== "",1,ToExpression[stoich]]},{3}];
	Join[Map[{#[[1]],Scol}->-#[[2]]&,SrowIn],Map[{#[[1]],Scol}->#[[2]]&,SrowOut]]]

Squeal[spec_,rname_]:=DialogInput[DialogNotebook[{TextCell[Style[
		"Inconsistent data - reading abandoned.\n\nMetabolite "<>spec<>"\nin reaction "<>rname<>
		"\nnot found in SBML listofSpecies section ! ",Purple,"Panel"]],
		Button["OK",DialogReturn[1]]},WindowTitle->"SBML Error"]];

ReactantSBML[i_]:=Table[
	"<speciesReference species=\"" ~~Reactions[[i,2,j,1]]~~ "\""~~ 
	" stoichiometry=\""~~ ToString[Reactions[[i,2,j,2]]]~~ "\" />",{j,1,Length[ Reactions[[i,2]]]}];

ProductSBML[i_]:=Table[
	"<speciesReference species=\"" ~~Reactions[[i,3,j,1]]~~ "\""~~ 
	" stoichiometry=\""~~ ToString[Reactions[[i,3,j,2]]]~~ "\" />",{j,1,Length[ Reactions[[i,3]]]}];

PutAttribute[xml_, element_, attribute_, ValueSet_] := 
 Module[{XML = DeleteCases[xml, (attribute -> _), 3]}, 
  Do[XML[[i]]=XML[[i]] /. 
     XMLElement[element, {tags___}, specs___] -> 
      XMLElement[element, {tags, Rule[attribute, ValueSet[[i]]]}, specs],
      	{i, Length[XML]}]; XML]

SBMLfix[compids_] := Module[{fixed, duplis},
  (*SBML rules only allow letter,digit or understroke,no digit at start of ID*)
  fixed = StringReplace[compids, 
  	{"-" -> "_", "+" -> "x", Except[WordCharacter] .. -> "_",
      StartOfString ~~ s : Except[LetterCharacter | "_"] -> "_" ~~ s}];
  duplis = Length[fixed] - Length@DeleteDuplicates[fixed];
  If[duplis > 0, 
   DialogInput[DialogNotebook[{TextCell[
       "Forced conformation to SBML ID naming rules \ncaused duplicate ID's:\n" <> 
        ToString@Take[Commonest[fixed], Min[3, duplis]] <> 
       "\nManual editing of ID's required to fix this."], 
      Button["OK", DialogReturn[1]]}]]];
  fixed]
   
SBMLExport[Title_,Blocklisting_]:=Module[{BlockIntRows,BlockIntCols,BlockRows,BlockExtRows,
	Inflows,Outflows,Crossflows,BlockStoi,Header,
	BlockCompartIDs,BlockCompartOuters,BlockCompartDims,BlockCompartConstant,BlockCompartSize,BlockCompartUnit,
	CompartmentSBL,Speclist,IntIDs, ExtIDs,BlockBC,BlockSpeclist, SpeciesSBML, BlockReactlist,BlockReactNames,
	SBMLReverselist,BlockRev,Stoicol,Reacts,Prods,ReactionSBML,Outfile,i},
	Outfile=StringJoin[StringSplit[Title][[1;;2]],".sbml"];
	Outfile=Title[[1]]<>".sbml";
	If[Length[Blocklisting]>0,
		BlockIntRows=Flatten[Table[Position[Complist,Blocklisting[[i]]],{i,1,Length[Blocklisting]}]];
		BlockIntCols=Nonzerows[Normal[Transpose[S[[BlockIntRows]]]]];
		BlockRows=Nonzerows[Normal[S[[All,BlockIntCols]]]];
		BlockExtRows=Complement[BlockRows,BlockIntRows];
		Inflows=BlockExtRows\[Intersection]InputExternalrows;
		Outflows=BlockExtRows\[Intersection]OutputExternalrows;
		Crossflows=Complement[BlockExtRows,Inflows\[Union]Outflows];
		BlockExtRows=Join[Inflows,Crossflows,Outflows];
		BlockRows=Join[BlockIntRows,BlockExtRows];
		BlockStoi=S[[BlockRows,BlockIntCols]];
		Header={"<?xml version=\"1.0\" encoding=\"UTF-8\"?> <sbml ",
			"xmlns=\"http://www.sbml.org/sbml/level2\"",
			"xmlns:sbml=\"http://www.sbml.org/sbml/level2\" version=\"4\" level=\"2\"",
			"xmlns:math=\"http://www.w3.org/1998/Math/MathML\"",
			"xmlns:html=\"http://www.w3.org/1999/xhtml\"   >",
			"<model id=\"" ~~Title[[1]]~~ "\" name=\"" ~~Title[[2]] ~~"\" >",
			"<notes>Metabolic subnetwork extracted by NetSplitter on "~~DateString[]~~
			".\nStoichiometry file processed is: " ~~Infile ~~ "</notes>"};
		BlockCompartIDs=Union[Compartlist[[BlockRows]]];
		BlockCompartOuters=CompartOuters[[Flatten[Table[Position[CompartIDs,
			BlockCompartIDs[[i]]],{i,1,Length[BlockCompartIDs]}]]]];
		BlockCompartDims=CompartDims[[Flatten[Table[Position[CompartIDs,
			BlockCompartIDs[[i]]],{i,1,Length[BlockCompartIDs]}]]]];
		BlockCompartSize=CompartSizes[[Flatten[Table[Position[CompartIDs,
			BlockCompartIDs[[i]]],{i,1,Length[BlockCompartIDs]}]]]];
		BlockCompartUnit=CompartUnits[[Flatten[Table[Position[CompartIDs,
			BlockCompartIDs[[i]]],{i,1,Length[BlockCompartIDs]}]]]];
		BlockCompartConstant=CompartConstant[[Flatten[Table[Position[CompartIDs,
			BlockCompartIDs[[i]]],{i,1,Length[BlockCompartIDs]}]]]];
		CompartmentSBL=Table["<compartment id=\""~~ BlockCompartIDs[[i]]~~
			"\" spatialDimensions=\""~~BlockCompartDims[[i]]~~"\""~~
			If[BlockCompartOuters[[i]]== "Missing",""," outside=\""~~ BlockCompartOuters[[i]]~~"\""]~~
			If[BlockCompartSize[[i]]== "Missing",""," size=\""~~BlockCompartSize[[i]]~~"\""]~~
			If[BlockCompartUnit[[i]]== "Missing",""," units=\""~~BlockCompartUnit[[i]]~~"\""]~~
			If[BlockCompartConstant[[i]]== "Missing",""," constant=\""~~BlockCompartConstant[[i]]~~"\""]~~
			" />",{i,1,Length[BlockCompartIDs]}];

	  If[SpeciesXML == {},
		IntIDs = SBMLfix[CompIDs[[BlockIntRows]]]; ExtIDs = SBMLfix[CompIDs[[BlockExtRows]]];
		Speclist = Join[IntIDs, ExtIDs];
(*Dummy is a workaround for CellnetAnalyzer bug, 
	name field in first record not read and upsets internal/external recognition *)
		SpeciesSBML= Join[{"<species id=\"DUMMY\" compartment=\"SYSTEM\" name=\"Dummy\" />"}, 
			Table["<species id=\""~~IntIDs[[i]]~~
				"\" compartment=\""~~Compartlist[[BlockIntRows[[i]]]]~~
				"\" boundaryCondition=\"false\""~~ 
				If[CompNames[[BlockIntRows[[i]]]]=="NoName", "", 
					" name=\""~~CompNames[[BlockIntRows[[i]]]]~~"\""]~~
					" />", {i, 1, Length[BlockIntRows]}], 
			Table["<species id=\""~~ExtIDs[[i]]~~
				"\" compartment=\""~~Compartlist[[BlockExtRows[[i]]]]~~
				"\" boundaryCondition=\"true\""~~ 
				If[CompNames[[BlockExtRows[[i]]]]=="NoName", "", 
					" name=\""~~CompNames[[BlockExtRows[[i]]]]~~"\""]~~
					" />", {i, 1, Length[BlockExtRows]}]];,

		(*Else, the full SBML specification of each species as read from an SBML input file is
			transmitted, except that BoundaryCondition is adjusted according to 
			external/internal status used by Netsplitter*)
		BlockBC=Join[ConstantArray["false",Length[BlockIntRows]],ConstantArray["true",Length[BlockExtRows]]];
		BlockSpeclist=PutAttribute[SpeciesXML[[BlockRows]], "species", "boundaryCondition", BlockBC];
		SpeciesSBML=XMLtoText[BlockSpeclist];
		  ];
		  
	  If[ReactionXML == {}, 
	  	BlockReactNames=ReactNames[[BlockIntCols]];
 		BlockReactlist = SBMLfix[Reactlist[[BlockIntCols]]]; 
 		SBMLReverselist = SBMLfix[Reverselist];
		BlockReactlist = Table[{BlockReactlist[[i]], 
			If[Count[SBMLReverselist, BlockReactlist[[i]]] > 0, True, False]},
			 {i, 1, Length[BlockReactlist]}];
 		Stoicol=(Transpose@Normal@BlockStoi);
		Reacts=Map[Flatten[Position[#,x_/;x<0]]&,Stoicol];
		Prods=Map[Flatten[Position[#,x_/;x>0]]&,Stoicol];
		Reactions=Table[{BlockReactlist[[i]],
			Transpose[{Speclist[[Reacts[[i]]]],-Stoicol[[i,Reacts[[i]]]]}],
			Transpose[{Speclist[[Prods[[i]]]],Stoicol[[i,Prods[[i]]]]}]},{i,1,Length[BlockReactlist]}];
		ReactionSBML=Flatten@Table[Join[{"<reaction id=\""~~ Reactions[[i,1,1]]~~"\"" ~~
			If[ Reactions[[i,1,2]],""," reversible=\"false\" "]~~
			If[BlockReactNames[[i]]=="NoName", ""," name=\"" ~~ BlockReactNames[[i]]~~"\""] ~~
			" >"},
			{"<listOfReactants>"},ReactantSBML[i],
			{"</listOfReactants>","<listOfProducts>"},ProductSBML[i],
			{"</listOfProducts>","</reaction>"}],{i,1,Length[Reactions]}];,
		
	(*Else, the full SBML specification of each reaction as read from an SBML input file is
	transmitted, except that reversibility is adjusted according to that used by Netsplitter*)
		BlockRev=Table[If[Count[Reversibles,BlockIntCols[[i]]]>0,"true","false"],{i,1,Length[BlockIntCols]}];
		BlockReactlist=PutAttribute[ReactionXML[[BlockIntCols]], "reaction", "reversible", BlockRev];
		ReactionSBML=XMLtoText[BlockReactlist];
			
	   ];
	Export[Outfile,Join[Header,FuncDefs,UnitDefs,CompartTypes,SpecieTypes,
		{"<listOfCompartments>"},CompartmentSBL,{"</listOfCompartments>",
		"<listOfSpecies>"},SpeciesSBML,{"</listOfSpecies>"},
		Parameters,InitAss,Rules,Constraints,
		{"<listOfReactions>"},ReactionSBML,{"</listOfReactions>"},Events,{"</model>","</sbml>"}],"List"];

]]

End[] (* End Private Context *)

EndPackage[]
