package Blastmap;

use strict;
use Carp;
use Cwd;
use Tk::widgets qw(Dialog);
use Tk;
use Tk::BrowseEntry;
use Bio::SearchIO;
use GQ_comm;
use vars qw( @ISA $AUTOLOAD );
use subs qw();


sub log {
	my $dbg = 0;
	return unless $dbg;
	open CHECK, ">>/tmp/blastoutput.txt" or die "can't open output file";
	print CHECK @_;
	close CHECK;
}

BEGIN {
    {
	my %_attr_data =	#     DEFAULT    ACCESSIBILITY
	  ( -exportselection => [1,         "read/write"],
	    #-background      => ["#bbeeff", "read/write"],
	    -foreground      => ["#000000", "read/write"],
	    -height          => [30,        "read/write"],
	    -width           => [60,        "read/write"],
	    -relief          => ["sunken",  "read/write"],
	    -takefocus       => [1,         "read/write"],
	    -wrap            => ['word',    "read/write"],
	    -scrollbars		=> ['se', 		"read/write"],
	    BlastDirectory => [undef,			"read/write"],
	    BlastBinaries => [undef,			"read/write"],
	    SequenceText    => [undef,     "read/write"],
	    BlastResultText => [undef,     "read/write"],
	    BlastResultDetails => [undef,     "read/write"],
	    FreeTextFrame   => [undef, 		"read/write"],
	    BlastParsedFrame   => [undef, 		"read/write"],
	    BlastRawFrame   => [undef, 		"read/write"],
	    Top				=> [undef, 		"read/write"],
	    LFrame          => [undef, 		"read/write"],
	    RFrame          => [undef, 		"read/write"],
	    BlastResults		=> [{}, 		'read/write'], # holds a list of feature objects coming from the BPLite parse of Blast
	  );


	# Is a specified object attribute accessible in a given mode
	sub _accessible  {
	    my ($self, $attr, $mode) = @_;
	    return $_attr_data{$attr}[1] =~ /$mode/
	}

	# Classwide default value for a specified object attribute
	sub _default_for {
	    my ($self, $attr) = @_;
	    return $_attr_data{$attr}[0]
	}

	# List of names of all specified object attributes
	# A hash so names are not repeated
	sub _standard_keys { keys %_attr_data }

	use subs (keys %_attr_data)
    }
} #end of BEGIN block

sub new {
    my ($caller, %args) = @_;
    my $caller_is_obj = ref($caller);
    my $class = $caller_is_obj || $caller;

    my $top = MainWindow->new(-title => "Blast Mapper");
    my $self=bless {},$class;

    foreach my $attrname ( $self->_standard_keys() ) {
    	if (exists $args{$attrname}) {
    	    $self->{$attrname} = $args{$attrname} }
    	elsif ($caller_is_obj) {
    	    $self->{$attrname} = $caller->{$attrname} }
    	else {
    	    $self->{$attrname} = $self->_default_for($attrname) }
    }

	die "blast binaries not found" unless $self->BlastBinaries;
	die "working directory not found" unless $self->BlastDirectory;
	
	my $BD = $self->BlastDirectory;
	my $BB = $self->BlastBinaries;
	
	my $char;
	if ($BD=~/\\/){$char="\\"}
	elsif ($BD=~/\//){$char="/"}
	else {die "can't determine operating system path delimiter from available path information"}
	
	$self->{"_delimiter"} = $char;


	unless ($BD =~ /$char$/){$BD .= $char};
	unless ($BB =~ /$char$/){$BB .= $char};
	
	$self->BlastBinaries($BB);  # paths now fixed with / or \ at end
	$self->BlastDirectory($BD);
	
	$self->Top($top);

    my $tframe = $self->Top->Frame->pack(-side => 'top', -expand => '1', -fill => 'both');
    my $bframe = $self->Top->Frame->pack(-side => 'bottom', -expand => '1', -fill => 'both');

    my $tlframe = $tframe->Frame->pack(-side => 'left');
    my $trframe = $tframe->Frame->pack(-side => 'right');
	
	my $tllframe = $tlframe->Frame->pack(-side => 'left');
    my $tlrframe = $tlframe->Frame->pack(-side => 'right');

    my $lframe = $bframe->Frame->pack(-side => 'left', -anchor => 'nw',-expand => '1', -fill => 'both');
    my $rframe = $bframe->Frame->pack(-side => 'right', -anchor => 'ne',-expand => '1', -fill => 'both');

    my $parsed_result_frame = $rframe ->Frame->pack(-side => 'top', -anchor => 'ne',-expand => '1', -fill => 'both');
    my $raw_result_frame = $rframe->Frame->pack(-side => 'top', -anchor => 'se',-expand => '1', -fill => 'both');

    $self->FreeTextFrame($lframe);
    $self->BlastParsedFrame($parsed_result_frame);
    $self->BlastRawFrame($raw_result_frame);

    $tllframe->Label(-text => "copy/paste your sequence data here")->pack(-side => "top", -expand => "1", -fill => "x");
    $tllframe->Button(-text => "Create Database", -command => sub {$self->CreateDatabase()})->pack(-side => "top");
	$tllframe->Button(-text => "BLAST", -command => sub {$self->Blast()})->pack(-side => "top");

    $tlrframe->Label(-text => "Blast Cutoff")->pack(-side => "top", -expand => "1", -fill => "x")->pack(-side => "left");
    my $Cutoff = $tlrframe->BrowseEntry(-width => 5, -listwidth => 25, -background => '#ffffff', -relief=>'sunken', -variable=>\$self->{cutoff})->pack(-side => "left");
    $tlrframe->Label(-text => "Blast Type")->pack(-side => "top", -expand => "1", -fill => "x")->pack(-side => "left");
    my $BlastType = $tlrframe->BrowseEntry(-width => 8, -listwidth => 40, -background => '#ffffff', -relief=>'sunken', -variable=>\$self->{type})->pack(-side => "left");

    $Cutoff->insert(1, "10-40"); $self->{cutoff} = "10-40";
	$Cutoff->insert(1, "10-30");
	$Cutoff->insert(1, "10-20");
	$Cutoff->insert(1, "10-10");
	$Cutoff->insert(1, "10-9");
	$Cutoff->insert(1, "10-8");
	$Cutoff->insert(1, "10-7");
	$Cutoff->insert(1, "10-6");
	$Cutoff->insert(1, "10-5");
	$Cutoff->insert(1, "10-4");
	$Cutoff->insert(1, "10-3");
	$Cutoff->insert(1, "10-2");
	$Cutoff->insert(1, "10-1");
	$Cutoff->insert(1, "1");
	$Cutoff->insert(1, "10");
	

    $BlastType->insert(1, "tblastx"); $self->{type} = "blastn";
    $BlastType->insert(1, "blastn");
	$BlastType->insert(1, "tblastn");

    $trframe->Label(-text => "Blast Results:  Single click to light up hit on chromosome map\nDouble click to open Annotation Canvas of that hit")->pack(-side => "top", -expand => "1", -fill => "x");

    $self->SequenceText($self->FreeTextFrame->Scrolled('Text', -height => 20, -width => 50, -background => "#FFFFFF")->pack(-side => 'top', -expand => '1', -fill => 'both' ));
    $self->BlastResultText($self->BlastParsedFrame->Scrolled('Text', -height => 10, -width => 30, -background => "#bbeeff")->pack(-side => 'top', -expand => '1', -fill => 'both' ));
    $self->BlastResultDetails($self->BlastRawFrame->Scrolled('Text', -wrap => 'none', -background => "#eeeeee", -height => 5, -width => 30, -scrollbars => "s")->pack(-side => 'top', -expand => '1', -fill => 'x' ));

	$SIG{'INT'}=sub {$self->Quit};
    $SIG{'TERM'}=sub {$self->Quit};

	$top->OnDestroy(sub {Quit()});
    return $self;
}

sub Quit {
	print "<KILLED>";
	close STDOUT;
	close STDIN;
	exit 1;
}


sub _showRawDetails {
	my ($self, $hsp) = @_;
	my $BlastDetails = $self->BlastResultDetails;
	my $query = $hsp->query_string;
	my $subj = $hsp->hit_string;
	my $compar = $hsp->homology_string;
	$BlastDetails->delete("1.0", "end");
	$BlastDetails->insert("1.0", "       " . $hsp->start('query') . "\n");
	$BlastDetails->insert("2.0", "Query: $query\n       $compar\nSbjct: $subj\n");
	$BlastDetails->insert("5.0", "       " . $hsp->start('sbjct'));
}


sub Blast {
	my ($self) = @_;
	my $seq = $self->SequenceText->get("1.0", "end");
	$seq =~ s/[\t ]//g;
	$seq =~ s/\n\n/\n/g;
	#$seq =~ s/^\n//g;
	return unless ($seq =~ /[ATCG]/);  # leave if it is empty or has just N's
	
	my $alphabet = $self->_guess_alphabet($seq);  # nicked from BioPerl...
	return 0 if ($alphabet eq "0");
    $self->BlastResultText->insert("end", "molecule type $alphabet");
	
	if ($alphabet eq "protein"){$self->{type} = "tblastn"}  # can only tblastn a protein sequence
	elsif ($self->{type} eq "tblastn"){   # can not tblastn a non-protein sequence
		$self->BlastResultText->delete("1.0", "end");
		$self->BlastResultText->insert("end", "Can't tblastn a nucleotide sequence.  Please chose another Blast method");
		return 0
	}
		
	
	if (!($seq =~ />\w+/)){$seq = ">no_name_given\n" . $seq;}  # add fasta haeader to query
	$seq =~ s/\n\n/\n/g;

	my $version = $self->getVersion;
	my $orgname = $self->getOrganism;
	$orgname =~ s/\s+//g;
	$orgname =~ s/\W//g;

    my $txtOutput = $self->BlastResultText;
	my $db_name = $orgname . "_version_" . $version;      # create the databse name from the organism and version number

	$txtOutput->delete("1.0", 'end');
	$txtOutput->insert('end', "Blasting against $db_name");
	$self->Top->Busy;
	my $BD = $self->BlastDirectory;
	my $char = $self->{"_delimiter"};
	
	my $WD = $BD . "$db_name$char";

	open OUT, ">$BD"."WB_Local_Blast.fas" or die "can't open fasta file $BD"."WB_Local+Blast.fas for writing";
	print OUT $seq;
	close OUT;

	my $BlastDir = $self->BlastBinaries;

	system "$BlastDir" . "blastall -p " . $self->{type} . " -i $BD"."WB_Local_Blast.fas -o $BD"."WB_Local_Blast.blast -d $WD$db_name.fas";

	$self->Top->Unbusy;
	
	#open IN, "$BD"."WB_Local_Blast.blast" or die "\n\ncan't open blast results because $!\n\n";

    my $blastObj = Bio::SearchIO->new(-file=>"$BD"."WB_Local_Blast.blast", -format => 'blast'); # use BioPerl BPLite parsing
    return 0 unless $blastObj;

    unlink "$BD"."WB_Local_Blast.blast";	# comment this line to prevent removal of the blast report
											# but it is sometimes useful to keep them for troubleshooting
    my $count;

    $self->{cutoff} =~ /10\-(\d+)/;
    my $cutoff = 0.1 ** $1;


	if ($txtOutput) {
		$txtOutput->delete("1.0", 'end');
		
		$txtOutput->insert('end', "Blast Results\n");     # write it to the text window

		while (my $result = $blastObj->next_result()){
			while (my $hit = $result->next_hit){
				my $seqname =  $hit->name;
				$seqname =~ s/\s$//g;
				$txtOutput->tagBind($seqname, "<Double-Button-1>" =>
						sub{&log("Button clicked\n");
							$self->lightContig($seqname);
							$self->openContig($seqname);
							$self->mapHits($hit, $seqname);
							});
				my $hitflag;
				while (my $hsp = $hit->next_hsp){
		
					next if ($hsp->evalue > $cutoff);#
					unless($hitflag){$txtOutput->insert('end', "\nhit on contig $seqname\n", [$seqname])}
					$hitflag = 1;
					my $score = $hsp->score;          # the raw score
					my $start = $hsp->hit->start;
					my $end = $hsp->hit->end;
					my $hsp_tag = "$start|$end";
					$txtOutput->insert('end', "\tstart $start\tend $end\tscore $score\n", [$seqname, $hsp_tag]);
					$txtOutput->tagBind($hsp_tag, "<Button-1>" =>
							sub{$self->_showRawDetails($hsp);
								$self->lightContig($seqname);
								}
					);
				}
			}
		}

		++$count;
		
	} #end of if $txtOutput
	if ($count == 0) {
		$txtOutput->delete("1.0", "end");
		$txtOutput->insert("end", "*** No Hits Found ***\nusing your cutoff parameters");
	}
	#close IN;

}


sub _guess_alphabet {
   my ($self, $str) = @_;
   my ($str2,$total,$atgc,$u,$type);

   $str =~ s/\-\.\?//g;

   $total = length($str);
   if( $total == 0 ) {
       $self->BlastResultText->insert("end", "cannot guess alphabet [$str]");
	   return 0 ;
   }

   $str2 = $str;

   $str2 =~ s/[ATGCNatgcn]//g;
   $atgc = $total - length $str2;
   $str = $str2;
   $str2 =~ s/[Uu]//g;

   $u = length($str) - length($str2);


   if( ($atgc / $total) > 0.85 ) {
       $type = 'dna';
   } elsif( (($atgc + $u) / $total) > 0.85 ) {
       $type = 'rna';
   } else {
       $type = 'protein';
   }

   return $type;

}

sub mapHits {
	my ($self, $hit, $blastcontig) = @_;
	
	$hit->{_iterator} = 0;
	my $result = &askGenquire("getActiveSequences","", $self->Top); # contig should have been opened by now
		
	return unless ($result =~ /contig\sid/);
	#&log("result1 $result\n\n");
	my @contigs = ($result =~ /(<contig.*?\/contig>)/g);
	foreach my $contigXML(@contigs){
		#&log("contig1 $contigXML\n\n");
		my $contig = (($contigXML =~ /<contig id\=\'(.*?)\'>/) && $1);
		&log("\n\ncontig name $contig\n\n");
		next unless ($contig =~ /$blastcontig/);  # must be the same contig
		next unless ($contigXML =~ /<start>\s?1\s?<\/start>/);  # must start at 1
		while (my $hsp = $hit->next_hsp){
			my $output = "<contig>$contig</contig>";
			my $strand = ($hsp->hit->strand =~ /\-/)?"-":"+"; 
			my $frame = ".";
				
			$output .="<start>".($hsp->hit->start)."</start>";
			$output .="<end>".($hsp->hit->end)."</end>";
			$output .="<feature>BlastHit</feature>";
			$output .="<source>Blast</source>";
			$output .="<strand>$strand</strand>";
			$output .="<frame>$frame</frame>";
			$output .="<score>".($hsp->score)."</score>";
			$output .="<attributes></attributes>";
	
			&log("output $output\n\n");
			&askGenquire("mapFeature", $output, $self->Top);
		}
		
	}
}

 
sub lightContig {
	my ($self, $seqname) = @_;
	my $string = "<contig>$seqname</contig>"; # this is a single element of a potential XML list
	my $ok = askGenquire("selectContigs", $string, $self->Top);
	return $ok
}

sub openContig {
	my ($self, $seqname) = @_;
	my $string = "<contig>$seqname</contig><start></start><stop></stop>"; # start and stop have no value because we want to open the entire thing
	my $ok = askGenquire("openContig", $string, $self->Top);
	return $ok
}

sub getOrganism {
	my ($self) = @_;
	my $name = askGenquire("getOrganismName","", $self->Top);
	return $name;
}

sub getVersion {
	my ($self) = @_;
	my $version = askGenquire("getSequenceVersion","", $self->Top);
	return $version;
}

sub getContigIDs{
	my ($self) = @_;
	my $contigs = askGenquire("getContigIDs", "", $self->Top);
	# $contigs has the format
	# <contig>blahblah</contig><contig>blahblah</contig>...
	my @contigs = ($contigs =~ /<contig>\n?\n?(.*?)\n?\n?<\/contig>/g);
	my $txt = $self->BlastResultText;

	return @contigs;
	
}

sub getUnSequence {
	my ($self, $name) = @_;
	my $seq_answer = askGenquire("getFullContigSequence",$name, $self->Top);  # this had to change to get teh hits to map properly.
	# hits will  be redundant using this method, but that can't be helped.
	my $sequence = (($seq_answer =~ /<seq>(.*?)<\/seq>/) && $1);	
	return $sequence;
}

sub CreateDatabase {
    my ($self) = @_;

	my $confirmDlg= $self->Top->DialogBox(-title => 'Really Create Database?',
    					 -default_button=> 'No',
    					 -buttons => [qw/Yes No/]
    					);
	$confirmDlg->add("Label",
			 text => "Creating a database can take several minutes or longer.\nThis should only be done if you do not have a current version of the Blast database for the chosen organism.\n\nDo you wish to continue?"
			)->pack;
	my $continue=$confirmDlg->Show;
	return if $continue eq 'No';


    my $WD = $self->BlastDirectory;
	my $char = $self->{"_delimiter"};
	
	($WD) or die "WD returned * $WD *\n";
    my $version = $self->getVersion;
    my $orgname = $self->getOrganism;
    $orgname =~ s/\s+//g;
    $orgname =~ s/\W//g;

    my $db_name = $orgname . "_version_" . $version;

    my $ORIG_DIR = cwd;

    chdir $WD or warn "can't change to working directory $!; this is a big problem!!\n";
    my $success = mkdir $db_name, 0777;
    unless ($success){$self->FAILURE("Can't create $WD$db_name organism directory because $!.\nAny data existing in this folder will now be overwritten")}
	
    chdir "$db_name" or die "can't change to organism directory $!\n";

	my $text = $self->BlastResultText;
    $text->delete("1.0", "end");
	$text->insert("end", "Querying Database for sequences (slow!)\n");
	$text->update;
    open FASTA, ">$db_name.fas" or die "cant open the output FASTA file for database creation.  Aborted.";
	my @contigs = $self->getContigIDs;
	#exit 0;
	foreach my $contig(@contigs){
		$text->insert("end", ".");
		$text->see("end");
		$text->update;
		my $thisseq = $self->getUnSequence($contig);
		
		print FASTA ">$contig\n$thisseq\n\n\n";	
	}
	close FASTA;
    my $BLASTBIN = $self->BlastBinaries;
    $text->delete("1.0", "end");
	$text->insert("end", "Running Blast FormatDB program\n");
    system "$BLASTBIN" . "formatdb -t $db_name -i $db_name.fas -p F -o T";
    $text->delete("1.0", "end");
	$text->insert("end", "Blast Database of $db_name Has Been Created"); 

    chdir $ORIG_DIR or warn "couldn't return to original working folder.  This may be a problem...";

}

sub FAILURE {
	my ($self, $message) = @_;
	my $txt = $self->SequenceText;
	$txt->delete('1.0', 'end');
	$txt->insert('end', $message);
	return;
}


sub AUTOLOAD {
    no strict "refs";
    my ($self, $newval) = @_;

    $AUTOLOAD =~ /.*::(\w+)/;

    my $attr=$1;
    if ($self->_accessible($attr,'write')) {

	*{$AUTOLOAD} = sub {
	    if (defined $_[1]) { $_[0]->{$attr} = $_[1] }
	    return $_[0]->{$attr};
	};    ### end of created subroutine

###  this is called first time only
	if (defined $newval) {
	    $self->{$attr} = $newval
	}
	return $self->{$attr};

    } elsif ($self->_accessible($attr,'read')) {

	*{$AUTOLOAD} = sub {
	    return $_[0]->{$attr} }; ### end of created subroutine
	return $self->{$attr}  }


    # Must have been a mistake then...
    croak "No such method: $AUTOLOAD";
}

1;
