package GQ::Client::Utilities;
use strict;
use HTTP::Request::Common qw(POST GET);
use Bio::Tools::Blast;
use Bio::SearchIO;

require Exporter;
use vars qw(@ISA @EXPORT); #keep 'use strict' happy
@ISA = qw(Exporter); 
@EXPORT = qw(sort_by_start sort_by_start_features CGI_Blast _addSlash);

=pod

=head sort_by_start

  Title   : sort_by_start
  Usage   : $sortedListRef=sort_by_start($MappedSeq,$listref);
  Returns : 2 lists:
             1)  a reference to the sorted list of FeatureIDs, and
             2)  a reference to the list of Bio::SeqFeatureI compliant objects,
                 sorted in order of sequence, either forward or reverse
  Args    : $MappedSeq, a Bio::SeqCanvas object, and
            $listref, a list of FID***** Feature IDs that you want sorted
  

=cut

sub sort_by_start {
    my ($MapObj,$list) = @_;
    my $hash=$MapObj->getFeaturesWithTag($list);   #get the hash of features to be sorted
    my ($dir,@sortedFIDs,@sortedfeatures);            #indexed by FID number

    foreach (keys %$hash) {
	$dir=$hash->{$_}->strand;    #get strand
	last if $dir;              # of first feature - assume all on the same strand!
    }
    if ($dir == 1) { #forward strand	
	@sortedFIDs=sort { $hash->{$a}->start <=> $hash->{$b}->start } keys %$hash;  	
    } elsif ($dir == -1) { #reverse strand	
	@sortedFIDs=sort { $hash->{$b}->end <=> $hash->{$a}->end} keys %$hash;		
    }
    foreach (@sortedFIDs) {
	push @sortedfeatures,$hash->{$_};
    }

    return (\@sortedFIDs,\@sortedfeatures);
}

sub sort_by_start_features {
    my ($MapObj,$hash) = @_;   #get the hash of features to be sorted
    my ($dir,@sortedFIDs,@sortedfeatures);            #indexed by FID number

    foreach (keys %$hash) {
	$dir=$hash->{$_}->strand;    #get strand
	last if $dir;              # of first feature - assume all on the same strand!
    }
    if ($dir == 1) { #forward strand	
	@sortedFIDs=sort { $hash->{$a}->start <=> $hash->{$b}->start } keys %$hash;  	
    } elsif ($dir == -1) { #reverse strand	
	@sortedFIDs=sort { $hash->{$b}->end <=> $hash->{$a}->end} keys %$hash;		
    }
    foreach (@sortedFIDs) {
	push @sortedfeatures,$hash->{$_};
    }

    return (\@sortedFIDs,\@sortedfeatures);
}

=head CGI_blast

  Title   : CGI_blast
  Usage   : $BlastObj=$self->CGI_blast($DatabaseID, $sequence, $source_tag);
  Returns : a BioPerl SearchIO Object
  Args    : the database id of the sequence you are blasting.
                        the sequence (plain text, no newlines, no header)
            the source_tag of the sequence you are blasting.
            Of these, only the sequence must be "accurate", the other
            two are only used to build up a fasta header for the Blast input
  Comments: This routine sets up a LWP net connection to the address specified
            for $BLAST_URL in your wb.conf file.   The CGI **must** return a properly
            formatted Blast response - anything else will kill the parser
            
=cut

sub CGI_blast {

    my ($self, $DB_ID, $seq, $source_tag)=@_;
    my ($flag, $seqin, $seqout, $seqobj);
    my ($count)=0;
    my $txtOutput = $self->TextOutput;
    my $tmpdir = $self->TEMP_DIR;
    my $BLAST_URL = $self->BLAST_URL;
	my $BLAST_CONFIG = $self->BLAST_CONFIG;

    $seq =~ s/\n//g;                                         # get rid of newlines
	$seq =~ s/([^\n]{70})/$1\n/g;
    $seq = ">FEATURE_INDEX_$DB_ID\_$source_tag\n" . $seq;    # add a FASTA header
    my $ua = LWP::UserAgent->new();                          # prepare for net connection

	my %CGI_params;
	foreach my $param(keys %{$BLAST_CONFIG}){
		my ($cgi, $def) = @{$BLAST_CONFIG->{$param}};
		if ($param eq '-i'){  # this is the input sequence, which we deal with separately 
			$CGI_params{$cgi} = $seq;  # special case
			next;
		} else {
			next unless ($cgi && $def);	# get rid of the undefs, i.e. the parameters that this CGI instance doesn't understand
			$CGI_params{$cgi} = $def;	# normal case
		}
	}


 ################################################################
 #         Make Your Local Blast Modifications to the line below
 ################################################################
	
    my $req = POST $BLAST_URL, [%CGI_params]; # prepare net command line

 ################################################################
 ################################################################
 ################################################################


    
    my $content = $ua->request($req);             # execute HTTP call and receive respone

	unless ($content->is_success){
		$self->lblSysMess->configure(-text => 'failure or timeout on CGI request');
		$self->lblSysMess->update;
		return 0;
	}
    $content = $content->as_string;
    my @content = split /\n/, $content;                      # split into an array
    open OUT, ">$tmpdir/blastout" or die "can't open outfile $!";       # write to a temp file
    my $line;
    
    my $parsed = 0;   # used as a flag to get rid of the HTML headers
    while ($parsed == 0){
		$line = shift @content;  # we are SHIFTING the content - ie ditching it!
		last unless ($line || scalar(@content));  # if there is no line data, and no additional lines, so we have reached the end of file without success
		if ($line =~ /^BLAST\w/){
			print OUT "$line\n"; 
			$parsed = 1; 
			last
		}     # throw away CGI headers etc. up to this line (the first line of a Blast header...)
	}                                                            # eg. BLASTX

	unless ($parsed) {  # if the parse failed, then abort mission
		close OUT;
		return 0;
	}
	

    foreach my $line(@content){print OUT "$line\n"}  # otherwise print the rest of the file
    close OUT;
	
    #my $blastObj = Bio::Tools::Blast->new(-file => "$tmpdir/blastout" , -parse => '1', -signif => '1e-2', -strict => "1"); # use BioPerl Blast parsing
	
	my $blastObj = Bio::SearchIO->new(-file => "$tmpdir/blastout" , -format => 'blast'); # use BioPerl Blast parsing
        
    return 0 unless $blastObj;

	my $result;
    #unlink "$tmpdir/blastout";                              # uncomment this line to remove the old blast reports
                                                             # but it is sometimes useful to keep them for troubleshooting
    if ($txtOutput) {
        $txtOutput->delete("1.0", 'end');

        my $BlastScreenReport = "NCBI gi\t Significance \cI Description\n";  # this is a simplistic BLAST report formatter
        $BlastScreenReport .=   "-------\t ------------ \cI -----------\n";  # for quick viewing in the text window

		$result = $blastObj->next_result();  # get the one and only result
		if ($result){
			while (my $hit = $result->next_hit){
				my $hsp = $hit->next_hsp;
				my (@elements) = split /\|/, $hit->name;       # the hit name looks like gi|1273464|gb|blahblah
				my $gi = $elements[1];                         # so break it on the pipe and take individual gi element
				
				unless ($gi){									# if Blast is done with -I=F flag, then fail this report
					$self->lblSysMess->configure(-text => 'Report lacks "gi" field. Parse Failed!');
					return 0;
				}
							 
				$BlastScreenReport .= "$gi\cI " . $hsp->evalue . "   \cI " . $hit->description . "\n\n";
				$hit->{'_iterator'} = 0;
			}	
			$txtOutput->insert('end', $BlastScreenReport);     # write it to the text window
		}
    }
    return ($result);  # pass it back to the databse for parsing
}


sub _addSlash {
    my ($string) = @_;
    if ($string =~ /\//){#unix style paths
	if (!($string =~ /\/$/)){$string .= '/'}
    } elsif ($string =~ /\\/){#windows style paths
	if (!($string =~ /\\$/)){$string .= "\\"}
    }

    return $string;
}


1;



