(************** Content-type: application/mathematica **************
                     CreatedBy='Mathematica 5.0'

                    Mathematica-Compatible Notebook

This notebook can be used with any Mathematica-compatible
application, such as Mathematica, MathReader or Publicon. The data
for the notebook starts with the line containing stars above.

To get the notebook into a Mathematica-compatible application, do
one of the following:

* Save the data starting with the line of stars above into a file
  with a name ending in .nb, then open the file inside the
  application;

* Copy the data starting with the line of stars above to the
  clipboard, then use the Paste menu command inside the application.

Data for notebooks contains only printable 7-bit ASCII and can be
sent directly in email or through ftp in text mode.  Newlines can be
CR, LF or CRLF (Unix, Macintosh or MS-DOS style).

NOTE: If you modify the data for this notebook not in a Mathematica-
compatible application, you must delete the line below containing
the word CacheID, otherwise Mathematica-compatible applications may
try to use invalid cache data.

For more information on notebooks and Mathematica-compatible 
applications, contact Wolfram Research:
  web: http://www.wolfram.com
  email: info@wolfram.com
  phone: +1-217-398-0700 (U.S.)

Notebook reader applications are available free of charge from 
Wolfram Research.
*******************************************************************)

(*CacheID: 232*)


(*NotebookFileLineBreakTest
NotebookFileLineBreakTest*)
(*NotebookOptionsPosition[     70198,       2359]*)
(*NotebookOutlinePosition[     71107,       2390]*)
(*  CellTagsIndexPosition[     71018,       2384]*)
(*WindowFrame->Normal*)



Notebook[{
Cell[TextData[StyleBox["Analyzing a metabolic network of moderate \
complexity.",
  FontSize->18,
  FontSlant->"Italic"]], "Title"],

Cell[CellGroupData[{

Cell["\<\
Make sure you are in the SNA tutorials directory, before loading \
SNAsym.\
\>", "Subsubsection",
  Evaluatable->False,
  TextJustification->1,
  FontFamily->"Times",
  FontSize->14,
  FontWeight->"Plain",
  FontSlant->"Italic",
  FontVariations->{"CompatibilityType"->0}],

Cell[CellGroupData[{

Cell[BoxData[
    \(\(<< "\<../mathcode/SNAsym.m\>";\)\)], "Input"],

Cell[BoxData[
    \(LinkObject["../mathcode/../pairelvs/pairelvsIFsh", 2, 2]\)], "Print"]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["\<\
The following reads in the network, acknowledges the source and \
displays some basic information about the network.\
\>", "Subsubsection",
  Evaluatable->False,
  TextJustification->1,
  FontFamily->"Times",
  FontSize->14,
  FontWeight->"Plain",
  FontSlant->"Italic",
  FontVariations->{"CompatibilityType"->0}],

Cell[CellGroupData[{

Cell[BoxData[{
    \(\({name, 
          humit}\  = \ << \((SNApath <> \
"\<tutorials/HumanMitochondria.m\>")\);\)\), "\n", 
    \(name\)}], "Input"],

Cell[BoxData[
    \("Metabolic network of human cardiac mitochondria adapted from\n\n    \
Vo, T.D., Greenberg, H.J., and Palsson, B.O.\n    Journal of Biological \
Chemistry, 2004.\n\nOnly reactions involving mitochondrial metabolites are \
included\nin the present network."\)], "Output"]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
    \(reactions@humit\  // Length\)], "Input"],

Cell[BoxData[
    \(123\)], "Output"]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
    \(metabolites@humit\)], "Input"],

Cell[BoxData[
    \({"10fthf"\^"m", "12dgr_m"\^"m", "2mop"\^"m", "34hpp"\^"m", 
      "3hacoa"\^"m", "3oacoa"\^"m", "5aop"\^"m", "5fthf"\^"m", "aacoa"\^"m", 
      "acac"\^"m", "accoa"\^"m", "adp"\^"m", "aglyc3p_m"\^"m", "ahcys"\^"m", 
      "akg"\^"m", "amet"\^"m", "amp"\^"m", "arg-L"\^"m", "ascb-L"\^"m", 
      "asp-L"\^"m", "atp"\^"m", "c204coa"\^"m", "c204crn"\^"m", 
      "c226coa"\^"m", "c226crn"\^"m", "cbp"\^"m", "cdp"\^"m", 
      "cdpdag_m"\^"m", "cdpea"\^"m", "chol"\^"m", "cit"\^"m", "citr-L"\^"m", 
      "clpn_m"\^"m", "cmp"\^"m", "co2"\^"m", "coa"\^"m", "creat"\^"m", 
      "crn"\^"m", "ctp"\^"m", "dgmp"\^"m", "dgsn"\^"m", "dhap"\^"m", 
      "dhdascb"\^"m", "dhor-S"\^"m", "dtmp"\^"m", "dump"\^"m", "dutp"\^"m", 
      "facoa_m"\^"m", "fad"\^"m", "fadh2"\^"m", "fe2"\^"m", "ficytC"\^"m", 
      "focytC"\^"m", "for"\^"m", "fum"\^"m", "gdp"\^"m", "glu-L"\^"m", 
      "gly"\^"m", "glyc"\^"m", "glyc3p"\^"m", "gthox"\^"m", "gthrd"\^"m", 
      "gtp"\^"m", "gudac"\^"m", "h"\^"m", "h2co3"\^"m", "h2o"\^"m", 
      "h2o2"\^"m", "hco3"\^"m", "hmgcoa"\^"m", "icit"\^"m", "idp"\^"m", 
      "itp"\^"m", "lys-L"\^"m", "mal-L"\^"m", "methf"\^"m", "mlthf"\^"m", 
      "mmcoa-S"\^"m", "nad"\^"m", "nadh"\^"m", "nadp"\^"m", "nadph"\^"m", 
      "nh4"\^"m", "o2"\^"m", "o2-"\^"m", "oaa"\^"m", "ocdycacoa"\^"m", 
      "ocdycrn"\^"m", "odecoa"\^"m", "odecrn"\^"m", "orn"\^"m", "orot"\^"m", 
      "pa_m"\^"m", "pc_m"\^"m", "pcreat"\^"m", "pe_m"\^"m", "pep"\^"m", 
      "pg_m"\^"m", "pgp_m"\^"m", "pheme"\^"m", "pi"\^"m", "pmtcoa"\^"m", 
      "pmtcrn"\^"m", "ppcoa"\^"m", "ppi"\^"m", "ppp9"\^"m", "pppg9"\^"m", 
      "ps_m"\^"m", "pyr"\^"m", "q10"\^"m", "q10h2"\^"m", "ser-L"\^"m", 
      "stcoa"\^"m", "stcrn"\^"m", "succ"\^"m", "succoa"\^"m", "thf"\^"m", 
      "thymd"\^"m", "tyr-L"\^"m", "udp"\^"m", "5aop"\_Xt\%\("c"\), 
      "adp"\_Xt\%\("c"\), "akg"\_Xt\%\("c"\), "arg-L"\_Xt\%\("c"\), 
      "asp-L"\_Xt\%\("c"\), "atp"\_Xt\%\("c"\), "c204crn"\_Xt\%\("c"\), 
      "c226crn"\_Xt\%\("c"\), "cdp"\_Xt\%\("c"\), "cit"\_Xt\%\("c"\), 
      "citr-L"\_Xt\%\("c"\), "co2"\_Xt\%\("c"\), "coa"\_Xt\%\("c"\), 
      "crn"\_Xt\%\("c"\), "fe2"\_Xt\%\("c"\), "fum"\_Xt\%\("c"\), 
      "gdp"\_Xt\%\("c"\), "glu-L"\_Xt\%\("c"\), "gly"\_Xt\%\("c"\), 
      "glyc"\_Xt\%\("c"\), "glyc3p"\_Xt\%\("c"\), "gthrd"\_Xt\%\("c"\), 
      "gtp"\_Xt\%\("c"\), "h"\_Xt\%\("c"\), "h2o"\_Xt\%\("c"\), 
      "lys-L"\_Xt\%\("c"\), "mal-L"\_Xt\%\("c"\), "nh4"\_Xt\%\("c"\), 
      "o2"\_Xt\%\("c"\), "ocdycrn"\_Xt\%\("c"\), "odecrn"\_Xt\%\("c"\), 
      "orn"\_Xt\%\("c"\), "pc_m"\_Xt\%\("c"\), "pep"\_Xt\%\("c"\), 
      "pheme"\_Xt\%\("e"\), "pi"\_Xt\%\("c"\), "PL"\_Xt\%\("e"\), 
      "pmtcrn"\_Xt\%\("c"\), "pppg9"\_Xt\%\("c"\), "ps_m"\_Xt\%\("c"\), 
      "pyr"\_Xt\%\("c"\), "stcrn"\_Xt\%\("c"\), "succ"\_Xt\%\("c"\), 
      "udp"\_Xt\%\("c"\)}\)], "Output"]
}, Open  ]],

Cell[TextData[{
  "Note that mitochondrial metabolites have the superscript  ",
  StyleBox["\"m\"",
    FontWeight->"Bold",
    FontSlant->"Plain",
    FontVariations->{"CompatibilityType"->0}],
  " ,  metabolites in the cytosol have ",
  StyleBox["\"c\"",
    FontWeight->"Bold",
    FontSlant->"Plain",
    FontVariations->{"CompatibilityType"->0}],
  " , and extracellular metabolites ",
  StyleBox["\"e\"",
    FontWeight->"Bold",
    FontSlant->"Plain",
    FontVariations->{"CompatibilityType"->0}],
  " .   Mitochondrial metabolites are treated as internal, whereas the others \
are external."
}], "Text",
  Evaluatable->False,
  FontFamily->"Times",
  FontSize->14,
  FontWeight->"Plain",
  FontSlant->"Italic"]
}, Open  ]],

Cell[CellGroupData[{

Cell["\<\
While listing all 123 reactions yields a result which is still just \
about readable,  this is a good occasion to show how to pick  parts of a \
network. 
The following lists all reaction involving  asp-L.\
\>", "Subsubsection",
  Evaluatable->False,
  TextJustification->1,
  FontFamily->"Times",
  FontSize->14,
  FontWeight->"Plain",
  FontSlant->"Italic",
  FontVariations->{"CompatibilityType"->0}],

Cell[CellGroupData[{

Cell[BoxData[
    \(with["\<asp-L\>", \ reactions@humit]\)], "Input"],

Cell[BoxData[
    RowBox[{"{", 
      RowBox[{
        RowBox[{\("akg"\^"m" + "asp-L"\^"m"\), 
          StyleBox["\[Equilibrium]",
            FontSize->18], \("glu-L"\^"m" + "oaa"\^"m"\)}], ",", 
        RowBox[{\("asp-L"\^"m" + "glu-L"\_Xt\%\("c"\) + "h"\_Xt\%\("c"\)\), 
          StyleBox["\[Equilibrium]",
            FontSize->18], \("glu-L"\^"m" + "h"\^"m" + 
            "asp-L"\_Xt\%\("c"\)\)}]}], "}"}]], "Output"]
}, Open  ]],

Cell["And this, all reactions involving asp-L or glu-L.", "Text",
  Evaluatable->False,
  FontFamily->"Times",
  FontSize->14,
  FontWeight->"Plain",
  FontSlant->"Italic"],

Cell[CellGroupData[{

Cell[BoxData[
    \(with[{"\<asp-L\>", "\<glu-L\>"}, reactions@humit]\ \  // 
      TableForm\)], "Input"],

Cell[BoxData[
    InterpretationBox[GridBox[{
          {
            RowBox[{\("akg"\^"m" + "asp-L"\^"m"\), 
              StyleBox["\[Equilibrium]",
                FontSize->18], \("glu-L"\^"m" + "oaa"\^"m"\)}]},
          {
            RowBox[{\("akg"\^"m" + "tyr-L"\^"m"\), 
              StyleBox["\[Equilibrium]",
                FontSize->18], \("34hpp"\^"m" + "glu-L"\^"m"\)}]},
          {
            RowBox[{\("glu-L"\_Xt\%\("c"\) + "h"\_Xt\%\("c"\)\), 
              StyleBox["\[Equilibrium]",
                FontSize->18], \("glu-L"\^"m" + "h"\^"m"\)}]},
          {
            
            RowBox[{\("asp-L"\^"m" + "glu-L"\_Xt\%\("c"\) + 
                "h"\_Xt\%\("c"\)\), 
              StyleBox["\[Equilibrium]",
                FontSize->18], \("glu-L"\^"m" + "h"\^"m" + 
                "asp-L"\_Xt\%\("c"\)\)}]}
          },
        RowSpacings->1,
        ColumnSpacings->3,
        RowAlignments->Baseline,
        ColumnAlignments->{Left}],
      TableForm[ {
        Equilibrium[ 
          Plus[ 
            meta[ Int, "akg", "m"], 
            meta[ Int, "asp-L", "m"]], 
          Plus[ 
            meta[ Int, "glu-L", "m"], 
            meta[ Int, "oaa", "m"]]], 
        Equilibrium[ 
          Plus[ 
            meta[ Int, "akg", "m"], 
            meta[ Int, "tyr-L", "m"]], 
          Plus[ 
            meta[ Int, "34hpp", "m"], 
            meta[ Int, "glu-L", "m"]]], 
        Equilibrium[ 
          Plus[ 
            meta[ Xt, "glu-L", "c"], 
            meta[ Xt, "h", "c"]], 
          Plus[ 
            meta[ Int, "glu-L", "m"], 
            meta[ Int, "h", "m"]]], 
        Equilibrium[ 
          Plus[ 
            meta[ Int, "asp-L", "m"], 
            meta[ Xt, "glu-L", "c"], 
            meta[ Xt, "h", "c"]], 
          Plus[ 
            meta[ Int, "glu-L", "m"], 
            meta[ Int, "h", "m"], 
            meta[ Xt, "asp-L", "c"]]]}]]], "Output"]
}, Open  ]],

Cell[TextData[{
  StyleBox["with ",
    FontWeight->"Bold",
    FontSlant->"Plain",
    FontVariations->{"CompatibilityType"->0}],
  "does not only work for reactions, but for lists in general."
}], "Text",
  Evaluatable->False,
  FontFamily->"Times",
  FontSize->14,
  FontWeight->"Plain",
  FontSlant->"Italic"],

Cell[CellGroupData[{

Cell[BoxData[
    \(with[Xt, \ metabolites@humit]\)], "Input"],

Cell[BoxData[
    \({"5aop"\_Xt\%\("c"\), "adp"\_Xt\%\("c"\), "akg"\_Xt\%\("c"\), 
      "arg-L"\_Xt\%\("c"\), "asp-L"\_Xt\%\("c"\), "atp"\_Xt\%\("c"\), 
      "c204crn"\_Xt\%\("c"\), "c226crn"\_Xt\%\("c"\), "cdp"\_Xt\%\("c"\), 
      "cit"\_Xt\%\("c"\), "citr-L"\_Xt\%\("c"\), "co2"\_Xt\%\("c"\), 
      "coa"\_Xt\%\("c"\), "crn"\_Xt\%\("c"\), "fe2"\_Xt\%\("c"\), 
      "fum"\_Xt\%\("c"\), "gdp"\_Xt\%\("c"\), "glu-L"\_Xt\%\("c"\), 
      "gly"\_Xt\%\("c"\), "glyc"\_Xt\%\("c"\), "glyc3p"\_Xt\%\("c"\), 
      "gthrd"\_Xt\%\("c"\), "gtp"\_Xt\%\("c"\), "h"\_Xt\%\("c"\), 
      "h2o"\_Xt\%\("c"\), "lys-L"\_Xt\%\("c"\), "mal-L"\_Xt\%\("c"\), 
      "nh4"\_Xt\%\("c"\), "o2"\_Xt\%\("c"\), "ocdycrn"\_Xt\%\("c"\), 
      "odecrn"\_Xt\%\("c"\), "orn"\_Xt\%\("c"\), "pc_m"\_Xt\%\("c"\), 
      "pep"\_Xt\%\("c"\), "pheme"\_Xt\%\("e"\), "pi"\_Xt\%\("c"\), 
      "PL"\_Xt\%\("e"\), "pmtcrn"\_Xt\%\("c"\), "pppg9"\_Xt\%\("c"\), 
      "ps_m"\_Xt\%\("c"\), "pyr"\_Xt\%\("c"\), "stcrn"\_Xt\%\("c"\), 
      "succ"\_Xt\%\("c"\), "udp"\_Xt\%\("c"\)}\)], "Output"]
}, Open  ]],

Cell[TextData[{
  "yields all external metabolites.\nIf you need the list indices of the \
elements and not  elements themselves,  ",
  StyleBox["iwith",
    FontWeight->"Bold",
    FontSlant->"Plain",
    FontVariations->{"CompatibilityType"->0}],
  " can be used instead of ",
  StyleBox["with",
    FontWeight->"Bold",
    FontSlant->"Plain",
    FontVariations->{"CompatibilityType"->0}],
  ". In fact the definition of  ",
  StyleBox["with",
    FontWeight->"Bold",
    FontSlant->"Plain",
    FontVariations->{"CompatibilityType"->0}],
  "  is just: "
}], "Text",
  Evaluatable->False,
  FontFamily->"Times",
  FontSize->14,
  FontWeight->"Plain",
  FontSlant->"Italic"],

Cell[CellGroupData[{

Cell[BoxData[
    \(\(?with\)\)], "Input"],

Cell["SNAsym`with", "Print",
  CellTags->"Info3340454019-2206978"],

Cell[BoxData[
    InterpretationBox[GridBox[{
          {GridBox[{
                {\(with[pat_, l_List] := 
                    l\[LeftDoubleBracket]iwith[pat, 
                        l]\[RightDoubleBracket]\)}
                },
              GridBaseline->{Baseline, {1, 1}},
              ColumnWidths->0.999,
              ColumnAlignments->{Left}]}
          },
        GridBaseline->{Baseline, {1, 1}},
        ColumnAlignments->{Left}],
      Definition[ "with"],
      Editable->False]], "Print",
  CellTags->"Info3340454019-2206978"]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[{
  "Many of the cytosolic compounds can, in fact, be synthesized by the \
mitochondria. So there is no need to feed the mitochondria with them.  Hence, \
we change the role of some  cytosolic compounds to  ",
  StyleBox["Xtout",
    FontWeight->"Bold",
    FontSlant->"Plain"],
  "."
}], "Subsubsection",
  Evaluatable->False,
  TextJustification->1,
  FontFamily->"Times",
  FontSize->14,
  FontWeight->"Plain",
  FontSlant->"Italic",
  FontVariations->{"CompatibilityType"->0}],

Cell[BoxData[
    \(\(humit1 = \ 
        setrole[\[IndentingNewLine]humit\ , \[IndentingNewLine]{"\<adp\>"\_Xt\
\%\("\<c\>"\), "\<atp\>"\_Xt\%\("\<c\>"\), "\<gdp\>"\_Xt\%\("\<c\>"\), "\<gtp\
\>"\_Xt\%\("\<c\>"\), "\<glyc3p\>"\_Xt\%\("\<c\>"\), "\<akg\>"\_Xt\%\("\<c\>"\
\), "\<cit\>"\_Xt\%\("\<c\>"\), "\<citr-L\>"\_Xt\%\("\<c\>"\), "\<succ\>"\_Xt\
\%\("\<c\>"\), "\<mal-L\>"\_Xt\%\("\<c\>"\), "\<fum\>"\_Xt\%\("\<c\>"\)}\ , \
\[IndentingNewLine]Xtout];\)\)], "Input"],

Cell[TextData[{
  "The above syntax for the metabolites is a bit more readable than",
  StyleBox[" meta[...]",
    FontWeight->"Bold",
    FontSlant->"Plain",
    FontVariations->{"CompatibilityType"->0}],
  ". There are arcane ways of actually typing things like ",
  Cell[BoxData[
      StyleBox[\("adp"\_Xt\%\("c"\)\),
        FontWeight->"Bold",
        FontSlant->"Plain",
        FontTracking->"Plain",
        FontVariations->{"Underline"->False,
        "Outline"->False,
        "Shadow"->False,
        "StrikeThrough"->False,
        "Masked"->False,
        "CompatibilityType"->0,
        "RotationAngle"->0}]]],
  " in ",
  StyleBox["Mathematica",
    FontSlant->"Italic"],
  ". I tend to avoid this by just pasting the output of ",
  StyleBox["metabolites",
    FontWeight->"Bold",
    FontSlant->"Plain",
    FontVariations->{"CompatibilityType"->0}],
  "."
}], "Text",
  Evaluatable->False,
  FontFamily->"Times",
  FontSize->14,
  FontWeight->"Plain",
  FontSlant->"Italic"],

Cell["\<\
Further, we don't want the mitochondria to produce or consume \
protons.\
\>", "Text",
  Evaluatable->False,
  FontFamily->"Times",
  FontSize->14,
  FontWeight->"Plain",
  FontSlant->"Italic"],

Cell[BoxData[
    \(\(humit1\  = \ 
        setrole[humit1\ , \ "\<h\>"\_Xt\%\("\<c\>"\), Int];\)\)], "Input"],

Cell["", "Subtitle",
  Evaluatable->False,
  FontFamily->"Times",
  FontSize->14,
  FontWeight->"Plain",
  FontSlant->"Italic"]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[{
  "The flux cone of ",
  StyleBox["humit1",
    FontWeight->"Bold",
    FontSlant->"Plain"]
}], "Section",
  Evaluatable->False,
  TextJustification->1,
  FontFamily->"Times",
  FontSize->14,
  FontWeight->"Plain",
  FontSlant->"Italic",
  FontVariations->{"CompatibilityType"->0}],

Cell[CellGroupData[{

Cell[TextData[{
  "First we study the flux cone of   ",
  StyleBox["humit1 ",
    FontWeight->"Bold",
    FontSlant->"Plain"],
  " by computing a minimal generating set of the cone."
}], "Subsubsection",
  Evaluatable->False,
  TextJustification->1,
  FontFamily->"Times",
  FontSize->14,
  FontWeight->"Plain",
  FontSlant->"Italic",
  FontVariations->{"CompatibilityType"->0}],

Cell[CellGroupData[{

Cell[BoxData[{
    \(\({fluxes, nrev}\  = \ symfluxgset[humit1];\)\), "\n", 
    \(nrev\)}], "Input"],

Cell[BoxData[
    \(1\)], "Output"]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
    \(fluxes // Length\)], "Input"],

Cell[BoxData[
    \(1150\)], "Output"]
}, Open  ]],

Cell["\<\
So the first one of the 1150 fluxes is reversible.  Let's inspect \
this flux.\
\>", "Text",
  Evaluatable->False,
  FontFamily->"Times",
  FontSize->14,
  FontWeight->"Plain",
  FontSlant->"Italic"],

Cell[CellGroupData[{

Cell[BoxData[
    \(fluxes[\([1]\)]\)], "Input"],

Cell[BoxData[
    \(R["CITRtm"] - R["ORNt3m"] + R["ORNt4m"]\)], "Output"]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
    \(with[{R["\<CITRtm\>"], R["\<ORNt3m\>"], R["\<ORNt4m\>"]}, \ 
        trpairs@humit1]\  // TableForm\)], "Input"],

Cell[BoxData[
    TagBox[GridBox[{
          {\(R["CITRtm"]\), 
            RowBox[{\("citr-L"\^"m"\), 
              StyleBox["\[Equilibrium]",
                FontSize->18], \("citr-L"\_Xtout\%\("c"\)\)}]},
          {\(R["ORNt3m"]\), 
            RowBox[{\("h"\^"c" + "orn"\^"m"\), 
              StyleBox["\[Equilibrium]",
                FontSize->18], \("h"\^"m" + "orn"\_Xt\%\("c"\)\)}]},
          {\(R["ORNt4m"]\), 
            RowBox[{\("h"\^"c" + "orn"\^"m" + "citr-L"\_Xtout\%\("c"\)\), 
              StyleBox["\[Equilibrium]",
                FontSize->18], \("citr-L"\^"m" + "h"\^"m" + 
                "orn"\_Xt\%\("c"\)\)}]}
          },
        RowSpacings->1,
        ColumnSpacings->3,
        RowAlignments->Baseline,
        ColumnAlignments->{Left}],
      Function[ BoxForm`e$, 
        TableForm[ BoxForm`e$]]]], "Output"]
}, Open  ]],

Cell["The second flux is already much more complicated:", "Text",
  Evaluatable->False,
  FontFamily->"Times",
  FontSize->14,
  FontWeight->"Plain",
  FontSlant->"Italic"],

Cell[CellGroupData[{

Cell[BoxData[
    \(fluxes[\([2]\)]\)], "Input"],

Cell[BoxData[
    RowBox[{\(R["ACONTm"]\), "-", \(R["ASPGLUm"]\), "+", \(R["ASPTAm"]\), 
      "+", \(R["ATPS4m"]\), "+", \(R["CO2tm"]\), "+", \(R["CSm"]\), 
      "-", \(R["FUMm"]\), "+", \(R["HCO3Em"]\), "+", \(R["ICDHxm"]\), 
      "-", \(R["MDHm"]\), "+", \(R["NADH2-u10m"]\), "+", \(R["PCm"]\), 
      "+", \(R["PDHm"]\), "-", \(R["PIt2m"]\), "+", \(2\ R["PYRt2m"]\), 
      "-", \(R["SUCCt2m"]\), "-", \(R["SUCD1m"]\), "-", \(R["SUCD3-u10m"]\), 
      "-", 
      RowBox[{"Rx", "[", 
        RowBox[{\("asp-L"\_Xt\%\("c"\)\), 
          StyleBox["\[Equilibrium]",
            FontSize->18], "0"}], "]"}], "-", 
      RowBox[{"Rx", "[", 
        RowBox[{\("co2"\_Xt\%\("c"\)\), 
          StyleBox["\[Equilibrium]",
            FontSize->18], "0"}], "]"}], "+", 
      RowBox[{"Rx", "[", 
        RowBox[{\("glu-L"\_Xt\%\("c"\)\), 
          StyleBox["\[Equilibrium]",
            FontSize->18], "0"}], "]"}], "-", 
      RowBox[{"2", " ", 
        RowBox[{"Rx", "[", 
          RowBox[{\("pyr"\_Xt\%\("c"\)\), 
            StyleBox["\[Equilibrium]",
              FontSize->18], "0"}], "]"}]}], "+", 
      RowBox[{"Rx", "[", 
        RowBox[{\("succ"\_Xtout\%\("c"\)\), 
          StyleBox["\[RightVector]",
            FontSize->18], "0"}], "]"}]}]], "Output"]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[{
  "Clearly,  just printing ",
  StyleBox["fluxes",
    FontWeight->"Bold",
    FontSlant->"Plain"],
  " does not make good bedtime reading.  So we use ",
  StyleBox["Mathematica",
    FontSlant->"Italic"],
  " to analyze\n ",
  StyleBox["fluxes",
    FontWeight->"Bold",
    FontSlant->"Plain"],
  " focussing on phospholipid synthesis.\n The phospholipid reaction is:"
}], "Subsubsection",
  Evaluatable->False,
  TextJustification->1,
  FontFamily->"Times",
  FontSize->14,
  FontWeight->"Plain",
  FontSlant->"Italic",
  FontVariations->{"CompatibilityType"->0}],

Cell[CellGroupData[{

Cell[BoxData[
    \(with["\<PL\>", \ trpairs@humit1]\)], "Input"],

Cell[BoxData[
    RowBox[{"{", 
      RowBox[{"{", 
        RowBox[{\(R["DMPL"]\), ",", 
          RowBox[{\(18\ "clpn_m"\^"m" + 43\ "pc_m"\^"m" + 34\ "pe_m"\^"m"\), 
            StyleBox["\[RightVector]",
              FontSize->18], \(100\ "PL"\_Xt\%\("e"\)\)}]}], "}"}], 
      "}"}]], "Output"]
}, Open  ]],

Cell["So we can pick all fluxes generating PL by", "Text",
  Evaluatable->False,
  FontFamily->"Times",
  FontSize->14,
  FontWeight->"Plain",
  FontSlant->"Italic"],

Cell[CellGroupData[{

Cell[BoxData[{
    \(\(plfluxes\  = \ with["\<DMPL\>", fluxes];\)\), "\n", 
    \(plfluxes\  // Length\)}], "Input"],

Cell[BoxData[
    \(791\)], "Output"]
}, Open  ]],

Cell[TextData[{
  "Note,that since the exchange reactions are part of a flux, we could also \
have obtained ",
  StyleBox["plfluxes",
    FontWeight->"Bold",
    FontSlant->"Plain"],
  " by asking for fluxes with PL."
}], "Text",
  Evaluatable->False,
  FontFamily->"Times",
  FontSize->14,
  FontWeight->"Plain",
  FontSlant->"Italic"],

Cell[CellGroupData[{

Cell[BoxData[
    \(plfluxes\  === \ \ with["\<PL\>", fluxes]\)], "Input"],

Cell[BoxData[
    \(True\)], "Output"]
}, Open  ]],

Cell["\<\
Next, we obtain all internal reactions which cam be involved in PL \
synthesis by\
\>", "Text",
  Evaluatable->False,
  FontFamily->"Times",
  FontSize->14,
  FontWeight->"Plain",
  FontSlant->"Italic"],

Cell[CellGroupData[{

Cell[BoxData[{
    \(\(somewhere\  = \ 
        Cases[plfluxes, R[_], \[Infinity]]\  // 
          Union;\)\), "\[IndentingNewLine]", 
    \(somewhere\  // Length\)}], "Input"],

Cell[BoxData[
    \(89\)], "Output"]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
    \(\(\(somewhere\)\(\ \)\)\)], "Input"],

Cell[BoxData[
    \({R["5AOPtm"], R["ACONTm"], R["ADK1m"], R["AGATm"], R["AKGDm"], 
      R["AKGMALtm"], R["ALASm"], R["ASPGLUm"], R["ASPTAm"], R["ATPS4m"], 
      R["C160CPT2"], R["C160CRN"], R["C180CPT2"], R["C180CRN"], 
      R["C181CRN2"], R["C181CRN3"], R["C182CRN2"], R["C182CRN3"], 
      R["C204CRN2"], R["C204CRN3"], R["C226CRN2"], R["C226CRN3"], R["CATm"], 
      R["CBMKm"], R["CITRtm"], R["CITtam"], R["CITtbm"], R["CLPNSm"], 
      R["CO2tm"], R["CRNtim"], R["CSm"], R["CYOOm3"], R["CYOR-u10m"], 
      R["DAGKm"], R["DASYNm"], R["DMheme"], R["DMPL"], R["FAOXC160"], 
      R["FAOXC180"], R["FAOXC181"], R["FAOXC182"], R["FAOXC204"], 
      R["FAOXC226"], R["FASYNm"], R["FCLTm"], R["FE2tm"], R["FUMm"], 
      R["G3PATm"], R["GLYCtm"], R["GLYKm"], R["GLYtm"], R["GTHOm"], 
      R["GTHPm"], R["H2Otm"], R["HCO3Em"], R["HMGCOASm"], R["HMGLm"], 
      R["ICDHxm"], R["ICDHym"], R["MALtm"], R["MDHm"], R["ME2m"], 
      R["NADH2-u10m"], R["NH4tm"], R["O2tm"], R["OCBTm"], R["OCOAT1m"], 
      R["ORNt3m"], R["PAPAm"], R["PCm"], R["PCtm"], R["PDHm"], R["PEPCKm"], 
      R["PGPPm"], R["PGSAm"], R["PIt2m"], R["PPAm"], R["PPPG9tm"], 
      R["PPPGOm"], R["PSDm"], R["PStm"], R["PYRt2m"], R["Satpctp"], 
      R["SPODMm"], R["SUCCt2m"], R["SUCD1m"], R["SUCD3-u10m"], R["SUCOASm"], 
      R["THD1m"]}\)], "Output"]
}, Open  ]],

Cell[TextData[{
  "and compare this with the reactions which occur in every flux of  ",
  StyleBox["plfluxes",
    FontWeight->"Bold",
    FontSlant->"Plain"],
  "."
}], "Text",
  Evaluatable->False,
  FontFamily->"Times",
  FontSize->14,
  FontWeight->"Plain",
  FontSlant->"Italic"],

Cell[CellGroupData[{

Cell[BoxData[{
    \(\(everywhere\  = \ 
        Intersection @@ 
          Map[\ Cases[#, R[_], \[Infinity]] &, 
            plfluxes];\)\), "\[IndentingNewLine]", 
    \(everywhere\  // Length\)}], "Input"],

Cell[BoxData[
    \(34\)], "Output"]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
    \(with[everywhere, \ trpairs@humit1]\  // TableForm\)], "Input"],

Cell[BoxData[
    TagBox[GridBox[{
          {\(R["ASPTAm"]\), 
            RowBox[{\("akg"\^"m" + "asp-L"\^"m"\), 
              StyleBox["\[Equilibrium]",
                FontSize->18], \("glu-L"\^"m" + "oaa"\^"m"\)}]},
          {\(R["ADK1m"]\), 
            RowBox[{\("amp"\^"m" + "atp"\^"m"\), 
              StyleBox["\[Equilibrium]",
                FontSize->18], \(2\ "adp"\^"m"\)}]},
          {\(R["PGSAm"]\), 
            RowBox[{\("cdpdag_m"\^"m" + "glyc3p"\^"m"\), 
              StyleBox["\[Equilibrium]",
                FontSize->18], \("cmp"\^"m" + "h"\^"m" + "pgp_m"\^"m"\)}]},
          {\(R["DASYNm"]\), 
            RowBox[{\("ctp"\^"m" + "h"\^"m" + "pa_m"\^"m"\), 
              StyleBox["\[Equilibrium]",
                FontSize->18], \("cdpdag_m"\^"m" + "ppi"\^"m"\)}]},
          {\(R["CLPNSm"]\), 
            RowBox[{\(2\ "pg_m"\^"m"\), 
              StyleBox["\[Equilibrium]",
                FontSize->18], \("clpn_m"\^"m" + "glyc"\^"m"\)}]},
          {\(R["CO2tm"]\), 
            RowBox[{\("co2"\_Xt\%\("c"\)\), 
              StyleBox["\[Equilibrium]",
                FontSize->18], \("co2"\^"m"\)}]},
          {\(R["ASPGLUm"]\), 
            RowBox[{\("asp-L"\^"m" + "h"\^"c" + "glu-L"\_Xt\%\("c"\)\), 
              StyleBox["\[Equilibrium]",
                FontSize->18], \("glu-L"\^"m" + "h"\^"m" + 
                "asp-L"\_Xt\%\("c"\)\)}]},
          {\(R["GLYCtm"]\), 
            RowBox[{\("glyc"\_Xt\%\("c"\)\), 
              StyleBox["\[Equilibrium]",
                FontSize->18], \("glyc"\^"m"\)}]},
          {\(R["H2Otm"]\), 
            RowBox[{\("h2o"\_Xt\%\("c"\)\), 
              StyleBox["\[Equilibrium]",
                FontSize->18], \("h2o"\^"m"\)}]},
          {\(R["PIt2m"]\), 
            RowBox[{\("h"\^"c" + "pi"\_Xt\%\("c"\)\), 
              StyleBox["\[Equilibrium]",
                FontSize->18], \("h"\^"m" + "pi"\^"m"\)}]},
          {\(R["Satpctp"]\), 
            RowBox[{\("atp"\^"m" + "cmp"\^"m"\), 
              StyleBox["\[RightVector]",
                FontSize->18], \("amp"\^"m" + "ctp"\^"m"\)}]},
          {\(R["C204CRN3"]\), 
            RowBox[{\("c204crn"\^"m" + "coa"\^"m"\), 
              StyleBox["\[RightVector]",
                FontSize->18], \("c204coa"\^"m" + "crn"\^"m"\)}]},
          {\(R["C226CRN3"]\), 
            RowBox[{\("c226crn"\^"m" + "coa"\^"m"\), 
              StyleBox["\[RightVector]",
                FontSize->18], \("c226coa"\^"m" + "crn"\^"m"\)}]},
          {\(R["CRNtim"]\), 
            RowBox[{\("crn"\^"m"\), 
              StyleBox["\[RightVector]",
                FontSize->18], \("crn"\_Xt\%\("c"\)\)}]},
          {\(R["AGATm"]\), 
            RowBox[{\("aglyc3p_m"\^"m" + "facoa_m"\^"m"\), 
              StyleBox["\[RightVector]",
                FontSize->18], \("coa"\^"m" + "pa_m"\^"m"\)}]},
          {\(R["GLYKm"]\), 
            RowBox[{\("atp"\^"m" + "glyc"\^"m"\), 
              StyleBox["\[RightVector]",
                FontSize->18], \("adp"\^"m" + "glyc3p"\^"m" + "h"\^"m"\)}]},
          {\(R["G3PATm"]\), 
            RowBox[{\("facoa_m"\^"m" + "glyc3p"\^"m"\), 
              StyleBox["\[RightVector]",
                FontSize->18], \("aglyc3p_m"\^"m" + "coa"\^"m"\)}]},
          {\(R["C182CRN3"]\), 
            RowBox[{\("coa"\^"m" + "ocdycrn"\^"m"\), 
              StyleBox["\[RightVector]",
                FontSize->18], \("crn"\^"m" + "ocdycacoa"\^"m"\)}]},
          {\(R["C181CRN3"]\), 
            RowBox[{\("coa"\^"m" + "odecrn"\^"m"\), 
              StyleBox["\[RightVector]",
                FontSize->18], \("crn"\^"m" + "odecoa"\^"m"\)}]},
          {\(R["DMPL"]\), 
            
            RowBox[{\(18\ "clpn_m"\^"m" + 43\ "pc_m"\^"m" + 34\ "pe_m"\^"m"\),
               
              StyleBox["\[RightVector]",
                FontSize->18], \(100\ "PL"\_Xt\%\("e"\)\)}]},
          {\(R["PGPPm"]\), 
            RowBox[{\("h2o"\^"m" + "pgp_m"\^"m"\), 
              StyleBox["\[RightVector]",
                FontSize->18], \("pg_m"\^"m" + "pi"\^"m"\)}]},
          {\(R["ATPS4m"]\), 
            RowBox[{\("adp"\^"m" + 4\ "h"\^"c" + "pi"\^"m"\), 
              StyleBox["\[RightVector]",
                FontSize->18], \("atp"\^"m" + 3\ "h"\^"m" + "h2o"\^"m"\)}]},
          {\(R["PPAm"]\), 
            RowBox[{\("h2o"\^"m" + "ppi"\^"m"\), 
              StyleBox["\[RightVector]",
                FontSize->18], \("h"\^"m" + 2\ "pi"\^"m"\)}]},
          {\(R["PSDm"]\), 
            RowBox[{\("h"\^"m" + "ps_m"\^"m"\), 
              StyleBox["\[RightVector]",
                FontSize->18], \("co2"\^"m" + "pe_m"\^"m"\)}]},
          {\(R["NADH2-u10m"]\), 
            RowBox[{\(5\ "h"\^"m" + "nadh"\^"m" + "q10"\^"m"\), 
              StyleBox["\[RightVector]",
                FontSize->18], \(4\ "h"\^"c" + "nad"\^"m" + 
                "q10h2"\^"m"\)}]},
          {\(R["FASYNm"]\), 
            
            RowBox[{\("c204coa"\^"m"\/4 + "c226coa"\^"m"\/20 + 
                "ocdycacoa"\^"m"\/5 + "odecoa"\^"m"\/10 + "pmtcoa"\^"m"\/5 + 
                "stcoa"\^"m"\/5\), 
              StyleBox["\[RightVector]",
                FontSize->18], \("facoa_m"\^"m"\)}]},
          {\(R["C180CPT2"]\), 
            RowBox[{\("coa"\^"m" + "stcrn"\^"m"\), 
              StyleBox["\[RightVector]",
                FontSize->18], \("crn"\^"m" + "stcoa"\^"m"\)}]},
          {\(R["C204CRN2"]\), 
            RowBox[{\("c204crn"\_Xt\%\("c"\)\), 
              StyleBox["\[RightVector]",
                FontSize->18], \("c204crn"\^"m"\)}]},
          {\(R["C226CRN2"]\), 
            RowBox[{\("c226crn"\_Xt\%\("c"\)\), 
              StyleBox["\[RightVector]",
                FontSize->18], \("c226crn"\^"m"\)}]},
          {\(R["C182CRN2"]\), 
            RowBox[{\("ocdycrn"\_Xt\%\("c"\)\), 
              StyleBox["\[RightVector]",
                FontSize->18], \("ocdycrn"\^"m"\)}]},
          {\(R["C181CRN2"]\), 
            RowBox[{\("odecrn"\_Xt\%\("c"\)\), 
              StyleBox["\[RightVector]",
                FontSize->18], \("odecrn"\^"m"\)}]},
          {\(R["PCtm"]\), 
            RowBox[{\("pc_m"\_Xt\%\("c"\)\), 
              StyleBox["\[RightVector]",
                FontSize->18], \("pc_m"\^"m"\)}]},
          {\(R["PStm"]\), 
            RowBox[{\("ps_m"\_Xt\%\("c"\)\), 
              StyleBox["\[RightVector]",
                FontSize->18], \("ps_m"\^"m"\)}]},
          {\(R["C180CRN"]\), 
            RowBox[{\("stcrn"\_Xt\%\("c"\)\), 
              StyleBox["\[RightVector]",
                FontSize->18], \("stcrn"\^"m"\)}]}
          },
        RowSpacings->1,
        ColumnSpacings->3,
        RowAlignments->Baseline,
        ColumnAlignments->{Left}],
      Function[ BoxForm`e$, 
        TableForm[ BoxForm`e$]]]], "Output"]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[{
  "The irreversible reactions in ",
  StyleBox["everywhere ",
    FontWeight->"Bold",
    FontSlant->"Plain"],
  "are truly essential for PL-synthesis. However, for the reversible \
reactions in ",
  StyleBox["everywhere ",
    FontWeight->"Bold",
    FontSlant->"Plain"],
  "this might not be the case, since cancellations could occur when combining \
the vectors in ",
  StyleBox["fluxes. \n",
    FontWeight->"Bold",
    FontSlant->"Plain"],
  "To check this, you might want to carry out the above analysis using the \
elementary vectors of the flux cone of ",
  StyleBox["humit1,",
    FontWeight->"Bold",
    FontSlant->"Plain"],
  "  starting with being a little patient until following computation \
terminates:\n"
}], "Subsubsection",
  Evaluatable->False,
  TextJustification->1,
  FontFamily->"Times",
  FontSize->14,
  FontWeight->"Plain",
  FontSlant->"Italic",
  FontVariations->{"CompatibilityType"->0}],

Cell[CellGroupData[{

Cell[BoxData[{
    \(\({efluxes, nrev}\  = \ 
        symfluxelvs[humit1];\)\), "\[IndentingNewLine]", 
    \(nrev\)}], "Input"],

Cell[BoxData[
    RowBox[{\(General::"spell1"\), \(\(:\)\(\ \)\), "\<\"Possible spelling \
error: new symbol name \\\"\\!\\(efluxes\\)\\\" is similar to existing symbol \
\\\"\\!\\(fluxes\\)\\\". \\!\\(\\*ButtonBox[\\\"More\[Ellipsis]\\\", \
ButtonStyle->\\\"RefGuideLinkText\\\", ButtonFrame->None, \
ButtonData:>\\\"General::spell1\\\"]\\)\"\>"}]], "Message"],

Cell[BoxData[
    \(1\)], "Output"]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
    \(efluxes // Length\)], "Input"],

Cell[BoxData[
    \(34690\)], "Output"]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
    \(with["\<DMPL\>", efluxes] // Length\)], "Input"],

Cell[BoxData[
    \(31410\)], "Output"]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["\<\
Sometimes it is convenient to transform symbolic representations \
like\
\>", "Subsubsection",
  Evaluatable->False,
  FontFamily->"Times",
  FontSize->14,
  FontWeight->"Plain",
  FontSlant->"Italic"],

Cell[CellGroupData[{

Cell[BoxData[
    \(fluxes[\([2]\)]\)], "Input"],

Cell[BoxData[
    RowBox[{\(R["ACONTm"]\), "-", \(R["ASPGLUm"]\), "+", \(R["ASPTAm"]\), 
      "+", \(R["ATPS4m"]\), "+", \(R["CO2tm"]\), "+", \(R["CSm"]\), 
      "-", \(R["FUMm"]\), "+", \(R["HCO3Em"]\), "+", \(R["ICDHxm"]\), 
      "-", \(R["MDHm"]\), "+", \(R["NADH2-u10m"]\), "+", \(R["PCm"]\), 
      "+", \(R["PDHm"]\), "-", \(R["PIt2m"]\), "+", \(2\ R["PYRt2m"]\), 
      "-", \(R["SUCCt2m"]\), "-", \(R["SUCD1m"]\), "-", \(R["SUCD3-u10m"]\), 
      "-", 
      RowBox[{"Rx", "[", 
        RowBox[{\("asp-L"\_Xt\%\("c"\)\), 
          StyleBox["\[Equilibrium]",
            FontSize->18], "0"}], "]"}], "-", 
      RowBox[{"Rx", "[", 
        RowBox[{\("co2"\_Xt\%\("c"\)\), 
          StyleBox["\[Equilibrium]",
            FontSize->18], "0"}], "]"}], "+", 
      RowBox[{"Rx", "[", 
        RowBox[{\("glu-L"\_Xt\%\("c"\)\), 
          StyleBox["\[Equilibrium]",
            FontSize->18], "0"}], "]"}], "-", 
      RowBox[{"2", " ", 
        RowBox[{"Rx", "[", 
          RowBox[{\("pyr"\_Xt\%\("c"\)\), 
            StyleBox["\[Equilibrium]",
              FontSize->18], "0"}], "]"}]}], "+", 
      RowBox[{"Rx", "[", 
        RowBox[{\("succ"\_Xtout\%\("c"\)\), 
          StyleBox["\[RightVector]",
            FontSize->18], "0"}], "]"}]}]], "Output"]
}, Open  ]],

Cell[TextData[{
  "into proper vectors.\nThis is achieved with the function",
  StyleBox[" factorsof",
    FontWeight->"Bold",
    FontSlant->"Plain",
    FontVariations->{"CompatibilityType"->0}],
  ", which extracts the pre-factors of the specified symbols, e.g."
}], "Text",
  Evaluatable->False,
  FontFamily->"Times",
  FontSize->14,
  FontWeight->"Plain",
  FontSlant->"Italic"],

Cell[CellGroupData[{

Cell[BoxData[
    \(factorsof[{\ R["\<PYRt2m\>"], R["\<PPAm\>"], R["\<ASPGLUm\>"]}, 
      fluxes[\([2]\)]]\)], "Input"],

Cell[BoxData[
    \({2, 0, \(-1\)}\)], "Output"]
}, Open  ]],

Cell[TextData[{
  StyleBox["factorsof ",
    FontWeight->"Bold",
    FontSlant->"Plain",
    FontVariations->{"CompatibilityType"->0}],
  " will not only  work with a single symbolic flux but also with a list of \
such fluxes (or more generally with a list of sums of products ).\nHence, the \
following extracts the flows for all internal reactions  in the fluxes \
generating PL."
}], "Text",
  Evaluatable->False,
  FontFamily->"Times",
  FontSize->14,
  FontWeight->"Plain",
  FontSlant->"Italic"],

Cell[CellGroupData[{

Cell[BoxData[{
    \(\(intcoefficientmatrix\  = \ 
        factorsof[tags@humit1, plfluxes];\)\), "\[IndentingNewLine]", 
    \(intcoefficientmatrix\  // Dimensions\)}], "Input"],

Cell[BoxData[
    \({791, 123}\)], "Output"]
}, Open  ]],

Cell[TextData[{
  "Note that a more numeric way of computing the set  ",
  StyleBox["everywhere",
    FontWeight->"Bold",
    FontSlant->"Plain",
    FontVariations->{"CompatibilityType"->0}],
  " could have been to use the above matrix. e.g."
}], "Text",
  Evaluatable->False,
  FontFamily->"Times",
  FontSize->14,
  FontWeight->"Plain",
  FontSlant->"Italic"],

Cell[CellGroupData[{

Cell[BoxData[
    \(\((tags@humit1)\)[\([iwith[Length@plfluxes, \ 
          Apply[Plus, Abs@\(Sign@intcoefficientmatrix\)]]]\)]\)], "Input"],

Cell[BoxData[
    \({R["ASPTAm"], R["ADK1m"], R["PGSAm"], R["DASYNm"], R["CLPNSm"], 
      R["CO2tm"], R["ASPGLUm"], R["GLYCtm"], R["H2Otm"], R["PIt2m"], 
      R["Satpctp"], R["C204CRN3"], R["C226CRN3"], R["CRNtim"], R["AGATm"], 
      R["GLYKm"], R["G3PATm"], R["C182CRN3"], R["C181CRN3"], R["DMPL"], 
      R["PGPPm"], R["ATPS4m"], R["PPAm"], R["PSDm"], R["NADH2-u10m"], 
      R["FASYNm"], R["C180CPT2"], R["C204CRN2"], R["C226CRN2"], 
      R["C182CRN2"], R["C181CRN2"], R["PCtm"], R["PStm"], 
      R["C180CRN"]}\)], "Output"]
}, Open  ]],

Cell[TextData[{
  "To get at flows through the exchange reactions can require a lot of \
typing. For convenience, the function ",
  StyleBox["exch",
    FontWeight->"Bold",
    FontSlant->"Plain",
    FontVariations->{"CompatibilityType"->0}],
  " construct the reaction tag for a given external  metabolite. "
}], "Text",
  Evaluatable->False,
  FontFamily->"Times",
  FontSize->14,
  FontWeight->"Plain",
  FontSlant->"Italic"],

Cell[CellGroupData[{

Cell[BoxData[
    \(exch["\<co2\>"\_Xt\%\("\<c\>"\)]\)], "Input"],

Cell[BoxData[
    RowBox[{"Rx", "[", 
      RowBox[{\("co2"\_Xt\%\("c"\)\), 
        StyleBox["\[Equilibrium]",
          FontSize->18], "0"}], "]"}]], "Output"]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
    \(exch[{"\<co2\>"\_Xt\%\("\<c\>"\), \ "\<coa\>"\^"\<m\>", \ \
"\<succ\>"\_Xtout\%\("\<c\>"\)}]\)], "Input"],

Cell[BoxData[
    RowBox[{"{", 
      RowBox[{
        RowBox[{"Rx", "[", 
          RowBox[{\("co2"\_Xt\%\("c"\)\), 
            StyleBox["\[Equilibrium]",
              FontSize->18], "0"}], "]"}], ",", 
        RowBox[{"Rx", "[", 
          RowBox[{\("succ"\_Xtout\%\("c"\)\), 
            StyleBox["\[RightVector]",
              FontSize->18], "0"}], "]"}]}], "}"}]], "Output"]
}, Open  ]],

Cell[TextData[{
  "Note, that ",
  StyleBox["exch ",
    FontWeight->"Bold",
    FontSlant->"Plain",
    FontVariations->{"CompatibilityType"->0}],
  "skips internal metabolites. So to get at  the flows through the exchange \
reactions in ",
  StyleBox["plfluxes",
    FontWeight->"Bold",
    FontSlant->"Plain",
    FontVariations->{"CompatibilityType"->0}],
  " you can simply use:"
}], "Text",
  Evaluatable->False,
  FontFamily->"Times",
  FontSize->14,
  FontWeight->"Plain",
  FontSlant->"Italic"],

Cell[CellGroupData[{

Cell[BoxData[{
    \(\(exchcoefficientmatrix\  = \ 
        factorsof[exch@\(metabolites@humit1\), 
          plfluxes];\)\), "\[IndentingNewLine]", 
    \(exchcoefficientmatrix\  // Dimensions\)}], "Input"],

Cell[BoxData[
    \({791, 43}\)], "Output"]
}, Open  ]]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[{
  "The conversion  cone of ",
  StyleBox["humit1",
    FontWeight->"Bold",
    FontSlant->"Plain"]
}], "Section",
  Evaluatable->False,
  TextJustification->1,
  FontFamily->"Times",
  FontSize->14,
  FontWeight->"Plain",
  FontSlant->"Italic",
  FontVariations->{"CompatibilityType"->0}],

Cell[CellGroupData[{

Cell["\<\
We first a compute a minimal generating set of the conversion \
cone\
\>", "Subsubsection",
  Evaluatable->False,
  FontFamily->"Times",
  FontSize->14,
  FontWeight->"Plain",
  FontSlant->"Italic"],

Cell[CellGroupData[{

Cell[BoxData[
    \(conversiongset[humit1]\ \  // Length\)], "Input"],

Cell[BoxData[
    \(641\)], "Output"]
}, Open  ]],

Cell["\<\
Since this generating set is not all that large, we go straight for \
the elementary conversions.\
\>", "Text",
  Evaluatable->False,
  FontFamily->"Times",
  FontSize->14,
  FontWeight->"Plain",
  FontSlant->"Italic"],

Cell[CellGroupData[{

Cell[BoxData[{
    \(\(econvs\  = \ conversionelvs[humit1]\ ;\)\), "\[IndentingNewLine]", 
    \(econvs\  // Length\)}], "Input"],

Cell[BoxData[
    \(1535\)], "Output"]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
    \(econvs\  // Last\)], "Input"],

Cell[BoxData[
    RowBox[{\(9629190\ "c204crn"\_Xt\%\("c"\) + 
        1925838\ "c226crn"\_Xt\%\("c"\) + 23685632\ "fe2"\_Xt\%\("c"\) + 
        11243430\ "gly"\_Xt\%\("c"\) + 28887570\ "glyc"\_Xt\%\("c"\) + 
        57292212\ "o2"\_Xt\%\("c"\) + 7703352\ "ocdycrn"\_Xt\%\("c"\) + 
        3851676\ "odecrn"\_Xt\%\("c"\) + 23003065\ "pc_m"\_Xt\%\("c"\) + 
        19258380\ "pi"\_Xt\%\("c"\) + 23685632\ "pppg9"\_Xt\%\("c"\) + 
        18188470\ "ps_m"\_Xt\%\("c"\) + 11243430\ "pyr"\_Xt\%\("c"\) + 
        15800046\ "stcrn"\_Xt\%\("c"\)\), 
      StyleBox["\[RightVector]",
        FontSize->18], \(11243430\ "5aop"\_Xt\%\("c"\) + 
        40675330\ "co2"\_Xt\%\("c"\) + 38910102\ "crn"\_Xt\%\("c"\) + 
        131007666\ "h2o"\_Xt\%\("c"\) + 23685632\ "pheme"\_Xt\%\("e"\) + 
        53495500\ "PL"\_Xt\%\("e"\)\)}]], "Output"]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["\<\
Similarly to the above analysis, we now ask which external \
metabolites can /must occur as inputs in PL-synthesis.\
\>", "Subsubsection",
  Evaluatable->False,
  FontFamily->"Times",
  FontSize->14,
  FontWeight->"Plain",
  FontSlant->"Italic"],

Cell[CellGroupData[{

Cell[BoxData[{
    \(\(pleconvs\  = \ with["\<PL\>", econvs];\)\), "\[IndentingNewLine]", 
    \(pleconvs\  // Length\)}], "Input"],

Cell[BoxData[
    \(1155\)], "Output"]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
    \(pleconvs // Last\)], "Input"],

Cell[BoxData[
    RowBox[{\(9629190\ "c204crn"\_Xt\%\("c"\) + 
        1925838\ "c226crn"\_Xt\%\("c"\) + 23685632\ "fe2"\_Xt\%\("c"\) + 
        11243430\ "gly"\_Xt\%\("c"\) + 28887570\ "glyc"\_Xt\%\("c"\) + 
        57292212\ "o2"\_Xt\%\("c"\) + 7703352\ "ocdycrn"\_Xt\%\("c"\) + 
        3851676\ "odecrn"\_Xt\%\("c"\) + 23003065\ "pc_m"\_Xt\%\("c"\) + 
        19258380\ "pi"\_Xt\%\("c"\) + 23685632\ "pppg9"\_Xt\%\("c"\) + 
        18188470\ "ps_m"\_Xt\%\("c"\) + 11243430\ "pyr"\_Xt\%\("c"\) + 
        15800046\ "stcrn"\_Xt\%\("c"\)\), 
      StyleBox["\[RightVector]",
        FontSize->18], \(11243430\ "5aop"\_Xt\%\("c"\) + 
        40675330\ "co2"\_Xt\%\("c"\) + 38910102\ "crn"\_Xt\%\("c"\) + 
        131007666\ "h2o"\_Xt\%\("c"\) + 23685632\ "pheme"\_Xt\%\("e"\) + 
        53495500\ "PL"\_Xt\%\("e"\)\)}]], "Output"]
}, Open  ]],

Cell["\<\
The first thing is, to get rid of the outputs and of  the \
stoichiometric factors.\
\>", "Text",
  Evaluatable->False,
  FontFamily->"Times",
  FontSize->14,
  FontWeight->"Plain",
  FontSlant->"Italic"],

Cell[CellGroupData[{

Cell[BoxData[{
    RowBox[{
      RowBox[{
        RowBox[{"tmp", " ", "=", " ", 
          RowBox[{"pleconvs", " ", "/.", " ", 
            RowBox[{
              RowBox[{"a_", " ", 
                StyleBox["\[RightVector]",
                  FontSize->18], 
                StyleBox[
                  RowBox[{
                    StyleBox[" ",
                      FontSize->18], " "}]], "b_"}], " ", "\[Rule]", " ", 
              "a"}]}]}], ";"}], " "}], "\[IndentingNewLine]", \(tmp\  // 
      Last\)}], "Input"],

Cell[BoxData[
    \(9629190\ "c204crn"\_Xt\%\("c"\) + 1925838\ "c226crn"\_Xt\%\("c"\) + 
      23685632\ "fe2"\_Xt\%\("c"\) + 11243430\ "gly"\_Xt\%\("c"\) + 
      28887570\ "glyc"\_Xt\%\("c"\) + 57292212\ "o2"\_Xt\%\("c"\) + 
      7703352\ "ocdycrn"\_Xt\%\("c"\) + 3851676\ "odecrn"\_Xt\%\("c"\) + 
      23003065\ "pc_m"\_Xt\%\("c"\) + 19258380\ "pi"\_Xt\%\("c"\) + 
      23685632\ "pppg9"\_Xt\%\("c"\) + 18188470\ "ps_m"\_Xt\%\("c"\) + 
      11243430\ "pyr"\_Xt\%\("c"\) + 
      15800046\ "stcrn"\_Xt\%\("c"\)\)], "Output"]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[{
    \(\(plins\  = \ 
        Map[Cases[#, meta[__], \[Infinity]] &, \ tmp];\)\), "\n", 
    \(plins\  // Last\)}], "Input"],

Cell[BoxData[
    \({"c204crn"\_Xt\%\("c"\), "c226crn"\_Xt\%\("c"\), "fe2"\_Xt\%\("c"\), 
      "gly"\_Xt\%\("c"\), "glyc"\_Xt\%\("c"\), "o2"\_Xt\%\("c"\), 
      "ocdycrn"\_Xt\%\("c"\), "odecrn"\_Xt\%\("c"\), "pc_m"\_Xt\%\("c"\), 
      "pi"\_Xt\%\("c"\), "pppg9"\_Xt\%\("c"\), "ps_m"\_Xt\%\("c"\), 
      "pyr"\_Xt\%\("c"\), "stcrn"\_Xt\%\("c"\)}\)], "Output"]
}, Open  ]],

Cell["And now, we proceed as for the fluxes.", "Text",
  Evaluatable->False,
  FontFamily->"Times",
  FontSize->14,
  FontWeight->"Plain",
  FontSlant->"Italic"],

Cell[CellGroupData[{

Cell[BoxData[
    \(insomewhere\  = \ 
      Cases[plins, meta[__], \[Infinity]]\  // Union\)], "Input"],

Cell[BoxData[
    \({"asp-L"\_Xt\%\("c"\), "c204crn"\_Xt\%\("c"\), "c226crn"\_Xt\%\("c"\), 
      "fe2"\_Xt\%\("c"\), "glu-L"\_Xt\%\("c"\), "gly"\_Xt\%\("c"\), 
      "glyc"\_Xt\%\("c"\), "nh4"\_Xt\%\("c"\), "o2"\_Xt\%\("c"\), 
      "ocdycrn"\_Xt\%\("c"\), "odecrn"\_Xt\%\("c"\), "orn"\_Xt\%\("c"\), 
      "pc_m"\_Xt\%\("c"\), "pi"\_Xt\%\("c"\), "pmtcrn"\_Xt\%\("c"\), 
      "pppg9"\_Xt\%\("c"\), "ps_m"\_Xt\%\("c"\), "pyr"\_Xt\%\("c"\), 
      "stcrn"\_Xt\%\("c"\)}\)], "Output"]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
    \(ineverywhere\  = \ 
      Intersection @@ 
        Map[\ Cases[#, meta[__], \[Infinity]] &, plins]\)], "Input"],

Cell[BoxData[
    \({"c204crn"\_Xt\%\("c"\), "c226crn"\_Xt\%\("c"\), "glyc"\_Xt\%\("c"\), 
      "ocdycrn"\_Xt\%\("c"\), "odecrn"\_Xt\%\("c"\), "pc_m"\_Xt\%\("c"\), 
      "pi"\_Xt\%\("c"\), "ps_m"\_Xt\%\("c"\), 
      "stcrn"\_Xt\%\("c"\)}\)], "Output"]
}, Open  ]],

Cell[TextData[{
  "Since we have considered the elementary conversions, the metabolites in ",
  StyleBox["ineverywher",
    FontWeight->"Bold",
    FontSlant->"Plain",
    FontVariations->{"CompatibilityType"->0}],
  "e are essential for PL-synthesis. Of course we could , hopefully,  have \
arrived at the same result by considering the exchange reactions of the \
elementary fluxes.  But then we'd have to deal with   \[GreaterEqual]30000 \
fluxes."
}], "Text",
  Evaluatable->False,
  FontFamily->"Times",
  FontSize->14,
  FontWeight->"Plain",
  FontSlant->"Italic"],

Cell[TextData[{
  "Now we know that ",
  Cell[BoxData[
      StyleBox[\("stcrn"\_Xt\%\("c"\)\),
        FontWeight->"Bold",
        FontSlant->"Plain",
        FontTracking->"Plain",
        FontVariations->{"Underline"->False,
        "Outline"->False,
        "Shadow"->False,
        "StrikeThrough"->False,
        "Masked"->False,
        "CompatibilityType"->0,
        "RotationAngle"->0}]]],
  " is essential, we can ask what the production rates of Pl  can be."
}], "Text",
  Evaluatable->False,
  FontFamily->"Times",
  FontSize->14,
  FontWeight->"Plain",
  FontSlant->"Italic"],

Cell[CellGroupData[{

Cell[BoxData[{
    RowBox[{
      RowBox[{"stcrnfactors", " ", "=", 
        RowBox[{"factorsof", "[", " ", 
          RowBox[{\("\<stcrn\>"\_Xt\%\("\<c\>"\)\), ",", 
            RowBox[{"pleconvs", " ", "/.", " ", 
              RowBox[{
                RowBox[{"a_", " ", 
                  StyleBox["\[RightVector]",
                    FontSize->18], 
                  StyleBox[
                    RowBox[{
                      StyleBox[" ",
                        FontSize->18], " "}]], "b_"}], " ", "\[Rule]", " ", 
                "a"}]}]}], " ", "]"}]}], ";"}], "\n", 
    RowBox[{
      RowBox[{"plefactors", "      ", "=", 
        RowBox[{"factorsof", "[", 
          RowBox[{\("\<PL\>"\_Xt\%\("\<e\>"\)\), ",", " ", 
            RowBox[{"pleconvs", " ", "/.", " ", 
              RowBox[{
                RowBox[{"a_", " ", 
                  StyleBox["\[RightVector]",
                    FontSize->18], 
                  StyleBox[
                    RowBox[{
                      StyleBox[" ",
                        FontSize->18], " "}]], "b_"}], " ", "\[Rule]", " ", 
                "b"}]}]}], " ", "]"}]}], ";"}], "\n", \(<< 
      Graphics`Graphics`\), "\n", \(Histogram[
      plefactors/stcrnfactors]\)}], "Input"],

Cell[GraphicsData["PostScript", "\<\
%!
%%Creator: Mathematica
%%AspectRatio: .61803 
MathPictureStart
/Mabs {
Mgmatrix idtransform
Mtmatrix dtransform
} bind def
/Mabsadd { Mabs
3 -1 roll add
3 1 roll add
exch } bind def
%% Graphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
-0.129032 0.16129 0.0147151 0.000784805 [
[.03226 .00222 -3 -9 ]
[.03226 .00222 3 0 ]
[.19355 .00222 -3 -9 ]
[.19355 .00222 3 0 ]
[.35484 .00222 -3 -9 ]
[.35484 .00222 3 0 ]
[.51613 .00222 -3 -9 ]
[.51613 .00222 3 0 ]
[.67742 .00222 -3 -9 ]
[.67742 .00222 3 0 ]
[.83871 .00222 -3 -9 ]
[.83871 .00222 3 0 ]
[1 .00222 -3 -9 ]
[1 .00222 3 0 ]
[-0.0125 .0932 -18 -4.5 ]
[-0.0125 .0932 0 4.5 ]
[-0.0125 .17168 -18 -4.5 ]
[-0.0125 .17168 0 4.5 ]
[-0.0125 .25016 -18 -4.5 ]
[-0.0125 .25016 0 4.5 ]
[-0.0125 .32864 -18 -4.5 ]
[-0.0125 .32864 0 4.5 ]
[-0.0125 .40712 -18 -4.5 ]
[-0.0125 .40712 0 4.5 ]
[-0.0125 .4856 -18 -4.5 ]
[-0.0125 .4856 0 4.5 ]
[-0.0125 .56408 -18 -4.5 ]
[-0.0125 .56408 0 4.5 ]
[ 0 0 0 0 ]
[ 1 .61803 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 g
.25 Mabswid
[ ] 0 setdash
.03226 .01472 m
.03226 .02097 L
s
[(1)] .03226 .00222 0 1 Mshowa
.19355 .01472 m
.19355 .02097 L
s
[(2)] .19355 .00222 0 1 Mshowa
.35484 .01472 m
.35484 .02097 L
s
[(3)] .35484 .00222 0 1 Mshowa
.51613 .01472 m
.51613 .02097 L
s
[(4)] .51613 .00222 0 1 Mshowa
.67742 .01472 m
.67742 .02097 L
s
[(5)] .67742 .00222 0 1 Mshowa
.83871 .01472 m
.83871 .02097 L
s
[(6)] .83871 .00222 0 1 Mshowa
1 .01472 m
1 .02097 L
s
[(7)] 1 .00222 0 1 Mshowa
0 .01472 m
1 .01472 L
s
0 .0932 m
.00625 .0932 L
s
[(100)] -0.0125 .0932 1 0 Mshowa
0 .17168 m
.00625 .17168 L
s
[(200)] -0.0125 .17168 1 0 Mshowa
0 .25016 m
.00625 .25016 L
s
[(300)] -0.0125 .25016 1 0 Mshowa
0 .32864 m
.00625 .32864 L
s
[(400)] -0.0125 .32864 1 0 Mshowa
0 .40712 m
.00625 .40712 L
s
[(500)] -0.0125 .40712 1 0 Mshowa
0 .4856 m
.00625 .4856 L
s
[(600)] -0.0125 .4856 1 0 Mshowa
0 .56408 m
.00625 .56408 L
s
[(700)] -0.0125 .56408 1 0 Mshowa
.125 Mabswid
0 .03041 m
.00375 .03041 L
s
0 .04611 m
.00375 .04611 L
s
0 .0618 m
.00375 .0618 L
s
0 .0775 m
.00375 .0775 L
s
0 .10889 m
.00375 .10889 L
s
0 .12459 m
.00375 .12459 L
s
0 .14028 m
.00375 .14028 L
s
0 .15598 m
.00375 .15598 L
s
0 .18737 m
.00375 .18737 L
s
0 .20307 m
.00375 .20307 L
s
0 .21876 m
.00375 .21876 L
s
0 .23446 m
.00375 .23446 L
s
0 .26585 m
.00375 .26585 L
s
0 .28155 m
.00375 .28155 L
s
0 .29724 m
.00375 .29724 L
s
0 .31294 m
.00375 .31294 L
s
0 .34433 m
.00375 .34433 L
s
0 .36003 m
.00375 .36003 L
s
0 .37573 m
.00375 .37573 L
s
0 .39142 m
.00375 .39142 L
s
0 .42281 m
.00375 .42281 L
s
0 .43851 m
.00375 .43851 L
s
0 .45421 m
.00375 .45421 L
s
0 .4699 m
.00375 .4699 L
s
0 .50129 m
.00375 .50129 L
s
0 .51699 m
.00375 .51699 L
s
0 .53269 m
.00375 .53269 L
s
0 .54838 m
.00375 .54838 L
s
0 .57977 m
.00375 .57977 L
s
0 .59547 m
.00375 .59547 L
s
0 .61117 m
.00375 .61117 L
s
.25 Mabswid
0 0 m
0 .61803 L
s
0 0 m
1 0 L
1 .61803 L
0 .61803 L
closepath
clip
newpath
1 0 0 r
0 .01472 m
0 .01628 L
.03226 .01628 L
.03226 .01472 L
F
0 g
.5 Mabswid
0 .01472 m
0 .01628 L
.03226 .01628 L
.03226 .01472 L
0 .01472 L
s
1 0 0 r
.03226 .01472 m
.03226 .01472 L
.06452 .01472 L
.06452 .01472 L
F
0 g
.03226 .01472 m
.03226 .01472 L
.06452 .01472 L
.06452 .01472 L
.03226 .01472 L
s
1 0 0 r
.06452 .01472 m
.06452 .01785 L
.09677 .01785 L
.09677 .01472 L
F
0 g
.06452 .01472 m
.06452 .01785 L
.09677 .01785 L
.09677 .01472 L
.06452 .01472 L
s
1 0 0 r
.09677 .01472 m
.09677 .01472 L
.12903 .01472 L
.12903 .01472 L
F
0 g
.09677 .01472 m
.09677 .01472 L
.12903 .01472 L
.12903 .01472 L
.09677 .01472 L
s
1 0 0 r
.12903 .01472 m
.12903 .01472 L
.16129 .01472 L
.16129 .01472 L
F
0 g
.12903 .01472 m
.12903 .01472 L
.16129 .01472 L
.16129 .01472 L
.12903 .01472 L
s
1 0 0 r
.16129 .01472 m
.16129 .01472 L
.19355 .01472 L
.19355 .01472 L
F
0 g
.16129 .01472 m
.16129 .01472 L
.19355 .01472 L
.19355 .01472 L
.16129 .01472 L
s
1 0 0 r
.19355 .01472 m
.19355 .01628 L
.22581 .01628 L
.22581 .01472 L
F
0 g
.19355 .01472 m
.19355 .01628 L
.22581 .01628 L
.22581 .01472 L
.19355 .01472 L
s
1 0 0 r
.22581 .01472 m
.22581 .01472 L
.25806 .01472 L
.25806 .01472 L
F
0 g
.22581 .01472 m
.22581 .01472 L
.25806 .01472 L
.25806 .01472 L
.22581 .01472 L
s
1 0 0 r
.25806 .01472 m
.25806 .01472 L
.29032 .01472 L
.29032 .01472 L
F
0 g
.25806 .01472 m
.25806 .01472 L
.29032 .01472 L
.29032 .01472 L
.25806 .01472 L
s
1 0 0 r
.29032 .01472 m
.29032 .01628 L
.32258 .01628 L
.32258 .01472 L
F
0 g
.29032 .01472 m
.29032 .01628 L
.32258 .01628 L
.32258 .01472 L
.29032 .01472 L
s
1 0 0 r
.32258 .01472 m
.32258 .01942 L
.35484 .01942 L
.35484 .01472 L
F
0 g
.32258 .01472 m
.32258 .01942 L
.35484 .01942 L
.35484 .01472 L
.32258 .01472 L
s
1 0 0 r
.35484 .01472 m
.35484 .0312 L
.3871 .0312 L
.3871 .01472 L
F
0 g
.35484 .01472 m
.35484 .0312 L
.3871 .0312 L
.3871 .01472 L
.35484 .01472 L
s
1 0 0 r
.3871 .01472 m
.3871 .03198 L
.41935 .03198 L
.41935 .01472 L
F
0 g
.3871 .01472 m
.3871 .03198 L
.41935 .03198 L
.41935 .01472 L
.3871 .01472 L
s
1 0 0 r
.41935 .01472 m
.41935 .28155 L
.45161 .28155 L
.45161 .01472 L
F
0 g
.41935 .01472 m
.41935 .28155 L
.45161 .28155 L
.45161 .01472 L
.41935 .01472 L
s
1 0 0 r
.45161 .01472 m
.45161 .01628 L
.48387 .01628 L
.48387 .01472 L
F
0 g
.45161 .01472 m
.45161 .01628 L
.48387 .01628 L
.48387 .01472 L
.45161 .01472 L
s
1 0 0 r
.48387 .01472 m
.48387 .01472 L
.51613 .01472 L
.51613 .01472 L
F
0 g
.48387 .01472 m
.48387 .01472 L
.51613 .01472 L
.51613 .01472 L
.48387 .01472 L
s
1 0 0 r
.51613 .01472 m
.51613 .01472 L
.54839 .01472 L
.54839 .01472 L
F
0 g
.51613 .01472 m
.51613 .01472 L
.54839 .01472 L
.54839 .01472 L
.51613 .01472 L
s
1 0 0 r
.54839 .01472 m
.54839 .01472 L
.58065 .01472 L
.58065 .01472 L
F
0 g
.54839 .01472 m
.54839 .01472 L
.58065 .01472 L
.58065 .01472 L
.54839 .01472 L
s
1 0 0 r
.58065 .01472 m
.58065 .0155 L
.6129 .0155 L
.6129 .01472 L
F
0 g
.58065 .01472 m
.58065 .0155 L
.6129 .0155 L
.6129 .01472 L
.58065 .01472 L
s
1 0 0 r
.6129 .01472 m
.6129 .01628 L
.64516 .01628 L
.64516 .01472 L
F
0 g
.6129 .01472 m
.6129 .01628 L
.64516 .01628 L
.64516 .01472 L
.6129 .01472 L
s
1 0 0 r
.64516 .01472 m
.64516 .0155 L
.67742 .0155 L
.67742 .01472 L
F
0 g
.64516 .01472 m
.64516 .0155 L
.67742 .0155 L
.67742 .01472 L
.64516 .01472 L
s
1 0 0 r
.67742 .01472 m
.67742 .01472 L
.70968 .01472 L
.70968 .01472 L
F
0 g
.67742 .01472 m
.67742 .01472 L
.70968 .01472 L
.70968 .01472 L
.67742 .01472 L
s
1 0 0 r
.70968 .01472 m
.70968 .01472 L
.74194 .01472 L
.74194 .01472 L
F
0 g
.70968 .01472 m
.70968 .01472 L
.74194 .01472 L
.74194 .01472 L
.70968 .01472 L
s
1 0 0 r
.74194 .01472 m
.74194 .01472 L
.77419 .01472 L
.77419 .01472 L
F
0 g
.74194 .01472 m
.74194 .01472 L
.77419 .01472 L
.77419 .01472 L
.74194 .01472 L
s
1 0 0 r
.77419 .01472 m
.77419 .01472 L
.80645 .01472 L
.80645 .01472 L
F
0 g
.77419 .01472 m
.77419 .01472 L
.80645 .01472 L
.80645 .01472 L
.77419 .01472 L
s
1 0 0 r
.80645 .01472 m
.80645 .01472 L
.83871 .01472 L
.83871 .01472 L
F
0 g
.80645 .01472 m
.80645 .01472 L
.83871 .01472 L
.83871 .01472 L
.80645 .01472 L
s
1 0 0 r
.83871 .01472 m
.83871 .01472 L
.87097 .01472 L
.87097 .01472 L
F
0 g
.83871 .01472 m
.83871 .01472 L
.87097 .01472 L
.87097 .01472 L
.83871 .01472 L
s
1 0 0 r
.87097 .01472 m
.87097 .01472 L
.90323 .01472 L
.90323 .01472 L
F
0 g
.87097 .01472 m
.87097 .01472 L
.90323 .01472 L
.90323 .01472 L
.87097 .01472 L
s
1 0 0 r
.90323 .01472 m
.90323 .01472 L
.93548 .01472 L
.93548 .01472 L
F
0 g
.90323 .01472 m
.90323 .01472 L
.93548 .01472 L
.93548 .01472 L
.90323 .01472 L
s
1 0 0 r
.93548 .01472 m
.93548 .01472 L
.96774 .01472 L
.96774 .01472 L
F
0 g
.93548 .01472 m
.93548 .01472 L
.96774 .01472 L
.96774 .01472 L
.93548 .01472 L
s
1 0 0 r
.96774 .01472 m
.96774 .60332 L
1 .60332 L
1 .01472 L
F
0 g
.96774 .01472 m
.96774 .60332 L
1 .60332 L
1 .01472 L
.96774 .01472 L
s
% End of Graphics
MathPictureEnd
\
\>"], "Graphics",
  ImageSize->{250, 154.5},
  ImageMargins->{{0, 0}, {0, 0}},
  ImageRegion->{{0, 1}, {0, 1}},
  ImageCache->GraphicsData["Bitmap", "\<\
CF5dJ6E]HGAYHf4PAg9QL6QYHg<PAVmbKF5d0`40003j0000VQ000`40n007h00On_ll003joc`00?[o
?000n_ll003joc`001[o?0@0023o?0@0027o?08002Co?003003o??ll01oo?0<002;o?08002?o?003
003o??ll00;o?0007?ll00<00?lloc`07oll00<00?lloc`08Oll00@00?lloc`002?o?003003o??ll
02;o?003003o??ll01ko?004003o??ll000Roc`00`00occo?002oc`001co?003003o??ll023o?003
003o??ll02?o?003003o??ll01ko?0D002?o?003003o??ll01ko?004003o??ll000Soc`00`00occo
?001oc`001co?003003o??ll027o?003003o??ll023o?080027o?004003o??ll000Qoc`3000Qoc`3
000Toc`00`00occo?001oc`001co?003003o??ll02;o?003003o??ll027o?003003o??ll01oo?003
003o?000027o?003003o??ll027o?003003o??ll02Go?040007o?07o?0006_ll0`008Oll00@00?ll
oc`0023o?004003o??ll000Roc`2000Qoc`00`00occo?00Qoc`00`00occo?00Roc`01000occo?000
0_ll000Doc`00`00occo?005oc`00`00occo?00Poc`2000Roc`2000Toc`00`00occo?00Ooc`4000Q
oc`3000Qoc`40002oc`001Co?003003o??ll0>?o?0005?ll00<00?lloc`0holl000Doc`00`00occo
?005oc`00`00occo?004oc`00`00occo?004oc`00`00occo?005oc`00`00occo?004oc`00`00occo
?004oc`00`00occo?004oc`00`00occo?005oc`00`00occo?004oc`00`00occo?00Yoc`00`00occo
?004oc`00`00occo?004oc`00`00occo?004oc`00`00occo?00Coc`00`00occo?004oc`00`00occo
?005oc`00`00occo?004oc`00`00occo?004oc`00`00occo?004oc`00`00occo?005oc`00`00occo
?004oc`00`00occo?004oc`00`00occo?004oc`00`00occo?009oc`001Co?><000?o?0005?ll00<0
0?lloc`01Oll00<00?lloc`08Oll00<00?lloc`06_ll20001_P000<00?P0n0001OP000<00?P0n000
1?P000<00?lloc`03?ll00<00?lloc`08Oll00<00?lloc`08Oll00<00?lloc`06_ll00<00?P0n000
1OP000<00?lloc`00Oll000Doc`00`00occo?01=oc`00`00n03h0004n0000`00n03h0005n0000`00
n03h0004n0000`00occo?01aoc`00`00n03h0005n0000`00occo?001oc`001Co?08004ko?0P000Oh
0003003h0?P000Ch0003003o??ll077o?003003h0?P000Gh0003003o??ll007o?0005?ll00<00?ll
oc`0E?ll2@001_P000<00?lloc`0LOll00<00?P0n0001OP000<00?lloc`00Oll000Doc`00`00occo
?01Loc`00`00n03h0004n0000`00occo?01aoc`00`00n03h0005n0000`00occo?001oc`001Co?003
003o??ll05co?003003h0?P000Ch0003003o??ll077o?003003h0?P000Gh0003003o??ll007o?000
5?ll0P00GOll00<00?P0n0001?P000<00?lloc`0LOll00<00?P0n0001OP000<00?lloc`00Oll000D
oc`00`00occo?01Loc`00`00n03h0004n0000`00occo?01aoc`00`00n03h0005n0000`00occo?001
oc`001Co?003003o??ll05co?003003h0?P000Ch0003003o??ll077o?003003h0?P000Gh0003003o
??ll007o?0005?ll0P00GOll00<00?P0n0001?P000<00?lloc`0LOll00<00?P0n0001OP000<00?ll
oc`00Oll000Doc`00`00occo?01Loc`00`00n03h0004n0000`00occo?01aoc`00`00n03h0005n000
0`00occo?001oc`001Co?003003o??ll05co?003003h0?P000Ch0003003o??ll077o?003003h0?P0
00Gh0003003o??ll007o?0005?ll00<00?lloc`0G?ll00<00?P0n0001?P000<00?lloc`0LOll00<0
0?P0n0001OP000<00?lloc`00Oll000Doc`2001Moc`00`00n03h0004n0000`00occo?01aoc`00`00
n03h0005n0000`00occo?001oc`00003oc`00000008000?o?08000Co?08000Co?003003o??ll05co
?003003h0?P000Ch0003003o??ll077o?003003h0?P000Gh0003003o??ll007o?0000oll00D00?ll
occo?00000;o?004003o??ll0002oc`01@00occo??ll0000G_ll00<00?P0n0001?P000<00?lloc`0
LOll00<00?P0n0001OP000<00?lloc`00Oll0003oc`01@00occo??ll00000_ll00@00?lloc`000;o
?004003o??lloc`3001Loc`00`00n03h0004n0000`00occo?01aoc`00`00n03h0005n0000`00occo
?001oc`000?o?005003o??lloc`00002oc`01000occo?0000_ll00D00?llocco?00005ko?003003h
0?P000Ch0003003o??ll077o?003003h0?P000Gh0003003o??ll007o?0000oll00D00?llocco?000
00;o?004003o??ll0002oc`01@00occo??ll0000G_ll00<00?P0n0001?P000<00?lloc`0LOll00<0
0?P0n0001OP000<00?lloc`00Oll00001?ll000000000oll00@00?lloc`000;o?004003o??ll0003
oc`00`00occo?01Loc`00`00n03h0004n0000`00occo?01aoc`00`00n03h0005n0000`00occo?001
oc`000?o?003003o??ll00;o?08000Co?08000Co?08005go?003003h0?P000Ch0003003o??ll077o
?003003h0?P000Gh0003003o??ll007o?0005?ll00<00?lloc`0G?ll00<00?P0n0001?P000<00?ll
oc`0LOll00<00?P0n0001OP000<00?lloc`00Oll000Doc`00`00occo?01Loc`00`00n03h0004n000
0`00occo?01aoc`00`00n03h0005n0000`00occo?001oc`001Co?08005go?003003h0?P000Ch0003
003o??ll077o?003003h0?P000Gh0003003o??ll007o?0005?ll00<00?lloc`0G?ll00<00?P0n000
1?P000<00?lloc`0LOll00<00?P0n0001OP000<00?lloc`00Oll000Doc`00`00occo?01Loc`00`00
n03h0004n0000`00occo?01aoc`00`00n03h0005n0000`00occo?001oc`001Co?003003o??ll05co
?003003h0?P000Ch0003003o??ll077o?003003h0?P000Gh0003003o??ll007o?0005?ll0P00GOll
00<00?P0n0001?P000<00?lloc`0LOll00<00?P0n0001OP000<00?lloc`00Oll000Doc`00`00occo
?01Loc`00`00n03h0004n0000`00occo?01aoc`00`00n03h0005n0000`00occo?001oc`001Co?003
003o??ll05co?003003h0?P000Ch0003003o??ll077o?003003h0?P000Gh0003003o??ll007o?000
5?ll00<00?lloc`0G?ll00<00?P0n0001?P000<00?lloc`0LOll00<00?P0n0001OP000<00?lloc`0
0Oll000Doc`2001Moc`00`00n03h0004n0000`00occo?01aoc`00`00n03h0005n0000`00occo?001
oc`00003oc`00000008000?o?08000Co?08000Co?003003o??ll05co?003003h0?P000Ch0003003o
??ll077o?003003h0?P000Gh0003003o??ll007o?00000?o?000oc`01?ll00@00?lloc`000;o?004
003o??ll0003oc`00`00occo?01Loc`00`00n03h0004n0000`00occo?01aoc`00`00n03h0005n000
0`00occo?001oc`000;o?003003o??ll00;o?004003o??ll0002oc`01000occo?0000oll0`00G?ll
00<00?P0n0001?P000<00?lloc`0LOll00<00?P0n0001OP000<00?lloc`00Oll0003oc`01@00occo
??ll00000_ll00@00?lloc`000;o?005003o??lloc`0001Noc`00`00n03h0004n0000`00occo?01a
oc`00`00n03h0005n0000`00occo?001oc`000Co?004003o??ll0002oc`01000occo?0000_ll00D0
0?llocco?00005ko?003003h0?P000Ch0003003o??ll077o?003003h0?P000Gh0003003o??ll007o
?00000Go?000occo?00000;o?004003o??ll0002oc`01000occo?0000oll00<00?lloc`0G?ll00<0
0?P0n0001?P000<00?lloc`0LOll00<00?P0n0001OP000<00?lloc`00Oll0002oc`20004oc`20004
oc`20004oc`2001Moc`00`00n03h0004n0000`00occo?01aoc`00`00n03h0005n0000`00occo?001
oc`001Co?003003o??ll05co?003003h0?P000Ch0003003o??ll077o?003003h0?P000Gh0003003o
??ll007o?0005?ll00<00?lloc`0G?ll00<00?P0n0001?P000<00?lloc`0LOll00<00?P0n0001OP0
00<00?lloc`00Oll000Doc`2001Moc`00`00n03h0004n0000`00occo?01aoc`00`00n03h0005n000
0`00occo?001oc`001Co?003003o??ll05co?003003h0?P000Ch0003003o??ll077o?003003h0?P0
00Gh0003003o??ll007o?0005?ll00<00?lloc`0G?ll00<00?P0n0001?P000<00?lloc`0LOll00<0
0?P0n0001OP000<00?lloc`00Oll000Doc`00`00occo?01Loc`00`00n03h0004n0000`00occo?01a
oc`00`00n03h0005n0000`00occo?001oc`001Co?08005go?003003h0?P000Ch0003003o??ll077o
?003003h0?P000Gh0003003o??ll007o?0005?ll00<00?lloc`0G?ll00<00?P0n0001?P000<00?ll
oc`0LOll00<00?P0n0001OP000<00?lloc`00Oll000Doc`00`00occo?01Loc`00`00n03h0004n000
0`00occo?01aoc`00`00n03h0005n0000`00occo?001oc`001Co?08005go?003003h0?P000Ch0003
003o??ll077o?003003h0?P000Gh0003003o??ll007o?0005?ll00<00?lloc`0G?ll00<00?P0n000
1?P000<00?lloc`0LOll00<00?P0n0001OP000<00?lloc`00Oll0002oc`20004oc`20004oc`20004
oc`00`00occo?01Loc`00`00n03h0004n0000`00occo?01aoc`00`00n03h0005n0000`00occo?001
oc`00005oc`00?lloc`00002oc`01000occo?0000_ll00@00?lloc`000?o?003003o??ll05co?003
003h0?P000Ch0003003o??ll077o?003003h0?P000Gh0003003o??ll007o?0001?ll00@00?lloc`0
00;o?004003o??ll0002oc`01000occo??ll0`00G?ll00<00?P0n0001?P000<00?lloc`0LOll00<0
0?P0n0001OP000<00?lloc`00Oll0002oc`20003oc`01000occo?0000_ll00@00?lloc`000?o?003
003o??ll05co?003003h0?P000Ch0003003o??ll077o?003003h0?P000Gh0003003o??ll007o?000
1?ll00@00?lloc`000;o?004003o??ll0002oc`01@00occo??ll0000G_ll00<00?P0n0001?P000<0
0?lloc`0LOll00<00?P0n0001OP000<00?lloc`00Oll00001Oll003o??ll00000_ll00@00?lloc`0
00;o?004003o??ll0003oc`2001Moc`00`00n03h0004n0000`00occo?01aoc`00`00n03h0005n000
0`00occo?001oc`000;o?08000Co?08000Co?08000Co?003003o??ll05co?003003h0?P000Ch0003
003o??ll077o?003003h0?P000Gh0003003o??ll007o?0005?ll00<00?lloc`0G?ll00<00?P0n000
1?P000<00?lloc`0LOll00<00?P0n0001OP000<00?lloc`00Oll000Doc`00`00occo?01Loc`00`00
n03h0004n0000`00occo?01aoc`00`00n03h0005n0000`00occo?001oc`001Co?08005go?0P007?o
?003003h0?P000Gh0003003o??ll007o?0005?ll00<00?lloc`0eoll00<00?P0n0001OP000<00?ll
oc`00Oll000Doc`00`00occo?03Goc`00`00n03h0005n0000`00occo?001oc`001Co?0800=So?003
003h0?P000Gh0003003o??ll007o?0005?ll00<00?lloc`0eoll00<00?P0n0001OP000<00?lloc`0
0Oll000Doc`00`00occo?03Goc`00`00n03h0005n0000`00occo?001oc`001Co?003003o??ll0=Oo
?003003h0?P000Gh0003003o??ll007o?0005?ll0P00f?ll00<00?P0n0001OP000<00?lloc`00Oll
0003oc`00`00occo?002oc`20004oc`20004oc`00`00occo?03Goc`00`00n03h0005n0000`00occo
?001oc`000?o?005003o??lloc`00002oc`01000occo?0000_ll00D00?llocco?0000=Wo?003003h
0?P000Gh0003003o??ll007o?0001@000_ll00@00?lloc`000;o?004003o??ll0003oc`3003Goc`0
0`00n03h0005n0000`00occo?001oc`00004003o??ll0003oc`01000occo?0000_ll00@00?lloc`0
00?o?003003o??ll0=Oo?003003h0?P000Gh0003003o??ll007o?00000Co?000oc`000?o?004003o
??ll0002oc`01000occo?0000oll00<00?lloc`0eoll00<00?P0n0001OP000<00?lloc`00Oll0002
oc`20003oc`01000occo?0000_ll00@00?lloc`000?o?003003o??ll0=Oo?003003h0?P000Gh0003
003o??ll007o?0000oll00<00?lloc`00_ll0P001?ll0P001?ll0P00f?ll00<00?P0n0001OP000<0
0?lloc`00Oll000Doc`00`00occo?03Goc`00`00n03h0005n0000`00occo?001oc`001Co?003003o
??ll0=Oo?003003h0?P000Gh0003003o??ll007o?0005?ll0P00f?ll00<00?P0n0001OP000<00?ll
oc`00Oll000Doc`00`00occo?03Goc`00`00n03h0005n0000`00occo?001oc`001Co?003003o??ll
0=Oo?003003h0?P000Gh0003003o??ll007o?0005?ll00<00?lloc`0eoll00<00?P0n0001OP000<0
0?lloc`00Oll000Doc`2003Hoc`00`00n03h0005n0000`00occo?001oc`001Co?003003o??ll0=Oo
?003003h0?P000Gh0003003o??ll007o?0005?ll00<00?lloc`0eoll00<00?P0n0001OP000<00?ll
oc`00Oll000Doc`00`00occo?03Goc`00`00n03h0005n0000`00occo?001oc`001Co?0800=So?003
003h0?P000Gh0003003o??ll007o?00000Co?000000000Co?08000Co?08000Co?003003o??ll0=Oo
?003003h0?P000Gh0003003o??ll007o?0001?ll00@00?lloc`000;o?004003o??ll0002oc`01@00
occo??ll0000fOll00<00?P0n0001OP000<00?lloc`00Oll0004oc`01000occo?0000_ll00@00?ll
oc`000;o?004003o??lloc`3003Goc`00`00n03h0005n0000`00occo?001oc`00004oc`000000003
oc`01000occo?0000_ll00@00?lloc`000?o?003003o??ll0=Oo?003003h0?P000Gh0003003o??ll
007o?00000?o?000oc`01?ll00@00?lloc`000;o?004003o??ll0003oc`00`00occo?03Goc`00`00
n03h0005n0000`00occo?001oc`00003oc`00?ll00Co?004003o??ll0002oc`01000occo?0000oll
00<00?lloc`0eoll00<00?P0n0001OP000<00?lloc`00Oll00000oll000000020003oc`20004oc`2
0004oc`2003Hoc`00`00n03h0005n0000`00occo?001oc`001Co?003003o??ll0=Oo?003003h0?P0
00Gh0003003o??ll007o?0005?ll00<00?lloc`0eoll00<00?P0n0001OP000<00?lloc`00Oll000D
oc`2003Hoc`00`00n03h0005n0000`00occo?001oc`001Co?003003o??ll0=Oo?003003h0?P000Gh
0003003o??ll007o?0005?ll00<00?lloc`0eoll00<00?P0n0001OP000<00?lloc`00Oll000Doc`0
0`00occo?03Goc`00`00n03h0005n0000`00occo?001oc`001Co?0800=So?003003h0?P000Gh0003
003o??ll007o?0005?ll00<00?lloc`0eoll00<00?P0n0001OP000<00?lloc`00Oll000Doc`00`00
occo?03Goc`00`00n03h0005n0000`00occo?001oc`001Co?0800=So?003003h0?P000Gh0003003o
??ll007o?0005?ll00<00?lloc`0eoll00<00?P0n0001OP000<00?lloc`00Oll0002oc`20004oc`2
0004oc`20004oc`00`00occo?03Goc`00`00n03h0005n0000`00occo?001oc`00005oc`00?lloc`0
0002oc`01000occo?0000_ll00@00?lloc`000?o?003003o??ll0=Oo?003003h0?P000Gh0003003o
??ll007o?00000Go?000occo?00000;o?004003o??ll0002oc`01000occo?0000oll0`00eoll00<0
0?P0n0001OP000<00?lloc`00Oll00001?ll000000000oll00@00?lloc`000;o?004003o??ll0003
oc`00`00occo?03Goc`00`00n03h0005n0000`00occo?001oc`00003oc`00?ll00Co?004003o??ll
0002oc`01000occo?0000oll00<00?lloc`0eoll00<00?P0n0001OP000<00?lloc`00Oll00000oll
003o?004oc`01000occo?0000_ll00@00?lloc`000?o?0800=So?003003h0?P000Gh0003003o??ll
007o?0000_ll0`000oll0P001?ll0P001?ll00<00?lloc`0eoll00<00?P0n0001OP000<00?lloc`0
0Oll000Doc`00`00occo?03Goc`00`00n03h0005n0000`00occo?001oc`001Co?003003o??ll0=Oo
?003003h0?P000Gh0003003o??ll007o?0005?ll0P00f?ll00<00?P0n0001OP000<00?lloc`00Oll
000Doc`00`00occo?03Goc`00`00n03h0005n0000`00occo?001oc`001Co?003003o??ll0=Oo?003
003h0?P000Gh0003003o??ll007o?0005?ll0P00f?ll00<00?P0n0001OP000<00?lloc`00Oll000D
oc`00`00occo?03Goc`00`00n03h0005n0000`00occo?001oc`001Co?003003o??ll0=Oo?003003h
0?P000Gh0003003o??ll007o?0005?ll00<00?lloc`0eoll00<00?P0n0001OP000<00?lloc`00Oll
000Doc`2003Hoc`00`00n03h0005n0000`00occo?001oc`000;o?003003o??ll00?o?08000Co?080
00Co?003003o??ll0=Oo?003003h0?P000Gh0003003o??ll007o?0000_ll00<00?lloc`00_ll00@0
0?lloc`000;o?004003o??ll0003oc`00`00occo?03Goc`00`00n03h0005n0000`00occo?001oc`0
00?o?005003o??lloc`00002oc`01000occo?0000_ll00@00?llocco?0<00=Oo?003003h0?P000Gh
0003003o??ll007o?0000oll00D00?llocco?00000;o?004003o??ll0002oc`01@00occo??ll0000
fOll00<00?P0n0001OP000<00?lloc`00Oll0004oc`01000occo?0000_ll00@00?lloc`000;o?005
003o??lloc`0003Ioc`00`00n03h0005n0000`00occo?001oc`00005oc`00?lloc`00002oc`01000
occo?0000_ll00@00?lloc`000?o?003003o??ll0=Oo?003003h0?P000Gh0003003o??ll007o?000
00?o?00000000P000oll0P001?ll0P001?ll0P00f?ll00<00?P0n0001OP000<00?lloc`00Oll000D
oc`00`00occo?03Goc`00`00n03h0005n0000`00occo?001oc`001Co?003003o??ll0=Oo?003003h
0?P000Gh0003003o??ll007o?0005?ll00<00?lloc`0eoll00<00?P0n0001OP000<00?lloc`00Oll
000Doc`2003Hoc`00`00n03h0005n0000`00occo?001oc`001Co?003003o??ll0=Oo?0T000?o?000
5?ll00<00?lloc`0holl000Doc`2003Toc`001Co?003003o??ll0>?o?0005?ll00<00?lloc`0holl
003joc`00?[o?000\
\>"],
  ImageRangeCache->{{{0, 249}, {153.5, 0}} -> {0.22687, -83.3428, 0.0275332, \
5.65852}}],

Cell[BoxData[
    TagBox[\(\[SkeletonIndicator]  Graphics  \[SkeletonIndicator]\),
      False,
      Editable->False]], "Output"]
}, Open  ]]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Feasible and infeasible reactions", "Section",
  Evaluatable->False,
  TextJustification->1,
  FontFamily->"Times",
  FontSize->14,
  FontWeight->"Plain",
  FontSlant->"Italic",
  FontVariations->{"CompatibilityType"->0}],

Cell[TextData[{
  "Before calculating elementary vectors , the above routines simplify the \
network. Among other things, this means that they determine which reactions \
are feasible. i.e. can actually have a non-zero flow  in steady state given \
the current assignment of roles.  Since this can be of independent interest, \
SNA provides the function ",
  StyleBox["feasiblemnet",
    FontWeight->"Bold",
    FontSlant->"Plain",
    FontVariations->{"CompatibilityType"->0}],
  ", which returns the subnetwork consisting of the feasible reactions."
}], "Text",
  Evaluatable->False,
  FontFamily->"Times",
  FontSize->14,
  FontWeight->"Plain",
  FontSlant->"Italic"],

Cell[CellGroupData[{

Cell[BoxData[
    \(reactions@feasiblemnet[humit]\  // Length\)], "Input"],

Cell[BoxData[
    \(94\)], "Output"]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
    \(reactions@feasiblemnet[humit1]\  // Length\)], "Input"],

Cell[BoxData[
    \(90\)], "Output"]
}, Open  ]],

Cell[TextData[{
  "So of the total 123 reactions, 94 are feasible if any non mitochondrial \
metabolite is treated as external. But the more restrictive assignment of \
roles used for ",
  StyleBox["humit1",
    FontWeight->"Bold",
    FontSlant->"Plain",
    FontVariations->{"CompatibilityType"->0}],
  " reduces this to 90."
}], "Text",
  Evaluatable->False,
  FontFamily->"Times",
  FontSize->14,
  FontWeight->"Plain",
  FontSlant->"Italic"]
}, Open  ]]
},
FrontEndVersion->"5.0 for X",
ScreenRectangle->{{0, 1280}, {0, 1024}},
WindowSize->{773, 799},
WindowMargins->{{94, Automatic}, {Automatic, 23}},
StyleDefinitions -> "Demo.nb"
]

(*******************************************************************
Cached data follows.  If you edit this Notebook file directly, not
using Mathematica, you must remove the line containing CacheID at
the top of  the file.  The cache data will then be recreated when
you save this file from within Mathematica.
*******************************************************************)

(*CellTagsOutline
CellTagsIndex->{
  "Info3340454019-2206978"->{
    Cell[12467, 363, 66, 1, 23, "Print",
      CellTags->"Info3340454019-2206978"],
    Cell[12536, 366, 544, 15, 23, "Print",
      CellTags->"Info3340454019-2206978"]}
  }
*)

(*CellTagsIndex
CellTagsIndex->{
  {"Info3340454019-2206978", 70822, 2375}
  }
*)

(*NotebookFileOutline
Notebook[{
Cell[1754, 51, 131, 3, 35, "Title"],

Cell[CellGroupData[{
Cell[1910, 58, 282, 10, 43, "Subsubsection",
  Evaluatable->False],

Cell[CellGroupData[{
Cell[2217, 72, 67, 1, 31, "Input"],
Cell[2287, 75, 89, 1, 23, "Print"]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{
Cell[2425, 82, 324, 10, 66, "Subsubsection",
  Evaluatable->False],

Cell[CellGroupData[{
Cell[2774, 96, 149, 4, 47, "Input"],
Cell[2926, 102, 290, 4, 142, "Output"]
}, Open  ]],

Cell[CellGroupData[{
Cell[3253, 111, 60, 1, 31, "Input"],
Cell[3316, 114, 37, 1, 46, "Output"]
}, Open  ]],

Cell[CellGroupData[{
Cell[3390, 120, 50, 1, 31, "Input"],
Cell[3443, 123, 2854, 40, 280, "Output"]
}, Open  ]],
Cell[6312, 166, 719, 23, 60, "Text",
  Evaluatable->False]
}, Open  ]],

Cell[CellGroupData[{
Cell[7068, 194, 413, 12, 89, "Subsubsection",
  Evaluatable->False],

Cell[CellGroupData[{
Cell[7506, 210, 69, 1, 31, "Input"],
Cell[7578, 213, 425, 9, 48, "Output"]
}, Open  ]],
Cell[8018, 225, 172, 5, 30, "Text",
  Evaluatable->False],

Cell[CellGroupData[{
Cell[8215, 234, 106, 2, 31, "Input"],
Cell[8324, 238, 1910, 56, 119, "Output"]
}, Open  ]],
Cell[10249, 297, 313, 11, 30, "Text",
  Evaluatable->False],

Cell[CellGroupData[{
Cell[10587, 312, 62, 1, 31, "Input"],
Cell[10652, 315, 1054, 15, 102, "Output"]
}, Open  ]],
Cell[11721, 333, 676, 23, 72, "Text",
  Evaluatable->False],

Cell[CellGroupData[{
Cell[12422, 360, 42, 1, 31, "Input"],
Cell[12467, 363, 66, 1, 23, "Print",
  CellTags->"Info3340454019-2206978"],
Cell[12536, 366, 544, 15, 23, "Print",
  CellTags->"Info3340454019-2206978"]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{
Cell[13129, 387, 494, 15, 66, "Subsubsection",
  Evaluatable->False],
Cell[13626, 404, 469, 7, 98, "Input"],
Cell[14098, 413, 992, 33, 47, "Text",
  Evaluatable->False],
Cell[15093, 448, 203, 8, 30, "Text",
  Evaluatable->False],
Cell[15299, 458, 110, 2, 32, "Input"],
Cell[15412, 462, 127, 5, 20, "Subtitle",
  Evaluatable->False]
}, Open  ]],

Cell[CellGroupData[{
Cell[15576, 472, 297, 12, 75, "Section",
  Evaluatable->False],

Cell[CellGroupData[{
Cell[15898, 488, 378, 13, 43, "Subsubsection",
  Evaluatable->False],

Cell[CellGroupData[{
Cell[16301, 505, 101, 2, 47, "Input"],
Cell[16405, 509, 35, 1, 46, "Output"]
}, Open  ]],

Cell[CellGroupData[{
Cell[16477, 515, 49, 1, 31, "Input"],
Cell[16529, 518, 38, 1, 46, "Output"]
}, Open  ]],
Cell[16582, 522, 209, 8, 30, "Text",
  Evaluatable->False],

Cell[CellGroupData[{
Cell[16816, 534, 48, 1, 31, "Input"],
Cell[16867, 537, 73, 1, 46, "Output"]
}, Open  ]],

Cell[CellGroupData[{
Cell[16977, 543, 132, 2, 31, "Input"],
Cell[17112, 547, 847, 21, 97, "Output"]
}, Open  ]],
Cell[17974, 571, 172, 5, 30, "Text",
  Evaluatable->False],

Cell[CellGroupData[{
Cell[18171, 580, 48, 1, 31, "Input"],
Cell[18222, 583, 1268, 28, 98, "Output"]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{
Cell[19539, 617, 581, 20, 89, "Subsubsection",
  Evaluatable->False],

Cell[CellGroupData[{
Cell[20145, 641, 65, 1, 31, "Input"],
Cell[20213, 644, 298, 7, 46, "Output"]
}, Open  ]],
Cell[20526, 654, 165, 5, 30, "Text",
  Evaluatable->False],

Cell[CellGroupData[{
Cell[20716, 663, 116, 2, 47, "Input"],
Cell[20835, 667, 37, 1, 46, "Output"]
}, Open  ]],
Cell[20887, 671, 336, 12, 45, "Text",
  Evaluatable->False],

Cell[CellGroupData[{
Cell[21248, 687, 74, 1, 31, "Input"],
Cell[21325, 690, 38, 1, 46, "Output"]
}, Open  ]],
Cell[21378, 694, 212, 8, 30, "Text",
  Evaluatable->False],

Cell[CellGroupData[{
Cell[21615, 706, 176, 4, 47, "Input"],
Cell[21794, 712, 36, 1, 46, "Output"]
}, Open  ]],

Cell[CellGroupData[{
Cell[21867, 718, 56, 1, 31, "Input"],
Cell[21926, 721, 1321, 19, 222, "Output"]
}, Open  ]],
Cell[23262, 743, 284, 11, 30, "Text",
  Evaluatable->False],

Cell[CellGroupData[{
Cell[23571, 758, 208, 5, 47, "Input"],
Cell[23782, 765, 36, 1, 46, "Output"]
}, Open  ]],

Cell[CellGroupData[{
Cell[23855, 771, 82, 1, 31, "Input"],
Cell[23940, 774, 6743, 151, 695, "Output"]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{
Cell[30732, 931, 933, 29, 158, "Subsubsection",
  Evaluatable->False],

Cell[CellGroupData[{
Cell[31690, 964, 128, 3, 47, "Input"],
Cell[31821, 969, 361, 5, 39, "Message"],
Cell[32185, 976, 35, 1, 46, "Output"]
}, Open  ]],

Cell[CellGroupData[{
Cell[32257, 982, 50, 1, 31, "Input"],
Cell[32310, 985, 39, 1, 46, "Output"]
}, Open  ]],

Cell[CellGroupData[{
Cell[32386, 991, 68, 1, 31, "Input"],
Cell[32457, 994, 39, 1, 46, "Output"]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{
Cell[32545, 1001, 211, 8, 43, "Subsubsection",
  Evaluatable->False],

Cell[CellGroupData[{
Cell[32781, 1013, 48, 1, 31, "Input"],
Cell[32832, 1016, 1268, 28, 98, "Output"]
}, Open  ]],
Cell[34115, 1047, 384, 12, 57, "Text",
  Evaluatable->False],

Cell[CellGroupData[{
Cell[34524, 1063, 120, 2, 31, "Input"],
Cell[34647, 1067, 48, 1, 46, "Output"]
}, Open  ]],
Cell[34710, 1071, 501, 14, 72, "Text",
  Evaluatable->False],

Cell[CellGroupData[{
Cell[35236, 1089, 178, 3, 47, "Input"],
Cell[35417, 1094, 44, 1, 46, "Output"]
}, Open  ]],
Cell[35476, 1098, 362, 12, 30, "Text",
  Evaluatable->False],

Cell[CellGroupData[{
Cell[35863, 1114, 141, 2, 31, "Input"],
Cell[36007, 1118, 532, 8, 110, "Output"]
}, Open  ]],
Cell[36554, 1129, 429, 13, 45, "Text",
  Evaluatable->False],

Cell[CellGroupData[{
Cell[37008, 1146, 65, 1, 32, "Input"],
Cell[37076, 1149, 161, 4, 46, "Output"]
}, Open  ]],

Cell[CellGroupData[{
Cell[37274, 1158, 125, 2, 32, "Input"],
Cell[37402, 1162, 382, 10, 46, "Output"]
}, Open  ]],
Cell[37799, 1175, 503, 18, 45, "Text",
  Evaluatable->False],

Cell[CellGroupData[{
Cell[38327, 1197, 207, 4, 47, "Input"],
Cell[38537, 1203, 43, 1, 46, "Output"]
}, Open  ]]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{
Cell[38641, 1211, 304, 12, 75, "Section",
  Evaluatable->False],

Cell[CellGroupData[{
Cell[38970, 1227, 208, 8, 43, "Subsubsection",
  Evaluatable->False],

Cell[CellGroupData[{
Cell[39203, 1239, 69, 1, 31, "Input"],
Cell[39275, 1242, 37, 1, 46, "Output"]
}, Open  ]],
Cell[39327, 1246, 228, 8, 30, "Text",
  Evaluatable->False],

Cell[CellGroupData[{
Cell[39580, 1258, 129, 2, 47, "Input"],
Cell[39712, 1262, 38, 1, 46, "Output"]
}, Open  ]],

Cell[CellGroupData[{
Cell[39787, 1268, 49, 1, 31, "Input"],
Cell[39839, 1271, 830, 13, 102, "Output"]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{
Cell[40718, 1290, 255, 8, 43, "Subsubsection",
  Evaluatable->False],

Cell[CellGroupData[{
Cell[40998, 1302, 131, 2, 47, "Input"],
Cell[41132, 1306, 38, 1, 46, "Output"]
}, Open  ]],

Cell[CellGroupData[{
Cell[41207, 1312, 49, 1, 31, "Input"],
Cell[41259, 1315, 830, 13, 102, "Output"]
}, Open  ]],
Cell[42104, 1331, 214, 8, 30, "Text",
  Evaluatable->False],

Cell[CellGroupData[{
Cell[42343, 1343, 521, 14, 47, "Input"],
Cell[42867, 1359, 530, 8, 84, "Output"]
}, Open  ]],

Cell[CellGroupData[{
Cell[43434, 1372, 138, 3, 47, "Input"],
Cell[43575, 1377, 362, 5, 66, "Output"]
}, Open  ]],
Cell[43952, 1385, 161, 5, 30, "Text",
  Evaluatable->False],

Cell[CellGroupData[{
Cell[44138, 1394, 104, 2, 31, "Input"],
Cell[44245, 1398, 483, 7, 66, "Output"]
}, Open  ]],

Cell[CellGroupData[{
Cell[44765, 1410, 131, 3, 31, "Input"],
Cell[44899, 1415, 254, 4, 48, "Output"]
}, Open  ]],
Cell[45168, 1422, 570, 15, 60, "Text",
  Evaluatable->False],
Cell[45741, 1439, 589, 20, 31, "Text",
  Evaluatable->False],

Cell[CellGroupData[{
Cell[46355, 1463, 1243, 30, 81, "Input"],
Cell[47601, 1495, 20737, 790, 171, 8013, 628, "GraphicsData", "PostScript", \
"Graphics"],
Cell[68341, 2287, 130, 3, 46, "Output"]
}, Open  ]]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{
Cell[68532, 2297, 227, 7, 75, "Section",
  Evaluatable->False],
Cell[68762, 2306, 670, 16, 75, "Text",
  Evaluatable->False],

Cell[CellGroupData[{
Cell[69457, 2326, 74, 1, 31, "Input"],
Cell[69534, 2329, 36, 1, 46, "Output"]
}, Open  ]],

Cell[CellGroupData[{
Cell[69607, 2335, 75, 1, 31, "Input"],
Cell[69685, 2338, 36, 1, 46, "Output"]
}, Open  ]],
Cell[69736, 2342, 446, 14, 45, "Text",
  Evaluatable->False]
}, Open  ]]
}
]
*)



(*******************************************************************
End of Mathematica Notebook file.
*******************************************************************)

