(************** Content-type: application/mathematica **************
                     CreatedBy='Mathematica 5.0'

                    Mathematica-Compatible Notebook

This notebook can be used with any Mathematica-compatible
application, such as Mathematica, MathReader or Publicon. The data
for the notebook starts with the line containing stars above.

To get the notebook into a Mathematica-compatible application, do
one of the following:

* Save the data starting with the line of stars above into a file
  with a name ending in .nb, then open the file inside the
  application;

* Copy the data starting with the line of stars above to the
  clipboard, then use the Paste menu command inside the application.

Data for notebooks contains only printable 7-bit ASCII and can be
sent directly in email or through ftp in text mode.  Newlines can be
CR, LF or CRLF (Unix, Macintosh or MS-DOS style).

NOTE: If you modify the data for this notebook not in a Mathematica-
compatible application, you must delete the line below containing
the word CacheID, otherwise Mathematica-compatible applications may
try to use invalid cache data.

For more information on notebooks and Mathematica-compatible 
applications, contact Wolfram Research:
  web: http://www.wolfram.com
  email: info@wolfram.com
  phone: +1-217-398-0700 (U.S.)

Notebook reader applications are available free of charge from 
Wolfram Research.
*******************************************************************)

(*CacheID: 232*)


(*NotebookFileLineBreakTest
NotebookFileLineBreakTest*)
(*NotebookOptionsPosition[     58103,       1895]*)
(*NotebookOutlinePosition[     58764,       1918]*)
(*  CellTagsIndexPosition[     58720,       1914]*)
(*WindowFrame->Normal*)



Notebook[{
Cell[TextData[{
  StyleBox["Analyzing genome scale networks",
    FontSize->18,
    FontSlant->"Italic",
    FontVariations->{"CompatibilityType"->0}],
  StyleBox["\n",
    FontSize->18,
    FontVariations->{"CompatibilityType"->0}]
}], "Title"],

Cell[CellGroupData[{

Cell[BoxData[
    \(\(<< "\<../mathcode/SNAsym.m\>";\)\)], "Input"],

Cell[BoxData[
    \(LinkObject["../mathcode/../pairelvs/pairelvsIFsh", 2, 2]\)], "Print"]
}, Open  ]],

Cell[CellGroupData[{

Cell["The network", "Subsubsection",
  Evaluatable->False,
  TextJustification->1,
  FontFamily->"Times",
  FontSize->14,
  FontWeight->"Plain",
  FontSlant->"Italic",
  FontVariations->{"CompatibilityType"->0}],

Cell[CellGroupData[{

Cell[BoxData[{
    \(\({name, yeast}\  = \ << yeast.m;\)\), "\[IndentingNewLine]", 
    \(name\)}], "Input"],

Cell[BoxData[
    \("S.cerevisiae iND750\n\nDuarte, N.C., Herrgard, M.J., and Palsson, B.O.\
\nGenome Research, 14: 1298-1309,(2004)"\)], "Output"]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
    \(reactions@yeast // Length\)], "Input"],

Cell[BoxData[
    \(1150\)], "Output"]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
    \(metabolites@yeast // Length\)], "Input"],

Cell[BoxData[
    \(1062\)], "Output"]
}, Open  ]],

Cell["\<\
Just calculating a generating set for the flux cone of a network  \
of this magnitude seems  computationally intractable.\
\>", "Text",
  Evaluatable->False,
  FontFamily->"Times",
  FontSize->14,
  FontWeight->"Plain",
  FontSlant->"Italic"]
}, Open  ]],

Cell[CellGroupData[{

Cell["\<\
Hence, we shall go for the conversion cone but even this is \
daunting, as the following long list of extracellular metabolites shows. \
\
\>", "Subsubsection",
  Evaluatable->False,
  TextJustification->1,
  FontFamily->"Times",
  FontSize->14,
  FontWeight->"Plain",
  FontSlant->"Italic",
  FontVariations->{"CompatibilityType"->0}],

Cell[CellGroupData[{

Cell[BoxData[
    \(with["\<e\>", metabolites@yeast]\)], "Input"],

Cell[BoxData[
    \({"13BDglcn"\^"e", "4abut"\^"e", "5aop"\^"e", "8aonn"\^"e", "abt"\^"e", 
      "ac"\^"e", "acald"\^"e", "ade"\^"e", "adn"\^"e", "akg"\^"e", 
      "ala-L"\^"e", "alltn"\^"e", "alltt"\^"e", "amet"\^"e", "arab-D"\^"e", 
      "arab-L"\^"e", "arg-L"\^"e", "asn-L"\^"e", "asp-L"\^"e", 
      "Biomass"\^"e", "btn"\^"e", "chol"\^"e", "cit"\^"e", "co2"\^"e", 
      "crn"\^"e", "csn"\^"e", "cys-L"\^"e", "cytd"\^"e", "dad-2"\^"e", 
      "dann"\^"e", "dcyt"\^"e", "dgsn"\^"e", "din"\^"e", "dttp"\^"e", 
      "duri"\^"e", "ergst"\^"e", "etoh"\^"e", "fmn"\^"e", "for"\^"e", 
      "fru"\^"e", "fum"\^"e", "gal"\^"e", "gam6p"\^"e", "gcald"\^"e", 
      "glc-D"\^"e", "gln-L"\^"e", "glu-L"\^"e", "gly"\^"e", "glyc"\^"e", 
      "gsn"\^"e", "gthox"\^"e", "gua"\^"e", "h"\^"e", "h2o"\^"e", 
      "hdca"\^"e", "hdcea"\^"e", "his-L"\^"e", "hxan"\^"e", "ile-L"\^"e", 
      "inost"\^"e", "ins"\^"e", "k"\^"e", "lac-L"\^"e", "leu-L"\^"e", 
      "lys-L"\^"e", "mal-L"\^"e", "malt"\^"e", "man"\^"e", "melib"\^"e", 
      "met-L"\^"e", "mmet"\^"e", "na1"\^"e", "nh4"\^"e", "nmn"\^"e", 
      "o2"\^"e", "ocdca"\^"e", "ocdcea"\^"e", "ocdcya"\^"e", "orn"\^"e", 
      "pap"\^"e", "pepd"\^"e", "phe-L"\^"e", "pi"\^"e", "pnto-R"\^"e", 
      "pro-L"\^"e", "ptrc"\^"e", "pyr"\^"e", "rib-D"\^"e", "ribflv"\^"e", 
      "sbt-D"\^"e", "sbt-L"\^"e", "ser-L"\^"e", "so4"\^"e", "spmd"\^"e", 
      "sprm"\^"e", "srb-L"\^"e", "succ"\^"e", "sucr"\^"e", "thm"\^"e", 
      "thmmp"\^"e", "thmpp"\^"e", "thr-L"\^"e", "thym"\^"e", "thymd"\^"e", 
      "tre"\^"e", "trp-L"\^"e", "ttdca"\^"e", "tyr-L"\^"e", "ura"\^"e", 
      "urea"\^"e", "uri"\^"e", "val-L"\^"e", "xan"\^"e", "xtsn"\^"e", 
      "xyl-D"\^"e", "xylt"\^"e", "zymst"\^"e"}\)], "Output"]
}, Open  ]],

Cell[TextData[{
  "But it is possible to analyze meaningful restricted scenarios.\nFirst, we \
choose to ignore the organisms  consumption/productions of water by simply \
zeroing ",
  Cell[BoxData[
      StyleBox[\("h2o"\^"e"\),
        FontWeight->"Bold",
        FontSlant->"Plain",
        FontTracking->"Plain",
        FontVariations->{"Underline"->False,
        "Outline"->False,
        "Shadow"->False,
        "StrikeThrough"->False,
        "Masked"->False,
        "CompatibilityType"->0,
        "RotationAngle"->0}]]]
}], "Text",
  Evaluatable->False,
  FontFamily->"Times",
  FontSize->14,
  FontWeight->"Plain",
  FontSlant->"Italic"],

Cell[BoxData[
    \(\(yeast1\  = \ 
        constructmnet[\[IndentingNewLine]\ \ \ \ \ \ \ \ \ \ \ \ \ \ \ \ \ \ \
\ \ \ reactions@yeast\ \  /. \ "\<h2o\>"\^"\<e\>"\  \[Rule] \ 
              0, \[IndentingNewLine]\ \ \ \ \ \ \ \ \ \ \ \ \ \ \ \ \ \ \ \ \ \
tags@yeast\  /. \ R[x_]\  \[Rule] \ x];\)\)], "Input"],

Cell[TextData[{
  "The transformation ",
  StyleBox[" R[x_] \[Rule] x ",
    FontWeight->"Bold",
    FontSlant->"Plain",
    FontVariations->{"CompatibilityType"->0}],
  " strips the ",
  StyleBox["R ",
    FontWeight->"Bold",
    FontSlant->"Plain",
    FontVariations->{"CompatibilityType"->0}],
  " from the tags, because ",
  StyleBox["constructmnet ",
    FontWeight->"Bold",
    FontSlant->"Plain",
    FontVariations->{"CompatibilityType"->0}],
  " insists on wrapping an  ",
  StyleBox["R ",
    FontWeight->"Bold",
    FontSlant->"Plain",
    FontVariations->{"CompatibilityType"->0}],
  " around them.\nMore importantly we shall only treat the following subset \
of the extracellular metabolites as external."
}], "Text",
  Evaluatable->False,
  FontFamily->"Times",
  FontSize->14,
  FontWeight->"Plain",
  FontSlant->"Italic"],

Cell[CellGroupData[{

Cell[BoxData[
    \(Xts\  = \ {"\<ac\>"\^"\<e\>", "\<acald\>"\^"\<e\>", \
"\<ala-L\>"\^"\<e\>", "\<Biomass\>"\^"\<e\>", "\<co2\>"\^"\<e\>", \
"\<csn\>"\^"\<e\>", "\<ergst\>"\^"\<e\>", "\<etoh\>"\^"\<e\>", "\<gam6p\>"\^"\
\<e\>", "\<glc-D\>"\^"\<e\>", "\<hdcea\>"\^"\<e\>", "\<ocdcea\>"\^"\<e\>", \
"\<ocdcya\>"\^"\<e\>", "\<so4\>"\^"\<e\>", "\<xylt\>"\^"\<e\>", \
"\<zymst\>"\^"\<e\>", "\<nh4\>"\^"\<e\>", "\<asp-L\>"\^"\<e\>", \
"\<ser-L\>"\^"\<e\>", "\<fum\>"\^"\<e\>", "\<gly\>"\^"\<e\>", \
"\<thr-L\>"\^"\<e\>"}\)], "Input"],

Cell[BoxData[
    \({"ac"\^"e", "acald"\^"e", "ala-L"\^"e", "Biomass"\^"e", "co2"\^"e", 
      "csn"\^"e", "ergst"\^"e", "etoh"\^"e", "gam6p"\^"e", "glc-D"\^"e", 
      "hdcea"\^"e", "ocdcea"\^"e", "ocdcya"\^"e", "so4"\^"e", "xylt"\^"e", 
      "zymst"\^"e", "nh4"\^"e", "asp-L"\^"e", "ser-L"\^"e", "fum"\^"e", 
      "gly"\^"e", "thr-L"\^"e"}\)], "Output"]
}, Open  ]],

Cell[BoxData[
    \(\(yeast2\  = \ setrole[yeast1, Xts, Xt];\)\)], "Input"]
}, Open  ]],

Cell[CellGroupData[{

Cell["\<\
Calculating the conversion cone\
\>", "Section",
  Evaluatable->False,
  FontFamily->"Times",
  FontSize->14,
  FontWeight->"Plain",
  FontSlant->"Italic"],

Cell[CellGroupData[{

Cell[TextData[{
  "While it might just about work, simply typing ",
  StyleBox["conversiongset[yeast2]",
    FontWeight->"Bold",
    FontSlant->"Plain"],
  " or  ",
  StyleBox["conversionelvs[yeast2]  ",
    FontWeight->"Bold",
    FontSlant->"Plain"],
  "is not a really a good idea for a network of this size.\nBut the first \
thing ",
  StyleBox["conversiongset",
    FontWeight->"Bold",
    FontSlant->"Plain"],
  " and ",
  StyleBox["conversionelvs",
    FontWeight->"Bold",
    FontSlant->"Plain"],
  " do, is to use ",
  StyleBox["convfullsimp ",
    FontWeight->"Bold",
    FontSlant->"Plain"],
  "to compute a smaller network which has the same conversion cone. This \
should always be done,even if takes a while."
}], "Subsubsection",
  Evaluatable->False,
  TextJustification->1,
  FontFamily->"Times",
  FontSize->14,
  FontWeight->"Plain",
  FontSlant->"Italic",
  FontVariations->{"CompatibilityType"->0}],

Cell[CellGroupData[{

Cell[BoxData[
    \(\(ysimp = convfullsimp[yeast2];\)\)], "Input"],

Cell[BoxData[
    RowBox[{\(General::"spell1"\), \(\(:\)\(\ \)\), "\<\"Possible spelling \
error: new symbol name \\\"\\!\\(ysimp\\)\\\" is similar to existing symbol \
\\\"\\!\\(simp\\)\\\". \\!\\(\\*ButtonBox[\\\"More\[Ellipsis]\\\", \
ButtonStyle->\\\"RefGuideLinkText\\\", ButtonFrame->None, \
ButtonData:>\\\"General::spell1\\\"]\\)\"\>"}]], "Message"]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
    \(reactions@ysimp\  // Length\)], "Input"],

Cell[BoxData[
    \(45\)], "Output"]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
    \(\(\(metabolites@ysimp\)\(\ \)\)\)], "Input"],

Cell[BoxData[
    \({"accoa"\^"m", "adp"\^"m", "atp"\^"m", "co2"\^"c", "coa"\^"m", 
      "dtmp"\^"c", "glu-L"\^"m", "hcys-L"\^"c", "nadp"\^"c", "nadp"\^"m", 
      "nadph"\^"c", "nadph"\^"m", "pep"\^"c", "pro-L"\^"c", "r5p"\^"c", 
      "s7p"\^"c", "thr-L"\^"c", "tyr-L"\^"c", "tyr-L"\^"m", "ura"\^"c", 
      "uri"\^"c", "xu5p-D"\^"c", "ac"\_Xt\%\("e"\), "acald"\_Xt\%\("e"\), 
      "ala-L"\_Xt\%\("e"\), "asp-L"\_Xt\%\("e"\), "Biomass"\_Xt\%\("e"\), 
      "co2"\_Xt\%\("e"\), "csn"\_Xt\%\("e"\), "ergst"\_Xt\%\("e"\), 
      "etoh"\_Xt\%\("e"\), "fum"\_Xt\%\("e"\), "gam6p"\_Xt\%\("e"\), 
      "glc-D"\_Xt\%\("e"\), "gly"\_Xt\%\("e"\), "hdcea"\_Xt\%\("e"\), 
      "nh4"\_Xt\%\("e"\), "ocdcea"\_Xt\%\("e"\), "ocdcya"\_Xt\%\("e"\), 
      "ser-L"\_Xt\%\("e"\), "so4"\_Xt\%\("e"\), "thr-L"\_Xt\%\("e"\), 
      "xylt"\_Xt\%\("e"\), "zymst"\_Xt\%\("e"\)}\)], "Output"]
}, Open  ]],

Cell[TextData[{
  "While, compared to",
  StyleBox[" yeast2,",
    FontWeight->"Bold",
    FontSlant->"Plain",
    FontVariations->{"CompatibilityType"->0}],
  "  the achieved reduction is impressive, it is not quite a large as the \
reaction count indicates.\nThe reasons is that the reactions of ",
  StyleBox["ysimp ",
    FontWeight->"Bold",
    FontSlant->"Plain",
    FontVariations->{"CompatibilityType"->0}],
  "tend to be much more complicated than in ",
  StyleBox[" yeast2. ",
    FontWeight->"Bold",
    FontSlant->"Plain",
    FontVariations->{"CompatibilityType"->0}],
  " For instance "
}], "Text",
  Evaluatable->False,
  FontFamily->"Times",
  FontSize->14,
  FontWeight->"Plain",
  FontSlant->"Italic"],

Cell[CellGroupData[{

Cell[BoxData[
    \(\((reactions@ysimp)\)[\([4]\)]\)], "Input"],

Cell[BoxData[
    RowBox[{\("adp"\^"m" + "nadp"\^"m" + "nadph"\^"c" + 2\ "r5p"\^"c" + 
        2\ "tyr-L"\^"m" + "ura"\^"c" + "xu5p-D"\^"c" + "acald"\_Xt\%\("e"\)\),
       
      StyleBox["\[RightVector]",
        FontSize->18], \("atp"\^"m" + "nadp"\^"c" + "nadph"\^"m" + 
        "pep"\^"c" + "s7p"\^"c" + 2\ "tyr-L"\^"c" + "uri"\^"c" + 
        "etoh"\_Xt\%\("e"\)\)}]], "Output"]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[{
  "From  ",
  StyleBox["ysimp ",
    FontWeight->"Bold",
    FontSlant->"Plain",
    FontVariations->{"CompatibilityType"->0}],
  "the functions ",
  StyleBox["conversiongset",
    FontWeight->"Bold",
    FontSlant->"Plain"],
  " and ",
  StyleBox["conversionelvs",
    FontWeight->"Bold",
    FontSlant->"Plain"],
  "  calculate  in one go a minimal generating set or, respectively, the \
elementary vectors of the conversion cone.\nDoing this in one go , may not be \
a good idea. The reason is that a few reactions in ",
  StyleBox["ysimp",
    FontWeight->"Bold",
    FontSlant->"Plain",
    FontVariations->{"CompatibilityType"->0}],
  "  do no longer involve any internal metabolites, e.g."
}], "Subsubsection",
  Evaluatable->False,
  TextJustification->1,
  FontFamily->"Times",
  FontSize->14,
  FontWeight->"Plain",
  FontSlant->"Italic",
  FontVariations->{"CompatibilityType"->0}],

Cell[CellGroupData[{

Cell[BoxData[
    \(\((reactions@ysimp)\)[\([13]\)]\)], "Input"],

Cell[BoxData[
    RowBox[{\("glu-L"\^"m" + "r5p"\^"c" + "tyr-L"\^"c" + "xu5p-D"\^"c" + 
        "asp-L"\_Xt\%\("e"\) + "etoh"\_Xt\%\("e"\)\), 
      StyleBox["\[RightVector]",
        FontSize->18], \("pep"\^"c" + "pro-L"\^"c" + "s7p"\^"c" + 
        "tyr-L"\^"m" + "acald"\_Xt\%\("e"\) + "fum"\_Xt\%\("e"\) + 
        "nh4"\_Xt\%\("e"\)\)}]], "Output"]
}, Open  ]],

Cell[TextData[{
  "and are thus already in the conversion cone..\nSo we can split ",
  StyleBox["ysimp ",
    FontWeight->"Bold",
    FontSlant->"Plain",
    FontVariations->{"CompatibilityType"->0}],
  " into two subnets ",
  StyleBox["ni",
    FontWeight->"Bold"],
  " and ",
  StyleBox["nxt",
    FontWeight->"Bold"],
  ", where the reactions in ",
  StyleBox["nxt",
    FontWeight->"Bold"],
  " do not involve internal metabolites while the ones in",
  StyleBox[" ni",
    FontWeight->"Bold"],
  " do. We then find a minimal generating set of conversion for ",
  StyleBox["ni",
    FontWeight->"Bold"],
  "  and append the reactions from  ",
  StyleBox["nxt",
    FontWeight->"Bold"],
  "  to this set. This results in a generating set for the conversions cone \
of  ",
  StyleBox["ysimp. ",
    FontWeight->"Bold",
    FontSlant->"Plain",
    FontVariations->{"CompatibilityType"->0}],
  " This is just what the following two commands do:"
}], "Text",
  Evaluatable->False,
  FontFamily->"Times",
  FontSize->14,
  FontWeight->"Plain",
  FontSlant->"Italic"],

Cell[BoxData[{
    \(\(ints\  = \ with[Int, metabolites@ysimp];\)\), "\n", 
    \(\(yeastgset\  = \ partialconversions[ysimp, ints];\)\)}], "Input"],

Cell[TextData[{
  "As you will have noted, the computation took quite long, even though we \
focussed on the subnetwork ",
  StyleBox["ni",
    FontWeight->"Bold"],
  " .\nWhile we now have a generating set for the conversion cone of ",
  StyleBox["yeast2,",
    FontWeight->"Bold",
    FontSlant->"Plain",
    FontVariations->{"CompatibilityType"->0}],
  "   this set may not be minimal. The reason is that, due to splitting, any \
redundancies between the conversions of  ",
  StyleBox[" ni",
    FontWeight->"Bold"],
  "  and the reactions of ",
  StyleBox["nxt",
    FontWeight->"Bold"],
  " have been ignored. \nSince  ",
  StyleBox["partialconversions ",
    FontWeight->"Bold",
    FontSlant->"Plain",
    FontVariations->{"CompatibilityType"->0}],
  "returns an intermediate result, to facilitate further processing,  the \
output is  not a list of reactions but an ",
  StyleBox["mnet. ",
    FontWeight->"Bold",
    FontSlant->"Plain",
    FontVariations->{"CompatibilityType"->0}],
  "So to count the number of conversions:"
}], "Text",
  Evaluatable->False,
  FontFamily->"Times",
  FontSize->14,
  FontWeight->"Plain",
  FontSlant->"Italic"],

Cell[CellGroupData[{

Cell[BoxData[
    \(reactions@yeastgset\  // Length\)], "Input"],

Cell[BoxData[
    \(950\)], "Output"]
}, Open  ]],

Cell[TextData[{
  "Finally, a minimal generating set of the conversion cone of ",
  StyleBox["yeast2,  ",
    FontWeight->"Bold",
    FontSlant->"Plain",
    FontVariations->{"CompatibilityType"->0}],
  "is computed by"
}], "Text",
  Evaluatable->False,
  FontFamily->"Times",
  FontSize->14,
  FontWeight->"Plain",
  FontSlant->"Italic"],

Cell[CellGroupData[{

Cell[BoxData[{
    \(\(mingsetyeast2conv\  = \ conversions[yeastgset, ZH2gset];\)\), "\n", 
    \(reactions@mingsetyeast2conv\  // Length\)}], "Input"],

Cell[BoxData[
    \(372\)], "Output"]
}, Open  ]],

Cell[TextData[{
  StyleBox["conversions",
    FontWeight->"Bold",
    FontSlant->"Plain"],
  "  computes conversions of the network given as the first argument  even \
if, as in the above case, there  a no longer any internal metabolites.  The \
reactions of the output ",
  StyleBox["mnet",
    FontWeight->"Bold",
    FontSlant->"Plain",
    FontVariations->{"CompatibilityType"->0}],
  "  are a minimal generating set if the second argument is the  SNAmat \
routine",
  StyleBox[" ZH2gset;  ",
    FontWeight->"Bold",
    FontSlant->"Plain"],
  " if  instead ",
  StyleBox["ZH2elvs",
    FontWeight->"Bold",
    FontSlant->"Plain"],
  " is specified, all elementary vectors of the conversion coen are \
enumerated."
}], "Text",
  Evaluatable->False,
  FontFamily->"Times",
  FontSize->14,
  FontWeight->"Plain",
  FontSlant->"Italic"]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[{
  "So, if this is the  right way to do it, why isn't it just build into ",
  StyleBox["conversiongset",
    FontWeight->"Bold",
    FontSlant->"Plain"],
  "? The reason is, that it may not be the best way of doing it.\nNote that \
",
  StyleBox["partialconversions ",
    FontWeight->"Bold",
    FontSlant->"Plain",
    FontVariations->{"CompatibilityType"->0}],
  "has a second argument and, above, we gave the list\nof all internal \
metabolites for this argument. One can also pass a subset of internal \
metabolites and then ",
  StyleBox["partialconversions ",
    FontWeight->"Bold",
    FontSlant->"Plain",
    FontVariations->{"CompatibilityType"->0}],
  "will proceed as if only the metabolites in the subset where internal , \
treating all other metabolites as ",
  StyleBox["Xt",
    FontWeight->"Bold",
    FontSlant->"Plain"],
  ". In the next step, one can then throw out the remaining internal \
metabolites."
}], "Subsubsection",
  Evaluatable->False,
  TextJustification->1,
  FontFamily->"Times",
  FontSize->14,
  FontWeight->"Plain",
  FontSlant->"Italic",
  FontVariations->{"CompatibilityType"->0}],

Cell[CellGroupData[{

Cell[BoxData[{
    \(\(si = {"\<accoa\>"\^"\<m\>", "\<adp\>"\^"\<m\>", "\<atp\>"\^"\<m\>", "\
\<coa\>"\^"\<m\>", "\<glu-L\>"\^"\<m\>", "\<nadp\>"\^"\<m\>", \
"\<nadph\>"\^"\<m\>", "\<pro-L\>"\^"\<c\>", "\<tyr-L\>"\^"\<m\>"};\)\), "\n", \

    \(\(ysimp1 = partialconversions[ysimp, si];\)\), "\n", 
    \(reactions@mingsetyeast2conv\ \  === \ \ reactions@
        conversions[ysimp1, ZH2gset]\)}], "Input"],

Cell[BoxData[
    \(True\)], "Output"]
}, Open  ]],

Cell["\<\
So we have the same result as before, but you will have noticed the \
more than fivefold reduction in computing time.\
\>", "Text",
  Evaluatable->False,
  FontFamily->"Times",
  FontSize->14,
  FontWeight->"Plain",
  FontSlant->"Italic"]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[{
  "Of course you can choose any subset of the internal metabolites of  ",
  StyleBox["ysimp ",
    FontWeight->"Bold",
    FontSlant->"Plain",
    FontVariations->{"CompatibilityType"->0}],
  " for ",
  StyleBox["si. ",
    FontWeight->"Bold",
    FontSlant->"Plain"],
  " While the final result does not depend on the choice, the computation \
time does.\nThe problem of finding the optimal choice for ",
  StyleBox["si, ",
    FontWeight->"Bold",
    FontSlant->"Plain"],
  " probably is at least as hard as the one of computing a generating set for \
the conversions. However, SNA can help a bit  by making it easy to inspect \
the network.\nThis is how I arrived at the above choice for ",
  StyleBox["si",
    FontWeight->"Bold",
    FontSlant->"Plain"],
  ".  As a starting point I thought that throwing out  all of the remaining \
mitochondrial metabolites might be good idea. To investigate this:"
}], "Subsubsection",
  Evaluatable->False,
  TextJustification->1,
  FontFamily->"Times",
  FontSize->14,
  FontWeight->"Plain",
  FontSlant->"Italic",
  FontVariations->{"CompatibilityType"->0}],

Cell[CellGroupData[{

Cell[BoxData[
    \(si1\  = \ with["\<m\>", metabolites@ysimp]\)], "Input"],

Cell[BoxData[
    \({"accoa"\^"m", "adp"\^"m", "atp"\^"m", "coa"\^"m", "glu-L"\^"m", 
      "nadp"\^"m", "nadph"\^"m", "tyr-L"\^"m"}\)], "Output"]
}, Open  ]],

Cell[BoxData[
    \(\({nisi1, nxtsi1}\  = \ splitmnet[ysimp, si1];\)\)], "Input"],

Cell[TextData[{
  "This returns two ",
  StyleBox["mnet",
    FontWeight->"Bold",
    FontSlant->"Plain",
    FontVariations->{"CompatibilityType"->0}],
  "s with ",
  StyleBox["nisi1",
    FontWeight->"Bold",
    FontSlant->"Plain",
    FontVariations->{"CompatibilityType"->0}],
  " having all reaction in",
  StyleBox[" ysimp",
    FontWeight->"Bold",
    FontSlant->"Plain",
    FontVariations->{"CompatibilityType"->0}],
  " which use at least one metabolite of ",
  StyleBox["si1",
    FontWeight->"Bold",
    FontSlant->"Plain",
    FontVariations->{"CompatibilityType"->0}],
  " and  ",
  StyleBox["nxtsi1",
    FontWeight->"Bold",
    FontSlant->"Plain",
    FontVariations->{"CompatibilityType"->0}],
  " all other reactions. Now"
}], "Text",
  Evaluatable->False,
  FontFamily->"Times",
  FontSize->14,
  FontWeight->"Plain",
  FontSlant->"Italic"],

Cell[CellGroupData[{

Cell[BoxData[
    \(reactions@nisi1 // Length\)], "Input"],

Cell[BoxData[
    \(18\)], "Output"]
}, Open  ]],

Cell[TextData[{
  "So to get rid of the mitochondrial  metabolites, the function ",
  StyleBox["partialconversions",
    FontWeight->"Bold",
    FontSlant->"Plain",
    FontVariations->{"CompatibilityType"->0}],
  " just has to consider 18 reactions. This is not all that demanding. \
However"
}], "Text",
  Evaluatable->False,
  FontFamily->"Times",
  FontSize->14,
  FontWeight->"Plain",
  FontSlant->"Italic"],

Cell[CellGroupData[{

Cell[BoxData[
    \(with[Int, Complement[metabolites@nisi1, metabolites@nxtsi1]]\)], "Input"],

Cell[BoxData[
    \({"accoa"\^"m", "adp"\^"m", "atp"\^"m", "coa"\^"m", "glu-L"\^"m", 
      "nadp"\^"m", "nadph"\^"m", "pro-L"\^"c", "tyr-L"\^"m"}\)], "Output"]
}, Open  ]],

Cell[TextData[{
  "is a superset of ",
  StyleBox["si1",
    FontWeight->"Bold",
    FontSlant->"Plain",
    FontVariations->{"CompatibilityType"->0}],
  ", containing in addition",
  StyleBox[" ",
    FontWeight->"Bold",
    FontSlant->"Plain",
    FontVariations->{"CompatibilityType"->0}],
  Cell[BoxData[
      \("pro-L"\^"c"\)],
    FontWeight->"Bold",
    FontSlant->"Plain",
    FontVariations->{"CompatibilityType"->0}],
  ". So ",
  Cell[BoxData[
      \("pro-L"\^"c"\)],
    FontWeight->"Bold",
    FontSlant->"Plain",
    FontVariations->{"CompatibilityType"->0}],
  " is only used by ",
  StyleBox["nisi1",
    FontWeight->"Bold",
    FontSlant->"Plain",
    FontVariations->{"CompatibilityType"->0}],
  ".  Hence, it makes sense to add ",
  StyleBox[" ",
    FontWeight->"Bold",
    FontSlant->"Plain",
    FontVariations->{"CompatibilityType"->0}],
  Cell[BoxData[
      \("pro-L"\^"c"\)],
    FontWeight->"Bold",
    FontSlant->"Plain",
    FontVariations->{"CompatibilityType"->0}],
  " to the list of metabolites we eliminate already in the first step because \
this does not increase the subnetwork  ",
  StyleBox["partialconversions",
    FontWeight->"Bold",
    FontSlant->"Plain",
    FontVariations->{"CompatibilityType"->0}],
  " has to consider."
}], "Text",
  Evaluatable->False,
  FontFamily->"Times",
  FontSize->14,
  FontWeight->"Plain",
  FontSlant->"Italic"],

Cell[CellGroupData[{

Cell[BoxData[
    \(si\  === \ si1 \[Union] {"\<pro-L\>"\^"\<c\>"}\)], "Input"],

Cell[BoxData[
    \(True\)], "Output"]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[{
  "Finally, to compute all elementary conversion  of ",
  StyleBox["yeast2,  ",
    FontWeight->"Bold",
    FontSlant->"Plain",
    FontVariations->{"CompatibilityType"->0}],
  "we can again use ",
  StyleBox["conversions",
    FontWeight->"Bold",
    FontSlant->"Plain"],
  StyleBox[" ",
    FontWeight->"Bold",
    FontSlant->"Plain",
    FontVariations->{"CompatibilityType"->0}],
  "but with the SNAmat routine ",
  StyleBox["ZH2elvs",
    FontWeight->"Bold",
    FontSlant->"Plain"],
  " instead of ",
  StyleBox[" ZH2gset",
    FontWeight->"Bold",
    FontSlant->"Plain"],
  ".  As first argument we might as well  use ",
  StyleBox["mingsetyeast2conv",
    FontWeight->"Bold",
    FontSlant->"Plain"],
  ", the minimal generating set of the conversion cone."
}], "Subsubsection",
  Evaluatable->False,
  TextJustification->1,
  FontFamily->"Times",
  FontSize->14,
  FontWeight->"Plain",
  FontSlant->"Italic",
  FontVariations->{"CompatibilityType"->0}],

Cell[CellGroupData[{

Cell[BoxData[{
    \(\(elvsyeast2conv\  = \ 
        conversions[mingsetyeast2conv, ZH2elvs];\)\), "\n", 
    \(reactions@elvsyeast2conv\  // Length\)}], "Input"],

Cell[BoxData[
    \(40969\)], "Output"]
}, Open  ]]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["\<\
Decomposing elementary conversions into elementary fluxes\
\>", \
"Section",
  Evaluatable->False,
  FontFamily->"Times",
  FontSize->14,
  FontWeight->"Plain",
  FontSlant->"Italic"],

Cell[CellGroupData[{

Cell["Allow me to pick my favorite conversion.", "Subsubsection",
  Evaluatable->False,
  FontFamily->"Times",
  FontSize->14,
  FontWeight->"Plain",
  FontSlant->"Italic"],

Cell[CellGroupData[{

Cell[BoxData[
    \(myconv = with[27266477, reactions@elvsyeast2conv] // First\)], "Input"],

Cell[BoxData[
    RowBox[{\(90626\ "ac"\_Xt\%\("e"\) + 23710211\ "acald"\_Xt\%\("e"\) + 
        5530\ "csn"\_Xt\%\("e"\) + 35\ "ergst"\_Xt\%\("e"\) + 
        23052269\ "fum"\_Xt\%\("e"\) + 9885\ "gam6p"\_Xt\%\("e"\) + 
        4033772\ "glc-D"\_Xt\%\("e"\) + 476\ "hdcea"\_Xt\%\("e"\) + 
        23401575\ "nh4"\_Xt\%\("e"\) + 672\ "ocdcea"\_Xt\%\("e"\) + 
        252\ "ocdcya"\_Xt\%\("e"\) + 3865\ "so4"\_Xt\%\("e"\) + 
        75\ "zymst"\_Xt\%\("e"\)\), 
      StyleBox["\[RightVector]",
        FontSize->18], \(22812604\ "asp-L"\_Xt\%\("e"\) + 
        50000\ "Biomass"\_Xt\%\("e"\) + 15522497\ "co2"\_Xt\%\("e"\) + 
        27266477\ "etoh"\_Xt\%\("e"\) + 323346\ "ser-L"\_Xt\%\("e"\) + 
        12475\ "thr-L"\_Xt\%\("e"\)\)}]], "Output"]
}, Open  ]],

Cell["\<\
The goal will be to find out how many elementary fluxes give rise \
to this single conversion.
For this first invert the conversion\
\>", "Text",
  Evaluatable->False,
  FontFamily->"Times",
  FontSize->14,
  FontWeight->"Plain",
  FontSlant->"Italic"],

Cell[CellGroupData[{

Cell[BoxData[
    RowBox[{"vnocym", " ", "=", " ", 
      RowBox[{"myconv", " ", "/.", " ", 
        RowBox[{
          RowBox[{"(", 
            RowBox[{"a_", " ", 
              StyleBox["\[RightVector]",
                FontSize->18], " ", "b_"}], ")"}], " ", "\[Rule]", "  ", 
          RowBox[{"(", 
            RowBox[{"b", " ", 
              StyleBox["\[RightVector]",
                FontSize->18], 
              StyleBox[" ",
                FontSize->18], "a"}], ")"}]}]}]}]], "Input"],

Cell[BoxData[
    RowBox[{\(22812604\ "asp-L"\_Xt\%\("e"\) + 
        50000\ "Biomass"\_Xt\%\("e"\) + 15522497\ "co2"\_Xt\%\("e"\) + 
        27266477\ "etoh"\_Xt\%\("e"\) + 323346\ "ser-L"\_Xt\%\("e"\) + 
        12475\ "thr-L"\_Xt\%\("e"\)\), 
      StyleBox["\[RightVector]",
        FontSize->18], \(90626\ "ac"\_Xt\%\("e"\) + 
        23710211\ "acald"\_Xt\%\("e"\) + 5530\ "csn"\_Xt\%\("e"\) + 
        35\ "ergst"\_Xt\%\("e"\) + 23052269\ "fum"\_Xt\%\("e"\) + 
        9885\ "gam6p"\_Xt\%\("e"\) + 4033772\ "glc-D"\_Xt\%\("e"\) + 
        476\ "hdcea"\_Xt\%\("e"\) + 23401575\ "nh4"\_Xt\%\("e"\) + 
        672\ "ocdcea"\_Xt\%\("e"\) + 252\ "ocdcya"\_Xt\%\("e"\) + 
        3865\ "so4"\_Xt\%\("e"\) + 75\ "zymst"\_Xt\%\("e"\)\)}]], "Output"]
}, Open  ]],

Cell[TextData[{
  "append  ",
  StyleBox["vnocym",
    FontWeight->"Bold",
    FontSlant->"Plain",
    FontVariations->{"CompatibilityType"->0}],
  " to the reactions in ",
  StyleBox["yeast2 ",
    FontWeight->"Bold",
    FontSlant->"Plain",
    FontVariations->{"CompatibilityType"->0}],
  "and make all metabolites internal."
}], "Text",
  Evaluatable->False,
  FontFamily->"Times",
  FontSize->14,
  FontWeight->"Plain",
  FontSlant->"Italic"],

Cell[BoxData[{
    \(\(restricted\  = \ 
        joinmnet[\[IndentingNewLine]\ \ \ \ \ \ \ \ \ \ \ \ \ \ \ \ \ \ \ \ \ \
\ \ \ \ \ \ \ \ yeast2, \[IndentingNewLine]\ \ \ \ \ \ \ \ \ \ \ \ \ \ \ \ \ \
\ \ \ \ \ \ \ \ \ \ \ \ constructmnet[{vnocym}, {"\<vnocym\>"}]];\)\), "\n", 
    \(\(restricted\  = \ 
        setrole[restricted, metabolites@restricted, Int];\)\)}], "Input"],

Cell[TextData[{
  "Now any elementary flux of",
  StyleBox[" restricted ",
    FontWeight->"Bold",
    FontSlant->"Plain",
    FontVariations->{"CompatibilityType"->0}],
  "corresponds to either a futile cycle of  ",
  StyleBox["yeast2 ",
    FontWeight->"Bold",
    FontSlant->"Plain",
    FontVariations->{"CompatibilityType"->0}],
  "or to an elementary flux of  ",
  StyleBox["yeast2",
    FontWeight->"Bold",
    FontSlant->"Plain",
    FontVariations->{"CompatibilityType"->0}],
  " which has",
  StyleBox[" myconv ",
    FontWeight->"Bold",
    FontSlant->"Plain",
    FontVariations->{"CompatibilityType"->0}],
  "as conversion. \nWe first calculate a minimal generating set of the flux \
cone by"
}], "Text",
  Evaluatable->False,
  FontFamily->"Times",
  FontSize->14,
  FontWeight->"Plain",
  FontSlant->"Italic"],

Cell[CellGroupData[{

Cell[BoxData[{
    \(\({gfluxes, nrev}\  = \ 
        symfluxgset[restricted];\)\), "\[IndentingNewLine]", 
    \(nrev\)}], "Input"],

Cell[BoxData[
    \(9\)], "Output"]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
    \({gfluxes // Length, \ with["\<vnocym\>", gfluxes] // Length}\)], "Input"],

Cell[BoxData[
    \({1044, 1024}\)], "Output"]
}, Open  ]],

Cell[TextData[{
  "So there only  20 futile cycles and the 1024 fluxes have ",
  StyleBox["myconv ",
    FontWeight->"Bold",
    FontSlant->"Plain",
    FontVariations->{"CompatibilityType"->0}],
  "as conversion."
}], "Text",
  Evaluatable->False,
  FontFamily->"Times",
  FontSize->14,
  FontWeight->"Plain",
  FontSlant->"Italic"]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[{
  "Now, one  might further wish to enumerate all elementary fluxes for ",
  StyleBox["restricted.  ",
    FontWeight->"Bold",
    FontSlant->"Plain",
    FontVariations->{"CompatibilityType"->0}],
  "But their number is  3.7 \[Times] 10",
  Cell[BoxData[
      \(\^7\)]],
  ",  so trying to use ",
  StyleBox["symfluxelvs ",
    FontWeight->"Bold",
    FontSlant->"Plain",
    FontVariations->{"CompatibilityType"->0}],
  "would fail disastrously.\nThe method presented below, however, might just \
about make it possible to enumerate all  the elementary fluxes, even if we \
only illustrate it for the minimal generating set.\nInitially we proceed as \
in Tutorial 7 and find the generating fluxes in vector form for the reduced \
network of ",
  StyleBox["restricted.",
    FontWeight->"Bold",
    FontSlant->"Plain",
    FontVariations->{"CompatibilityType"->0}]
}], "Subsubsection",
  Evaluatable->False,
  FontFamily->"Times",
  FontSize->14,
  FontWeight->"Plain",
  FontSlant->"Italic"],

Cell[CellGroupData[{

Cell[BoxData[{
    \(\(rrestricted = fluxsimp[restricted];\)\), "\[IndentingNewLine]", 
    \({stoich, nrev} = mnet2stoich[rrestricted]; {rgfluxes, rnrev}\  = \ 
      fluxgset[stoich, nrev];\), "\[IndentingNewLine]", 
    \(rnrev\)}], "Input"],

Cell[BoxData[
    RowBox[{\(General::"spell1"\), \(\(:\)\(\ \)\), "\<\"Possible spelling \
error: new symbol name \\\"\\!\\(rrestricted\\)\\\" is similar to existing \
symbol \\\"\\!\\(restricted\\)\\\". \
\\!\\(\\*ButtonBox[\\\"More\[Ellipsis]\\\", ButtonStyle->\\\"RefGuideLinkText\
\\\", ButtonFrame->None, ButtonData:>\\\"General::spell1\\\"]\\)\"\>"}]], \
"Message"],

Cell[BoxData[
    RowBox[{\(General::"spell1"\), \(\(:\)\(\ \)\), "\<\"Possible spelling \
error: new symbol name \\\"\\!\\(rgfluxes\\)\\\" is similar to existing \
symbol \\\"\\!\\(gfluxes\\)\\\". \\!\\(\\*ButtonBox[\\\"More\[Ellipsis]\\\", \
ButtonStyle->\\\"RefGuideLinkText\\\", ButtonFrame->None, \
ButtonData:>\\\"General::spell1\\\"]\\)\"\>"}]], "Message"],

Cell[BoxData[
    RowBox[{\(General::"spell"\), \(\(:\)\(\ \)\), "\<\"Possible spelling \
error: new symbol name \\\"\\!\\(rnrev\\)\\\" is similar to existing symbols \
\\!\\({nnrev, nrev}\\). \\!\\(\\*ButtonBox[\\\"More\[Ellipsis]\\\", \
ButtonStyle->\\\"RefGuideLinkText\\\", ButtonFrame->None, \
ButtonData:>\\\"General::spell\\\"]\\)\"\>"}]], "Message"],

Cell[BoxData[
    \(9\)], "Output"]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
    \(rgfluxes // Length\)], "Input"],

Cell[BoxData[
    \(144\)], "Output"]
}, Open  ]],

Cell[TextData[{
  "So the reduced network has fewer generating fluxes than  ",
  StyleBox["restricted.",
    FontWeight->"Bold",
    FontSlant->"Plain",
    FontVariations->{"CompatibilityType"->0}],
  "\nThe reason, becomes apparent by inspecting the tags of  ",
  StyleBox["rrestricted.",
    FontWeight->"Bold",
    FontSlant->"Plain",
    FontVariations->{"CompatibilityType"->0}]
}], "Text",
  Evaluatable->False,
  FontFamily->"Times",
  FontSize->14,
  FontWeight->"Plain",
  FontSlant->"Italic"],

Cell[CellGroupData[{

Cell[BoxData[
    \({\(with[ALT, tags@rrestricted]\)[\([2]\)]} // TableForm\)], "Input"],

Cell[BoxData[
    InterpretationBox[GridBox[{
          {\(65250\ ALT[R["CRNCARtm"] + R["CSNATifm"], 
                  R["ACRNtim"] + R["CRNtim"] + R["CSNATifm"]] + 
              25925\ ALT[R["GLCS2"], R["GBEZ"] + R["GLYGS"]] + 
              4033772\ ALT[\(-R["G6PI"]\) + R["GLUK"], R["HEX1"]] + 
              56740\ R["13GS"] + 14310\ R["2OXOADPtim"] - 
              28050\ R["3MOBtm"] - 9635\ R["3MOPtm"] + 14310\ R["AASAD2"] + 
              14310\ R["AATA"] + 23710211\ R["ACALDt"] + 8120\ R["ACCOACr"] + 
              8035\ R["ACGKm"] + 9635\ R["ACHBSm"] + 28050\ R["ACLSm"] + 
              92455\ R["ACOAH"] + 8035\ R["ACOTAim"] + 90626\ R["ACt2r"] + 
              900\ R["ADNK1"] + 2865\ R["ADSK"] + 5795\ R["ADSL1r"] + 
              4900\ R["ADSL2r"] + 5795\ R["ADSS"] + 1235\ R["AGAT_SC"] + 
              8035\ R["AGPRim"] + 900\ R["AHCi"] + 2535\ R["AHSERL2"] + 
              8215\ R["AICART"] + 4900\ R["AIRCr"] - 22940\ R["ALATA_L"] - 
              27266477\ R["ALCD2x"] + 1420\ R["ANPRT"] + 1420\ R["ANS"] + 
              8035\ R["ARGSL"] + 8035\ R["ARGSSr"] + 34215\ R["ASADi"] + 
              5085\ R["ASNS1"] + 34215\ R["ASPKi"] - 22812604\ R["ASPt2r"] + 
              3315\ R["ATPPRT"] + 8035\ R["ATPS3m"] + 50000\ R["BioM"] + 
              2865\ R["BPNT"] + 8035\ R["CBPS"] + 11795\ R["CHORM"] + 
              13215\ R["CHORS"] - 50940\ R["CITtam"] - 15522497\ R["CO2t"] - 
              187070\ R["CO2tm"] + 50940\ R["CSm"] + 65250\ R["CSNAT"] + 
              5530\ R["CSND"] + 5530\ R["CSNt2"] + 2235\ R["CTPS1"] + 
              330\ R["CYSS"] - 1360\ R["CYTK1"] - 180\ R["DADK"] + 
              330\ R["DAGPYP_SC"] + 875\ R["DASYN_SC"] - 120\ R["DCMPDA"] + 
              13215\ R["DDPA"] - 120\ R["DGK1"] + 28050\ R["DHAD1m"] + 
              9635\ R["DHAD2m"] + 180\ R["DHFRim"] + 180\ R["DHFtm"] + 
              13215\ R["DHQS"] + 13215\ R["DHQTi"] + 40395\ R["DOLPMMer"] + 
              40395\ R["DOLPMTcer"] - 40395\ R["DOLPt2er"] + 
              3535401\ R["ENO"] + 35\ R["ERGSTt"] - 27266477\ R["ETOHt"] + 
              476\ R["FACOAL161"] + 672\ R["FACOAL181"] + 
              252\ R["FACOAL182"] + 1400\ R["FAS100COA"] + 
              1344\ R["FAS120COA"] + 1176\ R["FAS140COA"] + 
              1400\ R["FAS80COA_L"] + 2690\ R["FDH"] - 2690\ R["FTHFL"] + 
              23052269\ R["FUMt2r"] + 1235\ R["G3PD1ir"] + 
              8235\ R["G5SADr"] + 8235\ R["G5SD2"] + 9885\ R["G6PDA"] + 
              11710431\ R["G6PDH2"] + 9885\ R["GAM6Pt"] + 
              3889797\ R["GAPD"] + 4900\ R["GARFTi"] + 1235\ R["GAT1_SC"] + 
              4033772\ R["GLCt1"] + 35345\ R["GLNS"] + 8235\ R["GLU5K"] - 
              23388925\ R["GLUDy"] + 4900\ R["GLUPRT"] + 2420\ R["GMPS2"] + 
              11710431\ R["GND"] + 11346714\ R["H2Ot"] + 14310\ R["HACNHm"] + 
              14310\ R["HCITSm"] + 16155\ R["HCO3E"] + 476\ R["HDCEAt"] + 
              14310\ R["HICITDm"] + 3315\ R["HISTD"] + 3315\ R["HISTP"] + 
              34215\ R["HSDxi"] + 2535\ R["HSERTA"] + 31680\ R["HSK"] + 
              3315\ R["HSTPT"] + 50940\ R["ICDHy"] + 3315\ R["IG3PS"] + 
              3315\ R["IGPDH"] + 1420\ R["IGPS"] - 9635\ R["ILETA"] - 
              8215\ R["IMPC"] + 2420\ R["IMPD"] + 14820\ R["IPMD"] - 
              14820\ R["IPPMIa"] - 14820\ R["IPPMIb"] + 14820\ R["IPPS"] + 
              28050\ R["KARA1im"] + 9635\ R["KARA2im"] - 14820\ R["LEUTA"] + 
              40395\ R["MAN1PT"] - 40395\ R["MAN6PI"] - 
              40395\ R["MANNANter"] + 14310\ R["MCITDm"] + 
              23039409\ R["MDH"] - 102960\ R["MDHm"] + 88650\ R["ME1m"] + 
              45900\ R["ME2m"] + 900\ R["METAT"] + 3435\ R["METS"] + 
              300\ R["MFAPS_SC"] + 265\ R["MI1PP"] + 265\ R["MI1PS"] + 
              15805\ R["MTHFC"] + 15805\ R["MTHFD"] + 3435\ R["MTHFR3"] - 
              1360\ R["NDPK3"] + 23401575\ R["NH4t"] - 9635\ R["NH4tm"] + 
              8035\ R["OCBTi"] + 672\ R["OCDCEAt"] + 252\ R["OCDCYAt"] + 
              14820\ R["OMCDC"] + 8035\ R["ORNt3m"] + 8035\ R["ORNTACim"] + 
              14310\ R["OXAGm"] + 8235\ R["P5CR"] + 
              2865\ R["PAPSR"] - \(21\ R["PEtm_SC"]\)\/4 + 
              300\ R["PETOHM_SC"] + 354396\ R["PGCD"] - 7761929\ R["PGI"] - 
              3889797\ R["PGK"] + 11710431\ R["PGL"] - 3535401\ R["PGM"] - 
              83835\ R["PGMT"] - 6695\ R["PHETA1"] + 265\ R["PINOS_SC"] - 
              40395\ R["PMANM"] + 300\ R["PMETM_SC"] + 170205\ R["PPA"] - 
              5530\ R["PPM"] + 5100\ R["PPND2"] + 6695\ R["PPNDH"] + 
              4900\ R["PRAGSr"] + 1420\ R["PRAIi"] + 4900\ R["PRAIS"] + 
              3315\ R["PRAMPC"] + 4900\ R["PRASCS"] + 3315\ R["PRATPP"] + 
              4900\ R["PRFGS"] + 3315\ R["PRMICIi"] + 9635\ R["PRPPS"] + 
              13215\ R["PSCVTi"] + 525\ R["PSERDm_SC"] + 610\ R["PSERS_SC"] + 
              354396\ R["PSERT"] + 
              354396\ R["PSP_L"] + \(21\ R["PStm_SC"]\)\/4 + 
              3508971\ R["PYK"] + 3556266\ R["PYRDC"] - 68815\ R["PYRt2m"] + 
              7792439\ R["RPE"] - 3917992\ R["RPI"] + 14310\ R["SACCD1"] + 
              14310\ R["SACCD2"] + 330\ R["SERATi"] - 323346\ R["SERt2r"] + 
              13215\ R["SHK3D"] + 13215\ R["SHKK"] + 2865\ R["SLFAT"] + 
              3865\ R["SO4ti"] - 2865\ R["SULR"] + 3902827\ R["TALA"] + 
              9635\ R["THRD_Lm"] + 31680\ R["THRS"] + 9635\ R["THRt2m"] - 
              12475\ R["THRt2r"] + 3902827\ R["TKT1"] + 3889612\ R["TKT2"] + 
              180\ R["TMDS"] - 1235\ R["TPI"] + 2865\ R["TRDR"] + 
              1170\ R["TRE6PP"] + 1170\ R["TRE6PS"] + 330\ R["TRIGS_SC"] + 
              1420\ R["TRPS1"] + 2535\ R["UMPK"] - 13230\ R["VALTA"] + 
              R["vnocym"] + 75\ R["ZYMSTt"]\)}
          },
        RowSpacings->1,
        ColumnSpacings->3,
        RowAlignments->Baseline,
        ColumnAlignments->{Left}],
      TableForm[ {
        Plus[ 
          Times[ 65250, 
            ALT[ 
              Plus[ 
                R[ "CRNCARtm"], 
                R[ "CSNATifm"]], 
              Plus[ 
                R[ "ACRNtim"], 
                R[ "CRNtim"], 
                R[ "CSNATifm"]]]], 
          Times[ 25925, 
            ALT[ 
              R[ "GLCS2"], 
              Plus[ 
                R[ "GBEZ"], 
                R[ "GLYGS"]]]], 
          Times[ 4033772, 
            ALT[ 
              Plus[ 
                Times[ -1, 
                  R[ "G6PI"]], 
                R[ "GLUK"]], 
              R[ "HEX1"]]], 
          Times[ 56740, 
            R[ "13GS"]], 
          Times[ 14310, 
            R[ "2OXOADPtim"]], 
          Times[ -28050, 
            R[ "3MOBtm"]], 
          Times[ -9635, 
            R[ "3MOPtm"]], 
          Times[ 14310, 
            R[ "AASAD2"]], 
          Times[ 14310, 
            R[ "AATA"]], 
          Times[ 23710211, 
            R[ "ACALDt"]], 
          Times[ 8120, 
            R[ "ACCOACr"]], 
          Times[ 8035, 
            R[ "ACGKm"]], 
          Times[ 9635, 
            R[ "ACHBSm"]], 
          Times[ 28050, 
            R[ "ACLSm"]], 
          Times[ 92455, 
            R[ "ACOAH"]], 
          Times[ 8035, 
            R[ "ACOTAim"]], 
          Times[ 90626, 
            R[ "ACt2r"]], 
          Times[ 900, 
            R[ "ADNK1"]], 
          Times[ 2865, 
            R[ "ADSK"]], 
          Times[ 5795, 
            R[ "ADSL1r"]], 
          Times[ 4900, 
            R[ "ADSL2r"]], 
          Times[ 5795, 
            R[ "ADSS"]], 
          Times[ 1235, 
            R[ "AGAT_SC"]], 
          Times[ 8035, 
            R[ "AGPRim"]], 
          Times[ 900, 
            R[ "AHCi"]], 
          Times[ 2535, 
            R[ "AHSERL2"]], 
          Times[ 8215, 
            R[ "AICART"]], 
          Times[ 4900, 
            R[ "AIRCr"]], 
          Times[ -22940, 
            R[ "ALATA_L"]], 
          Times[ -27266477, 
            R[ "ALCD2x"]], 
          Times[ 1420, 
            R[ "ANPRT"]], 
          Times[ 1420, 
            R[ "ANS"]], 
          Times[ 8035, 
            R[ "ARGSL"]], 
          Times[ 8035, 
            R[ "ARGSSr"]], 
          Times[ 34215, 
            R[ "ASADi"]], 
          Times[ 5085, 
            R[ "ASNS1"]], 
          Times[ 34215, 
            R[ "ASPKi"]], 
          Times[ -22812604, 
            R[ "ASPt2r"]], 
          Times[ 3315, 
            R[ "ATPPRT"]], 
          Times[ 8035, 
            R[ "ATPS3m"]], 
          Times[ 50000, 
            R[ "BioM"]], 
          Times[ 2865, 
            R[ "BPNT"]], 
          Times[ 8035, 
            R[ "CBPS"]], 
          Times[ 11795, 
            R[ "CHORM"]], 
          Times[ 13215, 
            R[ "CHORS"]], 
          Times[ -50940, 
            R[ "CITtam"]], 
          Times[ -15522497, 
            R[ "CO2t"]], 
          Times[ -187070, 
            R[ "CO2tm"]], 
          Times[ 50940, 
            R[ "CSm"]], 
          Times[ 65250, 
            R[ "CSNAT"]], 
          Times[ 5530, 
            R[ "CSND"]], 
          Times[ 5530, 
            R[ "CSNt2"]], 
          Times[ 2235, 
            R[ "CTPS1"]], 
          Times[ 330, 
            R[ "CYSS"]], 
          Times[ -1360, 
            R[ "CYTK1"]], 
          Times[ -180, 
            R[ "DADK"]], 
          Times[ 330, 
            R[ "DAGPYP_SC"]], 
          Times[ 875, 
            R[ "DASYN_SC"]], 
          Times[ -120, 
            R[ "DCMPDA"]], 
          Times[ 13215, 
            R[ "DDPA"]], 
          Times[ -120, 
            R[ "DGK1"]], 
          Times[ 28050, 
            R[ "DHAD1m"]], 
          Times[ 9635, 
            R[ "DHAD2m"]], 
          Times[ 180, 
            R[ "DHFRim"]], 
          Times[ 180, 
            R[ "DHFtm"]], 
          Times[ 13215, 
            R[ "DHQS"]], 
          Times[ 13215, 
            R[ "DHQTi"]], 
          Times[ 40395, 
            R[ "DOLPMMer"]], 
          Times[ 40395, 
            R[ "DOLPMTcer"]], 
          Times[ -40395, 
            R[ "DOLPt2er"]], 
          Times[ 3535401, 
            R[ "ENO"]], 
          Times[ 35, 
            R[ "ERGSTt"]], 
          Times[ -27266477, 
            R[ "ETOHt"]], 
          Times[ 476, 
            R[ "FACOAL161"]], 
          Times[ 672, 
            R[ "FACOAL181"]], 
          Times[ 252, 
            R[ "FACOAL182"]], 
          Times[ 1400, 
            R[ "FAS100COA"]], 
          Times[ 1344, 
            R[ "FAS120COA"]], 
          Times[ 1176, 
            R[ "FAS140COA"]], 
          Times[ 1400, 
            R[ "FAS80COA_L"]], 
          Times[ 2690, 
            R[ "FDH"]], 
          Times[ -2690, 
            R[ "FTHFL"]], 
          Times[ 23052269, 
            R[ "FUMt2r"]], 
          Times[ 1235, 
            R[ "G3PD1ir"]], 
          Times[ 8235, 
            R[ "G5SADr"]], 
          Times[ 8235, 
            R[ "G5SD2"]], 
          Times[ 9885, 
            R[ "G6PDA"]], 
          Times[ 11710431, 
            R[ "G6PDH2"]], 
          Times[ 9885, 
            R[ "GAM6Pt"]], 
          Times[ 3889797, 
            R[ "GAPD"]], 
          Times[ 4900, 
            R[ "GARFTi"]], 
          Times[ 1235, 
            R[ "GAT1_SC"]], 
          Times[ 4033772, 
            R[ "GLCt1"]], 
          Times[ 35345, 
            R[ "GLNS"]], 
          Times[ 8235, 
            R[ "GLU5K"]], 
          Times[ -23388925, 
            R[ "GLUDy"]], 
          Times[ 4900, 
            R[ "GLUPRT"]], 
          Times[ 2420, 
            R[ "GMPS2"]], 
          Times[ 11710431, 
            R[ "GND"]], 
          Times[ 11346714, 
            R[ "H2Ot"]], 
          Times[ 14310, 
            R[ "HACNHm"]], 
          Times[ 14310, 
            R[ "HCITSm"]], 
          Times[ 16155, 
            R[ "HCO3E"]], 
          Times[ 476, 
            R[ "HDCEAt"]], 
          Times[ 14310, 
            R[ "HICITDm"]], 
          Times[ 3315, 
            R[ "HISTD"]], 
          Times[ 3315, 
            R[ "HISTP"]], 
          Times[ 34215, 
            R[ "HSDxi"]], 
          Times[ 2535, 
            R[ "HSERTA"]], 
          Times[ 31680, 
            R[ "HSK"]], 
          Times[ 3315, 
            R[ "HSTPT"]], 
          Times[ 50940, 
            R[ "ICDHy"]], 
          Times[ 3315, 
            R[ "IG3PS"]], 
          Times[ 3315, 
            R[ "IGPDH"]], 
          Times[ 1420, 
            R[ "IGPS"]], 
          Times[ -9635, 
            R[ "ILETA"]], 
          Times[ -8215, 
            R[ "IMPC"]], 
          Times[ 2420, 
            R[ "IMPD"]], 
          Times[ 14820, 
            R[ "IPMD"]], 
          Times[ -14820, 
            R[ "IPPMIa"]], 
          Times[ -14820, 
            R[ "IPPMIb"]], 
          Times[ 14820, 
            R[ "IPPS"]], 
          Times[ 28050, 
            R[ "KARA1im"]], 
          Times[ 9635, 
            R[ "KARA2im"]], 
          Times[ -14820, 
            R[ "LEUTA"]], 
          Times[ 40395, 
            R[ "MAN1PT"]], 
          Times[ -40395, 
            R[ "MAN6PI"]], 
          Times[ -40395, 
            R[ "MANNANter"]], 
          Times[ 14310, 
            R[ "MCITDm"]], 
          Times[ 23039409, 
            R[ "MDH"]], 
          Times[ -102960, 
            R[ "MDHm"]], 
          Times[ 88650, 
            R[ "ME1m"]], 
          Times[ 45900, 
            R[ "ME2m"]], 
          Times[ 900, 
            R[ "METAT"]], 
          Times[ 3435, 
            R[ "METS"]], 
          Times[ 300, 
            R[ "MFAPS_SC"]], 
          Times[ 265, 
            R[ "MI1PP"]], 
          Times[ 265, 
            R[ "MI1PS"]], 
          Times[ 15805, 
            R[ "MTHFC"]], 
          Times[ 15805, 
            R[ "MTHFD"]], 
          Times[ 3435, 
            R[ "MTHFR3"]], 
          Times[ -1360, 
            R[ "NDPK3"]], 
          Times[ 23401575, 
            R[ "NH4t"]], 
          Times[ -9635, 
            R[ "NH4tm"]], 
          Times[ 8035, 
            R[ "OCBTi"]], 
          Times[ 672, 
            R[ "OCDCEAt"]], 
          Times[ 252, 
            R[ "OCDCYAt"]], 
          Times[ 14820, 
            R[ "OMCDC"]], 
          Times[ 8035, 
            R[ "ORNt3m"]], 
          Times[ 8035, 
            R[ "ORNTACim"]], 
          Times[ 14310, 
            R[ "OXAGm"]], 
          Times[ 8235, 
            R[ "P5CR"]], 
          Times[ 2865, 
            R[ "PAPSR"]], 
          Times[ 
            Rational[ -21, 4], 
            R[ "PEtm_SC"]], 
          Times[ 300, 
            R[ "PETOHM_SC"]], 
          Times[ 354396, 
            R[ "PGCD"]], 
          Times[ -7761929, 
            R[ "PGI"]], 
          Times[ -3889797, 
            R[ "PGK"]], 
          Times[ 11710431, 
            R[ "PGL"]], 
          Times[ -3535401, 
            R[ "PGM"]], 
          Times[ -83835, 
            R[ "PGMT"]], 
          Times[ -6695, 
            R[ "PHETA1"]], 
          Times[ 265, 
            R[ "PINOS_SC"]], 
          Times[ -40395, 
            R[ "PMANM"]], 
          Times[ 300, 
            R[ "PMETM_SC"]], 
          Times[ 170205, 
            R[ "PPA"]], 
          Times[ -5530, 
            R[ "PPM"]], 
          Times[ 5100, 
            R[ "PPND2"]], 
          Times[ 6695, 
            R[ "PPNDH"]], 
          Times[ 4900, 
            R[ "PRAGSr"]], 
          Times[ 1420, 
            R[ "PRAIi"]], 
          Times[ 4900, 
            R[ "PRAIS"]], 
          Times[ 3315, 
            R[ "PRAMPC"]], 
          Times[ 4900, 
            R[ "PRASCS"]], 
          Times[ 3315, 
            R[ "PRATPP"]], 
          Times[ 4900, 
            R[ "PRFGS"]], 
          Times[ 3315, 
            R[ "PRMICIi"]], 
          Times[ 9635, 
            R[ "PRPPS"]], 
          Times[ 13215, 
            R[ "PSCVTi"]], 
          Times[ 525, 
            R[ "PSERDm_SC"]], 
          Times[ 610, 
            R[ "PSERS_SC"]], 
          Times[ 354396, 
            R[ "PSERT"]], 
          Times[ 354396, 
            R[ "PSP_L"]], 
          Times[ 
            Rational[ 21, 4], 
            R[ "PStm_SC"]], 
          Times[ 3508971, 
            R[ "PYK"]], 
          Times[ 3556266, 
            R[ "PYRDC"]], 
          Times[ -68815, 
            R[ "PYRt2m"]], 
          Times[ 7792439, 
            R[ "RPE"]], 
          Times[ -3917992, 
            R[ "RPI"]], 
          Times[ 14310, 
            R[ "SACCD1"]], 
          Times[ 14310, 
            R[ "SACCD2"]], 
          Times[ 330, 
            R[ "SERATi"]], 
          Times[ -323346, 
            R[ "SERt2r"]], 
          Times[ 13215, 
            R[ "SHK3D"]], 
          Times[ 13215, 
            R[ "SHKK"]], 
          Times[ 2865, 
            R[ "SLFAT"]], 
          Times[ 3865, 
            R[ "SO4ti"]], 
          Times[ -2865, 
            R[ "SULR"]], 
          Times[ 3902827, 
            R[ "TALA"]], 
          Times[ 9635, 
            R[ "THRD_Lm"]], 
          Times[ 31680, 
            R[ "THRS"]], 
          Times[ 9635, 
            R[ "THRt2m"]], 
          Times[ -12475, 
            R[ "THRt2r"]], 
          Times[ 3902827, 
            R[ "TKT1"]], 
          Times[ 3889612, 
            R[ "TKT2"]], 
          Times[ 180, 
            R[ "TMDS"]], 
          Times[ -1235, 
            R[ "TPI"]], 
          Times[ 2865, 
            R[ "TRDR"]], 
          Times[ 1170, 
            R[ "TRE6PP"]], 
          Times[ 1170, 
            R[ "TRE6PS"]], 
          Times[ 330, 
            R[ "TRIGS_SC"]], 
          Times[ 1420, 
            R[ "TRPS1"]], 
          Times[ 2535, 
            R[ "UMPK"]], 
          Times[ -13230, 
            R[ "VALTA"]], 
          R[ "vnocym"], 
          Times[ 75, 
            R[ "ZYMSTt"]]]}]]], "Output"]
}, Open  ]],

Cell[BoxData[""], "Input"],

Cell[TextData[{
  "The symbol ",
  StyleBox["ALT",
    FontWeight->"Bold",
    FontSlant->"Plain",
    FontVariations->{"CompatibilityType"->0}],
  " indicates that the simplification procedure has generated duplicate \
reactions. To illustrate this, we look at the second  ",
  StyleBox["ALT",
    FontWeight->"Bold",
    FontSlant->"Plain",
    FontVariations->{"CompatibilityType"->0}],
  "  in the line above."
}], "Text",
  Evaluatable->False,
  FontFamily->"Times",
  FontSize->14,
  FontWeight->"Plain",
  FontSlant->"Italic"],

Cell[CellGroupData[{

Cell[BoxData[
    \(with[{"\<GLCS2\>", "\<GBEZ\>", "\<GLYGS\>"}, \ trpairs@restricted]\  // 
      TableForm\)], "Input"],

Cell[BoxData[
    TagBox[GridBox[{
          {\(R["GBEZ"]\), 
            RowBox[{\("14glun"\^"c"\), 
              StyleBox["\[RightVector]",
                FontSize->18], \("glycogen"\^"c" + "h2o"\^"c"\)}]},
          {\(R["GLCS2"]\), 
            RowBox[{\("udpg"\^"c"\), 
              StyleBox["\[RightVector]",
                FontSize->18], \("glycogen"\^"c" + "h"\^"c" + 
                "udp"\^"c"\)}]},
          {\(R["GLYGS"]\), 
            RowBox[{\("h2o"\^"c" + "udpg"\^"c"\), 
              StyleBox["\[RightVector]",
                FontSize->18], \("14glun"\^"c" + "h"\^"c" + "udp"\^"c"\)}]}
          },
        RowSpacings->1,
        ColumnSpacings->3,
        RowAlignments->Baseline,
        ColumnAlignments->{Left}],
      Function[ BoxForm`e$, 
        TableForm[ BoxForm`e$]]]], "Output"]
}, Open  ]],

Cell[TextData[{
  "The simplification procedure has found that ",
  StyleBox["GBEZ ",
    FontSlant->"Plain",
    FontVariations->{"CompatibilityType"->0}],
  "and ",
  StyleBox["GLYGS",
    FontSlant->"Plain",
    FontVariations->{"CompatibilityType"->0}],
  " can only run in tandem and are then equivalent to ",
  Cell[BoxData[
      RowBox[{
        StyleBox[\("h2o"\^"c" + \ "udpg"\^"c"\),
          FontWeight->"Plain",
          FontSlant->"Plain",
          FontTracking->"Plain",
          FontVariations->{"Underline"->False,
          "Outline"->False,
          "Shadow"->False,
          "StrikeThrough"->False,
          "Masked"->False,
          "CompatibilityType"->0,
          "RotationAngle"->0}], 
        StyleBox["\[RightVector]",
          FontSize->18,
          FontSlant->"Plain",
          FontVariations->{"CompatibilityType"->0}], 
        StyleBox[\("h2o"\^"c" + \ "glycogen"\^"c" + "h"\^"c" + "udp"\^"c"\),
          FontWeight->"Plain",
          FontSlant->"Plain",
          FontTracking->"Plain",
          FontVariations->{"Underline"->False,
          "Outline"->False,
          "Shadow"->False,
          "StrikeThrough"->False,
          "Masked"->False,
          "CompatibilityType"->0,
          "RotationAngle"->0}]}]]],
  ",  which is the same as ",
  StyleBox["GLCS2",
    FontSlant->"Plain",
    FontVariations->{"CompatibilityType"->0}],
  ".  So for any elementary flux using ",
  StyleBox["GLCS2",
    FontSlant->"Plain",
    FontVariations->{"CompatibilityType"->0}],
  " there is another one  using  ",
  StyleBox["GBEZ  ",
    FontSlant->"Plain",
    FontVariations->{"CompatibilityType"->0}],
  "as well as ",
  StyleBox["GLYGS. \n",
    FontSlant->"Plain",
    FontVariations->{"CompatibilityType"->0}],
  "Computationally it makes sense to generate such alternatives only at the \
very end of the calculation. This is just what ",
  StyleBox["symfluxgset ",
    FontWeight->"Bold",
    FontSlant->"Plain",
    FontVariations->{"CompatibilityType"->0}],
  " does, applying  ",
  StyleBox["ALTexpfull ",
    FontWeight->"Bold",
    FontSlant->"Plain",
    FontVariations->{"CompatibilityType"->0}],
  "to the intermediate result of ",
  StyleBox["fluxgset",
    FontWeight->"Bold",
    FontSlant->"Plain",
    FontVariations->{"CompatibilityType"->0}],
  " to obtain the final 1044 fluxes."
}], "Text",
  Evaluatable->False,
  FontFamily->"Times",
  FontSize->14,
  FontWeight->"Plain",
  FontSlant->"Italic"],

Cell[CellGroupData[{

Cell[BoxData[
    \(\(\(\ \)\(\(Map[ALTexpfull, \ rgfluxes . \((tags@rrestricted)\)] // 
        Flatten\) // Length\)\)\)], "Input"],

Cell[BoxData[
    \(1044\)], "Output"]
}, Open  ]],

Cell[TextData[{
  "But of course, one  can also expand the elements of  ",
  StyleBox["rgfluxes ",
    FontWeight->"Bold",
    FontSlant->"Plain",
    FontVariations->{"CompatibilityType"->0}],
  "  individually."
}], "Text",
  Evaluatable->False,
  FontFamily->"Times",
  FontSize->14,
  FontWeight->"Plain",
  FontSlant->"Italic"],

Cell[CellGroupData[{

Cell[BoxData[
    \(ALTexpfull[\ rgfluxes[\([1]\)] . \((tags@rrestricted)\)] // 
      Length\)], "Input"],

Cell[BoxData[
    \(1\)], "Output"]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
    \(ALTexpfull[\ rgfluxes[\([20]\)] . \((tags@rrestricted)\)] // 
      Length\)], "Input"],

Cell[BoxData[
    \(8\)], "Output"]
}, Open  ]],

Cell[TextData[{
  "So ",
  StyleBox["symfluxelvs[",
    FontWeight->"Bold",
    FontSlant->"Plain",
    FontVariations->{"CompatibilityType"->0}],
  " ",
  StyleBox["restricted] ",
    FontWeight->"Bold",
    FontSlant->"Plain",
    FontVariations->{"CompatibilityType"->0}],
  " is bound to run out of memory in the final stage of the calculation, due \
to the sheer size of the result. Enumerating the elementary fluxes of  ",
  StyleBox["rrestricted ",
    FontWeight->"Bold",
    FontSlant->"Plain",
    FontVariations->{"CompatibilityType"->0}],
  "is a much more reasonable proposition, since their number is only  4.6 \
\[Times] 10",
  Cell[BoxData[
      \(\^6\)]],
  ". So if you have 4GB of memory and can do without your computer for upto \
10 hours,  you might try ",
  StyleBox["fluxelvs[stoich,nrev];\n",
    FontWeight->"Bold",
    FontSlant->"Plain",
    FontVariations->{"CompatibilityType"->0}],
  "In fact, SNAmat provides the function  ",
  StyleBox["signfluxelvs, ",
    FontWeight->"Bold",
    FontSlant->"Plain",
    FontVariations->{"CompatibilityType"->0}],
  "which manages to squeeze the computation into 2GB by returning just the \
signs of the flows in each flux and using a compressed output format. But \
this is so messy that I am not sure whether I really consider  ",
  StyleBox["signfluxelvs ",
    FontWeight->"Bold",
    FontSlant->"Plain",
    FontVariations->{"CompatibilityType"->0}],
  "to be 'officially' part of SNAmat."
}], "Text",
  Evaluatable->False,
  FontFamily->"Times",
  FontSize->14,
  FontWeight->"Plain",
  FontSlant->"Italic"]
}, Open  ]]
}, Open  ]]
},
FrontEndVersion->"5.0 for X",
ScreenRectangle->{{0, 1280}, {0, 1024}},
WindowSize->{683, 700},
WindowMargins->{{83, Automatic}, {Automatic, 37}},
StyleDefinitions -> "Demo.nb"
]

(*******************************************************************
Cached data follows.  If you edit this Notebook file directly, not
using Mathematica, you must remove the line containing CacheID at
the top of  the file.  The cache data will then be recreated when
you save this file from within Mathematica.
*******************************************************************)

(*CellTagsOutline
CellTagsIndex->{}
*)

(*CellTagsIndex
CellTagsIndex->{}
*)

(*NotebookFileOutline
Notebook[{
Cell[1754, 51, 245, 8, 65, "Title"],

Cell[CellGroupData[{
Cell[2024, 63, 67, 1, 31, "Input"],
Cell[2094, 66, 89, 1, 23, "Print"]
}, Open  ]],

Cell[CellGroupData[{
Cell[2220, 72, 211, 7, 43, "Subsubsection",
  Evaluatable->False],

Cell[CellGroupData[{
Cell[2456, 83, 108, 2, 47, "Input"],
Cell[2567, 87, 147, 2, 94, "Output"]
}, Open  ]],

Cell[CellGroupData[{
Cell[2751, 94, 58, 1, 31, "Input"],
Cell[2812, 97, 38, 1, 46, "Output"]
}, Open  ]],

Cell[CellGroupData[{
Cell[2887, 103, 60, 1, 31, "Input"],
Cell[2950, 106, 38, 1, 46, "Output"]
}, Open  ]],
Cell[3003, 110, 252, 8, 45, "Text",
  Evaluatable->False]
}, Open  ]],

Cell[CellGroupData[{
Cell[3292, 123, 345, 11, 66, "Subsubsection",
  Evaluatable->False],

Cell[CellGroupData[{
Cell[3662, 138, 65, 1, 31, "Input"],
Cell[3730, 141, 1736, 24, 222, "Output"]
}, Open  ]],
Cell[5481, 168, 651, 21, 72, "Text",
  Evaluatable->False],
Cell[6135, 191, 312, 5, 64, "Input"],
Cell[6450, 198, 838, 28, 87, "Text",
  Evaluatable->False],

Cell[CellGroupData[{
Cell[7313, 230, 528, 8, 83, "Input"],
Cell[7844, 240, 357, 5, 78, "Output"]
}, Open  ]],
Cell[8216, 248, 75, 1, 31, "Input"]
}, Open  ]],

Cell[CellGroupData[{
Cell[8328, 254, 165, 7, 75, "Section",
  Evaluatable->False],

Cell[CellGroupData[{
Cell[8518, 265, 919, 31, 135, "Subsubsection",
  Evaluatable->False],

Cell[CellGroupData[{
Cell[9462, 300, 66, 1, 31, "Input"],
Cell[9531, 303, 357, 5, 39, "Message"]
}, Open  ]],

Cell[CellGroupData[{
Cell[9925, 313, 60, 1, 31, "Input"],
Cell[9988, 316, 36, 1, 46, "Output"]
}, Open  ]],

Cell[CellGroupData[{
Cell[10061, 322, 64, 1, 31, "Input"],
Cell[10128, 325, 871, 12, 116, "Output"]
}, Open  ]],
Cell[11014, 340, 720, 23, 87, "Text",
  Evaluatable->False],

Cell[CellGroupData[{
Cell[11759, 367, 63, 1, 31, "Input"],
Cell[11825, 370, 384, 7, 62, "Output"]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{
Cell[12258, 383, 908, 29, 112, "Subsubsection",
  Evaluatable->False],

Cell[CellGroupData[{
Cell[13191, 416, 64, 1, 31, "Input"],
Cell[13258, 419, 353, 6, 64, "Output"]
}, Open  ]],
Cell[13626, 428, 1063, 36, 102, "Text",
  Evaluatable->False],
Cell[14692, 466, 148, 2, 47, "Input"],
Cell[14843, 470, 1154, 34, 144, "Text",
  Evaluatable->False],

Cell[CellGroupData[{
Cell[16022, 508, 64, 1, 31, "Input"],
Cell[16089, 511, 37, 1, 46, "Output"]
}, Open  ]],
Cell[16141, 515, 338, 12, 30, "Text",
  Evaluatable->False],

Cell[CellGroupData[{
Cell[16504, 531, 151, 2, 47, "Input"],
Cell[16658, 535, 37, 1, 46, "Output"]
}, Open  ]],
Cell[16710, 539, 837, 27, 75, "Text",
  Evaluatable->False]
}, Open  ]],

Cell[CellGroupData[{
Cell[17584, 571, 1136, 32, 181, "Subsubsection",
  Evaluatable->False],

Cell[CellGroupData[{
Cell[18745, 607, 406, 7, 81, "Input"],
Cell[19154, 616, 38, 1, 46, "Output"]
}, Open  ]],
Cell[19207, 620, 248, 8, 45, "Text",
  Evaluatable->False]
}, Open  ]],

Cell[CellGroupData[{
Cell[19492, 633, 1117, 30, 181, "Subsubsection",
  Evaluatable->False],

Cell[CellGroupData[{
Cell[20634, 667, 75, 1, 31, "Input"],
Cell[20712, 670, 146, 2, 46, "Output"]
}, Open  ]],
Cell[20873, 675, 81, 1, 31, "Input"],
Cell[20957, 678, 859, 32, 45, "Text",
  Evaluatable->False],

Cell[CellGroupData[{
Cell[21841, 714, 58, 1, 31, "Input"],
Cell[21902, 717, 36, 1, 46, "Output"]
}, Open  ]],
Cell[21953, 721, 412, 13, 45, "Text",
  Evaluatable->False],

Cell[CellGroupData[{
Cell[22390, 738, 93, 1, 31, "Input"],
Cell[22486, 741, 160, 2, 46, "Output"]
}, Open  ]],
Cell[22661, 746, 1389, 49, 60, "Text",
  Evaluatable->False],

Cell[CellGroupData[{
Cell[24075, 799, 79, 1, 32, "Input"],
Cell[24157, 802, 38, 1, 46, "Output"]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{
Cell[24244, 809, 977, 34, 89, "Subsubsection",
  Evaluatable->False],

Cell[CellGroupData[{
Cell[25246, 847, 162, 3, 47, "Input"],
Cell[25411, 852, 39, 1, 46, "Output"]
}, Open  ]]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{
Cell[25511, 860, 193, 8, 75, "Section",
  Evaluatable->False],

Cell[CellGroupData[{
Cell[25729, 872, 172, 5, 43, "Subsubsection",
  Evaluatable->False],

Cell[CellGroupData[{
Cell[25926, 881, 91, 1, 31, "Input"],
Cell[26020, 884, 748, 12, 118, "Output"]
}, Open  ]],
Cell[26783, 899, 262, 9, 57, "Text",
  Evaluatable->False],

Cell[CellGroupData[{
Cell[27070, 912, 497, 13, 31, "Input"],
Cell[27570, 927, 748, 12, 118, "Output"]
}, Open  ]],
Cell[28333, 942, 447, 17, 30, "Text",
  Evaluatable->False],
Cell[28783, 961, 377, 6, 79, "Input"],
Cell[29163, 969, 824, 28, 72, "Text",
  Evaluatable->False],

Cell[CellGroupData[{
Cell[30012, 1001, 132, 3, 47, "Input"],
Cell[30147, 1006, 35, 1, 46, "Output"]
}, Open  ]],

Cell[CellGroupData[{
Cell[30219, 1012, 93, 1, 31, "Input"],
Cell[30315, 1015, 46, 1, 46, "Output"]
}, Open  ]],
Cell[30376, 1019, 333, 12, 30, "Text",
  Evaluatable->False]
}, Open  ]],

Cell[CellGroupData[{
Cell[30746, 1036, 1009, 28, 158, "Subsubsection",
  Evaluatable->False],

Cell[CellGroupData[{
Cell[31780, 1068, 244, 4, 79, "Input"],
Cell[32027, 1074, 371, 6, 39, "Message"],
Cell[32401, 1082, 363, 5, 39, "Message"],
Cell[32767, 1089, 357, 5, 39, "Message"],
Cell[33127, 1096, 35, 1, 46, "Output"]
}, Open  ]],

Cell[CellGroupData[{
Cell[33199, 1102, 51, 1, 31, "Input"],
Cell[33253, 1105, 37, 1, 46, "Output"]
}, Open  ]],
Cell[33305, 1109, 503, 16, 57, "Text",
  Evaluatable->False],

Cell[CellGroupData[{
Cell[33833, 1129, 88, 1, 31, "Input"],
Cell[33924, 1132, 17637, 537, 65, "Output"]
}, Open  ]],
Cell[51576, 1672, 26, 0, 31, "Input"],
Cell[51605, 1674, 533, 18, 45, "Text",
  Evaluatable->False],

Cell[CellGroupData[{
Cell[52163, 1696, 121, 2, 31, "Input"],
Cell[52287, 1700, 815, 21, 97, "Output"]
}, Open  ]],
Cell[53117, 1724, 2464, 76, 134, "Text",
  Evaluatable->False],

Cell[CellGroupData[{
Cell[55606, 1804, 133, 2, 31, "Input"],
Cell[55742, 1808, 38, 1, 46, "Output"]
}, Open  ]],
Cell[55795, 1812, 332, 12, 30, "Text",
  Evaluatable->False],

Cell[CellGroupData[{
Cell[56152, 1828, 106, 2, 31, "Input"],
Cell[56261, 1832, 35, 1, 46, "Output"]
}, Open  ]],

Cell[CellGroupData[{
Cell[56333, 1838, 107, 2, 31, "Input"],
Cell[56443, 1842, 35, 1, 46, "Output"]
}, Open  ]],
Cell[56493, 1846, 1582, 45, 164, "Text",
  Evaluatable->False]
}, Open  ]]
}, Open  ]]
}
]
*)



(*******************************************************************
End of Mathematica Notebook file.
*******************************************************************)

