/*	Copyright (C) 2000	by Indraneel Majumdar<indraneel@indialine.org> */

/*                 GNU LESSER GENERAL PUBLIC LICENSE
                       Version 2.1, February 1999

    You should have received a copy of the GNU Lesser General Public
    License along with this library; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
*/
/* Program to plot Chaos Game Representation of a DNA or RNA sequence. */


/* this file should be linked with 
-lreadline -lncurses -lm -lvgagl -lvga -lpng -lz */

#include<stdio.h>
#include<stdlib.h>
#include<vgagl.h>
#include<vga.h>
#include<math.h>
#include<readline/readline.h>
#include<png.h>

#define VGAMODE	    12		/* 1024 x 768 at 256 colors */
#define TEXT	    0		/* text mode */

/* initialise buffers for svgalib */
GraphicsContext *physicalscreen;
GraphicsContext *templatescreen, *fileplotscreen, *randomplotscreen;

/* stores probability data */
struct P_ATGC
{
  double a, t, g, c;
}
P;

struct point
{
  double x, y;
}
A, T, G, C;

/* stores input file name */
struct file_identity
{
  char name[105];		/* filepath and name has to be < 100 characters */
  int flag;			/* set this 0 for random plot */
}
seqfile, outfile;

/* global flags for user choice */
struct all_flags
{
  int file, prob, cycles, grid, seq, reload, counts, refresh, print;
}
flags;

char sequence_table[16][16][5];	/* stores sequence chart */
int probability_table[16][16];	/* stores counts of bases per 10000 */
long int CYCLES, CYCLES_FIRST, CYCLES_LAST;	/* iterations */

/* user input for probabilities for random plot */
void
get_atgc_probabilities ()
{
  vga_setmode (TEXT);
  P.a = P.t = P.g = P.c = 1.000000;
  while ((P.a + P.t + P.g) > 1.000001)
    {
      printf ("\nEnter probabilities for A, T, G\n"
	      "(probability for C is calculated automagically)\n"
	      "Probabilities=>");
      scanf ("%lf,%lf,%lf", &P.a, &P.t, &P.g);
      getchar ();		/* discard the entered newline character */
    }
  P.c = 1.000001 - (P.a + P.t + P.g);
  flags.prob = 1 - flags.prob;	/* reverse global flag */
  vga_setmode (VGAMODE);
}

/* user input for iterations */
int
get_cycles ()
{
  long int cycles;
  cycles = CYCLES_FIRST = 0;
  vga_setmode (TEXT);
  if (seqfile.flag == 0)
    {
      printf ("\nEnter number of points to plot\n" "Points to plot=>");
      scanf ("%d", &cycles);
    }
  else
    {
      do
	{			/* filter user's input for errors, discard -ve numbers */
	  printf ("\nEnter start point for plot\n" "First base=>");
	  scanf ("%d", &CYCLES_FIRST);
	  printf ("\nEnter end point for plot\n"
		  "(enter '0' for entire file)\n" "Last base=>");
	  scanf ("%d", &cycles);
	}
      while ((cycles != 0) && ((cycles < 0) || (cycles < CYCLES_FIRST)));
      if ((cycles == 0) || (CYCLES_FIRST < 1))
	CYCLES_FIRST = 1;	/*so display looks better */
    }
  getchar ();			/* discard newline character */
  vga_setmode (VGAMODE);
  flags.cycles = 1 - flags.cycles;	/* reverse global flag */
  return (cycles);		/* return end point of loop */
}

/* user's input for sequence input file */
void
get_seqfile ()
{
  FILE *fopen (), *file_pointer;	/* to check if file exists */
  char *filename_pointer, *malloc ();	/*readline requires this */
  int length;
  filename_pointer = malloc (100);	/* length of filename<100 */
  vga_setmode (TEXT);
  file_pointer = 0;		/* initialise pointer to NULL */
  seqfile.flag = 1;
  while ((file_pointer == 0) && (seqfile.flag == 1))
    {
      printf ("\nEnter full path of DNA sequence file\n"
	      "(enter '*' for random number plot)\n");
      filename_pointer = readline ("Filename=> ");	/* makes it easy to locate files */
      strncpy (seqfile.name, filename_pointer, 100);
      if (seqfile.name[0] == '*')
	{
	  sprintf (seqfile.name, "RANDOM NUMBER PLOT");	/* give a title to the plot */
	  seqfile.flag = 0;	/* set flag for plot type */
	}
      else
	{
	  length = strlen (seqfile.name);
	  if (seqfile.name[length - 1] == ' ')
	    seqfile.name[length - 1] = '\0';	/* discard last blank which maybe entered by readline input */
	  seqfile.flag = 1;	/* set flag that disk file is being used */
	  file_pointer = fopen (seqfile.name, "r");	/* check if file exists */
	}
    }
  if (seqfile.flag == 1)
    fclose (file_pointer);	/* close file if exists */
  free (filename_pointer);
  flags.file = 1 - flags.file;	/* reverse global flag */
  vga_setmode (VGAMODE);
}

/* generate (x,y) random point from 0 to 1 */
struct point
get_random_point ()
{
  struct point p;
  p.x = ((double) rand () / RAND_MAX);
  p.y = ((double) rand () / RAND_MAX);
  return (p);
}

/* make border of plot and print filename */
void
draw_border ()
{
  gl_line (A.x, A.y, T.x, T.y, 3);
  gl_line (T.x, T.y, G.x, G.y, 3);
  gl_line (G.x, G.y, C.x, C.y, 3);
  gl_line (C.x, C.y, A.x, A.y, 3);
  gl_setfontcolors (0, 32);
  gl_write (A.x - 25, A.y - 25, "DNA base 'A'");
  gl_write (T.x - 25, T.y + 30, "DNA base 'T'");
  gl_write (G.x - 50, G.y + 30, "DNA base 'G'");
  gl_write (C.x - 50, C.y - 25, "DNA base 'C'");
  gl_setfontcolors (0, 42);	/*same color as file data */
  gl_printf (A.x + 100, T.y + 50, "%s", seqfile.name);
}

/* draw the grid (lines are 37.5 pixels apart) */
void
draw_grid ()
{
  int num;
  float x, y, loopx, loopy = 0;
  x = C.x;
  y = T.y;
  loopx = A.x;
  loopy = A.y;
  num = 0;
  gl_setfontcolors (0, 52);
  gl_write (500, A.y - 20, "sequence resolution = 9 bp");	/* for a 600x600 pixel plot  difference in base in any of last 9 bases gives different points (advertise this) */
  while (loopx <= x)
    {
      gl_line (loopx, A.y - 10, loopx, y + 10, 8);
      gl_printf (loopx, y + 15, "%d", num);	/* grid number */
      loopx = loopx + 37.5;
      num++;
    }
  num = 0;
  while (loopy <= y)
    {
      gl_line (A.x - 10, loopy, x + 10, loopy, 8);
      gl_printf (x + 15, loopy, "%d", num);	/* grid number */
      loopy = loopy + 37.5;
      num++;
    }
  flags.grid = 1 - flags.grid;	/* reverse global flag */
}

/* plot random plot to virtual screen and copy to real screen (this makes plot and refresh faster)*/
void
plot_random_sequence (long int cycles)
{
  long int loop;
  double a, t, g, c;
  struct point p, pix;
  gl_setcontext (templatescreen);	/*change to template virtual screen */
  gl_copyscreen (randomplotscreen);	/*copy template to working virtual screen ie randomplotscreen */
  gl_setcontext (randomplotscreen);	/* change to working virtual screen */
  a = P.a;
  t = a + P.t;
  g = t + P.g;
  c = g + P.c;			/* convert user input to 0 to 1 */
  loop = 0;
  pix.x = 600;
  pix.y = 350;
  while (loop < cycles)
    {
      p = get_random_point ();
/* test against user input probability and scale pixel to fit within borders. point is located midway between previous point and corner for chosen base. This is the CGR */
      if (p.x <= a)
	{
	  pix.x = (pix.x + A.x) / 2;
	  pix.y = (pix.y + A.y) / 2;
	}
      else if (p.x <= t)
	{
	  pix.x = (pix.x + T.x) / 2;
	  pix.y = (pix.y + T.y) / 2;
	}
      else if (p.x <= g)
	{
	  pix.x = (pix.x + G.x) / 2;
	  pix.y = (pix.y + G.y) / 2;
	}
      else if (p.x <= c)
	{
	  pix.x = (pix.x + C.x) / 2;
	  pix.y = (pix.y + C.y) / 2;
	}
      gl_setpixel (pix.x, pix.y, 44);	/*plot the pixel */
      loop++;
    }
  gl_copyscreen (physicalscreen);	/* copy to monitor or stdout */
  gl_setcontext (physicalscreen);	/* change working screen to real screen */
  CYCLES = loop;		/* record actual number of points plotted */
}

/* plots CGR from disk file sequence to virtual screen and copies to real screen (makes plotting and refresh faster)*/
void
plot_file_sequence (long int cycles)
{
  long int loop, base_count, first_base;
  int base, array_row, array_col, array_box, stop;
  int a, t, g, c;
  struct point pix;
  char file_header[200];	/* non sequence lines at start of file */
  FILE *fopen (), *file_pointer;
  gl_setcontext (templatescreen);	/* change to template virtual screen */
  gl_copyscreen (fileplotscreen);	/* copy template to working screen */
  gl_setcontext (fileplotscreen);	/* change to working virtual screen */
  loop = stop = base_count = base = 0;
  first_base = CYCLES_FIRST - 1;
  a = t = g = c = 0;
/* initialise the counts table for each new plot */
  for (array_row = 0; array_row < 16; array_row++)
    {
      for (array_col = 0; array_col < 16; array_col++)
	{
	  probability_table[array_row][array_col] = 0;
	}
    }
  pix.x = 600;
  pix.y = 350;
  file_pointer = fopen (seqfile.name, "r");	/* open file for reading */
  fgets (file_header, 200, file_pointer);	/* first line is not checked for sequence (assumed to contain file information and is shown on display */
  gl_setfontcolors (0, 42);
  gl_write (50, T.y + 60, "File data:");
  gl_printf (50, T.y + 80, "%s", file_header);	/* show first line */
  base = getc (file_pointer);	/* get first base of next line */
  base = tolower (base);	/* convert to lowercase */
/* sequence description must start with a,t,u,g,c, or blank (fasta format), please write checkpoints for other sequence formats and add here */
  while ((base != 'a') && (base != 't') && (base != 'u') && (base != 'g')
	 && (base != 'c') && (base != ' '))
    {
      ungetc (base, file_pointer);	/* move back */
      fgets (file_header, 200, file_pointer);	/* remove the whole line */
      base = getc (file_pointer);	/* get first base of next line */
      base = tolower (base);
    }
  ungetc (base, file_pointer);	/* move back if sequence has started */
/* check for EOF and whether last base to plot has been reached */
  while ((stop == 0) && ((base = getc (file_pointer)) != EOF))
    {
      base = tolower (base);
      if ((base == 'a') || (base == 't') || (base == 'u') || (base == 'g')
	  || (base == 'c'))
	{
/* check whether base is to be plotted (user input for start and end points) */
	  if ((cycles == 0) || ((loop >= first_base) && (loop < cycles)))
	    {
/* set pixel according to CGR theory (point is halfway between previous point and corner of current chosen base) */
	      if (base == 'a')
		{
		  pix.x = (pix.x + A.x) / 2;
		  pix.y = (pix.y + A.y) / 2;
		  a++;
		}
	      else if ((base == 't') || (base == 'u'))
		{
		  pix.x = (pix.x + T.x) / 2;
		  pix.y = (pix.y + T.y) / 2;
		  t++;
		}
	      else if (base == 'g')
		{
		  pix.x = (pix.x + G.x) / 2;
		  pix.y = (pix.y + G.y) / 2;
		  g++;
		}
	      else if (base == 'c')
		{
		  pix.x = (pix.x + C.x) / 2;
		  pix.y = (pix.y + C.y) / 2;
		  c++;
		}
	      gl_setpixel (pix.x, pix.y, 44);	/* plot the pixel */
	      array_row = (pix.x - 300) / 37.5;
	      array_col = (pix.y - 50) / 37.5;	/* scale the pixel location and update the counts table */
	      probability_table[array_row][array_col]++;
	      base = 'x';	/* make base non atugc blank */
	      base_count++;	/* update count of actual points plotted */
	    }
	  loop++;		/* increment counter for bases checked */
	  if ((cycles != 0) && (loop == cycles))
	    stop = 1;		/* check for end point */
	}			/* no counter increments for non atugc even if not plotted */
    }
  fclose (file_pointer);
  CYCLES = base_count;		/* record actual points plotted */
  CYCLES_LAST = loop;		/* record actual last base checked */
  if (base_count == 0)
    CYCLES_FIRST = base_count = 1;	/* if user input for first base was > total bases in file */
/* calculate probability of atgc based on total points plotted */
  P.a = (double) a / base_count;
  P.t = (double) t / base_count;
  P.g = (double) g / base_count;
  P.c = (double) c / base_count;
/* prepare counts table to show counts out of 10000 */
  for (array_row = 0; array_row < 16; array_row++)
    {
      for (array_col = 0; array_col < 16; array_col++)
	{
/* convert display to integer */
	  array_box =
	    (probability_table[array_row][array_col] * 10000) / base_count;
/* update table */
	  probability_table[array_row][array_col] = array_box;
	}
    }
  gl_copyscreen (physicalscreen);	/* copy working screen to monitor or stdout */
  gl_setcontext (physicalscreen);	/* change working screen to real screen */
}

/* to show the sequence chart at the drop of a finger */
void
make_sequence_table ()
{
  int row, col, position;
  int base;
  row = col = position = 0;
/* pass all calculations to helper for all 1024 positions */
  while (position < 4)
    {				/* no place to print more than 4 bases */
      while (col < 16)
	{
	  while (row < 16)
	    {
	      base = help_sequence_table (col, row, position);
	      sequence_table[row][col][position] = base;
	      row++;
	    }
	  col++;
	  row = 0;
	}
      position++;
      col = 0;
    }
/* add a blank to all ends so they can be printed as strings */
  row = col = 0;
  while (col < 16)
    {
      while (row < 16)
	{
	  sequence_table[row][col][4] = '\0';
	  row++;
	}
      col++;
      row = 0;
    }
}

/* this returns value of a base at any position of any table */
/* (first position is position 0 and so on..) */
int
help_sequence_table (int row, int col, int position)
{
  int base;
  int divisor, checkx, checky;	/*checkxy - even is 0, odd is 1 */
  divisor = pow (2, position);	/* convert divisor to position power of 2 */
  row = row / divisor;
  col = col / divisor;		/* now it is easier to handle as everything becomes odd or even */
  checkx = checky = 1;
/* check odd or even */
  if ((row % 2) == 0)
    checkx = 0;
  if ((col % 2) == 0)
    checky = 0;
/* the bottom is true when ATGC represents a square */
  if ((checkx == 0) && (checky == 0))
    base = 'A';
  else if ((checkx == 1) && (checky == 0))
    base = 'T';
  else if ((checkx == 1) && (checky == 1))
    base = 'G';
  else if ((checkx == 0) && (checky == 1))
    base = 'C';
  return (base);		/* done */
}

/* display the table when asked */
void
display_sequence_table ()
{
  int row, col, startx, starty;
  gl_setfontcolors (0, 92);
  gl_printf (20, 200, "(sequence ending for point)");
  startx = 302;
  starty = 53;
  row = col = 0;
  while (col < 16)
    {
      while (row < 16)
	{
/* print the table as strings of 4 chars */
	  gl_write (startx + row * 37.5, starty + col * 37.5,
		    sequence_table[row][col]);
/* oops, must have goofed up somewhere, interchange row and col for correct output */
	  row++;
	}
      col++;
      row = 0;
    }
}

/* display the counts table */
void
display_probability_table ()
{
  int row, col, startx, starty;
  gl_setfontcolors (0, 40);
  gl_printf (20, 220, "(counts per 10'000 total points)");
  gl_setfontcolors (31, 40);	/* change background to white to see better */
  startx = 302;
  starty = 63;
  row = col = 0;
  while (col < 16)
    {
      while (row < 16)
	{
	  gl_printf (startx + row * 37.5, starty + col * 37.5, "%d",
		     probability_table[row][col]);
	  row++;
	}
      col++;
      row = 0;
    }
}

/* display total probability, points plotted, user's input of first base, lase base checked */
void
display_parameters ()
{
  gl_setfontcolors (0, 48);
  gl_write (50, 60, "Probabilities are:");
  gl_printf (20, 80, "A = %6.4f", P.a);
  gl_printf (20, 100, "T = %6.4f", P.t);
  gl_printf (150, 100, "G = %6.4f", P.g);
  gl_printf (150, 80, "C = %6.4f", P.c);
  gl_printf (50, 120, "Total points: %d", CYCLES);	/* total points plotted */
/* only if disk file is read */
  if (seqfile.flag == 1)
    {
      gl_printf (20, 140, "First base: %d", CYCLES_FIRST);	/* user's input echoed with minor changes in case of wrong input */
      gl_printf (20, 160, "Last base: %d", CYCLES_LAST);	/* display last base actually checked */
    }
}

/*keep template for use by fileplotscreen and randomplotscreen for fresh plots*/
void
make_templatescreen ()
{
  gl_setcontext (templatescreen);	/*change to virtual screen */
  gl_clearscreen (0);
  gl_setfontcolors (0, 44);
  gl_printf (5, 5,
	     " HELP MENU:    file    iterations    probability    counts    grid    sequence    refresh    Print    Reload    about    quit");
  gl_setfontcolors (0, vga_white ());
  gl_printf (50, 650, "dnacgr-0.2");
  draw_grid ();
  draw_border ();
  gl_setcontext (physicalscreen);	/*change to real screen */
}

/* prepare virtual screens randomplotscreen and fileplot screen and show on real screen */
void
make_display (long int cycles)
{
  make_templatescreen ();
  if (seqfile.flag == 0)
    {
      plot_random_sequence (cycles);
    }
  else
    plot_file_sequence (cycles);
  display_parameters ();
  flags.reload = 1 - flags.reload;	/* reverse global flag */
}

/* to save time just refresh the screen, do not replot any data */
void
display_refresh (int input_grid)
{
  if (seqfile.flag == 0)
    {
      gl_setcontext (randomplotscreen);
      gl_copyscreen (physicalscreen);
      gl_setcontext (physicalscreen);
    }
  else
    {
      gl_setcontext (fileplotscreen);
      gl_copyscreen (physicalscreen);
      gl_setcontext (physicalscreen);
    }
  if (input_grid == 1)
    {
      draw_grid ();
      draw_border ();
    }				/* draw grid if required */
  display_parameters ();	/* parameters are always displayed */
  flags.refresh = 1 - flags.refresh;	/* reverse global flag */
}

/* save the color table to file for use with screen memory dump */
/* unused code */
void
save_color_table ()
{
  FILE *fopen (), *file_pointer;
  int color, r, g, b;
  color = 0;
  file_pointer = fopen ("clrtbl.dnacgr", "w");
  for (color = 0; color < 256; color++)
    {
      gl_getpalettecolor (color, &r, &g, &b);
      fprintf (file_pointer, "%d\t%d\t%d\t%d\n", color, r * 4, g * 4, b * 4);
    }
  fclose (file_pointer);
}

/* ask user for output filename. .png extension is added automatically */
void
get_output_filename ()
{
  char *filename_pointer, *malloc ();
  vga_setmode (TEXT);
  filename_pointer = malloc (100);
  printf ("\nEnter new filename for saving image file\n");
  filename_pointer = readline ("Filename=> ");
  strncpy (outfile.name, filename_pointer, 100);
  strcat (outfile.name, ".png");
  free (filename_pointer);
  vga_setmode (VGAMODE);
}

/* this just dumps screen memory to file, you need color tables to use this dump */
/* unused code */
void
print_file ()
{
  FILE *fopen (), *file_pointer;
  char *saved_screen_ptr, *malloc ();
  saved_screen_ptr = malloc (1024 * 768);
  gl_getbox (0, 0, 1024, 768, saved_screen_ptr);
  file_pointer = fopen (outfile.name, "w");
  fwrite (saved_screen_ptr, 1, 1024 * 768, file_pointer);
  fclose (file_pointer);
  free (saved_screen_ptr);
  flags.print = 1 - flags.print;
}

/* print to png file. modified from code donated by Sergio Masci <sergio@titan.demon.co.uk> */
void
print_png_file ()
{
  FILE *fopen (), *fp;
  char *malloc ();
  int xred, xgreen, xblue;
  int adj, bits_per_colour, j;
  char *rp;
  png_struct *png_ptr;
  png_info *info_ptr;
  png_colorp palette;

  fp = fopen (outfile.name, "wb");

  // allocate the necessary structures
  png_ptr = (png_struct *) malloc (sizeof (png_struct));

  info_ptr = (png_info *) malloc (sizeof (png_info));

  // initialize the structures
  png_info_init (info_ptr);
  png_write_init (png_ptr);

  // set up the output control
  png_init_io (png_ptr, fp);

  // set the file information here
  info_ptr->width = 1024;
  info_ptr->height = 768;
  info_ptr->bit_depth = 8;
  info_ptr->color_type = PNG_COLOR_TYPE_PALETTE;

//----------------------

  // set the palette
  info_ptr->valid |= PNG_INFO_PLTE;
  info_ptr->palette = (png_color *) malloc (256 * sizeof (png_color));
  info_ptr->num_palette = 256;



  bits_per_colour = 6;

  adj = 8 - bits_per_colour;


  for (j = 0; j < 256; j++)
    {
      vga_getpalette (j, &xred, &xgreen, &xblue);

      info_ptr->palette[j].red = xred << adj;
      info_ptr->palette[j].green = xgreen << adj;
      info_ptr->palette[j].blue = xblue << adj;
    }

  // write the file information
  png_write_info (png_ptr, info_ptr);

  rp = malloc (1024);

  for (j = 0; j < 768; j++)
    {
      gl_getbox (0, j, 1024, 1, rp);

      // Write a row at a time.
      png_write_rows (png_ptr, &rp, 1);
    }

  free (rp);

//----------------------

  // write the rest of the file
  png_write_end (png_ptr, NULL);

  // clean up after the write, and free any memory allocated
  png_write_destroy (png_ptr);

  if (info_ptr->palette)
    {				// a palette was used, so free it
      free (info_ptr->palette);
    }

  // free the structures
  free (png_ptr);
  free (info_ptr);

  fclose (fp);
}


/* add your name here if you add a significant functionality to this code */
/* do not change the copyright notice */
void
show_about_screen ()
{
  int key;
  vga_setmode (VGAMODE);
  gl_clearscreen (31);
  gl_setfontcolors (31, 0);
  gl_printf (360, 250,
	     "        DNA-CGR (version 0.2)\nProgram to visualise patterns in DNA\n\n         Copyright (C) 2000\n\n                by\n\n         Indraneel Majumdar\n\n     <indraneel@123india.com>\n\n\n     Copyright Policy - GNU-LGPL\n\n   (see file LGPL-2.1 for details)\n\n     (press 'L' to show License)");
  key = getchar ();
  if (key == 'L')
    {
      vga_setmode (TEXT);
      system ("/usr/bin/less  /usr/share/common-licenses/LGPL-2.1");
      vga_setmode (VGAMODE);
    }
}

/* background process to handle all keypresses */
void
user_request_handler ()
{
  long int cycles;
  int user_key;
  struct all_flags input;
  /* set initial values of local flags for first run */
  input.file = 0;
  input.prob = 1;
  input.cycles = 0;
  input.grid = 0;
  input.seq = 0;
  input.reload = 0;
  input.counts = 0;
  input.refresh = 1;
  input.print = 1;
  user_key = 'x';
  while (user_key != 'q')
    {
      switch (user_key)
	{
	case 'f':
	  input.file = 1 - input.file;
	  input.reload = 1 - input.reload;	/* reload file (replot) */
	  break;
	case 'p':
	  if (seqfile.flag == 0)
	    {
	      input.prob = 1 - input.prob;
	      input.reload = 1 - input.reload;	/* replot */
	    }
	  break;
	case 'i':
	  input.cycles = 1 - input.cycles;
	  input.reload = 1 - input.reload;	/* replot from file */
	  break;
	case 'g':
	  input.grid = 1 - input.grid;
	  if (input.grid == 1)
	    {
	      draw_grid ();
	      draw_border ();
	    }
	  else
	    input.refresh = 1 - input.refresh;	/* just refresh screen */
	  break;
	case 's':
	  input.seq = 1 - input.seq;
	  if (input.seq == 0)
	    input.refresh = 1 - input.refresh;	/*just refresh screen */
	  break;
	case 'R':
	  input.reload = 1 - input.reload;	/* user's request to reload file */
	  break;
	case 'c':
	  if (seqfile.flag == 1)
	    input.counts = 1 - input.counts;
	  /* switch off counts display (refresh screen) */
	  if ((seqfile.flag == 1) && (input.counts == 0))
	    input.refresh = 1 - input.refresh;
	  break;
	case 'r':
	  input.refresh = 1 - input.refresh;	/* refresh screen */
	  break;
	case 'P':
	  get_output_filename ();
	  input.refresh = 1 - input.refresh;
	  input.print = 1 - input.print;
	  break;
	case 'a':
	  show_about_screen ();	/* version, names of author(s), copyright */
	  input.refresh = 1 - input.refresh;
	  break;
	default:
	  break;
	}
      user_key = 'x';
/* do not change the sequence below (the sequence of operations is required at least for first run) */
      if (input.file == flags.file)
	get_seqfile ();
/* next step explicitly checks for probabilities on first run (probabilities are required only for random number plot) */
      if (
	  ((seqfile.flag == 0) && ((P.a + P.t + P.g + P.c) == 0)
	   && (input.prob = 1 - input.prob)) || ((seqfile.flag == 0)
						 && (input.prob ==
						     flags.
						     prob)))
	  get_atgc_probabilities ();
      if (input.cycles == flags.cycles)
	cycles = get_cycles ();	/* iterations */
      if (input.reload == flags.reload)
	make_display (cycles);	/* replot */
      if (input.refresh == flags.refresh)
	display_refresh (input.grid);	/*just refresh */
      if (input.seq == 1)
	display_sequence_table ();
/* counts table displayed only if sequence file is used */
      if ((seqfile.flag == 1) && (input.counts == 1))
	display_probability_table ();
      user_key = vga_getkey ();	/* read key without waiting */
      if (input.print == flags.print)
	print_png_file ();
    }
}

int
main ()
{
  int x, y;
  x = y = 0;
  vga_init ();			/* initialise display for svgalib */
  if (vga_hasmode (VGAMODE) == 0)
    exit (1);			/* check if mode 12 is supported */
/* set corner points of border */
  A.x = 300;
  A.y = 50;
  T.x = 300;
  T.y = 650;
  G.x = 900;
  G.y = 650;
  C.x = 900;
  C.y = 50;
  srand (time (0));		/* initialise random number generator */

  vga_setmode (VGAMODE);	/* set 1024x768 at 256 colors */

/* define fileplotscreen as virtual screen */
  gl_setcontextvgavirtual (VGAMODE);
  fileplotscreen = gl_allocatecontext ();
  gl_getcontext (fileplotscreen);

/* define randomplotscreen as virtual screen */
  gl_setcontextvgavirtual (VGAMODE);
  randomplotscreen = gl_allocatecontext ();
  gl_getcontext (randomplotscreen);

/* define templatescreen as virtual screen */
  gl_setcontextvgavirtual (VGAMODE);
  templatescreen = gl_allocatecontext ();
  gl_getcontext (templatescreen);

/* define physicalscreen as real screen */
  gl_setcontextvga (VGAMODE);
  physicalscreen = gl_allocatecontext ();
  gl_getcontext (physicalscreen);

  gl_setcontext (physicalscreen);	/* change to physicalscreen */
/*	save_color_table();		 output to file clrtbl.dnacgr */

/* define fonts globally */
  gl_setfont (8, 8, gl_font8x8);
  gl_setwritemode (FONT_COMPRESSED + WRITEMODE_OVERWRITE);
  show_about_screen ();		/* whom you can't flame if you lose your job */
  make_sequence_table ();	/* this is not modified later */
  user_request_handler ();	/* main loop */

/* free all memory */
  gl_freecontext (fileplotscreen);
  gl_freecontext (randomplotscreen);
  gl_freecontext (templatescreen);
  vga_setmode (TEXT);		/* just to be safe */
}
