{***************************************************************************
 *
 * Author:     Alberto Pascual Montano (pascual@fis.ucm.es)
 *             http://www.dacya.ucm.es/apascual

 * Complutense University of Madrid (UCM). Madrid, Spain
 * The KEY Institute for Brain-Mind Research, Zurich, Switzerland
 * National Center for Biotechnology (CNB). Madrid, Spain
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 *  All comments concerning this program package may be sent to the
 *  e-mail address 'pascual@fis.ucm.es'
 ***************************************************************************}

unit processUnit;

interface

uses uMatToolsDyn,ComCtrls;

var
    MyStatusBar: ^TStatusBar;
    CancelOption: boolean;

procedure classNMF(infndat,outfn:string;
               ReUseWH,Order:boolean;
               minNF, maxNF, maxiter, sparsePower:integer;
               sparse,nosparse:extended;
               negmethod: integer;
               showgraphics: boolean;
               transpose: boolean;
               saveRR: boolean;
               numberOfRuns: integer;
               normMethod: integer;
               minimize: boolean
               );

procedure biNMF(infndat,outfn:string;
               ReUseWH,Order:boolean;
               nq,maxiter, sparsePower:integer;
               eps1,sparse,nosparse:extended;
               negmethod: integer;
               showgraphics: boolean;
               transpose: boolean;
               saveRR: boolean;
               numberOfRuns: integer;
               normMethod: integer;
               minimize: boolean
               );

function stNMF(infndat,outfn:string;
               ReUseWH,Order:boolean;
               nq,maxiter, sparsePower:integer;
               eps1,sparse,nosparse:extended;
               negmethod: integer;
               showgraphics: boolean;
               transpose: boolean;
               saveRR: boolean;
               numberOfRuns: integer;
               normMethod: integer;
               minimize: boolean
               ): boolean;

implementation

uses sysutils,windows,utilityUnit,Dialogs,strutils, microarray, consensus, math, forms;

var ws,sh,x1,vc:matrix; ev:extended; y:vector; iy:intvector;
var
    rowlabels, varlabels: stringvector;
    numberofrowheaders, numberofvarheaders: integer;
    x2: matrix;

function GetFunctional(nvars,nobjs:integer;
                         var x:matrix
                         ):extended;
label 1;
var t1:extended; i,j:integer;
begin
  t1:=0;
  for i:=1 to nvars do
    for j:=1 to nobjs do
      if x.m[j,i]<=0 then begin
        t1:=0;
        goto 1;
      end else
        t1:=t1+x.m[j,i]*(ln(x.m[j,i])-1.0);
  1:;
  GetFunctional:=t1;
end;


procedure ReadW(OutFN:string;
                           ITEMSxVARS:boolean;
                           nq,nobjs:integer; 
                           var w:matrix
                           );
var fn:string; txt1:textfile; i,j:integer;
begin
  fn:=OutFn+'-Wt.txt';
  assignfile(txt1,fn);
  filemode:=0;
  reset(txt1);
  if ITEMSxVARS then
    for i:=1 to NQ do
      for j:=1 to Nobjs do
        read(txt1,w.m[j,i])
  else
    for j:=1 to Nobjs do
      for i:=1 to NQ do
        read(txt1,w.m[j,i]);
  closefile(txt1);
end;

procedure ReadH(OutFN:string;
                   ITEMSxVARS:boolean;
                   nvars,nq:integer;
                   var h:matrix
                   );
var fn:string; txt1:textfile; i,j:integer; s:string;
begin
  fn:=OutFn+'-Ht.txt';
  assignfile(txt1,fn);
  filemode:=0;
  reset(txt1);
  if ITEMSxVARS then
    for i:=1 to Nvars do begin
      for j:=1 to NQ do
        read(txt1,h.m[j,i]);
      readln(txt1,s);
    end
  else
    for j:=1 to NQ do
      for i:=1 to Nvars do
        read(txt1,h.m[j,i]);
  closefile(txt1);
end;

procedure GetWH(ReUseWH,ITEMSxVARS:boolean;
                nvars,nq,nobjs:integer;
                OutFN:string;
                var w:matrix;
                var h:matrix;
                var x:matrix
                );
var i,j,k:integer;
begin
  if not ReUseWH then begin
    randomize;
    for j:=1 to nq do begin
      k:=random(nvars);
      for i:=1 to nobjs do
        w.m[i,j]:=x.m[i,k];
    end;
    for i:=1 to nq do
      for j:=1 to nvars do
        h.m[i,j]:=random;
  end else begin
    ReadW(OutFN,ITEMSxVARS,nq,nvars,w);
    ReadH(OutFN,ITEMSxVARS,nobjs,nq,h);
  end;
end;


procedure GetWH2(nvars,nq,nobjs:integer;
                var w:matrix;
                var h:matrix
                );
var i,j:integer;
begin
    randomize;
    for j:=1 to nq do
      for i:=1 to nobjs do
        w.m[i,j]:= random;
    for i:=1 to nq do
      for j:=1 to nvars do
        h.m[i,j]:=random;
end;


procedure WrtInfo1(OutFN,InFNdat:string;
                   ReUseWH:boolean;
                   nobjs,nvars,nq,maxiter,iter, randomRuns, bestRun:integer;
                   bestEV, eps1,tconv,functional,sparse:extended;
                   negmethod: integer;
                   nmfMethod: integer;
                   normmethod: integer
                   );
var fn:string; txt1:textfile; t:extended; i:integer;
var
  orgvars, orgitems: integer;
begin
  orgvars := nvars;
  orgitems:= nobjs;
  if (negmethod = 1) then
     orgvars := orgvars div 2;
  if (negmethod = 2) then
     orgitems := orgitems div 2;
  fn:=OutFn+'-info.txt';
  assignfile(txt1,fn);
  filemode:=1;
  rewrite(txt1);
  writeln(txt1,'bioNMF: Non-negative Matrix Factorization for gene expression analysis');
  writeln(txt1,'Complutense University of Madrid, Spain, National Center for Biotecnology, Spain and the KEY Institute for Brain-Mind Research, Zurich, Switzerland');
  writeln(txt1);
  if (nmfMethod = 0) then
    writeln(txt1,'Method: Standard NMF');
  if (nmfMethod = 1) then
    writeln(txt1,'Method: Biclustering Analysis');
  if (nmfMethod = 2) then
    writeln(txt1,'Method: Sample classification');

  writeln(txt1);
  writeln(txt1,'Input data: ',InFNdat);

  write(txt1,'Normalization method: ');
  if (normmethod = 0) then
    writeln(txt1,'No normalization')
  else if (normmethod = 1) then
    writeln(txt1,'Subtract global mean')
  else if (normmethod = 2) then
    writeln(txt1,'Scale columns, then normalize rows')
  else if (normmethod = 3) then
    writeln(txt1,'Mean = 0, SD = 1 by rows')
  else if (normmethod = 4) then
    writeln(txt1,'Mean = 0, SD = 1 by columns')
  else if (normmethod = 5) then
    writeln(txt1,'Subtract mean by rows')
  else if (normmethod = 6) then
    writeln(txt1,'Subtract mean by columns')
  else if (normmethod = 7) then
    writeln(txt1,'Subtract mean by rows and then by columns')
  else if (normmethod = 8) then
    writeln(txt1,'Iterative rows and columns normalization')
  else if (normmethod = 9) then
    writeln(txt1,'Log-Interactions normalization');

  if (negmethod = 1) then begin
    writeln(txt1,'Data originally negative. It was made positive by folding the columns ');
    writeln(txt1,'Original number of columns: ',orgvars);
  end else if (negmethod = 2) then begin
    writeln(txt1,'Data originally negative. It was made positive by folding the rows ');
    writeln(txt1,'Original number of rows: ',orgitems);
  end else if (negmethod = 3) then begin
    writeln(txt1,'Data originally negative. It was made positive by adding the absolute minimum ');
  end else if (negmethod = 4) then begin
    writeln(txt1,'Data originally negative. It was made positive by exponential scaling ');
  end;

  writeln(txt1,'Number of rows: ',NObjs);
  writeln(txt1,'Number of columns: ',NVars);

  writeln(txt1,'Number of factors: ',NQ);
  writeln(txt1,'Sparseness constant: ',FloatToStrF(sparse, ffFixed, 4, 2));
  writeln(txt1,'Stopping threshold:  ',FloatToStrF(eps1, ffExponent, 2, 2));
  writeln(txt1,'Maximum number of iterations: ',MaxIter);
  if (randomRuns >1) then begin
     writeln(txt1,'Number of random runs: ',randomRuns);
     writeln(txt1, 'Best run: ', bestRun);
     writeln(txt1, 'Best explained variance: ', FloatToStrF(bestEV, ffFixed, 4, 2));
  end;
  writeln(txt1);
  writeln(txt1,'Output info file (this file): ',fn);
  fn:=OutFn+'-W.txt';
  writeln(txt1,'Factors file (matrix W in the model): ',fn);
  fn:=OutFn+'-H.txt';
  writeln(txt1,'Encoding vectors (matrix H in the model): ',fn);
  fn:=OutFn+'-Wt.txt';
  writeln(txt1,'Plain text factors file (matrix W in the model): ',fn);
  fn:=OutFn+'-Ht.txt';
  writeln(txt1,'Plain text encoding vectors (matrix H in the model): ',fn);
  writeln(txt1);
  writeln(txt1,'% Explained Variance by full model = ',FloatToStrF(ev, ffFixed, 4, 2));
  if ReUseWH then
    writeln(txt1,'Re-using old input from previous run');
  writeln(txt1);
  t:=0;
  for i:=1 to nq do
    t:=t+y.v[i];
  writeln(txt1,'Importance (%) by factors:');
  for i:=1 to nq do
    if (t > 1e-16) then
      writeln(txt1,i:5,'  ',(100.0*y.v[i]/t):15:5)
    else
      writeln(txt1,i:5,'  ',0.0:15:5);
  writeln(txt1);
  closefile(txt1);

end;

procedure WrtInfo2(OutFN,InFNdat:string;
                   nobjs,nvars,minNF, maxNF,randomRuns:integer;
                   sparse:extended; negmethod: integer;
                   normmethod: integer;
                   var consensusRec: TConsensusRecord);
var fn:string; txt1:textfile;
var
  orgvars, orgitems,i: integer;
begin
  orgvars := nvars;
  orgitems:= nobjs;
  if (negmethod = 1) then
     orgvars := orgvars div 2;
  if (negmethod = 2) then
     orgitems := orgitems div 2;
  fn:=OutFn+'-info.txt';
  assignfile(txt1,fn);
  filemode:=1;
  rewrite(txt1);
  writeln(txt1,'bioNMF: Non-negative Matrix Factorization for gene expression analysis');
  writeln(txt1,'Complutense University of Madrid, Spain, National Center for Biotecnology, Spain and the KEY Institute for Brain-Mind Research, Zurich, Switzerland');
  writeln(txt1);
  writeln(txt1,'Method: Sample classification');

  writeln(txt1);
  writeln(txt1,'Input data: ',InFNdat);

  write(txt1,'Normalization method: ');
  if (normmethod = 0) then
    writeln(txt1,'No normalization')
  else if (normmethod = 1) then
    writeln(txt1,'Subtract global mean')
  else if (normmethod = 2) then
    writeln(txt1,'Scale columns, then normalize rows')
  else if (normmethod = 3) then
    writeln(txt1,'Mean = 0, SD = 1 by rows')
  else if (normmethod = 4) then
    writeln(txt1,'Mean = 0, SD = 1 by columns')
  else if (normmethod = 5) then
    writeln(txt1,'Subtract mean by rows')
  else if (normmethod = 6) then
    writeln(txt1,'Subtract mean by columns')
  else if (normmethod = 7) then
    writeln(txt1,'Subtract mean by rows and then by columns')
  else if (normmethod = 8) then
    writeln(txt1,'Iterative rows and columns normalization')
  else if (normmethod = 9) then
    writeln(txt1,'Log-Interactions normalization');

  if (negmethod = 1) then begin
    writeln(txt1,'Data originally negative. It was made positive by folding the columns ');
    writeln(txt1,'Original number of columns: ',orgvars);
  end else if (negmethod = 2) then begin
    writeln(txt1,'Data originally negative. It was made positive by folding the rows ');
    writeln(txt1,'Original number of rows: ',orgitems);
  end else if (negmethod = 3) then begin
    writeln(txt1,'Data originally negative. It was made positive by adding the absolute minimum ');
  end else if (negmethod = 4) then begin
    writeln(txt1,'Data originally negative. It was made positive by exponential scaling ');
  end;

  writeln(txt1,'Number of rows: ',NObjs);
  writeln(txt1,'Number of columns: ',NVars);

  writeln(txt1,'Minimum number of factors: ', minNF);
  writeln(txt1,'Maximum number of factors: ', maxNF);
  writeln(txt1,'Number of random runs: ',randomRuns);
  writeln(txt1);
  writeln(txt1,'Output info file (this file): ',fn);
  fn:=OutFn+'-cluster-#factors.txt';
  writeln(txt1,'Cluster files for each factorization: ',fn);
  writeln(txt1);

  writeln(txt1,'Cophenetic correlation coefficients: ');
  writeln(txt1);

  for i:= 1 to (maxNF-minNF)+1 do begin
      writeln(txt1, consensusRec.ConsMatvector[i].K, ' Factors: ', FloatToStrF(consensusRec.ConsMatvector[i].Coph, ffFixed, 4, 4));
  end;
  closefile(txt1);
end;


procedure WrtW1(OutFN:string;
                          ITEMSxVARS:boolean;
                          nq,nobjs:integer;
                          var w:matrix
                          );
var fn:string; txt1:textfile; i,j:integer;
begin
  fn:=OutFn+'-Wt.txt';
  assignfile(txt1,fn);
  filemode:=1;
  rewrite(txt1);
  if ITEMSxVARS then
    for i:=1 to NQ do begin
      for j:=1 to Nobjs do
        write(txt1,' ',w.m[j,iy.v[i]]:15);
      writeln(txt1);
    end
  else
    for j:=1 to Nobjs do begin
      for i:=1 to NQ do
        write(txt1,' ',w.m[j,iy.v[i]]:15);
      writeln(txt1);
    end;
  closefile(txt1);
end;

procedure WrtConsensus(OutFN:string;
                 var TT:matrix
                          );
var fn:string; txt1:textfile; i,j:integer;
begin
  fn:=OutFn+'-CONS.txt';
  assignfile(txt1,fn);
  filemode:=1;
  rewrite(txt1);
  for i:=1 to TT.nr do begin
      for j:=1 to TT.nc do begin
        write(txt1,' ',TT.m[i,j]);
      end;
      writeln(txt1);
  end;
  closefile(txt1);
end;


procedure WrtW2(OutFN:string;
                          nq,nobjs:integer;
                          var w:matrix;
                          negmethod: integer
                          );
var fn:string; txt1, txt2:textfile; i,j,k:integer;
begin
  if (negmethod <> 2) then begin
      fn:=OutFn+'-W.txt';
      assignfile(txt1,fn);
      filemode:=1;
      rewrite(txt1);

      write(txt1,'Item Labels',#9);

      for i:=1 to NQ-1 do
        write(txt1,'Factor-', inttostr(i),#9);
      write(txt1,'Factor-', inttostr(NQ),#13);
      for j:=1 to Nobjs do begin
        write(txt1,rowlabels[j],#9);
        for i:=1 to NQ do begin
            if (i <> NQ) then
                write(txt1,w.m[j,iy.v[i]]:15, #9)
            else
                write(txt1,w.m[j,iy.v[i]]:15, #13)
        end;
      end;
      closefile(txt1);
  end else begin
      fn:=OutFn+'-Wpos.txt';
      assignfile(txt1,fn);
      filemode:=1;
      rewrite(txt1);
      fn:=OutFn+'-Wneg.txt';
      assignfile(txt2,fn);
      filemode:=1;
      rewrite(txt2);
      write(txt1,'Item Labels',#9);
      write(txt2,'Item Labels',#9);


      for i:=1 to NQ-1 do begin
        write(txt1,'Factor-', inttostr(i),#9);
        write(txt2,'Factor-', inttostr(i),#9);
      end;
      write(txt1,'Factor-', inttostr(NQ),#13);
      write(txt2,'Factor-', inttostr(NQ),#13);
      k := 1;
      for j:=1 to (Nobjs div 2) do begin
        write(txt1,rowlabels[j],#9);
        write(txt2,rowlabels[j],#9);

        for i:=1 to NQ-1 do begin
           write(txt1,w.m[k,iy.v[i]]:15, #9);
           write(txt2,w.m[k+1,iy.v[i]]:15, #9);
        end;
        write(txt1,w.m[k,iy.v[NQ]]:15, #13);
        write(txt2,w.m[k+1,iy.v[NQ]]:15, #13);
        k := k+2;
      end;
      closefile(txt1);
      closefile(txt2);
  end;
end;

procedure WrtH1(OutFN:string;
                  ITEMSxVARS:boolean;
                  nq,nvars:integer;
                  var h:matrix
                  );
var fn:string; txt1:textfile; i,j:integer;
begin
  fn:=OutFn+'-Ht.txt';
  assignfile(txt1,fn);
  filemode:=1;
  rewrite(txt1);
  if ITEMSxVARS then
    for i:=1 to Nvars do begin
      for j:=1 to NQ do
        write(txt1,' ',h.m[iy.v[j],i]:15);
      writeln(txt1);
    end
  else
    for j:=1 to NQ do begin
      for i:=1 to Nvars do
        write(txt1,' ',h.m[iy.v[j],i]:15);
      writeln(txt1)
    end;
  closefile(txt1);
end;

procedure WrtH2(OutFN:string;
                  nq,nvars:integer;
                  var h:matrix;
                  negmethod: integer
                  );
var fn:string; txt1, txt2:textfile; i,j,k:integer;
begin
  if (negmethod <> 1) then begin
      fn:=OutFn+'-H.txt';
      assignfile(txt1,fn);
      filemode:=1;
      rewrite(txt1);

      write(txt1,'Var Labels',#9);

      for i:=1 to NQ-1 do
        write(txt1,'Factor-', inttostr(i),#9);
      write(txt1,'Factor-', inttostr(NQ),#13);

      for i:=1 to Nvars do begin
        write(txt1,varlabels[i+numberofrowheaders],#9);
        for j:=1 to NQ-1 do
           write(txt1,h.m[iy.v[j],i]:15,#9);
        write(txt1,h.m[iy.v[NQ],i]:15,#13);
      end;
      closefile(txt1);
  end else begin
      fn:=OutFn+'-Hpos.txt';
      assignfile(txt1,fn);
      filemode:=1;
      rewrite(txt1);
      fn:=OutFn+'-Hneg.txt';
      assignfile(txt2,fn);
      filemode:=1;
      rewrite(txt2);

      write(txt1,'Var Labels',#9);
      write(txt2,'Var Labels',#9);

      for i:=1 to NQ-1 do begin
        write(txt1,'Factor-', inttostr(i),#9);
        write(txt2,'Factor-', inttostr(i),#9);
      end;
      write(txt1,'Factor-', inttostr(NQ),#13);
      write(txt2,'Factor-', inttostr(NQ),#13);
      k:=1;
      for i:=1 to (Nvars div 2) do begin
       write(txt1,varlabels[i+numberofrowheaders],#9);
       write(txt2,varlabels[i+numberofrowheaders],#9);
       for j:=1 to NQ-1 do begin
           write(txt1,h.m[iy.v[j],k]:15,#9);
           write(txt2,h.m[iy.v[j],k+1]:15,#9);
        end;
        write(txt1,h.m[iy.v[NQ],k]:15,#13);
        write(txt2,h.m[iy.v[NQ],k+1]:15,#13);
        k:= k+2;
      end;
      closefile(txt1);
      closefile(txt2);
  end;
end;


procedure GobiNMF(infndat,outfn:string;
               ReUseWH,Order:boolean;
               nvars,nobjs,nq,maxiter, sparsePower:integer;
               eps1,sparse,nosparse:extended;
               var x:matrix;
               var w:matrix;
               var h:matrix;
               var iter:integer;
               var tconv,functional:extended;
               negmethod: integer;
               showgraphics: boolean;
               saveRR:boolean;
               numberOfRuns: integer;
               normMethod: integer
               );
var Functional1:extended; num,den2,den1,r1,t,mean, maxEV:extended; 
    a,b,c,i,j,k,kk,index,rr,maxRun:integer; convergence:boolean;
    sortedA, sortedB: vector;
    sortedAIndex, sortedBIndex: intvector;
    wp, wn, hp, hn: matrix;
    maxw, maxh: matrix;
begin


  // Run N number of times using random initialization (in case Re-use is not used)
 maxEV := -1;
 maxRun := 1;
 Create1(nq, nvars,maxh);
 Create1(nobjs, nq,maxw);
 for rr:=1 to numberOfRuns do begin

      myStatusBar^.SimpleText := ' Initializing...';
      Functional1:=GetFunctional(nvars,nobjs,x);
      GetWH(ReUseWH,true,nvars,nq,nobjs,OutFN,w,h,x);

      equate1(ws,w); // ws = w
      for kk := 1 to sparsePower do begin
        for i:=1 to nobjs do begin
          r1:=0;
          for j:=1 to nq do
            r1:=r1+ws.m[i,j];
          r1:=r1/nq;
          for j:=1 to nq do
            ws.m[i,j]:=nosparse*ws.m[i,j]+sparse*r1;
        end;
      end;

      equate1(sh,h); //sh = h
      for kk:=1 to sparsePower do begin
        for i:=1 to nvars do begin
          r1:=0;
          for j:=1 to nq do
            r1:=r1+sh.m[j,i];
          r1:=r1/nq;
          for j:=1 to nq do
            sh.m[j,i]:=nosparse*sh.m[j,i]+sparse*r1;
        end;
      end;

      iter:=0;
      functional:=0;
      repeat
        iter:=iter+1;

        myStatusBar^.SimpleText := ' Running Iteration ' + inttostr(iter) + ' for run ' + inttostr(rr);
        if (iter mod 10) = 0 then begin
            application.ProcessMessages;
            if cancelOption then exit;
        end;
         
        for b:=1 to nvars do
          for i:=1 to nobjs do begin
            den1:=0;
            for k:=1 to nq do
              den1:=den1+w.m[i,k]*sh.m[k,b];
            x1.m[i,b]:=den1;
            if den1 > 1e-16 then
              vc.m[i,b]:=x.m[i,b]/den1
            else
              vc.m[i,b]:=0;             
          end;

        t:=Functional1;
        for i:=1 to nobjs do
          for j:=1 to nvars do
            if x1.m[i,j] > 1e-16 then
              t:=t-x.m[i,j]*ln(x1.m[i,j])+x1.m[i,j];

        for a:=1 to nq do
          for b:=1 to nvars do begin
            num:=0;
            for i:=1 to nobjs do
              num:=num+ws.m[i,a]*vc.m[i,b];
            h.m[a,b]:=h.m[a,b]*num;
          end;

        equate1(sh,h); //sh = h
        for kk:=1 to sparsePower do begin
          for i:=1 to nvars do begin
            r1:=0;
            for j:=1 to nq do
              r1:=r1+sh.m[j,i];
            r1:=r1/nq;
            for j:=1 to nq do
              sh.m[j,i]:=nosparse*sh.m[j,i]+sparse*r1;
          end;
        end;

        for a:=1 to nq do begin
          for c:=1 to nobjs do begin
            num:=0;
            for j:=1 to nvars do
              num:=num+sh.m[a,j]*vc.m[c,j];
            w.m[c,a]:=w.m[c,a]*num;
          end;
          den2:=0;
          for i:=1 to nobjs do
            den2:=den2+w.m[i,a];
          for i:=1 to nobjs do
            if den2 > 1e-16 then
              w.m[i,a]:=w.m[i,a]/den2
            else
              w.m[i,a]:=0;
        end;

        equate1(ws,w); // ws = w
        for kk:=1 to sparsePower do begin
          for i:=1 to nobjs do begin
            r1:=0;
            for j:=1 to nq do
              r1:=r1+ws.m[i,j];
            r1:=r1/nq;
            for j:=1 to nq do
              ws.m[i,j]:=nosparse*ws.m[i,j]+sparse*r1;
          end;
        end;

        convergence:=(iter>=maxiter);
        if t > 1e-16 then
          tconv:=abs(1.0-functional/t)
        else
          tconv := 1;
        convergence:=convergence or (tconv<=eps1);
        functional:=t;
      until convergence;


      t:=0;
      den2:=0;
      for b:=1 to nvars do
        for i:=1 to nobjs do begin
          den1:=0;
          for k:=1 to nq do
            den1:=den1+w.m[i,k]*sh.m[k,b];
          t:=t+sqr(den1-x.m[i,b]);
          den2:=den2+sqr(x.m[i,b]);
        end;
      if den2 > 1e-16 then
        ev:=100-100*t/den2
      else
        ev := 0;

      if (maxEV < ev) then begin
         maxEV := ev;
         maxRun := rr;
         equate1(maxw,w); // maxw= w
         equate1(maxh,h); // maxh= h
      end;


      // Save random results
      if ((saveRR) and (numberOfRuns >1))  then begin
          for i:=1 to nq do begin
            t:=0;
            for j:=1 to nvars do
              t:=t+h.m[i,j];
            y.v[i]:=t;
            iy.v[i]:=i;
          end;
          if order then
            HEAPSORT1(nq,y,iy);

         WrtInfo1(OutFN+'-RandomRun'+IntToStr(rr),InFNdat,ReUseWH,nobjs,nvars,nq,
                   maxiter,iter,-1,-1,-1.0,eps1,tconv,functional,sparse, negmethod, 1, normMethod);
          WrtW1(OutFN+'-RandomRun'+IntToStr(rr),false,nq,nobjs,w);
          WrtW2(OutFN+'-RandomRun'+IntToStr(rr),nq,nobjs,w, negmethod);
          WrtH1(OutFN+'-RandomRun'+IntToStr(rr),true,nq,nvars,h);
          WrtH2(OutFN+'-RandomRun'+IntToStr(rr),nq,nvars,h,negmethod);
      end;


  end; // for rr (number of runs)

  myStatusBar^.SimpleText := ' Saving results...';

  equate1(w,maxw); // w = maxw
  equate1(h,maxh); // h = maxh
  ev := maxEV;
  Destroy1(maxw);
  Destroy1(maxh);

  for i:=1 to nq do begin
    t:=0;
    for j:=1 to nvars do
      t:=t+h.m[i,j];
    y.v[i]:=t;
    iy.v[i]:=i;
  end;
  if order then
    HEAPSORT1(nq,y,iy);

  WrtInfo1(OutFN,InFNdat,ReUseWH,nobjs,nvars,nq,
           maxiter,iter,numberOfRuns, maxRun, maxEV, eps1,tconv,functional,sparse, negmethod,1, normMethod);
  WrtW1(OutFN,false,nq,nobjs,w);
  WrtW2(OutFN,nq,nobjs,w, negmethod);
  WrtH1(OutFN,true,nq,nvars,h);
  WrtH2(OutFN,nq,nvars,h,negmethod);

  // Now the graphical tool
  if showgraphics then begin
      create1(microform.n,sortedA);
      create1(microform.p,sortedB);
      create1(microform.n,sortedAIndex);
      create1(microform.p,sortedBIndex);

      t:=0;
      for i:=1 to nq do
        t:=t+y.v[i];
      if (negmethod = 1) then begin // if (negmethod = 1) then begin
          // Fold by Columns
          Create1(nq, nvars div 2,hp);
          Create1(nq, nvars div 2,hn);
          k := 1;
          for j:=1 to (NVars div 2) do begin
              for i:=1 to nq do begin
                 hp.m[i,j] := h.m[i,k];
                 hn.m[i,j] := h.m[i,k+1];
              end;
              k := k+2;
          end;
          index := 2;

          for k := 1 to nq do begin
             // Sort matrix according to the factors
              for i:=1 to nobjs do
                sortedA.v[i] := w.m[i,iy.v[k]];
                indexx2(sortedA, sortedAIndex);
             //Positive
              for j:=1 to nvars div 2 do
                sortedB.v[j] := hp.m[iy.v[k],j];
               indexx2(sortedB, sortedBIndex);

              with microform.matvec[index] do begin
                    evar := 0;
                    if (t > 1e-16) then
                        evar := (100.0*y.v[k]/t);
                    name := 'Original data matrix sorted by positive factor ' + inttostr(k);
                    create1(nobjs,AIndex);
                    create1(nvars div 2,BIndex);
                    Equate1(AIndex,sortedAIndex);
                    Equate1(BIndex,sortedBIndex);
              end;

              with microform.factvec[index-1] do begin
                    evar := 0;
                    if (t > 1e-16) then
                        evar := (100.0*y.v[k]/t);
                    name := 'Sorted positive factor ' + inttostr(k);
                    create1(nobjs,AIndex);
                    create1(nvars div 2,BIndex);
                    create1(nobjs,Avec);
                    create1(nvars div 2,Bvec);
                    weight := 1;
                    amin:= 656000; amax := -656000;
                    bmin:= 656000; bmax := -656000;
                    for i:=1 to nobjs do begin
                        Avec.v[i] := w.m[sortedAindex.v[i], iy.v[k]];
                        if (amin > Avec.v[i]) then amin := Avec.v[i];
                        if (amax < Avec.v[i]) then amax := Avec.v[i];
                    end;
                    for j:=1 to nvars div 2 do begin
                        Bvec.v[j] := hp.m[iy.v[k], sortedBindex.v[j]];
                        if (bmin > Bvec.v[j]) then bmin := Bvec.v[j];
                        if (bmax < Bvec.v[j]) then bmax := Bvec.v[j];
                    end;
                    Equate1(AIndex,sortedAIndex);
                    Equate1(BIndex,sortedBIndex);
              end;
              index := index+1;
              // Negative
              for j:=1 to nvars div 2 do
               sortedB.v[j] := hn.m[iy.v[k],j];
              indexx2(sortedB, sortedBIndex);

              with microform.matvec[index] do begin
                    evar := 0;
                    if (t > 1e-16) then
                        evar := (100.0*y.v[k]/t);
                    name := 'Original data matrix sorted by negative factor ' + inttostr(k);
                    create1(nobjs,AIndex);
                    create1(nvars div 2,BIndex);
                    Equate1(AIndex,sortedAIndex);
                    Equate1(BIndex,sortedBIndex);
              end;

              with microform.factvec[index-1] do begin
                    evar := 0;
                    if (t > 1e-16) then
                        evar := (100.0*y.v[k]/t);
                    name := 'Sorted negative factor ' + inttostr(k);
                    create1(nobjs,AIndex);
                    create1(nvars div 2,BIndex);
                    create1(nobjs,Avec);
                    create1(nvars div 2,Bvec);
                    weight := 1;
                    amin:= 656000; amax := -656000;
                    bmin:= 656000; bmax := -656000;
                    for i:=1 to nobjs do begin
                        Avec.v[i] := w.m[sortedAindex.v[i], iy.v[k]];
                        if (amin > Avec.v[i]) then amin := Avec.v[i];
                        if (amax < Avec.v[i]) then amax := Avec.v[i];
                    end;
                    for j:=1 to nvars div 2 do begin
                        Bvec.v[j] := hn.m[iy.v[k], sortedBindex.v[j]];
                        if (bmin > Bvec.v[j]) then bmin := Bvec.v[j];
                        if (bmax < Bvec.v[j]) then bmax := Bvec.v[j];
                    end;
                    Equate1(AIndex,sortedAIndex);
                    Equate1(BIndex,sortedBIndex);
              end;
              index := index+1;
          end;
          Destroy1(wp);
          Destroy1(wn);
      end else if (negmethod = 2) then begin // end else if (negmethod = 2) then begin
          // Fold by rows
          Create1(nobjs div 2, nq, wp);
          Create1(nobjs div 2, nq, wn);
          k:=1;
          for i:=1 to (NObjs div 2) do begin
            for j:=1 to nq do begin
               wp.m[i, j] := w.m[k,j];
               wn.m[i, j] := w.m[k+1,j]; 
            end;
            k:=k+2;
          end;

          index := 2;
          for k := 1 to nq do begin
             // Sort matrix according to the factors
              for j:=1 to nvars do
                sortedB.v[j] := h.m[iy.v[k],j];
              indexx2(sortedB, sortedBIndex);

             //Positive
              for i:=1 to nobjs div 2 do
                sortedA.v[i] := wp.m[i, iy.v[k]];
              indexx2(sortedA, sortedAIndex);

              with microform.matvec[index] do begin
                    evar := 0;
                    if (t > 1e-16) then
                        evar := (100.0*y.v[k]/t);
                    name := 'Original data matrix sorted by positive factor ' + inttostr(k);
                    create1(nobjs div 2,AIndex);
                    create1(nvars,BIndex);
                    Equate1(AIndex,sortedAIndex);
                    Equate1(BIndex,sortedBIndex);
              end;

              with microform.factvec[index-1] do begin
                    evar := 0;
                    if (t > 1e-16) then
                        evar := (100.0*y.v[k]/t);
                    name := 'Sorted positive factor ' + inttostr(k);
                    create1(nobjs div 2,AIndex);
                    create1(nvars,BIndex);
                    create1(nobjs div 2,Avec);
                    create1(nvars,Bvec);
                    weight := 1;
                    amin:= 656000; amax := -656000;
                    bmin:= 656000; bmax := -656000;
                    for i:=1 to nobjs div 2 do begin
                        Avec.v[i] := wp.m[sortedAindex.v[i], iy.v[k]];
                        if (amin > Avec.v[i]) then amin := Avec.v[i];
                        if (amax < Avec.v[i]) then amax := Avec.v[i];
                    end;
                    for j:=1 to nvars do begin
                        Bvec.v[j] := h.m[iy.v[k], sortedBindex.v[j]];
                        if (bmin > Bvec.v[j]) then bmin := Bvec.v[j];
                        if (bmax < Bvec.v[j]) then bmax := Bvec.v[j];
                    end;
                    Equate1(AIndex,sortedAIndex);
                    Equate1(BIndex,sortedBIndex);
              end;
              index := index+1;
             //Negative
              for i:=1 to nobjs div 2 do
                sortedA.v[i] := wn.m[i, iy.v[k]];
              indexx2(sortedA, sortedAIndex);

              with microform.matvec[index] do begin
                    evar := 0;
                    if (t > 1e-16) then
                        evar := (100.0*y.v[k]/t);
                    name := 'Original data matrix sorted by negative factor ' + inttostr(k);
                    create1(nobjs div 2,AIndex);
                    create1(nvars,BIndex);
                    Equate1(AIndex,sortedAIndex);
                    Equate1(BIndex,sortedBIndex);
              end;

              with microform.factvec[index-1] do begin
                    evar := 0;
                    if (t > 1e-16) then
                        evar := (100.0*y.v[k]/t);
                    name := 'Sorted negative factor ' + inttostr(k);
                    create1(nobjs div 2,AIndex);
                    create1(nvars,BIndex);
                    create1(nobjs div 2,Avec);
                    create1(nvars,Bvec);
                    weight := 1;
                    amin:= 656000; amax := -656000;
                    bmin:= 656000; bmax := -656000;
                    for i:=1 to nobjs div 2 do begin
                        Avec.v[i] := wn.m[sortedAindex.v[i], iy.v[k]];
                        if (amin > Avec.v[i]) then amin := Avec.v[i];
                        if (amax < Avec.v[i]) then amax := Avec.v[i];
                    end;
                    for j:=1 to nvars do begin
                        Bvec.v[j] := h.m[iy.v[k], sortedBindex.v[j]];
                        if (bmin > Bvec.v[j]) then bmin := Bvec.v[j];
                        if (bmax < Bvec.v[j]) then bmax := Bvec.v[j];
                    end;
                    Equate1(AIndex,sortedAIndex);
                    Equate1(BIndex,sortedBIndex);
              end;
              index := index+1;
          end;
          Destroy1(hp);
          Destroy1(hn);
      end else begin
          // No folding
          for k := 1 to nq do begin
             // Sort matrix according to the factors
             for i:=1 to nobjs do
               sortedA.v[i] := w.m[i,iy.v[k]];
             for j:=1 to nvars do
               sortedB.v[j] := h.m[iy.v[k],j];

              // Sort both vectors
              indexx2(sortedA, sortedAIndex);
              indexx2(sortedB, sortedBIndex);

              with microform.matvec[k+1] do begin
                    evar := 0;
                    if (t > 1e-16) then
                        evar := (100.0*y.v[k]/t);
                    name := 'Original data matrix sorted by factor ' + inttostr(k);
                    create1(nobjs,AIndex);
                    create1(nvars,BIndex);
                    Equate1(AIndex,sortedAIndex);
                    Equate1(BIndex,sortedBIndex);
              end;

              with microform.factvec[k] do begin
                    evar := 0;
                    if (t > 1e-16) then
                        evar := (100.0*y.v[k]/t);
                    name := 'Sorted factor ' + inttostr(k);
                    create1(nobjs,AIndex);
                    create1(nvars,BIndex);
                    create1(nobjs,Avec);
                    create1(nvars,Bvec);
                    weight := 1;
                    amin:= 656000; amax := -656000;
                    bmin:= 656000; bmax := -656000;
                    for i:=1 to nobjs do begin
                        Avec.v[i] := w.m[sortedAindex.v[i],iy.v[k]];
                        if (amin > Avec.v[i]) then amin := Avec.v[i];
                        if (amax < Avec.v[i]) then amax := Avec.v[i];
                    end;
                    for j:=1 to nvars do begin
                        Bvec.v[j] := h.m[iy.v[k], sortedBindex.v[j]];
                        if (bmin > Bvec.v[j]) then bmin := Bvec.v[j];
                        if (bmax < Bvec.v[j]) then bmax := Bvec.v[j];
                    end;
                    Equate1(AIndex,sortedAIndex);
                    Equate1(BIndex,sortedBIndex);
              end;
          end;
      end;
      destroy1(sortedA);
      destroy1(sortedB);
      destroy1(sortedAIndex);
      destroy1(sortedBIndex);
  end; // if showgraohics

end;

var x,w,h:matrix; iter:integer; tconv,functional:extended;

procedure biNMF(infndat,outfn:string;
               ReUseWH,Order:boolean;
               nq,maxiter,sparsePower:integer;
               eps1,sparse,nosparse:extended;
               negmethod: integer;
               showgraphics: boolean;
               transpose: boolean;
               saveRR: boolean;
               numberOfRuns: integer;
               normMethod: integer;
               minimize: boolean
               );
var
  i,j,k, nobjs, nvars: integer;
  isneg: boolean;
  min, max, mean: real;
begin

   myStatusBar^.SimpleText := ' Detecting data format from ' + infndat + ' file...';
   i:=detectFormat(infndat);
   if (i = -1) then begin
      showmessage('The input file is not supported.');
      exit;
   end else if (i = 0) then begin // Pure data (no labels in rows and columns)
      myStatusBar^.SimpleText := ' Reading data dimension from ' + infndat + ' file...';
      if ((not getPureDataDimensions(infndat, nobjs, nvars, numberofrowheaders, numberofvarheaders))
           or (nvars<=0) or (nobjs<=0)) then begin
          showmessage('Wrong input file or format not supported.');
          exit;
      end;
      myStatusBar^.SimpleText := ' Reading pure data from ' + infndat + ' file...';
      create1(nobjs,nvars,x2);
      createdestroy(true,nvars+numberofrowheaders,varlabels);
      createdestroy(true,nobjs, rowlabels);
      for i:=1 to nvars do varlabels[i] := 'Var' + inttostr(i);
      for i:=1 to nobjs do rowlabels[i] := 'Item' + inttostr(i);
      if (not ReadPureData(infndat,nobjs,nvars,x2, mean, min, max, isneg)) then begin
          showmessage('Wrong input file or format not supported.');
          exit;
      end;
   end else if (i = 1) then begin // Data with column and row labels
      myStatusBar^.SimpleText := ' Reading data dimension from ' + infndat + ' file...';
      if ((not getLabeledDataDimensions(infndat, nobjs, nvars, numberofrowheaders, numberofvarheaders))
           or (nvars<=0) or (nobjs<=0)) then begin
          showmessage('Wrong input file or format not supported.');
          exit;
      end;
      myStatusBar^.SimpleText := ' Reading labeled data from ' + infndat + ' file...';
      create1(nobjs,nvars,x2);
      createdestroy(true,nvars+numberofrowheaders,varlabels);
      createdestroy(true,nobjs, rowlabels);
      for i:=1 to nvars do varlabels[i] := 'Var' + inttostr(i);
      for i:=1 to nobjs do rowlabels[i] := 'Item' + inttostr(i);
      if (not ReadLabeledData(infndat,nobjs,nvars, numberofrowheaders, numberofvarheaders, x2, rowlabels, varlabels, mean, min, max, isneg)) then begin
          showmessage('Wrong input file or format not supported.');
          exit;
      end;
   end else if (i = 2) then begin // Engene
      myStatusBar^.SimpleText := ' Reading data dimension from ' + infndat + ' file...';
      if ((not getEngeneDataDimensions(infndat, nobjs, nvars, numberofrowheaders, numberofvarheaders))
           or (nvars<=0) or (nobjs<=0)) then begin
           showmessage('Wrong input file or format not supported.');
           exit;
      end;
      myStatusBar^.SimpleText := ' Reading Engene data from ' + infndat + ' file...';
      create1(nobjs,nvars,x2);
      createdestroy(true,nvars+numberofrowheaders,varlabels);
      createdestroy(true,nobjs, rowlabels);
      for i:=1 to nvars do varlabels[i] := 'Var' + inttostr(i);
      for i:=1 to nobjs do rowlabels[i] := 'Item' + inttostr(i);
      if (not ReadEngeneData(infndat,nobjs,nvars,numberofrowheaders, numberofvarheaders, x2, rowlabels, varlabels, mean, min, max, isneg)) then begin
           showmessage('Wrong input file or format not supported.');
           exit;
      end;
   end;

  if (transpose) then  // this seems a contradiction but necessary in microarray
      TransposeData(x2, nobjs, nvars, numberofrowheaders, numberofvarheaders, rowlabels, varlabels);

  if (normMethod > 0) then
    if (not normalizeData(x2, normMethod, isneg, min, max, mean)) then begin
      showmessage('Normalization error. Exiting');
      exit;
    end;

  if (not isneg) then negmethod := 0;
  // Sets the stuff needed in the graphic part
  if showgraphics then begin
      microform.n := nobjs;
      microform.p := nvars;
      microform.q := nq;
      microform.importancemodelstr := 'Importance of factor: ';
      microform.numberofrowheaders := numberofrowheaders;
  end;

  if ((isneg) and (negmethod = 1)) then begin
     nvars := nvars*2;
     microform.dimmatvec := nq*2+1;
     microform.dimfactvec := nq*2;
  end else if ((isneg) and (negmethod = 2)) then begin
     nobjs := nobjs*2;
     if showgraphics then begin
        microform.dimmatvec := nq*2+1;
        microform.dimfactvec := nq*2;
     end;
  end else begin
     if showgraphics then begin
        microform.dimmatvec := nq+1;
        microform.dimfactvec := nq;
     end;
  end;

  if (nq > Math.Min(nvars, nobjs)) then begin
      showmessage('Number of factors should be less than number of items or variables');
      exit;
  end;

  if showgraphics then begin
      createdestroy(true,microform.dimmatvec, microform.matvec);
      createdestroy(true,microform.dimfactvec, microform.factvec);
  end;
  
  Create1(nobjs,nq,ws);
  Create1(nq,nvars,sh);
  Create1(nobjs,nvars,x1);
  Create1(nobjs,nvars,vc);
  Create1(nq,y);
  Create1(nq,iy);
  Create1(nobjs,nvars,x);
  Create1(nobjs,nq,w);
  Create1(nq,nvars,h);

  // If data is negative, then convert it to positive
  if ((isneg) and (negmethod = 0)) then begin
      showmessage('Data is negative. Select one method to make it positive.');
      exit;
  end else if ((isneg) and (negmethod = 1)) then begin
    // Folding by columns
    myStatusBar^.SimpleText := ' Folding data by columns...';
    for i:=1 to nobjs do begin
       k := 1;
       for j:=1 to (nvars div 2) do begin
          if (x2.m[i,j] >= 0) then begin
             x.m[i,k] := x2.m[i,j];
             x.m[i,k+1] := 0.0;
          end else begin
             x.m[i,k+1] := abs(x2.m[i,j]);
             x.m[i,k] := 0.0;
          end;
          k := k+2;
       end;
    end;
  end else if ((isneg) and (negmethod = 2)) then begin
    // Folding by rows
    myStatusBar^.SimpleText := ' Folding data by rows...';
    k := 1;
    for i:=1 to (nobjs div 2) do begin
       for j:=1 to nvars do begin
          if (x2.m[i,j] >= 0) then begin
             x.m[k,j] := x2.m[i,j];
             x.m[k+1,j] := 0;
          end else begin
             x.m[k+1,j] := abs(x2.m[i,j]);
             x.m[k,j] := 0;
          end;
       end;
       k := k+2;
     end;
  end else if ((isneg) and (negmethod = 3)) then begin
    // Substract absolute minimum to the matrix
    myStatusBar^.SimpleText := ' Making data positive by substracting the minimum...';
    for i:=1 to nobjs do
       for j:=1 to nvars do
          x.m[i,j] := x2.m[i,j] - min;
  end else if ((isneg) and (negmethod = 4)) then begin
    // Exponential scaling
    myStatusBar^.SimpleText := ' Making data positive by exponential scaling...';
    if (abs(min) < abs(max)) then
         max := abs(max)
    else
         max := abs(min);
    for i:=1 to nobjs do
       for j:=1 to nvars do
          if (max <= 200) then
              x.m[i,j] := exp(x2.m[i,j])
          else
              x.m[i,j] := exp(x2.m[i,j]*(3/max)); // scale it for large numbers
  end else begin
    Equate1(x,x2); // x=x2
  end;

   // Goes to the graphic part
  if showgraphics then begin
      with microform.matvec[1] do begin
            evar := 100;
            name := 'Original data matrix ';
            create1(microform.n,AIndex);
            create1(microform.p,BIndex);
            for i:=1 to microform.n do
                Aindex.v[i] := i;
            for j:=1 to microform.p do
                Bindex.v[j] := j;
      end;
  end;

  if minimize then
      Application.Minimize;
  GobiNMF(InFndat,OutFn,Reusewh,Order,
       nvars,nobjs,nq,maxiter,sparsePower,eps1,sparse,nosparse,
       x,w,h,iter,tconv,functional, negmethod, showgraphics,saveRR,numberOfRuns, normMethod);

  Destroy1(ws);
  Destroy1(sh);
  Destroy1(x1);
  Destroy1(vc);
  Destroy1(y);
  Destroy1(iy);
  Destroy1(x);
  Destroy1(w);
  Destroy1(h);

  if showgraphics then begin
      myStatusBar^.SimpleText := ' Loading graphical tools...';
      microform.rowlabels := rowlabels;
      microform.varlabels := varlabels;
      microform.orgmat := x2.m;
      if not CancelOption then
        microform.showmodal;
  end;
  Destroy1(x2);

  createdestroy(false,nvars+numberofrowheaders,varlabels);
  createdestroy(false,nobjs, rowlabels);

  if showgraphics then begin
      for i:=1 to microform.dimmatvec do begin
        Destroy1(microform.matvec[i].AIndex);
        Destroy1(microform.matvec[i].BIndex);
      end;
      for i:=1 to microform.dimfactvec do begin
        Destroy1(microform.factvec[i].AIndex);
        Destroy1(microform.factvec[i].BIndex);
        Destroy1(microform.factvec[i].Avec);
        Destroy1(microform.factvec[i].Bvec);
      end;
      createdestroy(false,microform.dimmatvec, microform.matvec);
      createdestroy(false,microform.dimfactvec, microform.factvec);
  end;
  if minimize then
      Application.Restore;

end;

procedure Go01stdNMF(infndat,outfn:string;
               ReUseWH,Order:boolean;
               nvars,nobjs,nq,maxiter, sparsePower:integer;
               eps1,sparse,nosparse:extended;
               var x:matrix;
               var w:matrix;
               var h:matrix;
               var iter:integer;
               var tconv,functional:extended;
               negmethod: integer;
               showgraphics: boolean;
               saveRR:boolean;
               numberOfRuns: integer;
               normMethod: integer
               );
var Functional1:extended; num,den2,den1,r1,t,mean, maxEV:extended;
    a,b,c,i,j,k,kk,index,rr,maxRun:integer; convergence:boolean;
    maxw, maxh, expH, expW: matrix;
begin

 // Standard NMF uses nsNMF code with sparse constant set to 0
 // Run N number of times using random initialization (in case Re-use is not used)

 maxEV := -1;
 maxRun := 1;
 Create1(nq, nvars,maxh);
 Create1(nobjs, nq,maxw);
 for rr:=1 to numberOfRuns do begin

      myStatusBar^.SimpleText := ' Initializing...';
      Functional1:=GetFunctional(nvars,nobjs,x);
      GetWH(ReUseWH,true,nvars,nq,nobjs,OutFN,w,h,x);

      equate1(ws,w); // ws = w
      for kk := 1 to sparsePower do begin
        for i:=1 to nobjs do begin
          r1:=0;
          for j:=1 to nq do
            r1:=r1+ws.m[i,j];
          r1:=r1/nq;
          for j:=1 to nq do
            ws.m[i,j]:=nosparse*ws.m[i,j]+sparse*r1;
        end;
      end;

      equate1(sh,h); //sh = h
      for kk:=1 to sparsePower do begin
        for i:=1 to nvars do begin 
          r1:=0;
          for j:=1 to nq do
            r1:=r1+sh.m[j,i];
          r1:=r1/nq;
          for j:=1 to nq do
            sh.m[j,i]:=nosparse*sh.m[j,i]+sparse*r1;
        end;
      end;

      iter:=0;
      functional:=0;
      repeat
        iter:=iter+1;

        myStatusBar^.SimpleText := ' Running Iteration ' + inttostr(iter) + ' for run ' + inttostr(rr);
        if (iter mod 10) = 0 then begin
            application.ProcessMessages;
            if cancelOption then exit;
        end;    

        for b:=1 to nvars do
          for i:=1 to nobjs do begin
            den1:=0;
            for k:=1 to nq do
              den1:=den1+w.m[i,k]*sh.m[k,b];
            x1.m[i,b]:=den1;
            if den1 > 1e-16 then
              vc.m[i,b]:=x.m[i,b]/den1
            else
              vc.m[i,b]:=0;
          end;

        t:=Functional1;
        for i:=1 to nobjs do
          for j:=1 to nvars do
            if x1.m[i,j] > 1e-16 then
              t:=t-x.m[i,j]*ln(x1.m[i,j])+x1.m[i,j];

        for a:=1 to nq do
          for b:=1 to nvars do begin
            num:=0;
            for i:=1 to nobjs do
              num:=num+ws.m[i,a]*vc.m[i,b];
            h.m[a,b]:=h.m[a,b]*num;
          end;

        equate1(sh,h); //sh = h
        for kk:=1 to sparsePower do begin
          for i:=1 to nvars do begin
            r1:=0;
            for j:=1 to nq do
              r1:=r1+sh.m[j,i];
            r1:=r1/nq;
            for j:=1 to nq do
              sh.m[j,i]:=nosparse*sh.m[j,i]+sparse*r1;
          end;
        end;

        for a:=1 to nq do begin
          for c:=1 to nobjs do begin
            num:=0;
            for j:=1 to nvars do
              num:=num+sh.m[a,j]*vc.m[c,j];
            w.m[c,a]:=w.m[c,a]*num;
          end;
          den2:=0;
          for i:=1 to nobjs do
            den2:=den2+w.m[i,a];
          for i:=1 to nobjs do
            if den2 > 0 then
              w.m[i,a]:=w.m[i,a]/den2
            else
              w.m[i,a]:=0;
        end;

        equate1(ws,w); // ws = w
        for kk:=1 to sparsePower do begin
          for i:=1 to nobjs do begin
            r1:=0;
            for j:=1 to nq do
              r1:=r1+ws.m[i,j];
            r1:=r1/nq;
            for j:=1 to nq do
              ws.m[i,j]:=nosparse*ws.m[i,j]+sparse*r1;
          end;
        end;

        convergence:=(iter>=maxiter);
        if t > 1e-16 then
          tconv:=abs(1.0-functional/t)
        else
          tconv := 1;
        convergence:=convergence or (tconv<=eps1);
        functional:=t;
      until convergence;


      t:=0;
      den2:=0;
      for b:=1 to nvars do
        for i:=1 to nobjs do begin
          den1:=0;
          for k:=1 to nq do
            den1:=den1+w.m[i,k]*sh.m[k,b];
          t:=t+sqr(den1-x.m[i,b]);
          den2:=den2+sqr(x.m[i,b]);
        end;
      if den2 > 1e-16 then
        ev:=100-100*t/den2
      else
        ev := 0;

      if (maxEV < ev) then begin
         maxEV := ev;
         maxRun := rr;
         equate1(maxw,w); // maxw= w
         equate1(maxh,h); // maxh= h
      end;


      // Save random results
      if (numberOfRuns >1) then begin
          if (saveRR) then begin
              for i:=1 to nq do begin
                t:=0;
                for j:=1 to nvars do
                  t:=t+h.m[i,j];
                y.v[i]:=t;
                iy.v[i]:=i;
              end;
              if order then
                HEAPSORT1(nq,y,iy);

               WrtInfo1(OutFN+'-RandomRun'+IntToStr(rr),InFNdat,ReUseWH,nobjs,nvars,nq,
                       maxiter,iter,-1,-1,-1.0,eps1,tconv,functional,sparse, negmethod, 0, normMethod);
              WrtW1(OutFN+'-RandomRun'+IntToStr(rr),false,nq,nobjs,w);
              WrtW2(OutFN+'-RandomRun'+IntToStr(rr),nq,nobjs,w, negmethod);
              WrtH1(OutFN+'-RandomRun'+IntToStr(rr),true,nq,nvars,h);
              WrtH2(OutFN+'-RandomRun'+IntToStr(rr),nq,nvars,h,negmethod);
          end else begin // combine results
              if (rr = 1) then begin
                 Create1(nq*numberOfRuns,nvars,expH);
                 Create1(nobjs,nq*numberOfRuns,expW);
              end;

              for i:=1 to nq do
                for j:=1 to nvars do
                  expH.m[(rr-1)*nq+i,j]:=h.m[i,j];
              for i:=1 to nobjs do
                for j:=1 to nq do
                  expW.m[i,(rr-1)*nq+j]:=w.m[i,j];

          end; // else
      end;

  end; // for rr (number of runs)

  if ((numberOfRuns >1) and (not saveRR) ) then begin
      Destroy1(y);
      Destroy1(iy);
      Create1(nq*numberOfRuns,y);
      Create1(nq*numberOfRuns,iy);
      for i:=1 to nq*numberOfRuns do begin
        t:=0;
        for j:=1 to nvars do
          t:=t+expH.m[i,j];
        y.v[i]:=t;
        iy.v[i]:=i;
      end;
      if order then
        HEAPSORT1(nq*numberOfRuns,y,iy);
      WrtW1(OutFN+'-CombinedRun',false,nq*numberOfRuns,nobjs,expW);
      WrtW2(OutFN+'-CombinedRun',nq*numberOfRuns,nobjs,expW, negmethod);
      WrtH1(OutFN+'-CombinedRun',true,nq*numberOfRuns,nvars,expH);
      WrtH2(OutFN+'-CombinedRun',nq*numberOfRuns,nvars,expH,negmethod);
      Destroy1(expW);
      Destroy1(expH);
  end;

  myStatusBar^.SimpleText := ' Saving results...';

  equate1(w,maxw);
  equate1(h,maxh);
  ev := maxEV;
  Destroy1(maxw);
  Destroy1(maxh);

  for i:=1 to nq do begin
    t:=0;
    for j:=1 to nvars do
      t:=t+h.m[i,j];
    y.v[i]:=t;
    iy.v[i]:=i;
  end;
  if order then
    HEAPSORT1(nq,y,iy);

  WrtInfo1(OutFN,InFNdat,ReUseWH,nobjs,nvars,nq, maxiter,iter,numberOfRuns, maxRun, maxEV, eps1,tconv,functional,sparse, negmethod, 0, normMethod);
  WrtW1(OutFN,false,nq,nobjs,w);
  WrtW2(OutFN,nq,nobjs,w, negmethod);
  WrtH1(OutFN,true,nq,nvars,h);
  WrtH2(OutFN,nq,nvars,h,negmethod);

end;

function stNMF(infndat,outfn:string;
               ReUseWH,Order:boolean;
               nq,maxiter,sparsePower:integer;
               eps1,sparse,nosparse:extended;
               negmethod: integer;
               showgraphics: boolean;
               transpose: boolean;
               saveRR: boolean;
               numberOfRuns: integer;
               normMethod: integer;
               minimize: boolean
               ): boolean;
var
  i,j,k, nobjs, nvars: integer;
  isneg: boolean;
  min, max, mean: real;

begin
   stNMF := false;
   myStatusBar^.SimpleText := ' Detecting data format from ' + infndat + ' file...';
   i:=detectFormat(infndat);
   if (i = -1) then begin
      showmessage('The input file is not supported.');
      exit;
   end else if (i = 0) then begin // Pure data (no labels in rows and columns)
      myStatusBar^.SimpleText := ' Reading data dimension from ' + infndat + ' file...';
      if ((not getPureDataDimensions(infndat, nobjs, nvars, numberofrowheaders, numberofvarheaders))
           or (nvars<=0) or (nobjs<=0)) then begin
          showmessage('Wrong input file or format not supported.');
          exit;
      end;
      myStatusBar^.SimpleText := ' Reading pure data from ' + infndat + ' file...';
      create1(nobjs,nvars,x2);
      createdestroy(true,nvars+numberofrowheaders,varlabels);
      createdestroy(true,nobjs, rowlabels);
      for i:=1 to nvars do varlabels[i] := 'Var' + inttostr(i);
      for i:=1 to nobjs do rowlabels[i] := 'Item' + inttostr(i);
      if (not ReadPureData(infndat,nobjs,nvars,x2, mean, min, max, isneg)) then begin
          showmessage('Wrong input file or format not supported.');
          exit;
      end;
   end else if (i = 1) then begin // Data with column and row labels
      myStatusBar^.SimpleText := ' Reading data dimension from ' + infndat + ' file...';
      if ((not getLabeledDataDimensions(infndat, nobjs, nvars, numberofrowheaders, numberofvarheaders))
           or (nvars<=0) or (nobjs<=0)) then begin
          showmessage('Wrong input file or format not supported.');
          exit;
      end;
      myStatusBar^.SimpleText := ' Reading labeled data from ' + infndat + ' file...';
      create1(nobjs,nvars,x2);
      createdestroy(true,nvars+numberofrowheaders,varlabels);
      createdestroy(true,nobjs, rowlabels);
      for i:=1 to nvars do varlabels[i] := 'Var' + inttostr(i);
      for i:=1 to nobjs do rowlabels[i] := 'Item' + inttostr(i);
      if (not ReadLabeledData(infndat,nobjs,nvars, numberofrowheaders, numberofvarheaders, x2, rowlabels, varlabels, mean, min, max, isneg)) then begin
          showmessage('Wrong input file or format not supported.');
          exit;
      end;
   end else if (i = 2) then begin // Engene
      myStatusBar^.SimpleText := ' Reading data dimension from ' + infndat + ' file...';
      if ((not getEngeneDataDimensions(infndat, nobjs, nvars, numberofrowheaders, numberofvarheaders))
           or (nvars<=0) or (nobjs<=0)) then begin
           showmessage('Wrong input file or format not supported.');
           exit;
      end;
      myStatusBar^.SimpleText := ' Reading Engene data from ' + infndat + ' file...';
      create1(nobjs,nvars,x2);
      createdestroy(true,nvars+numberofrowheaders,varlabels);
      createdestroy(true,nobjs, rowlabels);
      for i:=1 to nvars do varlabels[i] := 'Var' + inttostr(i);
      for i:=1 to nobjs do rowlabels[i] := 'Item' + inttostr(i);
      if (not ReadEngeneData(infndat,nobjs,nvars,numberofrowheaders, numberofvarheaders, x2, rowlabels, varlabels, mean, min, max, isneg)) then begin
           showmessage('Wrong input file or format not supported.');
           exit;
      end;
   end;


  if (transpose) then  // this seems a contradiction but necessary in microarray
      TransposeData(x2, nobjs, nvars, numberofrowheaders, numberofvarheaders, rowlabels, varlabels);

  if (normMethod > 0) then
    if (not normalizeData(x2, normMethod, isneg, min, max, mean)) then begin
      showmessage('Normalization error. Exiting');
      exit;
    end;

  if (not isneg) then negmethod := 0;

  if ((isneg) and (negmethod = 1)) then begin
     nvars := nvars*2;
  end else if ((isneg) and (negmethod = 2)) then begin
     nobjs := nobjs*2;
  end;

  if (nq > Math.Min(nvars, nobjs)) then begin
      showmessage('Number of factors should be less than number of items or variables');
      exit;
  end;

  Create1(nobjs,nq,ws);
  Create1(nq,nvars,sh);
  Create1(nobjs,nvars,x1);
  Create1(nobjs,nvars,vc);
  Create1(nq,y);
  Create1(nq,iy);
  Create1(nobjs,nvars,x);
  Create1(nobjs,nq,w);
  Create1(nq,nvars,h);

  // If data is negative, then convert it to positive
  if ((isneg) and (negmethod = 0)) then begin
      showmessage('Data is negative. Select one method to make it positive.');
      exit;
  end else if ((isneg) and (negmethod = 1)) then begin
    // Folding by columns
    myStatusBar^.SimpleText := ' Folding data by columns...';
    for i:=1 to nobjs do begin
       k := 1;
       for j:=1 to (nvars div 2) do begin
          if (x2.m[i,j] >= 0) then begin
             x.m[i,k] := x2.m[i,j];
             x.m[i,k+1] := 0.0;
          end else begin
             x.m[i,k+1] := abs(x2.m[i,j]);
             x.m[i,k] := 0.0;
          end;
          k := k+2;
       end;
    end;
  end else if ((isneg) and (negmethod = 2)) then begin
    // Folding by rows
    myStatusBar^.SimpleText := ' Folding data by rows...';
    k := 1;
    for i:=1 to (nobjs div 2) do begin
       for j:=1 to nvars do begin
          if (x2.m[i,j] >= 0) then begin
             x.m[k,j] := x2.m[i,j];
             x.m[k+1,j] := 0;
          end else begin
             x.m[k+1,j] := abs(x2.m[i,j]);
             x.m[k,j] := 0;
          end;
       end;
       k := k+2;
     end;
  end else if ((isneg) and (negmethod = 3)) then begin
    // Substract absolute minimum to the matrix
    myStatusBar^.SimpleText := ' Making data positive by substracting the minimum...';
    for i:=1 to nobjs do
       for j:=1 to nvars do
          x.m[i,j] := x2.m[i,j] - min;
  end else if ((isneg) and (negmethod = 4)) then begin
    // Exponential scaling
    myStatusBar^.SimpleText := ' Making data positive by exponential scaling...';
    if (abs(min) < abs(max)) then
         max := abs(max)
    else
         max := abs(min);
    for i:=1 to nobjs do
       for j:=1 to nvars do
          if (max <= 200) then
              x.m[i,j] := exp(x2.m[i,j])
          else
              x.m[i,j] := exp(x2.m[i,j]*(3/max)); // scale it for large numbers
  end else begin
    Equate1(x,x2); // x=x2
  end;

  if minimize then
      Application.Minimize;

  Go01stdNMF(InFndat,OutFn,Reusewh,Order,
       nvars,nobjs,nq,maxiter,sparsePower,eps1,sparse,nosparse,
       x,w,h,iter,tconv,functional, negmethod, showgraphics,saveRR,numberOfRuns, normMethod);

  if minimize then
      Application.Restore;

  Destroy1(ws);
  Destroy1(sh);
  Destroy1(x1);
  Destroy1(vc);
  Destroy1(y);
  Destroy1(iy);
  Destroy1(x);
  Destroy1(w);
  Destroy1(h);
  Destroy1(x2);

  createdestroy(false,nvars+numberofrowheaders,varlabels);
  createdestroy(false,nobjs, rowlabels);

  if cancelOption then
    stNMF := false
  else
    stNMF := true;

end;


function maxRowH(var _h:matrix; _c: integer): integer;
var
  i, maxi: integer;
  v: single;
begin
    v := 0;
    maxi := 1;
    for i:=1 to _h.nr do begin
        if _h.m[i,_c] > v then begin
           v := _h.m[i,_c];
           maxi:=i;
        end;
    end;
   maxRowH := maxi;
end;


function traceback(var ht:integervector; b:integer; m: integer): single;
var
  a, c: single;
  i: integer;
begin
   i := 1;
   if (ht^[b-m].v[i] > m) then
      a := traceback(ht, trunc(ht^[b-m].v[i]), m)
   else
      a := ht^[b-m].v[i];

   i := 2;
   if (ht^[b-m].v[i] > m) then
      c := traceback(ht, trunc(ht^[b-m].v[i]), m)
   else
      c := ht^[b-m].v[i];
   if (a <= c) then
    traceback := a
   else
    traceback := c;
end;


procedure transz(var ht:integervector; m: integer);
var
  i, index: integer;
  a, b : single;
begin
  for i:=1 to m-1 do begin
     index := 1;
     if ht^[i].v[index] > m then
        ht^[i].v[index] := traceback(ht, trunc(ht^[i].v[index]), m);
     index := 2;
     if ht^[i].v[index] > m then
        ht^[i].v[index] := traceback(ht, trunc(ht^[i].v[index]), m);
     index := 1;
     if ht^[i].v[index] > ht^[i].v[index+1] then begin
         a:= ht^[i].v[index]; b:= ht^[i].v[index+1];
         ht^[i].v[index] := b; ht^[i].v[index+1]:=a;
     end;
  end;
end;

function cophenet(var yy: matrix; var z :integervector): double;
var
  i, j, k, index, n,m1,m,msav: integer;
  t: single;
  link, listhead: intvector;
  sum1, sum2, s11, s22, s12,u,c: double;
begin
   n := yy.nr;

   create1(n, link);
   create1(n*2, listhead);
   create1(trunc(n*(n-1)/2), y);

   for i:=1 to n do begin
      link.v[i] := 0;
      listhead.v[i] := i;
   end;

   sum1:=0; sum2:=0; s11:=0; s12:=0; s22:=0; index:=1; msav :=1;
   for k:=1 to n-1 do begin
      i := trunc(z^[k].v[index]);
      j := trunc(z^[k].v[index+1]);
      t := z^[k].v[index+2];
      m1 := listhead.v[i];
      while m1 > 0 do begin
        m := listhead.v[j];
        while m > 0 do begin
           u := yy.m[m,m1];
           sum1 := sum1+t; sum2:= sum2+u;
           s11:= s11+t*t; s22 := s22 + u*u;
           s12 := s12+t*u;
           msav := m;
           m := link.v[m];
        end;
        m1 := link.v[m1];
      end;
      link.v[msav] := listhead.v[i];
      listhead.v[n+k] := listhead.v[j];
   end;
   t:=2/(n*(n-1));
   s11:=s11-sum1*sum1*t;
   s22:=s22-sum2*sum2*t;
   s12:=s12-sum1*sum2*t;
   if (s11*s22 > 1e-16) then
      c:=s12/sqrt(s11*s22)
   else
      c:=0;
   destroy1(listhead);
   destroy1(link);
   destroy1(y);
   cophenet:=c;
end;

procedure clusternum(var Z :integervector;  var indexclusters: intvector; k, c: integer);
var
  i, m: integer;
  children: intvector;
begin
  m := indexclusters.n;
  create1(2, children);
  // get the children of node at this level
  for i:=1 to 2 do
    children.v[i] := trunc(z^[k].v[i]);
  // Assign this node number to leaf children
  for i:=1 to 2 do begin
    if (children.v[i] <= m) then
       indexclusters.v[children.v[i]] := c
    else
       // move to next level
       clusternum(Z, indexclusters, children.v[i]-m, c);
  end;
  destroy1(children);
end;


procedure cluster(var Z :integervector;  var indexclusters: intvector; c: integer);
var
  i, m, classnum,k, index: integer;
  node: intvector;
begin
  index := 1;
  m := indexclusters.n;
  create1(m, node);
  if m <= c then begin
      for i:=1 to m do
          indexclusters.v[i] := i;
  end else if c = 1 then begin
      for i:=1 to m do
          indexclusters.v[i] := 1;
  end else begin
      for i:=1 to m do
          indexclusters.v[i] := 0;
      classnum := 1;
      for k := (m-c+1) to (m-1) do begin
        i := trunc(z^[k].v[index]); // left tree
        if i <= m then begin
           indexclusters.v[i] := classnum;
           node.v[classnum] := i;
           classnum := classnum  + 1;
        end else if (i < (2*m-c+1)) then begin
           clusternum(Z, indexclusters, i-m, classnum);
           node.v[classnum] := i;
           classnum := classnum  + 1;
        end;
        i := trunc(z^[k].v[index+1]); // right tree
        if i <= m then begin
           indexclusters.v[i] := classnum;
           node.v[classnum] := i;
           classnum := classnum  + 1;
        end else if (i < (2*m-c+1)) then begin
           clusternum(Z, indexclusters, i-m, classnum);
           node.v[classnum] := i;
           classnum := classnum  + 1;
        end;
      end;
  end;
  destroy1(node);
end;

procedure linkage(var _m: matrix; var z :integervector; method: integer);
var
  m,i,j,c,n,s, mini, newi, newj: integer;
  y1,y2,Nv,R,p,II, JJ, aux,W: vector;
  minv: double;
begin

  m := _m.nr;
  create1(trunc((m*(m-1)/2)), y1);

  c:=1;
  for i := 1 to _m.nr do
     for j := i+1 to _m.nc do begin
        y1.v[c] := _m.m[i,j];
        c := c+1;
     end;

  create1(trunc((m*(m-1))), Nv);
  for i:=1 to Nv.n do Nv.v[i] := 0.0;
  for i:=1 to m do Nv.v[i] := 1.0;
  n:=m;
  create1((m*(m-1)){n}, R);
  for i:=1 to n do R.v[i] := i;


  for s:=1 to n-1 do begin
    if method = 1 then begin // average
      create1(m-2, p);
      for i:=1 to m-2 do
        p.v[i]  := (m-1) - (i-1);
      create1(trunc(m*(m-1)/2), II);
      for i:=1 to II.n do
          II.v[i] := 0;

      create1(p.n+1, aux);
      aux.v[1] := 1;
      for i:=1 to p.n do
        aux.v[i+1] := aux.v[i] + p.v[i];
      for i:=1 to aux.n do
        II.v[trunc(aux.v[i])] := 1;
      for i:=2 to II.n do
        II.v[i] := II.v[i] + II.v[i-1];

      create1(trunc(m*(m-1)/2), JJ);
      for i:=1 to JJ.n do
          JJ.v[i] := 1;
      for i:=2 to aux.n do
          JJ.v[trunc(aux.v[i])] := (2-p.v[i-1]);
      JJ.v[1] :=2;
      for i:=2 to JJ.n do
        JJ.v[i] := JJ.v[i] + JJ.v[i-1];

      create1(y1.n, W);
      for i:=1 to W.n do
         W.v[i] := (Nv.v[trunc(R.v[trunc(II.v[i])])]*Nv.v[trunc(R.v[trunc(JJ.v[i])])]);

      minv := 65000;
      mini := 0;
      for i:= 1 to y1.n do begin
         if (w.v[i] > 1e-16) then begin
            if minv > (y1.v[i]/w.v[i]) then begin
               minv := (y1.v[i]/w.v[i]);
               mini := i;
            end;
         end;
      end;

      destroy1(W);
      destroy1(aux);
      destroy1(JJ);
      destroy1(II);
      destroy1(p);
    end; // if method

    newi := floor(m+1/2-sqrt((m*m)-m+1/4-2*(mini-1)));
    newj := trunc(mini - (newi-1)*(m-newi/2)+newi);

    c :=1; Z^[s].v[c] := R.v[newi];
    c :=2; Z^[s].v[c] := R.v[newj];
    c :=3; Z^[s].v[c] := minv;

    create1(trunc((m*(m-1)/2)), II);
    create1(trunc((m*(m-1)/2)), JJ);

    c := 1;
    for i := 1 to newi-1 do begin
       II.v[c] := i*(m-(i+1)/2)-m+newi;
       JJ.v[c] := i*(m-(i+1)/2)-m+newj;
       c := c+1;
    end;
    for i := newi+1 to newj-1 do begin
       II.v[c] :=  newi*(m-(newi+1)/2)-m+i;
       JJ.v[c] := i*(m-(i+1)/2)-m+newj;
       c := c+1;
    end;
    for i := newj+1 to m do begin
       II.v[c] := newi*(m-(newi+1)/2)-m+i;
       JJ.v[c] := newj*(m-(newj+1)/2)-m+i;
       c := c+1;
    end;

    if method = 1 then begin // average
        for i := 1 to c-1 do begin
          Y1.v[trunc(II.v[i])] := Y1.v[trunc(II.v[i])] + Y1.v[trunc(JJ.v[i])];
        end;
    end; // if method

   // Reduce Y1
   if (y1.n-c) > 0 then begin
       create1(y1.n-c, y2);
       for i:=1 to c-1 do
          Y1.v[trunc(JJ.v[i])] := -200;
       Y1.v[trunc(newi*(m-(newi+1)/2)-m+newj)] := -200;
       c:= 1;
       for i:=1 to y1.n do begin
         if y1.v[i] <> -200 then begin
            y2.v[c] := y1.v[i];
            c:=c+1;
         end;
       end;
       destroy1(y1);
       create1(y2.n, y1);
       equate1(y1,y2); // y1 = y2
       destroy1(y2);
   end;

   // update m, N, R
   m:= m-1;
   Nv.v[n+s] := Nv.v[trunc(R.v[newi])] + Nv.v[trunc(R.v[newj])];
   R.v[newi] := n+s;

   for i:= newj to n-1 do
     R.v[i] := R.v[i+1];

   destroy1(JJ);
   destroy1(II);

  end; // for s

  //sort Z
  if method = 1 then begin// average
      c:=1;
      for i:=1 to _m.nr-1 do
         if Z^[i].v[c] > Z^[i].v[c+1] then begin
            minv:= Z^[i].v[c+1];
            Z^[i].v[c+1] := Z^[i].v[c];
            Z^[i].v[c] := minv;
         end;
  end;
  destroy1(R);
  destroy1(y1);
  destroy1(Nv);

end;

procedure averagelinkage(var _m: matrix; var _sortedindex: intvector; var _indexclusters: intvector; var _coph: double; _nk: integer);
var
 i,j,k,s,index, g:integer;
 ht, WW :integervector;
 nsw, rsw, R: intvector;
 X: vector;

begin

   // Call average linkage
   createdestroy(true,_m.nr-1,ht);
   for i:= 1 to _m.nr-1 do
      create1(3, ht^[i]);
   linkage(_m, ht, 1);

    // calculate cophenetic coeficient
   _coph := cophenet(_m,ht);
   cluster(ht, _indexclusters, _nk);

   // Creates index vector (permutation vector)
    transz(ht, _m.nr);

    createdestroy(true,_m.nr-1,WW);
    create1(_m.nr, nsw);
    create1(_m.nr, rsw);
    create1(_m.nr, R);
    create1(_m.nr, X);

    for i:=1 to _m.nr-1 do begin
      create1(3, WW^[i]);
      for j:=1 to 3 do begin
          WW^[i].v[j] := 0.0;
      end;
    end;

    for i:=1 to _m.nr do begin
        nsw.v[i] := 0;
        rsw.v[i] := 0;
    end;

    for j:=1 to 3 do begin
          WW^[1].v[j] := ht^[1].v[j];
    end;
    rsw.v[1] := 1;
    index := 1;
    nsw.v[trunc(ht^[1].v[index])] := 1;
    nsw.v[trunc(ht^[1].v[index+1])] := 1;
    k := 2; s :=2;

    while (k< _m.nr) do begin
       i := s;
       while ((rsw.v[i] = 1) or not ((nsw.v[trunc(ht^[i].v[index])] = 1) or (nsw.v[trunc(ht^[i].v[index+1])] = 1))) do begin
          if ((rsw.v[i] = 1) and (i = s)) then s := s + 1;
          i := i + 1;
       end;
      for j:=1 to 3 do begin
          WW^[k].v[j] := ht^[i].v[j];
      end;
      nsw.v[trunc(ht^[i].v[index])] := 1;
      nsw.v[trunc(ht^[i].v[index+1])] := 1;
      rsw.v[i] := 1;
      if (s = i) then s := s + 1;
      k := k + 1;
    end;

    g := 1;
    for i:=1 to _m.nr do begin
        X.v[i] := i;
        R.v[i] := 0;
    end;

    for k:=1 to _m.nr-1 do begin
      i := trunc(WW^[k].v[index]);
      if (R.v[i] = 0) then begin
        X.v[i] := g;
        g := g+1;
        R.v[i] := 1;
      end;
      i := trunc(WW^[k].v[index+1]);
      if (R.v[i] = 0) then begin
        X.v[i] := g;
        g := g+1;
        R.v[i] := 1;
      end;
    end;

    for i:=1 to _m.nr do begin
        R.v[i] := i;
    end;

    HEAPSORT1(_m.nr,X, R);

    for i:=1 to _m.nr do begin
        _sortedindex.v[i] := R.v[_m.nr-i + 1];
    end;

    destroy1(nsw);
    destroy1(rsw);
    destroy1(R);
    destroy1(X);
    for i:=1 to _m.nr-1 do
      destroy1(WW^[i]);

   createdestroy(false,_m.nr-1,WW);
   createdestroy(false,_m.nr-1,ht);

end;


procedure calculateConsensus(infndat,outfn:string; minNF, maxNF, maxiter, sparsePower:integer;
               nvars,nobjs:integer;
               sparse,nosparse:extended;
               var iter,MsPerIter:integer;
               var x:matrix;
               showgraphics: boolean;
               saveRR: boolean;
               numberOfRuns: integer;
               negmethod: integer;
               normMethod: integer
               );
var Functional1:extended; num,den2,den1,r1,t,mean, maxEV:extended; tt0,tt1:longword;
    a,b,c,i,j,k,kk,index,rr,nk,maxRun, count, maxi:integer; convergence,testb:boolean;
    maxw, maxh, expH, expW: matrix;
    w,h, new_cc, old_cc, ave_cc, distm:matrix;
    fn:string; txt1:textfile;
    maxv: double;

begin

 myStatusBar^.SimpleText := ' Initializing...';
 Create1(nvars, nvars,new_cc);
 Create1(nvars, nvars,old_cc);
 Create1(nvars, nvars,ave_cc);
 Create1(nvars, nvars,distm);


  if showgraphics then begin
      consensusRec.name := '';
      consensusRec.NumbOfFactors := (maxNF-minNF)+1;
      createdestroy(true,(maxNF-minNF)+1,consensusRec.ConsMatvector);
  end;

 for nk := minNF to maxNF do begin  // iterates all factors
    myStatusBar^.SimpleText := ' Calculating for ' + inttostr(nk) + ' factors ...';
    Create1(nk, nvars,h);
    Create1(nobjs, nk,w);

    Create1(nk, 1,sh);
    Create1(1, nk,ws);

    for i:=1 to nvars do
      for j:=1 to nvars do
            ave_cc.m[i,j] := 0;

    for rr:=1 to numberOfRuns do begin

        myStatusBar^.SimpleText := ' Running random iteration ' + inttostr(rr) + ' for factor ' + inttostr(nk);
        application.ProcessMessages;
        if CancelOption then exit;


        count := 0;
        for i:=1 to nvars do
          for j:=1 to nvars do
            old_cc.m[i,j] := 0;

        // Random init of W and H
        GetWH2(nvars,nk,nobjs,w,h);

        // Calculate W and H using the divergence function
        iter:=0; convergence := false;
        repeat
          iter:=iter+1;

          // CALCULATE H
          for a:=1 to nk do begin
            ws.m[1,a] := 0;
            for i:=1 to nobjs do
                ws.m[1,a]:= ws.m[1,a] + w.m[i,a];
          end;

          for b:=1 to nvars do
            for i:=1 to nobjs do begin
              den1:=0;
              for k:=1 to nk do
                den1:=den1+w.m[i,k]*h.m[k,b];
              if den1 > 1e-16 then
                vc.m[i,b]:=x.m[i,b]/den1
              else
                vc.m[i,b]:=0;
            end;

          maxv := 0;
          for a:=1 to nk do
            for b:=1 to nvars do begin
              num:=0;
              for i:=1 to nobjs do
                num:=num+w.m[i,a]*vc.m[i,b];
              if (ws.m[1,a] > 1e-16) then
                  h.m[a,b]:=h.m[a,b]*num/ws.m[1,a]
              else
                  h.m[a,b] := 0;
              if maxv < h.m[a,b] then maxv := h.m[a,b];
            end;

            if maxv < 1e-16 then
              for a:=1 to nk do
                  for b:=1 to nvars do
                    h.m[a,b] := 1e-16;

          // CALCULATE W
          for a:=1 to nk do begin
            sh.m[a,1] := 0;
            for i:=1 to nvars do
                sh.m[a,1]:= sh.m[a,1] + h.m[a,i];
          end;

          for b:=1 to nvars do
            for i:=1 to nobjs do begin
              den1:=0;
              for k:=1 to nk do
                den1:=den1+w.m[i,k]*h.m[k,b];
              if den1 > 1e-16 then
                vc.m[i,b]:=x.m[i,b]/den1
              else
                vc.m[i,b]:=0;
            end;

          maxv := 0;
          for c:=1 to nobjs do begin
            for a:=1 to nk do begin
              num:=0;
              for j:=1 to nvars do
                num:=num+h.m[a,j]*vc.m[c,j];
              if (sh.m[a,1] > 0) then
                  w.m[c,a]:=w.m[c,a]*num/sh.m[a,1]
              else
                  w.m[c,a]:=0;
              if maxv < w.m[c,a] then maxv := w.m[c,a];
            end;
          end;

            if maxv < 1e-16 then
              for a:=1 to nk do
                  for c:=1 to nobjs do
                    w.m[c,a] := 1e-16;

          // Test convergence every 10 iterations
          if (iter mod 10 = 0) then begin

             if (iter mod 50 = 0) then begin
                application.ProcessMessages;
                if CancelOption then exit;
             end;

            // CALCULATE CONNECTIVITY MATRIX

              for i:=1 to nvars do begin
                  maxi := maxRowH(h,i);
                  for j:=i to nvars do begin
                      if (i = j) then
                          new_cc.m[i,j] := 1
                      else
                        if maxi = maxRowH(h,j) then
                          new_cc.m[i,j] := 1
                        else
                          new_cc.m[i,j] := 0;
                  end;
              end;

              // CHECK IF IT IS IDENTICAL TO THE PREVIOUS ONE, IF SO, FINISH
              testb := true;
              for i:=1 to nvars do begin
                  for j:=i to nvars do begin
                      if (new_cc.m[i,j] <> old_cc.m[i,j]) then
                          testb := false;
                          break;
                  end;
                  if not testb then break;
              end;

              for i:=1 to nvars do
                  for j:=i to nvars do
                      old_cc.m[i,j] := new_cc.m[i,j];

              if testb then count := count +1 else count := 0;

              convergence:=(iter>=maxiter);
              convergence:=convergence or (count>=MsPerIter);

          end; // iter mod 10
      until convergence;

      for i:=1 to nvars do
          for j:=i to nvars do
              ave_cc.m[i,j] := ave_cc.m[i,j]+new_cc.m[i,j];

    end; // for numberOfRuns

    destroy1(h);
    destroy1(w);
    destroy1(sh);
    destroy1(ws);

    // avarage consensus matrices for this nk:

    for i:=1 to nvars do
      for j:=i to nvars do
          ave_cc.m[i,j] := ave_cc.m[i,j]/numberOfRuns;

      // expand full matrix
      for i:=1 to nvars do
        for j:=i to nvars do
          ave_cc.m[j,i] := ave_cc.m[i,j];

     // Creates distance matrix from consensus matrix
      for i:=1 to nvars do
        for j:=1 to nvars do
          distm.m[i,j] := 1.0-ave_cc.m[i,j];

    // Calculates average linkage

    if showgraphics then begin
      consensusRec.ConsMatvector[nk-minNF+1].K := nk;
      Create1(nvars, consensusRec.ConsMatvector[nk-minNF+1].index);
      Create1(nvars, consensusRec.ConsMatvector[nk-minNF+1].indexcluster);
      for i := 1 to nvars do
         consensusRec.ConsMatvector[nk-minNF+1].index.v[i] := i;
      // Calculate average linkage, cophenetic coeficient and clusters
      averagelinkage(distm, consensusRec.ConsMatvector[nk-minNF+1].index, consensusRec.ConsMatvector[nk-minNF+1].indexcluster, consensusRec.ConsMatvector[nk-minNF+1].coph, nk);

      // Save cluster information
      if outfn <> '' then begin
          fn:=outfn + '-'+inttostr(nk) + '-cluster.txt';
          assignfile(txt1,fn);
          filemode:=1;
          rewrite(txt1);
          writeln(txt1,'Variable    Cluster');
          for i:=1 to nvars do begin
              writeln(txt1, trim(varlabels^[numberofrowheaders+i]), '    ', consensusRec.ConsMatvector[nk-minNF+1].indexcluster.v[i]);
          end;
          closefile(txt1);
      end;

      Create1(nvars, nvars,consensusRec.ConsMatvector[nk-minNF+1].mat);
      equate1(consensusRec.ConsMatvector[nk-minNF+1].mat, ave_cc);
    end;

 end; // for nk (minNF-maxNF)

 if showgraphics then begin
      WrtInfo2(OutFN,InFNdat,nobjs,nvars,minNF, maxNF,numberOfRuns, sparse, negmethod, normMethod,consensusRec);
      myStatusBar^.SimpleText := ' Loading graphical tools...';
      consensusForm.varlabels := @varlabels;
      consensusForm.numberofrowheaders := numberofrowheaders;
      consensusForm.showmodal;
      for i :=1 to (maxNF-minNF)+1 do
        Destroy1(consensusRec.ConsMatvector[i].mat);
      for i := 1 to (maxNF-minNF)+1 do begin
        Destroy1(consensusRec.ConsMatvector[i].index);
        Destroy1(consensusRec.ConsMatvector[i].indexcluster);
      end;
      createdestroy(false,(maxNF-minNF)+1,consensusRec.ConsMatvector);
  end;

end;


// Sample classification tab

procedure classNMF(infndat,outfn:string;
               ReUseWH,Order:boolean;
               minNF, maxNF,maxiter,sparsePower:integer;
               sparse,nosparse:extended;
               negmethod: integer;
               showgraphics: boolean;
               transpose: boolean;
               saveRR: boolean;
               numberOfRuns: integer;
               normMethod: integer;
               minimize: boolean
               );
var
  i,j,nobjs, nvars: integer;
  isneg: boolean;
  min, max, mean: real;
  MsPerIter: integer;
begin

   myStatusBar^.SimpleText := ' Detecting data format from ' + infndat + ' file...';
   i:=detectFormat(infndat);
   if (i = -1) then begin
      showmessage('The input file is not supported.');
      exit;
   end else if (i = 0) then begin // Pure data (no labels in rows and columns)
      myStatusBar^.SimpleText := ' Reading data dimension from ' + infndat + ' file...';
      if ((not getPureDataDimensions(infndat, nobjs, nvars, numberofrowheaders, numberofvarheaders))
           or (nvars<=0) or (nobjs<=0)) then begin
          showmessage('Wrong input file or format not supported.');
          exit;
      end;
      myStatusBar^.SimpleText := ' Reading pure data from ' + infndat + ' file...';
      create1(nobjs,nvars,x2);
      createdestroy(true,nvars+numberofrowheaders,varlabels);
      createdestroy(true,nobjs, rowlabels);
      for i:=1 to nvars do varlabels[i] := 'Var' + inttostr(i);
      for i:=1 to nobjs do rowlabels[i] := 'Item' + inttostr(i);
      if (not ReadPureData(infndat,nobjs,nvars,x2, mean, min, max, isneg)) then begin
          showmessage('Wrong input file or format not supported.');
          exit;
      end;
   end else if (i = 1) then begin // Data with column and row labels
      myStatusBar^.SimpleText := ' Reading data dimension from ' + infndat + ' file...';
      if ((not getLabeledDataDimensions(infndat, nobjs, nvars, numberofrowheaders, numberofvarheaders))
           or (nvars<=0) or (nobjs<=0)) then begin
          showmessage('Wrong input file or format not supported.');
          exit;
      end;
      myStatusBar^.SimpleText := ' Reading labeled data from ' + infndat + ' file...';
      create1(nobjs,nvars,x2);
      createdestroy(true,nvars+numberofrowheaders,varlabels);
      createdestroy(true,nobjs, rowlabels);
      for i:=1 to nvars do varlabels[i] := 'Var' + inttostr(i);
      for i:=1 to nobjs do rowlabels[i] := 'Item' + inttostr(i);
      if (not ReadLabeledData(infndat,nobjs,nvars, numberofrowheaders, numberofvarheaders, x2, rowlabels, varlabels, mean, min, max, isneg)) then begin
          showmessage('Wrong input file or format not supported.');
          exit;
      end;
   end else if (i = 2) then begin // Engene
      myStatusBar^.SimpleText := ' Reading data dimension from ' + infndat + ' file...';
      if ((not getEngeneDataDimensions(infndat, nobjs, nvars, numberofrowheaders, numberofvarheaders))
           or (nvars<=0) or (nobjs<=0)) then begin
           showmessage('Wrong input file or format not supported.');
           exit;
      end;
      myStatusBar^.SimpleText := ' Reading Engene data from ' + infndat + ' file...';
      create1(nobjs,nvars,x2);
      createdestroy(true,nvars+numberofrowheaders,varlabels);
      createdestroy(true,nobjs, rowlabels);
      for i:=1 to nvars do varlabels[i] := 'Var' + inttostr(i);
      for i:=1 to nobjs do rowlabels[i] := 'Item' + inttostr(i);
      if (not ReadEngeneData(infndat,nobjs,nvars,numberofrowheaders, numberofvarheaders, x2, rowlabels, varlabels, mean, min, max, isneg)) then begin
           showmessage('Wrong input file or format not supported.');
           exit;
      end;
   end;

  if (maxNF > Math.Min(nvars, nobjs)) then begin
      showmessage('Maximum number of factors should be less than number of items or variables');
      exit;
  end;

  if (transpose) then  // this seems a contradiction but necessary in microarray
      TransposeData(x2, nobjs, nvars, numberofrowheaders, numberofvarheaders, rowlabels, varlabels);

  if (normMethod > 0) then
    if (not normalizeData(x2, normMethod, isneg, min, max, mean)) then begin
      showmessage('Normalization error. Exiting');
      exit;
    end;

  if (not isneg) then negmethod := 0;

  Create1(nobjs,nvars,x1);
  Create1(nobjs,nvars,vc);
  Create1(nobjs,nvars,x);

  // If data is negative, then convert it to positive
  if ((isneg) and (negmethod = 0)) then begin
      showmessage('Data is negative. Select one method to make it positive.');
      exit;
  end else if ((isneg) and (negmethod = 3)) then begin
    // Substract absolute minimum to the matrix
    myStatusBar^.SimpleText := ' Making data positive by substracting the minimum...';
    for i:=1 to nobjs do
       for j:=1 to nvars do
          x.m[i,j] := x2.m[i,j] - min;
  end else if ((isneg) and (negmethod = 4)) then begin
    // Exponential scaling
    myStatusBar^.SimpleText := ' Making data positive by exponential scaling...';
    if (abs(min) < abs(max)) then
         max := abs(max)
    else
         max := abs(min);
    for i:=1 to nobjs do
       for j:=1 to nvars do
          if (max <= 200) then
              x.m[i,j] := exp(x2.m[i,j])
          else
              x.m[i,j] := exp(x2.m[i,j]*(3/max)); // scale it for large numbers

  end else begin
    Equate1(x,x2); // x=x2
  end;

  MsPerIter:=40;
  if minimize then
      Application.Minimize;
  calculateConsensus(infndat,outfn, minNF, maxNF, maxiter, sparsePower,
               nvars,nobjs,
               sparse,nosparse, iter, MsPerIter,
               x, showgraphics, saveRR, numberOfRuns, negmethod, normMethod);
  if minimize then
      Application.Restore;

  Destroy1(x1);
  Destroy1(vc);
  Destroy1(x);
  Destroy1(x2);

  createdestroy(false,nvars+numberofrowheaders,varlabels);
  createdestroy(false,nobjs, rowlabels);

end;


end.

