{***************************************************************************
 *
 * Author:     Alberto Pascual Montano (pascual@fis.ucm.es)
 *             http://www.dacya.ucm.es/apascual

 * Complutense University of Madrid (UCM). Madrid, Spain
 * The KEY Institute for Brain-Mind Research, Zurich, Switzerland
 * National Center for Biotechnology (CNB). Madrid, Spain
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 *  All comments concerning this program package may be sent to the
 *  e-mail address 'pascual@fis.ucm.es'
 ***************************************************************************}

unit uMatToolsDyn;

interface

uses sysutils;

(* GENERAL NOTE:
You must turn range checking off "{$R-}" if you want to access the elements
of vectors and matrices explicitly in your program!
*)


var CheckingOn:boolean=false;
    Ok_NoError:boolean;
{Set CheckingOn:=true to force checking of correct vector/matrix dimensions.
Then you will have to manually check the value of Ok_NoError by yourself, since
the program will not stop by error! Error checking is left up to you if you set
CheckingOn:=true!!}


type real=double; // change here if you like {single double extended}
     TInternalVector=array[1..1] of real;
     pvector=^TInternalVector;
     vector=record
       n:integer;
       v:pvector;
     end;
     TInternalMatrix=array[1..1] of pvector;
     pmatrix=^TInternalMatrix;
     matrix=record
       nr,nc:integer;
       m:pmatrix;
     end;
     TInternalIntVector=array[1..1] of integer;
     pintvector=^TInternalIntVector;
     intvector=record
       n:integer;
       v:pintvector;
     end;


{ It's your absolute responsibility to create and destroy variables, using
these procedures}
procedure Create1(n:integer; var v:vector); overload;
procedure Create1(n:integer; var v:intvector); overload;
procedure Create1(nr,nc:integer; var m:matrix); overload;
procedure Destroy1(var v:vector); overload;
procedure Destroy1(var v:intvector); overload;
procedure Destroy1(var m:matrix); overload;


{ GENERAL NOTES on accessing elements of vectors/matrices.
1. If v is a pvector, then v[i] is of type real. E.g., if a and b reals,
then the following expressions are valid: a:=b+v[5] and v[10]:=v[8]-a.
There's no need for writing v^[5] under delphi 7.
2. If a and b are pmatrices, then a[i,j] and b[i,j] are of type real;
no need for writing a^[i]^[j]]^ etc. a[1,5]-b[6,3] is valid.
3. if a is a pmatrix, then a^[i] is of type pvector. Here we do need the "^" symbol!
}


{NOTE: Never equate two vectors or two matrices. This must be done element
by element or using the "Equate" procedures.}
procedure Equate1(a,b:matrix); overload; // a=b
procedure Equate1(a,b:vector); overload; // a=b
procedure Equate1(a,b:intvector); overload; // a=b
{-----------------------------------------------}


{NOTE: Never use fillchar on vectors or matrices. This must be done element
by element or using the "Fill" procedures.}
procedure FillChar1(var a:matrix; val:real); overload; // a=val
procedure FillChar1(var a:vector; val:real); overload; // a=val
procedure FillChar1(var a:intvector; val:integer); overload; // a=val
{-----------------------------------------------}


function VxV(n:integer; var a,b:pvector):extended; overload; // plain scalar product

function VxV(var a,b:vector):extended; overload; // plain scalar product

procedure MxV(var a:vector; var b:matrix; var c:vector); // a=bc; creates a if a.n=0

procedure VxM(var r,v:vector; var m:matrix); // r=v*m; creates r if r.n=0

procedure MxM(var a,b,c:matrix); // a=bc; creates a if a.nr=0

procedure MxMt(var a,m:matrix); overload; // a=m*m'; creates a if a.nr=0

procedure MxMt(var a,b,c:matrix); overload; // a=bc'; creates a if a.nr=0

procedure MtxM(var a,m:matrix); overload; // a=m'*m; creates a if a.nr=0

procedure MtxM(var a,b,c:matrix); overload; // a=b'c; creates a if a.nr=0

PROCEDURE indexx1(VAR arr:vector; // no change
                  VAR indx:intvector // ascending
                  );
{Outputs indices pointing to ordered vector. Creates indx if indx.n=0}

PROCEDURE indexx2(VAR arr:vector; // no change
                  VAR indx:intvector // descending
                  );
{Outputs indices pointing to ordered vector. Creates indx if indx.n=0}

procedure TRANSP(VAR Mt,M:MATRIX); // mt is m transposed; creates mt if mt.nr=0

procedure RowNorms(var ns:vector; var x:matrix);
{ns will contain norms of rows of x. Creates ns if ns.n=0}

procedure ColumnNorms(var ns:vector; var x:matrix);
{ns will contain norms of columns of x. Creates ns if ns.n=0}

procedure RowMeans(var ns:vector; var x:matrix);
{ns will contain means of rows of x. Creates ns if ns.n=0}

procedure SubtractRowMeans(var ns:vector; var x:matrix); overload;
{Rows of x will have zero mean. ns will contain means of rows of x.
Creates ns if ns.n=0}

procedure SubtractRowMeans(var x:matrix); overload;
{Rows of x will have zero mean.}

procedure SubtractColumnMeans(var x:matrix);
{Columns of x will have zero mean.}

function ReadTxt1(nr,nc:integer; fn:string; var x:matrix):boolean; overload;
{Creates and reads in text matrix. Last column can be any string, but will be
ignored. Footer allowed and ignored.}

function ReadTxt1(fn:string; var x:matrix):boolean; overload;
{Reads in text matrix (previously created). Last column can be any string,
but will be ignored. Footer allowed and ignored.}

function WriteTxt1(fn:string; var x:matrix):boolean; overload;
{Writes text matrix.}

function WriteTxt1(fn:string; var x:vector):boolean; overload;
{Writes text vector.}

function WriteTxtT1(fn:string; var x:matrix):boolean;
{Writes transposed text matrix.}

function ReadTxtT1(nr,nc:integer; fn:string; var xt:matrix):boolean;
{Creates and reads in text matrix transposed, i.e., in file nr*nc, but placed
into xt nc*nr. Last column can be any string, but will be ignored. Footer
allowed and ignored.}

implementation

{$R-}

var eps1:extended=0.0;
    eps2:extended=0.0;
    precis:extended absolute eps1;
    eta:extended absolute eps2;




procedure Create1(n:integer; var v:vector); overload;
begin
  if n > 0 then begin
    v.n:=n;
    getmem(v.v,n*sizeof(real));
  end;
end;

procedure Destroy1(var v:vector); overload;
begin
  if v.n > 0 then begin
    freemem(v.v,v.n*sizeof(real));
    v.n:=0;
  end;
end;

procedure Create1(n:integer; var v:intvector); overload;
begin
  v.n:=n;
  getmem(v.v,n*sizeof(integer));
end;

procedure Destroy1(var v:intvector); overload;
begin
  freemem(v.v,v.n*sizeof(integer));
  v.n:=0;
end;

procedure Create1(nr,nc:integer; var m:matrix); overload;
var i:integer;
begin
  m.nr:=nr;
  m.nc:=nc;
  getmem(m.m,nr*sizeof(pointer));
  for i:=1 to nr do
    getmem(m.m[i],nc*sizeof(real))
end;

procedure Destroy1(var m:matrix); overload;
var i:integer;
begin
  for i:=m.nr downto 1 do
    freemem(m.m[i],m.nc*sizeof(real));
  freemem(m.m,m.nr*sizeof(pointer));
  m.nr:=0;
  m.nc:=0;
end;




procedure Check1(var a,b:matrix); overload;
begin
  Ok_NoError:=(a.nr=b.nr) and (a.nc=b.nc);
end;

procedure Check1(var a,b:vector); overload;
begin
  Ok_NoError:=(a.n=b.n)
end;

procedure Check1(var a,b:intvector); overload;
begin
  Ok_NoError:=(a.n=b.n)
end;

procedure Check1(var a:vector; var b:matrix; var c:vector); overload;
begin
  Ok_NoError:=(b.nc=c.n);
  if not Ok_NoError then exit;
  if a.n>0 then
    Ok_NoError:=(a.n=b.nr);
end;

procedure Check1(var r,v:vector; var m:matrix); overload;
begin
  Ok_NoError:=(m.nr=v.n);
  if not Ok_NoError then exit;
  if r.n>0 then
    Ok_NoError:=(r.n=m.nc);
end;

procedure Check1(var a,b,c:matrix); overload;
begin
  Ok_NoError:=(b.nc=c.nr);
  if not Ok_NoError then exit;
  if a.nr>0 then
    Ok_NoError:=(a.nr=b.nr) and (a.nc=c.nc);
end;




procedure Equate1(a,b:matrix); overload; // a=b
var i,j:integer;
begin
  if CheckingOn then begin
    Check1(a,b);
    if not Ok_NoError then exit;
  end;
  for i:=1 to a.nr do
    for j:=1 to a.nc do
      a.m[i,j]:=b.m[i,j];
end;

procedure Equate1(a,b:vector); overload; // a=b
var i:integer;
begin
  if CheckingOn then begin
    Check1(a,b);
    if not Ok_NoError then exit;
  end;
  for i:=1 to a.n do
    a.v[i]:=b.v[i];
end;

procedure Equate1(a,b:intvector); overload; // a=b
var i:integer;
begin
  if CheckingOn then begin
    Check1(a,b);
    if not Ok_NoError then exit;
  end;
  for i:=1 to a.n do
    a.v[i]:=b.v[i];
end;




procedure FillChar1(var a:matrix; val:real); overload; // a=val
var i,j:integer;
begin
  for i:=1 to a.nr do
    for j:=1 to a.nc do
      a.m[i,j]:=val;
end;

procedure FillChar1(var a:vector; val:real); overload; // a=val
var i:integer;
begin
  for i:=1 to a.n do
    a.v[i]:=val;
end;

procedure FillChar1(var a:intvector; val:integer); overload; // a=val
var i:integer;
begin
  for i:=1 to a.n do
    a.v[i]:=val;
end;




function VxV(n:integer; var a,b:pvector):extended; overload;
var t:extended; i:integer;
begin
  t:=0.0;
  for i:=1 to n do
    t:=t+a[i]*b[i];
  VxV:=t
end;




function VxV(var a,b:vector):extended; overload;
var t:extended; i:integer;
begin
  VxV:=0;
  if CheckingOn then begin
    Check1(a,b);
    if not Ok_NoError then exit;
  end;
  t:=0.0;
  for i:=1 to a.n do
    t:=t+a.v[i]*b.v[i];
  VxV:=t;
end;




procedure MxV(var a:vector; var b:matrix; var c:vector); // a=bc; creates a if a.n=0
var i:integer;
begin
  if CheckingOn then begin
    Check1(a,b,c);
    if not Ok_NoError then exit;
  end;
  if a.n=0 then
    Create1(b.nr,a);
  for i:=1 to a.n do
    a.v[i]:=VxV(b.nc,b.m^[i],c.v);
end;




procedure VxM(var r,v:vector; var m:matrix); // r=v*m; creates r if r.n=0
VAR I,J:INTEGER; T:extended;
BEGIN { VECxMAT }
  if CheckingOn then begin
    Check1(r,v,m);
    if not Ok_NoError then exit;
  end;
  if r.n=0 then
    Create1(m.nc,r);
  FOR J:=1 TO m.nc DO BEGIN
    T:=0.0;
    FOR I:=1 TO m.nr DO
      T:=T+V.v[I]*M.m[I,J];
    R.v[J]:=T
  END
END; { VECxMAT }




procedure MxM(var a,b,c:matrix); // a=bc; creates a if a.nr=0
var i,j,k:integer; t:extended;
begin
  if CheckingOn then begin
    Check1(a,b,c);
    if not Ok_NoError then exit;
  end;
  if a.nr=0 then
    Create1(b.nr,c.nc,a);
  for i:=1 to a.nr do
    for j:=1 to a.nc do begin
      t:=0;
      for k:=1 to b.nc do
        t:=t+b.m[i,k]*c.m[k,j];
      a.m[i,j]:=t
    end;
end;




procedure MxMt(var a,m:matrix); overload; // a=m*m'; creates a if a.nr=0
var i,j:integer;
begin

  if CheckingOn then
    if a.nr>0 then begin
      Ok_NoError:=(a.nr=m.nr) and ((a.nc=m.nr));
      if not Ok_NoError then exit;
    end;
  if a.nr=0 then
    Create1(m.nr,m.nr,a);

  for i:=1 to a.nr do
    for j:=i to a.nc do
      a.m[i,j]:=vxv(m.nc,m.m^[i],m.m^[j]);
  for i:=1 to a.nr do
    for j:=i+1 to a.nc do
      a.m[j,i]:=a.m[i,j];
end;




procedure MxMt(var a,b,c:matrix); overload; // a=bc'; creates a if a.nr=0
var i,j:integer;
begin
  if CheckingOn then begin
    Ok_NoError:=(b.nc=c.nc);
    if not Ok_NoError then exit;
    if a.nr>0 then begin
      Ok_NoError:=(a.nr=b.nr) and ((a.nc=c.nr));
      if not Ok_NoError then exit;
    end;
  end;
  if a.nr=0 then
    Create1(b.nr,c.nr,a);

  for i:=1 to a.nr do
    for j:=1 to a.nc do
      a.m[i,j]:=vxv(b.nc,b.m^[i],c.m^[j]);
end;




procedure MtxM(var a,m:matrix); overload; // a=m'*m; creates a if a.nr=0
var i,j,k:integer; t:extended;
begin
  if CheckingOn then
    if a.nr>0 then begin
      Ok_NoError:=(a.nr=m.nc) and ((a.nc=m.nc));
      if not Ok_NoError then exit;
    end;
  if a.nr=0 then
    Create1(m.nc,m.nc,a);

  for i:=1 to a.nr do
    for j:=i to a.nc do begin
      t:=0;
      for k:=1 to m.nr do
        t:=t+m.m[k,i]*m.m[k,j];
      a.m[i,j]:=t
    end;
  for i:=1 to a.nr do
    for j:=i+1 to a.nc do
      a.m[j,i]:=a.m[i,j];
end;




procedure MtxM(var a,b,c:matrix); overload; // a=b'c; creates a if a.nr=0
var i,j,k:integer; t:extended;
begin
  if CheckingOn then begin
    Ok_NoError:=(b.nr=c.nr);
    if not Ok_NoError then exit;
    if a.nr>0 then begin
      Ok_NoError:=(a.nr=b.nc) and ((a.nc=c.nc));
      if not Ok_NoError then exit;
    end;
  end;
  if a.nr=0 then
    Create1(b.nc,c.nc,a);

  for i:=1 to a.nr do
    for j:=1 to a.nc do begin
      t:=0;
      for k:=1 to b.nr do
        t:=t+b.m[k,i]*c.m[k,j];
      a.m[i,j]:=t
    end;
end;


PROCEDURE HEAPSORT(VAR A:vector); {descending}

VAR L,R:INTEGER; X:real;

  PROCEDURE SIFT;
  VAR I,J:INTEGER; P:BOOLEAN;
  BEGIN { SIFT }
    P:=TRUE;
    I:=L;
    J:=2*I;
    X:=A.v[I];
    WHILE (J<=R) AND P DO BEGIN
     IF J<R THEN
       IF A.v[J]>A.v[J+1] THEN // > descending
         J:=J+1;
     IF X>A.v[J] THEN BEGIN // > descending
       A.v[I]:=A.v[J];
       I:=J;
       J:=I*2
     END ELSE
       P:=FALSE
    END;
    A.v[I]:=X
  END; { SIFT }

BEGIN { HEAPSORT }
  L:=(a.n DIV 2)+1;
  R:=a.n;
  WHILE L>1 DO BEGIN
    L:=L-1;
    SIFT
  END;
  WHILE R>1 DO BEGIN
    X:=A.v[1];
    A.v[1]:=A.v[R];
    A.v[R]:=X;
    R:=R-1;
    SIFT
  END
END; { HEAPSORT }




PROCEDURE indexx1(VAR arr:vector; // no change
                  VAR indx:intvector // ascending
                  );
{Outputs indices pointing to ordered vector. Creates indx if indx.n=0}
LABEL 99;
VAR
   l,j,ir,indxt,i,n:integer; q: real;
BEGIN
  if indx.n=0 then
    create1(arr.n,indx);
  n:=arr.n;

  if CheckingOn then
    if (arr.n<>indx.n) then begin
      Ok_NoError:=false;
      exit;
    end;

   FOR j := 1 TO n DO
      indx.v[j] := j;
   IF n = 1 THEN GOTO 99;
   l := (n DIV 2) + 1;
   ir := n;
   WHILE true DO BEGIN
      IF l > 1 THEN BEGIN
         l := l-1;
         indxt := indx.v[l];
         q := arr.v[indxt]
      END
      ELSE BEGIN
         indxt := indx.v[ir];
         q := arr.v[indxt];
         indx.v[ir] := indx.v[1];
         ir := ir-1;
         IF ir = 1 THEN BEGIN
            indx.v[1] := indxt;
            GOTO 99
         END
      END;
      i := l;
      j := l+l;
      WHILE j <= ir DO BEGIN
         IF j < ir THEN
            IF arr.v[indx.v[j]] < arr.v[indx.v[j+1]] THEN j := j+1; // < ascending
         IF q < arr.v[indx.v[j]] THEN BEGIN // < ascending
            indx.v[i] := indx.v[j];
            i := j;
            j := j+j
         END
         ELSE
         j := ir+1
      END;
      indx.v[i] := indxt
   END;
99:
END;




PROCEDURE indexx2(VAR arr:vector; // no change
                  VAR indx:intvector // descending
                  );
{Outputs indices pointing to ordered vector. Creates indx if indx.n=0}
LABEL 99;
VAR
   l,j,ir,indxt,i,n: integer; q: real;
BEGIN
  if indx.n=0 then
    create1(arr.n,indx);
  n:=arr.n;

  if CheckingOn then
    if (arr.n<>indx.n) then begin
      Ok_NoError:=false;
      exit;
    end;

   FOR j := 1 TO n DO
      indx.v[j] := j;
   IF n = 1 THEN GOTO 99;
   l := (n DIV 2) + 1;
   ir := n;
   WHILE true DO BEGIN
      IF l > 1 THEN BEGIN
         l := l-1;
         indxt := indx.v[l];
         q := arr.v[indxt]
      END
      ELSE BEGIN
         indxt := indx.v[ir];
         q := arr.v[indxt];
         indx.v[ir] := indx.v[1];
         ir := ir-1;
         IF ir = 1 THEN BEGIN
            indx.v[1] := indxt;
            GOTO 99
         END
      END;
      i := l;
      j := l+l;
      WHILE j <= ir DO BEGIN
         IF j < ir THEN
            IF arr.v[indx.v[j]] > arr.v[indx.v[j+1]] THEN j := j+1; // > descending
         IF q > arr.v[indx.v[j]] THEN BEGIN // > descending
            indx.v[i] := indx.v[j];
            i := j;
            j := j+j
         END
         ELSE
         j := ir+1
      END;
      indx.v[i] := indxt
   END;
99:
END;


procedure TRANSP(VAR Mt,M:MATRIX); // mt is m transposed; creates mt if mt.nr=0
VAR I,J:integer;
BEGIN { TRANSP }
  if mt.nr=0 then
    create1(m.nc,m.nr,mt);
  if CheckingOn then
    if (mt.nr<>m.nc) or (mt.nc<>m.nr) then begin
      Ok_NoError:=false;
      exit;
    end;
  FOR I:=1 TO m.nr DO
    FOR J:=1 TO m.nc DO
      mt.m[J,I]:=M.m[I,J]
END; { TRANSP }



procedure RowNorms(var ns:vector; var x:matrix);
{ns will contain norms of rows of x. Creates ns if ns.n=0}
var i:integer;
begin
  if CheckingOn then
    if ns.n>0 then begin
      Ok_NoError:=(x.nr=ns.n);
      if not Ok_NoError then exit;
    end;
  if ns.n=0 then
    Create1(x.nr,ns);
  for i:=1 to ns.n do
    ns.v[i]:=sqrt(vxv(x.nc,x.m^[i],x.m^[i]));
end;

procedure ColumnNorms(var ns:vector; var x:matrix);
{ns will contain norms of columns of x. Creates ns if ns.n=0}
var i,j:integer; r1:extended;
begin
  if CheckingOn then
    if ns.n>0 then begin
      Ok_NoError:=(x.nc=ns.n);
      if not Ok_NoError then exit;
    end;
  if ns.n=0 then
    Create1(x.nc,ns);
  for i:=1 to ns.n do begin
    r1:=0;
    for j:=1 to x.nr do
      r1:=r1+sqr(x.m[j,i]);
    ns.v[i]:=sqrt(r1);
  end;
end;
procedure RowMeans(var ns:vector; var x:matrix);
{ns will contain means of rows of x. Creates ns if ns.n=0}
var i,j:integer; r1:extended;
begin
  if CheckingOn then
    if ns.n>0 then begin
      Ok_NoError:=(x.nr=ns.n);
      if not Ok_NoError then exit;
    end;
  if ns.n=0 then
    Create1(x.nr,ns);
  for i:=1 to x.nr do begin
    r1:=0;
    for j:=1 to x.nc do
      r1:=r1+x.m[i,j];
    ns.v[i]:=r1/x.nc;
  end;
end;

procedure SubtractRowMeans(var ns:vector; var x:matrix); overload;
{Rows of x will have zero mean. ns will contain means of rows of x.
Creates ns if ns.n=0}
var i,j:integer; r1:extended;
begin
  if CheckingOn then
    if ns.n>0 then begin
      Ok_NoError:=(x.nr=ns.n);
      if not Ok_NoError then exit;
    end;
  if ns.n=0 then
    Create1(x.nr,ns);
  for i:=1 to x.nr do begin
    r1:=0;
    for j:=1 to x.nc do
      r1:=r1+x.m[i,j];
    r1:=r1/x.nc;
    ns.v[i]:=r1;
    for j:=1 to x.nc do
      x.m[i,j]:=x.m[i,j]-r1;
  end;
end;

procedure SubtractRowMeans(var x:matrix); overload;
{Rows of x will have zero mean.}
var i,j:integer; r1:extended;
begin
  for i:=1 to x.nr do begin
    r1:=0;
    for j:=1 to x.nc do
      r1:=r1+x.m[i,j];
    r1:=r1/x.nc;
    for j:=1 to x.nc do
      x.m[i,j]:=x.m[i,j]-r1;
  end;
end;

procedure SubtractColumnMeans(var x:matrix);
{Columns of x will have zero mean.}
var i,j:integer; r1:extended;
begin
  for j:=1 to x.nc do begin
    r1:=0;
    for i:=1 to x.nr do
      r1:=r1+x.m[i,j];
    r1:=r1/x.nr;
    for i:=1 to x.nr do
      x.m[i,j]:=x.m[i,j]-r1;
  end;
end;


function ReadTxt1(nr,nc:integer; fn:string; var x:matrix):boolean; overload;
{Creates and reads in text matrix. Last column can be any string, but will be
ignored. Footer allowed and ignored.}
var txt:textfile; i,j:integer; s:string;
begin
  {$I+}
  Result:=true;
  create1(nr,nc,x);
  assignfile(txt,fn);
  filemode:=0;
  try
    reset(txt);
    try
      for i:=1 to x.nr do begin
        for j:=1 to x.nc do
          read(txt,x.m[i,j]);
        readln(txt,s);
      end;
    finally
      closeFile(txt);
    end;
  except
    on EInOutError do
      Result:=false;
    on EInvalidOp do
      Result:=false;
  end;
  if not Result then
    destroy1(x);
end;




function ReadTxt1(fn:string; var x:matrix):boolean; overload;
{Reads in text matrix (previously created). Last column can be any string,
but will be ignored. Footer allowed and ignored.}
var txt:textfile; i,j:integer; s:string;
begin
  {$I+}
  Result:=true;
  assignfile(txt,fn);
  filemode:=0;
  try
    reset(txt);
    try
      for i:=1 to x.nr do begin
        for j:=1 to x.nc do
          read(txt,x.m[i,j]);
        readln(txt,s);
      end;
    finally
      closeFile(txt);
    end;
  except
    on EInOutError do
      Result:=false;
    on EInvalidOp do
      Result:=false;
  end;
end;




function ReadTxtT1(nr,nc:integer; fn:string; var xt:matrix):boolean;
{Creates and reads in text matrix transposed, i.e., in file nr*nc, but placed
into xt nc*nr. Last column can be any string, but will be ignored. Footer
allowed and ignored.}
var txt:textfile; i,j:integer; s:string;
begin
  {$I+}
  Result:=true;
  create1(nc,nr,xt);
  assignfile(txt,fn);
  filemode:=0;
  try
    reset(txt);
    try
      for i:=1 to nr do begin
        for j:=1 to nc do
          read(txt,xt.m[j,i]);
        readln(txt,s);
      end;
    finally
      closeFile(txt);
    end;
  except
    on EInOutError do
      Result:=false;
    on EInvalidOp do
      Result:=false;
  end;
  if not Result then
    destroy1(xt);
end;




function WriteTxt1(fn:string; var x:matrix):boolean; overload;
{Writes text matrix.}
var txt:textfile; i,j:integer;
begin
  {$I+}
  Result:=true;
  assignfile(txt,fn);
  filemode:=1;
  try
    rewrite(txt);
    try
      for i:=1 to x.nr do begin
        for j:=1 to x.nc do
          write(txt,' ',x.m[i,j]:15);
        writeln(txt);
      end;
    finally
      closeFile(txt);
    end;
  except
    on EInOutError do
      Result:=false;
    on EInvalidOp do
      Result:=false;
  end;
end;




function WriteTxt1(fn:string; var x:vector):boolean; overload;
{Writes text vector.}
var txt:textfile; i:integer;
begin
  {$I+}
  Result:=true;
  assignfile(txt,fn);
  filemode:=1;
  try
    rewrite(txt);
    try
      for i:=1 to x.n do
        writeln(txt,' ',x.v[i]:15);
    finally
      closeFile(txt);
    end;
  except
    on EInOutError do
      Result:=false;
    on EInvalidOp do
      Result:=false;
  end;
end;




function WriteTxtT1(fn:string; var x:matrix):boolean;
{Writes transposed text matrix.}
var txt:textfile; i,j:integer;
begin
  {$I+}
  Result:=true;
  assignfile(txt,fn);
  filemode:=1;
  try
    rewrite(txt);
    try
      for j:=1 to x.nc do begin
        for i:=1 to x.nr do
          write(txt,' ',x.m[i,j]:15);
        writeln(txt);
      end;
    finally
      closeFile(txt);
    end;
  except
    on EInOutError do
      Result:=false;
    on EInvalidOp do
      Result:=false;
  end;
end;

end.

