/**++
 *   
 *   LICENSE
 *   -------
 *   
 *   Copyright (c) 2004 Renato Mancuso
 *   All rights reserved.
 *   
 *   Redistribution and use in source and binary forms, with or without modification, are 
 *   permitted provided that the following conditions are met:
 *   
 *   - Redistributions of source code must retain the above copyright notice, this list 
 *     of conditions and the following disclaimer.
 *   
 *   - Redistributions in binary form must reproduce the above copyright notice, this list
 *     of conditions and the following disclaimer in the documentation and/or other materials 
 *     provided with the distribution.
 *   
 *   - Neither the name of Renato Mancuso nor the names of its contributors may be used to 
 *     endorse or promote products derived from this software without specific prior written 
 *     permission.
 *   
 *   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS AS IS AND ANY EXPRESS 
 *   OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY 
 *   AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR 
 *   CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL 
 *   DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, 
 *   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER 
 *   IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT 
 *   OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *   
--**/


/*****************************************************************************
 * 
 *  A very simple flat file checker
 * 
 ****************************************************************************/

#include <cstdio>
#include <string>
#include <Phoenix.hpp>

using namespace OpenEMBL::Phoenix;

#define NO_ACCESSION_NUMBER "-"

class PARSER 
    : public FileParserHost
{
public:
    PARSER(char const * theFilename)
        : FileParserHost( theFilename )
        , _accession(NO_ACCESSION_NUMBER)
    {
    }

    ~PARSER() throw()
    {
    }

    METHOD onBeginEntry(long theLineNumber)
    {
        FileParserHost::onBeginEntry(theLineNumber);
        _accession = NO_ACCESSION_NUMBER;
    }

    METHOD onAC(long theLineNumber, const ACLine & theField)
    {
        if (theField.AccessionNumberList.size() > 0)
            _accession = theField.AccessionNumberList[0];
    }

    METHOD onEndEntry()
    {
        FileParserHost::onEndEntry();
        _accession = NO_ACCESSION_NUMBER;
    }

    void handleError(
        long          theLineNumber,
        int           theErrorCode,
        char const*   theDescription,
        char const*   theMessage)
    {
        if (NULL == theMessage)
            fprintf(
                stderr,
                "%s(%ld)[%s]: error P%05d: %s.\n",
                getSourceDisplayName(),
                theLineNumber,
                _accession.c_str(),
                theErrorCode,
                theDescription);
        else
            fprintf(
                stderr,
                "%s(%ld)[%s]: error P%05d: %s: %s.\n",
                getSourceDisplayName(),
                theLineNumber,
                _accession.c_str(),
                theErrorCode,
                theDescription,
                theMessage);
    }

    void handleWarning(
        long            theLineNumber,
        int             theWarningNumber,
        int             theWarningLevel,
        char const*    theDescription,
        char const*    theMessage)
    {
        if (NULL == theMessage)
            fprintf(
                stderr,
                "%s(%ld)[%s]: warning W%05d: %s.\n",
                getSourceDisplayName(),
                theLineNumber,
                _accession.c_str(),
                theWarningNumber,
                theDescription);
        else
            fprintf(
                stderr,
                "%s(%ld)[%s]: warning W%05d: %s: %s.\n",
                getSourceDisplayName(),
                theLineNumber,
                _accession.c_str(),
                theWarningNumber,
                theDescription,
                theMessage);
    }

private:
    std::string _accession;
};

int main(int argc, char* argv[])
{
    using namespace OpenEMBL;

    if (argc < 2)
        return 1;

    PARSER theParser(argv[1]);
    theParser.parse();
    return 0;
}

