/**++
 *   
 *   LICENSE
 *   -------
 *   
 *   Copyright (c) 2004 Renato Mancuso
 *   All rights reserved.
 *   
 *   Redistribution and use in source and binary forms, with or without modification, are 
 *   permitted provided that the following conditions are met:
 *   
 *   - Redistributions of source code must retain the above copyright notice, this list 
 *     of conditions and the following disclaimer.
 *   
 *   - Redistributions in binary form must reproduce the above copyright notice, this list
 *     of conditions and the following disclaimer in the documentation and/or other materials 
 *     provided with the distribution.
 *   
 *   - Neither the name of Renato Mancuso nor the names of its contributors may be used to 
 *     endorse or promote products derived from this software without specific prior written 
 *     permission.
 *   
 *   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS AS IS AND ANY EXPRESS 
 *   OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY 
 *   AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR 
 *   CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL 
 *   DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, 
 *   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER 
 *   IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT 
 *   OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *   
--**/

//=============================================================================
//
//  DESCRIPTION
//
//  This is a formatting utility that removes some of the artifacts
//  produced by text editors when saving EMBL flat files.
//
//  As described in section $3.4 of the EMBL User Manual:
//
//      Each line begins with a two-character line type code. 
//      This code is *always* followed by three blanks, so that the actual information 
//      in each line begins in character position 6.
//
//  Therefore for each line in the input file the following actions 
//  are performed:
//
//      a) trailing blanks are removed
//      b) the required 5 character gutter is recreated if the line
//         length is less than 5 characters
//
//=============================================================================

#include <iostream>
#include <fstream>
#include <string>
#include <vector>
#include <set>

#include <stdlib.h>
#include <string.h>
#include <stdio.h>

#if defined(_WIN32) || defined(__WIN32__)
#   define OPTION_STR  "/"
#   define OPTION_CHAR '/'
#   define SHOW_HELP_OPTION "/?"
#else
#   define OPTION_STR  "-"
#   define OPTION_CHAR '-'
#   define SHOW_HELP_OPTION "-help"
#endif

using namespace std;

namespace {

    struct AbortRun {};

    char const * theBanner   = 
        "EFMT ver 1.0 - written by Renato Mancuso\n"
        "               build: " __DATE__ ", " __TIME__ "\n"
        "\n"
        ;

    void printBanner()
    {
        cout << theBanner;
    }

    void printUsage()
    {
        printBanner();
        cout << "Description:\n";
        cout << "       prepares an EMBL flat file for processing by stripping trailing blanks\n";
        cout << "       from each line and creating a 5 character gutter if necessary.\n";
        cout << "\n";
        cout << "Usage: efmt [" OPTION_STR "nologo] [" SHOW_HELP_OPTION "] [input] [output]\n";
        cout << "\n";
        cout << "       where:\n";
        cout << "         input       name of input file  (stdin if no name given)\n";
        cout << "         output      name of output file (stdout if no name given)\n";
        cout << "\n";
        cout << "       options\n";
        cout << "         " SHOW_HELP_OPTION "          displays this help screen\n";
        cout << "         " OPTION_STR "nologo     suppress display of banner\n";
        cout << "\n";
    }

    void processLine(string & theLine, ostream & theOutput)
    {
        //
        // trim trailing blanks
        //
        theLine.erase(theLine.find_last_not_of(" \t") + 1);

        //
        // copy string to output making sure we have a
        // gutter of at least 5 characters
        //
        theOutput << theLine;

        if (theLine.length() < 5)
        {
            size_t theBlankCount = 5 - theLine.length();

            for (size_t i = 0; i < theBlankCount; ++i)
                theOutput.put(' ');
        }

        theOutput.put('\n');

        if (!theOutput.good())
        {
            cerr << "ERROR: failed to write to output file.\n";
            throw AbortRun();
        }
    }

}

int main(int argc, char* argv[])
{
    try
    {
        if (argc == 2 && !strcmp(argv[1], SHOW_HELP_OPTION))
        {
            printUsage();
            return EXIT_SUCCESS;
        }    
        
        vector<string>  theArguments;
        set<string>     theOptions;

        for (int i = 1; i < argc; ++i)
        {
            char*   theArgument     = argv[i];
            size_t  theArgLength    = strlen(theArgument);

            if (0 == theArgLength)
                continue;

            if (theArgument[0] == OPTION_CHAR)
            {
                if (theArgLength > 1)
                    theOptions.insert(theArgument + 1);
            }
            else
            {
                theArguments.push_back(theArgument);
            }
        }

        if (theOptions.find("nologo") == theOptions.end())      
            printBanner();

        istream* theInput   = NULL;
        ostream* theOutput  = NULL;

        if (theArguments.size() > 0)
            theInput = new ifstream(theArguments[0].c_str());
        else
            theInput = &cin;

        if (!theInput->good())
        {
            cerr << "ERROR: failed to open input file.\n";
            return EXIT_FAILURE;
        }

        if (theArguments.size() > 1)
        {
            if (theArguments[1] == theArguments[0])
            {            
                cerr << "ERROR: output filename must be different from input filename.\n";
                return EXIT_FAILURE;
            }

            theOutput = new ofstream(theArguments[1].c_str());
        }
        else
            theOutput = &cout;
        
        if (!theOutput->good())
        {
            cerr << "ERROR: failed to open output file.\n";
            return EXIT_FAILURE;
        }

        string theLine;

        while (getline(*theInput, theLine))
        {
            processLine(theLine, *theOutput);    
        }

        theOutput->flush();

        bool fSuccess = true;

        if (!theInput->eof())
        {
            cerr << "ERROR: failed to read from input file.\n";
            fSuccess = false;
        }

        if (!theOutput->good())
        {
            cerr << "ERROR: error writing output file.\n";
            fSuccess = false;   
        }

        if (theOutput != &cout)
            delete theOutput;

        if (theInput != &cin)
            delete theInput;

        return fSuccess ? EXIT_SUCCESS : EXIT_FAILURE;
    }
    catch (AbortRun &)
    {
    }

    return EXIT_FAILURE;
}

