/**++
 *   
 *   LICENSE
 *   -------
 *   
 *   Copyright (c) 2004 Renato Mancuso
 *   All rights reserved.
 *   
 *   Redistribution and use in source and binary forms, with or without modification, are 
 *   permitted provided that the following conditions are met:
 *   
 *   - Redistributions of source code must retain the above copyright notice, this list 
 *     of conditions and the following disclaimer.
 *   
 *   - Redistributions in binary form must reproduce the above copyright notice, this list
 *     of conditions and the following disclaimer in the documentation and/or other materials 
 *     provided with the distribution.
 *   
 *   - Neither the name of Renato Mancuso nor the names of its contributors may be used to 
 *     endorse or promote products derived from this software without specific prior written 
 *     permission.
 *   
 *   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS AS IS AND ANY EXPRESS 
 *   OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY 
 *   AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR 
 *   CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL 
 *   DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, 
 *   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER 
 *   IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT 
 *   OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *   
--**/


#ifndef FF2XML_XML_CONVERTER_INCLUDED
#define FF2XML_XML_CONVERTER_INCLUDED

#if !defined(OPEN_EMBL_NO_PRAGMA_ONCE)
#pragma once
#endif

#include <cstdio>

#include <Phoenix.hpp>
#include "EntryHeader.hpp"
#include "XmlWriter.hpp"

namespace OpenEMBL
{
namespace ff2xml
{

    class XmlConverter
        : public Phoenix::FileParserHost
    {
    
        typedef Phoenix::FileParserHost base;

        EntryHeader                             _entry;
        PublicationInfo                         _publication;
        Phoenix::FeatureKey                     _feature;
        std::vector<Phoenix::FeatureQualifier>  _qualifiers;
        TaxonomyInfo                            _taxonomyInfo;       

        XmlWriter           _xml;
        long                _processedCount;
        std::string         _destFilename;

        static const XmlAttributeList           EMPTY_ATTRIBUTE_LIST;
        static const std::vector<std::string>   EMPTY_EDITOR_LIST;
        static const std::vector<std::string>   EMPTY_LOCATOR;
        static const std::vector<std::string>   EMPTY_PATENT_APPLICANT_LIST;

    public:
        XmlConverter(const char * srcFilename, const char* destFilename);
        ~XmlConverter() throw();

        long processedCount() const throw() { return _processedCount; }

        METHOD parse();

    private:
        static std::string  makeString(std::vector<std::string> const & theLines);
        static std::string  makeDate(std::string const & str);
        static int          getMonth(std::string const & month);

        bool        suppressQualifierDisplay(std::string const & theQualifierName);
        std::string getOrganismNameFromQualifiers();
        std::string getTaxIdFromQualifiers();

        void getTaxonomyInfoForOrganism
            (
            std::string const &     theOrganism,
            std::string &           theCommonName,
            Phoenix::StringList &   theLineage
            );

        void getScientificAndCommonName
            (
            std::string const & str,
            std::string & theScientificName,
            std::string & theCommonName 
            );
        
        void getDatabaseAndIdentifier
            (
            std::string const & str,
            std::string &       theDatabase,
            std::string &       theIdentifier 
            );

        void removeQuotes(Phoenix::StringList & theLines);

        void renderCitation
            (
            XmlWriter::layout_type              layout,
            XmlAttributeList                    attributes,
            std::vector<std::string> const &    eds         = EMPTY_EDITOR_LIST,
            std::vector<std::string> const &    locator     = EMPTY_LOCATOR,
            std::vector<std::string> const &    patentApplicants = EMPTY_PATENT_APPLICANT_LIST
            );

        void renderPublicationData
            (
            std::vector<std::string> const & eds,
            std::vector<std::string> const & locator,
            std::vector<std::string> const & patentApplicants = EMPTY_PATENT_APPLICANT_LIST
            );

        void renderOrganismData();
        void renderQualifiers();
        void renderDbXrefsQualifiers();
        void renderLocation(Phoenix::FeatureLocation const & location);
        void renderLocationElement(Phoenix::LocationElement const & theLocationElement);
        void renderBasePosition(long x1, long x2, unsigned flags);


    public:
        // -----------------------------------------------------------
        // PARSER EVENTS
        // -----------------------------------------------------------

        METHOD onError(long theLineNumber, int theErrorCode, LPCSTR theMessage);

        // Entry
        METHOD onBeginEntry(long theLineNumber);
        METHOD onEndEntry();

        // Individual Fields
        METHOD onID(long theLineNumber, Phoenix::IDLine const & theField);
        METHOD onAC(long theLineNumber, Phoenix::ACLine const & theField);
        METHOD onSV(long theLineNumber, Phoenix::SVLine const & theField);
        METHOD onDT(long theLineNumber, Phoenix::DTLine const & theField);
        METHOD onDE(long theLineNumber, Phoenix::DELine const & theField);
        METHOD onKW(long theLineNumber, Phoenix::KWLine const & theField);
        METHOD onDR(long theLineNumber, Phoenix::DRLine const & theField);
        METHOD onCC(long theLineNumber, Phoenix::CCLine const & theField);
        METHOD onSQ(long theLineNumber, Phoenix::SQLine const & theField);

        // Taxonomy lines
        METHOD onBeginTaxonomyLines(long theLineNumber);
        METHOD onOS(long theLineNumber, Phoenix::OSLine const & theField);    
        METHOD onOC(long theLineNumber, Phoenix::OCLine const & theField);
        METHOD onOG(long theLineNumber, Phoenix::OGLine const & theField);        
        METHOD onEndTaxonomyLines();

        // Feature Table
        METHOD onBeginFeatureTable(long theLineNumber);
        METHOD onBeginFeatureKey(long theLineNumber, Phoenix::FeatureKey const & theField);
        METHOD onFeatureQualifier(long theLineNumber, Phoenix::FeatureQualifier const  & theQualifier);   
        METHOD onEndFeatureKey();
        METHOD onEndFeatureTable();

        // Sequence Data
        METHOD onBeginSequenceData(long theLineNumber);
        METHOD onSequenceLine(long theLineNumber, Phoenix::SequenceLine const & theField);
        METHOD onEndSequenceData();

        // TPA lines
        METHOD onBeginTPALines(long theLineNumber);
        METHOD onAS(long theLineNumber, Phoenix::ASLine const & theField);
        METHOD onEndTPALines();

        // CO lines
        METHOD onBeginCOLines(long theLineNumber, bool isComplement);
        METHOD onCOSegmentInfo (long theLineNumber, Phoenix::COSegmentInfo const & theSegmentInfo);
        METHOD onCOGapInfo (long theLineNumber, Phoenix::COGapInfo const & theGapInfo);
        METHOD onEndCOLines();

        // Publications
        METHOD onBeginPublication(long theLineNumber);

        METHOD onRN(long theLineNumber, Phoenix::RNLine const & theField); 
        METHOD onRC(long theLineNumber, Phoenix::RCLine const & theField);
        METHOD onRP(long theLineNumber, Phoenix::RPLine const & theField);
        METHOD onRX(long theLineNumber, Phoenix::RXLine const & theField);
        METHOD onRG(long theLineNumber, Phoenix::RGLine const & theField);
        METHOD onRA(long theLineNumber, Phoenix::RALine const & theField);
        METHOD onRT(long theLineNumber, Phoenix::RTLine const & theField);

        //>> RL line types
        METHOD onSubmission        (long theLineNumber, Phoenix::SubmissionInfo         const & theField);
        METHOD onUnpublished       (long theLineNumber, Phoenix::UnpublishedInfo        const & theField);
        METHOD onBook              (long theLineNumber, Phoenix::BookInfo               const & theField);
        METHOD onThesis            (long theLineNumber, Phoenix::ThesisInfo             const & theField);
        METHOD onPatent            (long theLineNumber, Phoenix::PatentInfo             const & theField);
        METHOD onOnlineJournal     (long theLineNumber, Phoenix::OnlineJournalInfo      const & theField);
        METHOD onElectronicResource(long theLineNumber, Phoenix::ElectronicResourceInfo const & theField);
		METHOD onMiscResource      (long theLineNumber, Phoenix::MiscResourceInfo       const & theField);
        METHOD onJournalArticle    (long theLineNumber, Phoenix::JournalArticleInfo     const & theField);

        METHOD onEndPublication();
    };

}
}

#endif // FF2XML_XML_CONVERTER_INCLUDED
