/**++
 *   
 *   LICENSE
 *   -------
 *   
 *   Copyright (c) 2004 Renato Mancuso
 *   All rights reserved.
 *   
 *   Redistribution and use in source and binary forms, with or without modification, are 
 *   permitted provided that the following conditions are met:
 *   
 *   - Redistributions of source code must retain the above copyright notice, this list 
 *     of conditions and the following disclaimer.
 *   
 *   - Redistributions in binary form must reproduce the above copyright notice, this list
 *     of conditions and the following disclaimer in the documentation and/or other materials 
 *     provided with the distribution.
 *   
 *   - Neither the name of Renato Mancuso nor the names of its contributors may be used to 
 *     endorse or promote products derived from this software without specific prior written 
 *     permission.
 *   
 *   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS AS IS AND ANY EXPRESS 
 *   OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY 
 *   AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR 
 *   CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL 
 *   DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, 
 *   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER 
 *   IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT 
 *   OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *   
--**/


#include "XmlWriter.hpp"

#include <cerrno>
#include <cstring>
#include <cstdio>
#include <sstream>

#include "Exception.hpp"

#include <cassert>

namespace OpenEMBL
{
namespace ff2xml
{

    XmlWriter::XmlWriter()
        : _destinationFile(NULL)
    {
    }

    XmlWriter::XmlWriter(const char* destFilename)
    {
        open(destFilename);
    }

    XmlWriter::~XmlWriter()
    {
        close();
    }

    void XmlWriter::open(const char* destFilename)
    {
        assert(NULL == _destinationFile);
        assert(NULL != destFilename);

        _destinationFile = ::fopen(destFilename, "w");

        if (NULL == _destinationFile)
        {
            raiseSysError("failed to create destination file");
        }
    }

    void XmlWriter::close()
    {
        if (NULL == _destinationFile)
            return;

        FILE* fh         = _destinationFile;
        _destinationFile = NULL;

        if (::fclose(fh) != 0)
        {
            raiseSysError("failed to close destination file");
        }
    }

    void XmlWriter::raiseSysError(char const * msg) const
    {
        int theErrorCode = errno;

        std::ostringstream oss;

        oss << msg 
            << ": "
            << ::strerror(theErrorCode)
            ;

        throw Exception(oss.str());                            
    }

    void XmlWriter::directWrite(char c)
    {
        assert(NULL != _destinationFile);

        if (EOF == ::fputc(c, _destinationFile))
            raiseSysError("failed to write into destination file");                        
    }

    void XmlWriter::directWrite(char const * str)
    {
        assert(NULL != _destinationFile);
        assert(NULL != str);

        if (::fprintf(_destinationFile, "%s", str) < 0)
            raiseSysError("failed to write into destination file");
    }

    void XmlWriter::writeElementText(char const * str)
    {
        assert(NULL != str);
        writeElementText(std::string(str));
    }

    void XmlWriter::writeElementText(std::string const & str)
    {
        static const char LT[]        = "&lt;";
        static const char GT[]        = "&gt;";
        static const char AMPERSAND[] = "&amp;";

        std::string::const_iterator first = str.begin();
        std::string::const_iterator last  = str.end();

        for (; first < last; ++first)
        {
            char c = *first;

            switch (c)
            {
            case '<':
                directWrite(LT);
                break;

            case '>':
                directWrite(GT);
                break;

            case '&':
                directWrite(AMPERSAND);
                break;

            default:
                directWrite(c);
            }
        }
    }

    void XmlWriter::newLine()
    {
        directWrite('\n');
    }

    void XmlWriter::writeAttribute(XmlAttribute const & attr)
    {
        directWrite(attr.getName());
        directWrite('=');
        writeAttributeValue(attr.getValue());
    }

    void XmlWriter::writeAttributeValue(char const * str)
    {
        assert(NULL != str);
        writeAttributeValue(std::string(str));
    }

    void XmlWriter::writeAttributeValue(std::string const & str)
    {
        static const char LT[]        = "&lt;";
        static const char GT[]        = "&gt;";
        static const char AMPERSAND[] = "&amp;";
        static const char QUOTES[]    = "&quot;";

        std::string::const_iterator first = str.begin();
        std::string::const_iterator last  = str.end();

        directWrite('"');

        for (; first < last; ++first)
        {
            char c = *first;

            switch (c)
            {
            case '<':
                directWrite(LT);
                break;

            case '>':
                directWrite(GT);
                break;

            case '&':
                directWrite(AMPERSAND);
                break;

            case '"':
                directWrite(QUOTES);
                break;

            default:
                directWrite(c);
            }
        }

        directWrite('"');
    }

    void XmlWriter::openTag()
    {
        directWrite('<');
    }

    void XmlWriter::closeTag(bool single /* = false */)
    {
        if (single)
            directWrite(" /");

        directWrite('>');
    }

    void XmlWriter::writeElementName(char const * str)
    {
        assert(NULL != str);
        writeElementName(std::string(str));
    }

    void XmlWriter::writeElementName(std::string const & str)
    {
        directWrite(str);
    }

    void XmlWriter::writeIndent(unsigned level)
    {
        for (unsigned i = 0; i < level; ++i)
        {
            directWrite("    ");
        }
    }

    void XmlWriter::writeXMLDeclaration(char const * version, char const * encoding)
    {
        if (NULL == version)
            version = "1.0";

        if (NULL == encoding)
            encoding = "UTF-8";

        XmlAttribute verAttr("version", version);
        XmlAttribute encodingAttr("encoding", encoding);

        directWrite("<?xml");     
        directWrite(' ');
        writeAttribute(verAttr);
        directWrite(' ');
        writeAttribute(encodingAttr);
        directWrite(" ?>");
        newLine();
    }

    void XmlWriter::writeEndTag(char const * tagName)
    {
        assert(NULL != tagName);

        openTag();
        directWrite('/');
        writeElementName(tagName);
        closeTag();
        newLine();
    }

    void XmlWriter::writeBeginTag
        (
        unsigned     indentLevel,
        char const * tagName
        )
    {
        assert(NULL != tagName);

        writeIndent(indentLevel);
        openTag();
        writeElementName(tagName);
        closeTag();
    }

    void XmlWriter::writeBeginTag
        (
        layout_type                 layout,
        unsigned                    indentLevel,
        char const *                tagName, 
        XmlAttributeList const &    attributes,
        bool                        isSingleElement
        )
    {
        if (layout == SINGLELINE)
            writeBeginTagSingleLine(indentLevel, tagName, attributes, isSingleElement);
        else
            writeBeginTagMultiLine(indentLevel, tagName, attributes, isSingleElement);
    }


    void XmlWriter::writeBeginTagSingleLine 
        (
        unsigned                    indentLevel,
        char const *                tagName, 
        XmlAttributeList const &    attributes,
        bool                        isSingleElement
        )
    {
        assert(NULL != tagName);

        writeIndent(indentLevel);
        openTag();
        writeElementName(tagName);

        for (size_t i = 0; i < attributes.size(); ++i)
        {
            directWrite(' ');
            writeAttribute(attributes[i]);
        }

        closeTag(isSingleElement);

        if (isSingleElement)
            newLine();
    }

    void XmlWriter::writeBeginTagMultiLine
        (
        unsigned                    indentLevel,
        char const *                tagName,         
        XmlAttributeList const &    attributes,
        bool                        isSingleElement /* = false */
        )
    {
        assert(NULL != tagName);

        writeIndent(indentLevel);
        openTag();
        writeElementName(tagName);
        newLine();

        for (size_t i = 0; i < attributes.size(); ++i)
        {
            writeIndent(indentLevel + 1);
            writeAttribute(attributes[i]);
            newLine();
        }
        
        writeIndent(indentLevel);
        closeTag(isSingleElement);        

        if (isSingleElement)
            newLine();
    }

    void XmlWriter::writeElement  
        (
        unsigned                    indentLevel,
        char const *                tagName,
        XmlAttributeList            attributes,
        std::string const &         text
        )
    {
        writeBeginTag(SINGLELINE, indentLevel, tagName, attributes);
        writeElementText(text);
        writeEndTag(tagName);
    }

    void XmlWriter::writeElement  
        (
        unsigned                indentLevel,
        char const *            tagName,
        XmlAttributeList        attributes,
        StringList const &      text
        )
    {
        writeBeginTag(SINGLELINE, indentLevel, tagName, attributes);

        for (size_t i = 0; i < text.size(); ++i)
        {
            writeElementText(text[0]);
            break;
        }

        for (size_t i = 1; i < text.size(); ++i)
        {
            newLine();
            writeIndent(indentLevel);
            writeElementText(text[i]);
        }

        writeEndTag(tagName);
    }

    void XmlWriter::writeSingleTag(
        layout_type                 layout,
        unsigned                    indentLevel,
        char const *                tagName,
        XmlAttributeList const &    attributes
        )
    {
        writeBeginTag(layout, indentLevel, tagName, attributes, true);
    }



}
}
